<?php

namespace Vimeify\Core\Traits;

trait ScriptQueue {

	/**
	 * Enqueue specific style
	 *
	 * @param string $handle
	 * @param string $virtual_path
	 * @param array|null $dependencies
	 * @param string|null $version
	 * @param string $media
	 *
	 * @return array
	 */
	public function prepare_style( string $handle, string $virtual_path, ?array $dependencies = [], ?string $version = 'auto', string $media = 'all' ): array {

		if ( ! isset( $this->plugin ) ) {
			return [];
		}
		$plugin_path  = rtrim( $this->plugin->path(), '/' ) . DIRECTORY_SEPARATOR;
		$plugin_url   = rtrim( $this->plugin->url(), '/' ) . '/';
		$virtual_path = ltrim( $virtual_path, '/' );
		$url          = $plugin_url . $virtual_path;
		$path         = $plugin_path . $virtual_path;

		// Handle version parameter
		if ( $version === 'auto' ) {
			$version = file_exists( $path ) ? filemtime( $path ) : null;
		}

		// Development mode - use non-minified files if available
		if ( strpos($virtual_path, '/resources/') === false && defined( 'VIMEIFY_DEVELOPMENT' ) && VIMEIFY_DEVELOPMENT ) {
			$url  = str_replace( '.min.css', '.css', $url );
			$path = str_replace( '.min.css', '.css', $path );
		}

		return [
			'handle' => $handle,
			'src'    => $url,
			'path'   => $path,
			'ver'    => $version,
			'deps'   => $dependencies,
			'media'  => $media,
		];
	}

	/**
	 * Enqueue the script
	 *
	 * @param string $handle
	 * @param string $virtual_path
	 * @param array|null $dependencies
	 * @param string|null $version
	 * @param array|bool $args
	 *
	 * @return array
	 */
	public function prepare_script( string $handle, string $virtual_path, ?array $dependencies = [], ?string $version = 'auto', $args = [] ): array {

		if ( ! isset( $this->plugin ) ) {
			return [];
		}
		$plugin_path  = rtrim( $this->plugin->path(), '/' ) . DIRECTORY_SEPARATOR;
		$plugin_url   = rtrim( $this->plugin->url(), '/' ) . '/';
		$virtual_path = ltrim( $virtual_path, '/' );
		$url          = $plugin_url . $virtual_path;
		$path         = $plugin_path . $virtual_path;

		// Handle version parameter
		if ( $version === 'auto' ) {
			$version = file_exists( $path ) ? filemtime( $path ) : null;
		}

		// Development mode - use non-minified files if available
		if ( strpos($virtual_path, '/resources/') === false && defined( 'VIMEIFY_DEVELOPMENT' ) && VIMEIFY_DEVELOPMENT ) {
			$url  = str_replace( '.min.js', '.js', $url );
			$path = str_replace( '.min.js', '.js', $path );
		}

		return [
			'handle' => $handle,
			'src'    => $url,
			'path'   => $path,
			'ver'    => $version,
			'deps'   => $dependencies,
			'args'   => $args,
		];
	}

	/**
	 * Register style
	 *
	 * @param string $handle
	 * @param string $virtual_path
	 * @param array|null $dependencies
	 * @param string|null $version
	 * @param string $media
	 *
	 * @return void
	 */
	public function register_style( string $handle, string $virtual_path, ?array $dependencies = [], ?string $version = 'auto', string $media = 'all' ): void {

		$params = $this->prepare_style( $handle, $virtual_path, $dependencies, $version, $media );

		if ( empty( $params ) ) {
			return;
		}

		wp_register_style(
			$params['handle'],
			$params['src'],
			$params['deps'],
			$params['ver'],
			$params['media']
		);

	}

	/**
	 * Register script
	 *
	 * @param string $handle
	 * @param string $virtual_path
	 * @param array|null $dependencies
	 * @param string|null $version
	 * @param array|bool $args
	 *
	 * @return void
	 */
	public function register_script( string $handle, string $virtual_path, ?array $dependencies = [], ?string $version = 'auto', $args = [] ): void {

		$params = $this->prepare_script( $handle, $virtual_path, $dependencies, $version, $args );

		if ( empty( $params ) ) {
			return;
		}

		wp_register_script(
			$params['handle'],
			$params['src'],
			$params['deps'],
			$params['ver'],
			$params['args']
		);
	}

	/**
	 * Enqueue specific style
	 *
	 * @param string $handle
	 * @param string $virtual_path
	 * @param array|null $dependencies
	 * @param string|null $version
	 * @param string $media
	 *
	 * @return void
	 */
	public function enqueue_style( string $handle, string $virtual_path = '', ?array $dependencies = [], ?string $version = 'auto', string $media = 'all' ): void {

		// If no virtual_path provided, assume script is already registered
		if ( empty( $virtual_path ) ) {
			wp_enqueue_style( $handle );
			return;
		}

		$params = $this->prepare_style( $handle, $virtual_path, $dependencies, $version, $media );

		if ( empty( $params ) ) {
			return;
		}

		wp_enqueue_style(
			$params['handle'],
			$params['src'],
			$params['deps'],
			$params['ver'],
			$params['media']
		);
	}

	/**
	 * Enqueue specific script
	 *
	 * @param string $handle
	 * @param string $virtual_path
	 * @param array|null $dependencies
	 * @param string|null $version
	 * @param array|bool $args
	 *
	 * @return void
	 */
	public function enqueue_script( string $handle, string $virtual_path = '', ?array $dependencies = [], ?string $version = 'auto', $args = [] ): void {

		// If no virtual_path provided, assume script is already registered
		if ( empty( $virtual_path ) ) {
			wp_enqueue_script( $handle );
			return;
		}

		$params = $this->prepare_script( $handle, $virtual_path, $dependencies, $version, $args );

		if ( empty( $params ) ) {
			return;
		}

		wp_enqueue_script(
			$params['handle'],
			$params['src'],
			$params['deps'],
			$params['ver'],
			$params['args']
		);
	}

}