<?php
/********************************************************************
 * Copyright (C) 2024 Darko Gjorgjijoski (https://darkog.com/)
 * Copyright (C) 2024 IDEOLOGIX MEDIA Dooel (https://ideologix.com/)
 *
 * This file is property of IDEOLOGIX MEDIA Dooel (https://ideologix.com)
 * This file is part of Vimeify Plugin - https://wordpress.org/plugins/vimeify/
 *
 * Vimeify - Formerly "WP Vimeo Videos" is free software: you can redistribute
 * it and/or modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the License,
 * or (at your option) any later version.
 *
 * Vimeify - Formerly "WP Vimeo Videos" is distributed in the hope that it
 * will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this plugin. If not, see <https://www.gnu.org/licenses/>.
 *
 * Code developed by Darko Gjorgjijoski <dg@darkog.com>.
 **********************************************************************/

namespace Vimeify\Core\Frontend;

use Vimeify\Core\Abstracts\BaseProvider;
use Vimeify\Core\Backend\Galleries;
use Vimeify\Core\Components\Database;
use Vimeify\Core\Frontend\Views\GalleryView;

class GalleryPreview extends BaseProvider {

	/**
	 * Registers specific piece of functionality
	 * @return void
	 */
	public function register() {
		add_action( 'template_redirect', [ $this, 'handle_gallery_preview' ] );
	}

	/**
	 * Handle gallery preview request
	 * @return void
	 */
	public function handle_gallery_preview() {
		// Check if preview parameter exists
		if ( ! isset( $_GET['vimeify-gallery-preview'] ) ) {
			return;
		}

		$gallery_id = intval( $_GET['vimeify-gallery-preview'] );

		if ( empty( $gallery_id ) ) {
			wp_die( __( 'Invalid gallery ID.', 'vimeify' ) );
		}

		// Verify user has permission to preview
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_die( __( 'You do not have permission to preview galleries.', 'vimeify' ) );
		}

		// Verify gallery exists
		$gallery = get_term( $gallery_id, Database::TAXONOMY_GALLERY );
		if ( ! $gallery || is_wp_error( $gallery ) ) {
			wp_die( __( 'Gallery not found.', 'vimeify' ) );
		}

		// Render preview page
		$this->render_preview_page( $gallery_id, $gallery );
		exit;
	}

	/**
	 * Render the preview page
	 *
	 * @param int $gallery_id
	 * @param object $gallery
	 *
	 * @return void
	 */
	private function render_preview_page( $gallery_id, $gallery ) {
		$gallery_type = Galleries::get_gallery_type( $gallery_id );
		$gallery_settings = Galleries::get_gallery_settings( $gallery_id );
		$display_style = isset( $gallery_settings['display_style'] ) ? $gallery_settings['display_style'] : 'slider';

		// Get site title
		$site_title = get_bloginfo( 'name' );

		// Enqueue gallery assets - do this AFTER scripts are registered
		add_action( 'wp_enqueue_scripts', function() {
			// Enqueue gallery assets (they're already registered by Frontend/Scripts)
            wp_enqueue_script('jquery');
			wp_enqueue_style( 'vimeify-gallery' );
			wp_enqueue_script( 'vimeify-gallery' );
		}, 999 ); // Very late priority to run after registration

		?>
		<!DOCTYPE html>
		<html <?php language_attributes(); ?>>
		<head>
			<meta charset="<?php bloginfo( 'charset' ); ?>">
			<meta name="viewport" content="width=device-width, initial-scale=1.0">
			<meta name="robots" content="noindex, nofollow">
			<title><?php echo esc_html( sprintf( __( 'Gallery Preview: %s - %s', 'vimeify' ), $gallery->name, $site_title ) ); ?></title>
			<?php wp_head(); ?>
			<style>
				body {
					margin: 0;
					padding: 0;
					font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
					background: #f5f5f5;
				}

				.vimeify-preview-header {
					background: #fff;
					border-bottom: 1px solid #e0e0e0;
					padding: 15px 20px;
					box-shadow: 0 2px 4px rgba(0,0,0,0.05);
					position: sticky;
					top: 0;
					z-index: 1000;
				}

				.vimeify-preview-header-content {
					max-width: 1200px;
					margin: 0 auto;
					display: flex;
					justify-content: space-between;
					align-items: center;
				}

				.vimeify-preview-title {
					margin: 0;
					font-size: 18px;
					font-weight: 600;
					color: #333;
				}

				.vimeify-preview-meta {
					font-size: 14px;
					color: #666;
				}

				.vimeify-preview-badge {
					display: inline-block;
					padding: 4px 12px;
					background: #6366F1;
					color: #fff;
					border-radius: 4px;
					font-size: 12px;
					font-weight: 500;
					margin-left: 10px;
				}

				.vimeify-preview-actions {
					display: flex;
					gap: 10px;
				}

				.vimeify-preview-btn {
					padding: 8px 16px;
					background: #fff;
					border: 1px solid #ddd;
					border-radius: 4px;
					text-decoration: none;
					color: #333;
					font-size: 14px;
					transition: all 0.3s ease;
				}

				.vimeify-preview-btn:hover {
					border-color: #6366F1;
					color: #6366F1;
				}

				.vimeify-preview-btn-primary {
					background: #6366F1;
					border-color: #6366F1;
					color: #fff;
				}

				.vimeify-preview-btn-primary:hover {
					background: #8B5CF6;
					border-color: #8B5CF6;
					color: #fff;
				}

				.vimeify-preview-container {
					max-width: 1200px;
					margin: 30px auto;
					padding: 0 20px;
				}

				.vimeify-preview-content {
					background: #fff;
					padding: 30px;
					border-radius: 8px;
					box-shadow: 0 2px 8px rgba(0,0,0,0.08);
				}

				@media (max-width: 768px) {
					.vimeify-preview-header-content {
						flex-direction: column;
						gap: 15px;
						align-items: flex-start;
					}

					.vimeify-preview-actions {
						width: 100%;
					}

					.vimeify-preview-btn {
						flex: 1;
						text-align: center;
					}

					.vimeify-preview-content {
						padding: 20px;
					}
				}
			</style>
		</head>
		<body class="vimeify-gallery-preview-page">

		<div class="vimeify-preview-header">
			<div class="vimeify-preview-header-content">
				<div>
					<h1 class="vimeify-preview-title">
						<?php echo esc_html( $gallery->name ); ?>
						<span class="vimeify-preview-badge"><?php esc_html_e( 'Preview', 'vimeify' ); ?></span>
					</h1>
					<div class="vimeify-preview-meta">
						<?php
						$type_label = '';
						$gallery_types = ( new Galleries( $this->plugin ) )->get_gallery_types();
						if ( isset( $gallery_types[ $gallery_type ] ) ) {
							$type_label = $gallery_types[ $gallery_type ]['label'];
						}
						printf(
							esc_html__( 'Type: %s | Style: %s', 'vimeify' ),
							esc_html( $type_label ),
							esc_html( ucfirst( $display_style ) )
						);
						?>
					</div>
					<div class="vimeify-preview-shortcode" style="margin-top: 10px; padding: 8px 12px; background: #f0f0f0; border-radius: 4px; font-family: monospace; font-size: 13px; color: #333;">
						<strong><?php esc_html_e( 'Shortcode:', 'vimeify' ); ?></strong>
						<code style="background: #fff; padding: 4px 8px; border-radius: 3px; margin-left: 5px; user-select: all;">[vimeify_gallery id="<?php echo esc_attr( $gallery_id ); ?>"]</code>
					</div>
				</div>
				<div class="vimeify-preview-actions">
					<a href="<?php echo esc_url( admin_url( 'admin.php?page=vimeify-galleries&action=edit&gallery_id=' . $gallery_id ) ); ?>" class="vimeify-preview-btn">
						<?php esc_html_e( 'Edit Gallery', 'vimeify' ); ?>
					</a>
					<a href="<?php echo esc_url( admin_url( 'admin.php?page=vimeify-galleries' ) ); ?>" class="vimeify-preview-btn-primary vimeify-preview-btn">
						<?php esc_html_e( 'Back to Galleries', 'vimeify' ); ?>
					</a>
				</div>
			</div>
		</div>

		<div class="vimeify-preview-container">
			<div class="vimeify-preview-content">
				<?php
				// Render gallery after wp_head has been called
				$view = new GalleryView( $this->plugin );
				echo $view->output( [ 'id' => $gallery_id ] );
				?>
			</div>
		</div>

		<?php wp_footer(); ?>
		</body>
		</html>
		<?php
	}
}
