<?php
/********************************************************************
 * Copyright (C) 2024 Darko Gjorgjijoski (https://darkog.com/)
 * Copyright (C) 2024 IDEOLOGIX MEDIA Dooel (https://ideologix.com/)
 *
 * This file is property of IDEOLOGIX MEDIA Dooel (https://ideologix.com)
 * This file is part of Vimeify Plugin - https://wordpress.org/plugins/vimeify/
 *
 * Vimeify - Formerly "WP Vimeo Videos" is free software: you can redistribute
 * it and/or modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the License,
 * or (at your option) any later version.
 *
 * Vimeify - Formerly "WP Vimeo Videos" is distributed in the hope that it
 * will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this plugin. If not, see <https://www.gnu.org/licenses/>.
 *
 * Code developed by Darko Gjorgjijoski <dg@darkog.com>.
 **********************************************************************/

namespace Vimeify\Core\Backend\Tools\Migrators;

use Vimeify\Core\Abstracts\BaseMigrator;
use Vimeify\Core\Backend\Tools\Migrators\Steps\FormsMigration;
use Vimeify\Core\Backend\Tools\Migrators\Steps\SettingsMigration;
use Vimeify\Core\Backend\Tools\Migrators\Steps\VideosMigration;
use WP_Error;

class LegacyMigrator extends BaseMigrator {

	/**
	 * The migrator ID
	 * @var string
	 */
	protected $id = 'legacy_migrator';

	/**
	 * The migrator name
	 * @var string
	 */
	protected $name = 'Video Uploads for Vimeo';

	/**
	 * Returns the migrator steps
	 * @return array|\WP_Error
	 */
	public function get_steps() {

		return [
			1 => array(
				'name'  => 'Settings',
				'pages' => 1
			),
			2 => array(
				'name'  => 'Videos',
				'pages' => ( new VideosMigration() )->get_page_count(),
			),
			3 => array(
				'name'  => 'Forms',
				'pages' => 2
			),
			4 => array(
				'name'  => 'Clean up',
				'pages' => 1
			)
		];
	}

	/**
	 * Check if it is possible to facilitate migration
	 * @return bool|\WP_Error
	 */
	public function check_availability() {
		$settings = get_option( 'dgv_settings', array() );
		if ( empty( $settings ) ) {
			return new WP_Error( 'no_legacy_settings', esc_html__( 'The legacy plugin settings not found. The migration was probably ran before', 'vimeify' ) );
		}

		return true;
	}

	/**
	 * Initializes the process
	 *
	 * @param array $data
	 *
	 * @return bool|\WP_Error
	 */
	public function init( $data = array() ) {
		return true;
	}

	/**
	 * Initializes the process
	 *
	 * @param $step
	 * @param $page
	 * @param array $data
	 *
	 * @return bool|\WP_Error
	 */
	public function do_step( $step, $page, $data = array() ) {

		$this->system->logger()->log( sprintf("Legacy Migrator: Step: %s, Page: %s, Data: %s", $step, $page, json_encode($data)), 'migration' );

		switch ( $step ) {
			case 1:
				$settings_migration = new SettingsMigration();
				$settings_migration->migrate();
				break;
			case 2:
				$videos_migration = new VideosMigration();
				$videos_migration->migrate_page( $page );
				break;
			case 3:
				$forms_migration = new FormsMigration( $this->system );
				$migration_needed = $forms_migration->is_migration_needed();

				$this->system->logger()->log( sprintf("Legacy Migrator: Forms migration needed: %s", json_encode($migration_needed)), 'migration' );

				if ( $page === 1 ) {
					if ( isset( $migration_needed['gravityforms'] ) && $migration_needed['gravityforms'] ) {
						$forms_migration->migrate_gravityforms();
					};
				} else {
					if ( isset( $migration_needed['wpforms'] ) && $migration_needed['wpforms'] ) {
						$forms_migration->migrate_wpforms();
					}
				}
				break;
			case 4:
				global $wpdb;
				$wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE 'dgv_%'");
				break;
		}

		return true;
	}

	/**
	 * Undoes the migration process
	 *
	 * @return bool
	 */
	public function undo() {
		return false;
	}
}