<?php
/**
 * This file comes from the "Digital License Manager" WordPress plugin.
 * https://darkog.com/p/vimeify/
 *
 * Copyright (C) 2020-present  Darko Gjorgjijoski. All Rights Reserved.
 * Copyright (C) 2020-present  IDEOLOGIX MEDIA DOOEL. All Rights Reserved.
 *
 * Digital License Manager is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * Digital License Manager program is distributed in the hope that it
 * will be useful,but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License v3
 * along with this program;
 *
 * If not, see: https://www.gnu.org/licenses/gpl-3.0.en.html
 *
 * Code written, maintained by Darko Gjorgjijoski (https://darkog.com)
 */

namespace Vimeify\Core\Backend\Tools;

use Vimeify\Core\Abstracts\BaseMigrator;
use Vimeify\Core\Abstracts\BaseTool;
use Vimeify\Core\Abstracts\Interfaces\SystemInterface;
use Vimeify\Core\Backend\Tools\Migrators\LegacyMigrator;
use Vimeify\Core\System;

/**
 * Migration tool
 */
class Migration extends BaseTool {

	/**
	 * The plugin classes
	 * @var array
	 */
	protected $plugins;

	/**
	 * Constructor
	 *
	 * @param SystemInterface|System $plugin
	 * @param string $id
	 */
	public function __construct( $id, $system ) {

		parent::__construct( $id, $system );

		$this->slug        = 'migration';
		$this->description = __( 'Migration tool that makes it possible to easily move from other plugins', 'vimeify' );

		$this->plugins = [
			new LegacyMigrator( $system ),
		];
	}

	/**
	 * Returns the view
	 * @return string
	 */
	public function get_view() {
		return $this->system->views()->get_view( 'admin/partials/tools/migration', [
			'tool'    => $this,
			'plugins' => $this->plugins,
		] );
	}


	/**
	 * Returns the migrator steps
	 * @return array|\WP_Error
	 */
	public function get_steps() {
		$plugin = $this->get_plugin();
		if ( is_wp_error( $plugin ) ) {
			return $plugin;
		}

		return $plugin->get_steps();
	}

	/**
	 * Initializes the process
	 * @return bool|\WP_Error
	 */
	public function init_process() {

		$canInit = $this->check_availability();
		if ( is_wp_error( $canInit ) ) {
			return $canInit;
		}
		$plugin = $this->get_plugin();
		if ( is_wp_error( $plugin ) ) {
			return $plugin;
		}

		return $plugin->init();
	}


	/**
	 * Returns a plugin
	 *
	 * @param $identifier
	 *
	 * @return LegacyMigrator|BaseMigrator|\WP_Error
	 */
	public function get_plugin() {

		$identifier = $this->get_identifier();

		foreach ( $this->plugins as $plugin ) {
			if ( $plugin->get_id() == $identifier ) {
				return $plugin;
			}
		}

		return new \WP_Error( '404', 'Plugin migrator not found.' );
	}

	/**
	 * Initializes the process
	 *
	 * @param $step
	 * @param $page
	 *
	 * @return bool|\WP_Error
	 */
	public function do_step( $step, $page ) {

		$identifier = $this->get_identifier();

		$plugin = $this->get_plugin( $identifier );
		if ( is_wp_error( $plugin ) ) {
			return $plugin;
		}

		return $plugin->do_step( $step, $page );
	}

	/**
	 * Check availability
	 *
	 * @return bool|\WP_Error
	 */
	public function check_availability() {

		$identifier = $this->get_identifier();

		$plugin = $this->get_plugin( $identifier );
		if ( is_wp_error( $plugin ) ) {
			return $plugin;
		}

		return $plugin->check_availability();

	}

	/**
	 * Return the identifier
	 * @return string
	 */
	public function get_identifier() {
		return isset( $_REQUEST['identifier'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['identifier'] ) ) : '';
	}

	/**
	 * Set the tool status
	 * @return void
	 */
	public function set_status( $args ) {
		$data = wp_parse_args( $args, [
			'completed_at' => null,
		] );
		update_option( $this->get_status_key(), $data );
	}

	/**
	 * Get the tool status
	 * @return mixed
	 */
	public function get_status() {
		return get_option( $this->get_status_key() );
	}

	/**
	 * Reset the tool status
	 * @return void
	 */
	public function reset_status() {
		delete_option( $this->get_status_key() );
	}

	/**
	 * Marks as complete
	 *
	 * @param $args
	 *
	 * @return void
	 */
	public function mark_as_complete() {
		parent::mark_as_complete();
		$this->set_status( [
			'completed_at' => gmdate( 'Y-m-d H:i:s' )
		] );
	}

	/**
	 * The status key
	 * @return string
	 */
	private function get_status_key() {
		return sprintf( 'vimeify_migration_%s', $this->get_identifier() );
	}
}