<?php
/*
Plugin Name: VertiMonial
Description: A WordPress plugin to display a scrolling VertiMonial slider with backend management and documentation.
Version: 1.2
Author: Haseeb Abid
Author URI: https://profiles.wordpress.org/haseebabidofficial/
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: vertimonial
*/

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

define('VERTIMONIAL_VERSION', '1.2');

// Register Custom Post Type for VertiMonials
function vertim_register_testimonial_cpt() {
    $labels = array(
        'name' => 'VertiMonials',
        'singular_name' => 'VertiMonial',
        'menu_name' => 'VertiMonials',
        'add_new' => 'Add New VertiMonial',
        'add_new_item' => 'Add New VertiMonial',
        'edit_item' => 'Edit VertiMonial',
        'new_item' => 'New VertiMonial',
        'view_item' => 'View VertiMonial',
        'search_items' => 'Search VertiMonials',
        'not_found' => 'No VertiMonials found',
        'not_found_in_trash' => 'No VertiMonials found in Trash',
    );

    $args = array(
        'labels' => $labels,
        'public' => false,
        'show_ui' => true,
        'supports' => array('title', 'editor', 'thumbnail'),
        'menu_icon' => 'dashicons-format-quote',
        'show_in_menu' => true,
    );

    register_post_type('vertim-testimonial', $args);
}
add_action('init', 'vertim_register_testimonial_cpt');

// Register Taxonomy for Testimonial Categories
function vertim_register_testimonial_taxonomy() {
    $labels = array(
        'name' => 'Categories',
        'singular_name' => 'Category',
        'menu_name' => 'Categories',
        'all_items' => 'All Categories',
        'edit_item' => 'Edit Category',
        'view_item' => 'View Category',
        'update_item' => 'Update Category',
        'add_new_item' => 'Add New Category',
        'new_item_name' => 'New Category Name',
        'parent_item' => 'Parent Category',
        'parent_item_colon' => 'Parent Category:',
        'search_items' => 'Search Categories',
    );

    $args = array(
        'labels' => $labels,
        'hierarchical' => true,
        'show_ui' => true,
        'show_in_rest' => true,
        'query_var' => true,
        'rewrite' => array('slug' => 'vertim-category'),
    );

    register_taxonomy('vertim-category', 'vertim-testimonial', $args);
}
add_action('init', 'vertim_register_testimonial_taxonomy');

// Add Meta Fields for Role, Company, and Rating
function vertim_add_testimonial_meta_box() {
    add_meta_box(
        'vertim_testimonial_meta',
        'VertiMonial Details',
        'vertim_testimonial_meta_callback',
        'vertim-testimonial',
        'normal',
        'high'
    );
}
add_action('add_meta_boxes', 'vertim_add_testimonial_meta_box');

function vertim_testimonial_meta_callback($post) {
    wp_nonce_field('vertim_save_testimonial_meta', 'vertim_testimonial_nonce');
    $role = get_post_meta($post->ID, '_vertim_testimonial_role', true);
    $company = get_post_meta($post->ID, '_vertim_testimonial_company', true);
    $rating = get_post_meta($post->ID, '_vertim_testimonial_rating', true);
    ?>
    <p>
        <label for="vertim_testimonial_role">Role/Designation:</label>
        <input type="text" id="vertim_testimonial_role" name="vertim_testimonial_role" value="<?php echo esc_attr($role); ?>" style="width: 100%;">
    </p>
    <p>
        <label for="vertim_testimonial_company">Company Name:</label>
        <input type="text" id="vertim_testimonial_company" name="vertim_testimonial_company" value="<?php echo esc_attr($company); ?>" style="width: 100%;">
    </p>
    <p>
        <label for="vertim_testimonial_rating">Rating (1-5):</label>
        <select id="vertim_testimonial_rating" name="vertim_testimonial_rating">
            <option value="1" <?php selected($rating, 1); ?>>1</option>
            <option value="2" <?php selected($rating, 2); ?>>2</option>
            <option value="3" <?php selected($rating, 3); ?>>3</option>
            <option value="4" <?php selected($rating, 4); ?>>4</option>
            <option value="5" <?php selected($rating, 5); ?>>5</option>
        </select>
    </p>
    <?php
}

function vertim_save_testimonial_meta($post_id) {
    // Check if nonce is set
    if (!isset($_POST['vertim_testimonial_nonce'])) {
        return;
    }

    // Verify the nonce
    $nonce = sanitize_text_field(wp_unslash($_POST['vertim_testimonial_nonce']));
    if (!wp_verify_nonce($nonce, 'vertim_save_testimonial_meta')) {
        return;
    }

    // Check user permissions
    if (!current_user_can('edit_post', $post_id)) {
        return;
    }

    // Make sure it's not an autosave
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }

    // Save the role field if it exists
    if (isset($_POST['vertim_testimonial_role'])) {
        $role = sanitize_text_field(wp_unslash($_POST['vertim_testimonial_role']));
        update_post_meta($post_id, '_vertim_testimonial_role', $role);
    }

    // Save the company field if it exists
    if (isset($_POST['vertim_testimonial_company'])) {
        $company = sanitize_text_field(wp_unslash($_POST['vertim_testimonial_company']));
        update_post_meta($post_id, '_vertim_testimonial_company', $company);
    }

    // Save the rating field if it exists
    if (isset($_POST['vertim_testimonial_rating'])) {
        $rating = intval($_POST['vertim_testimonial_rating']);
        if ($rating >= 1 && $rating <= 5) {
            update_post_meta($post_id, '_vertim_testimonial_rating', $rating);
        }
    }
}
add_action('save_post', 'vertim_save_testimonial_meta');

// Add Admin Menu Pages
function vertim_add_admin_menu() {
    add_submenu_page(
        'edit.php?post_type=vertim-testimonial',
        'VertiMonial Settings',
        'Settings',
        'manage_options',
        'vertim-settings',
        'vertim_settings_page_callback'
    );

    add_submenu_page(
        'edit.php?post_type=vertim-testimonial',
        'VertiMonial Documentation',
        'Documentation',
        'manage_options',
        'vertim-documentation',
        'vertim_documentation_page_callback'
    );
}
add_action('admin_menu', 'vertim_add_admin_menu');

function vertim_settings_page_callback() {
    ?>
    <div class="wrap">
        <h1>VertiMonial Settings</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('vertim_settings_group');
            do_settings_sections('vertim-settings');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

function vertim_documentation_page_callback() {
    ?>
    <div class="wrap">
        <?php
        $doc_file = plugin_dir_path(__FILE__) . 'documentation.html';
        if (file_exists($doc_file) && is_readable($doc_file)) {
            echo file_get_contents($doc_file);
        } else {
            echo '<div class="error"><p>Documentation file is missing or not readable. Please ensure <code>documentation.html</code> exists in the plugin directory and has proper permissions.</p></div>';
        }
        ?>
    </div>
    <?php
}

function vertim_register_settings() {
    register_setting('vertim_settings_group', 'vertim_animation_speed', [
    'sanitize_callback' => 'absint'
    ]);

    register_setting('vertim_settings_group', 'vertim_background_color', [
        'sanitize_callback' => 'sanitize_hex_color'
    ]);

    register_setting('vertim_settings_group', 'vertim_text_color', [
        'sanitize_callback' => 'sanitize_hex_color'
    ]);

    register_setting('vertim_settings_group', 'vertim_column_width', [
        'sanitize_callback' => 'absint'
    ]);

    register_setting('vertim_settings_group', 'vertim_column_height', [
        'sanitize_callback' => 'absint'
    ]);

    register_setting('vertim_settings_group', 'vertim_gap_size', [
        'sanitize_callback' => 'absint'
    ]);

    register_setting('vertim_settings_group', 'vertim_padding_size', [
        'sanitize_callback' => 'absint'
    ]);

    register_setting('vertim_settings_group', 'vertim_border_radius', [
        'sanitize_callback' => 'absint'
    ]);

    register_setting('vertim_settings_group', 'vertim_show_rating', [
        'sanitize_callback' => 'rest_sanitize_boolean'
    ]);

    register_setting('vertim_settings_group', 'vertim_pause_on_hover', [
        'sanitize_callback' => 'rest_sanitize_boolean'
    ]);

    register_setting('vertim_settings_group', 'vertim_single_column', [
        'sanitize_callback' => 'rest_sanitize_boolean'
    ]);


    add_settings_section(
        'vertim_general_section',
        'General Settings',
        null,
        'vertim-settings'
    );

    add_settings_field(
        'vertim_animation_speed',
        'Animation Speed (seconds)',
        'vertim_animation_speed_callback',
        'vertim-settings',
        'vertim_general_section'
    );

    add_settings_field(
        'vertim_background_color',
        'Testimonial Background Color',
        'vertim_background_color_callback',
        'vertim-settings',
        'vertim_general_section'
    );

    add_settings_field(
        'vertim_text_color',
        'Testimonial Text Color',
        'vertim_text_color_callback',
        'vertim-settings',
        'vertim_general_section'
    );

    add_settings_field(
        'vertim_column_width',
        'Column Width (px)',
        'vertim_column_width_callback',
        'vertim-settings',
        'vertim_general_section'
    );

    add_settings_field(
        'vertim_column_height',
        'Column Height (px)',
        'vertim_column_height_callback',
        'vertim-settings',
        'vertim_general_section'
    );

    add_settings_field(
        'vertim_gap_size',
        'Gap Between Columns (px)',
        'vertim_gap_size_callback',
        'vertim-settings',
        'vertim_general_section'
    );

    add_settings_field(
        'vertim_padding_size',
        'Testimonial Padding (px)',
        'vertim_padding_size_callback',
        'vertim-settings',
        'vertim_general_section'
    );

    add_settings_field(
        'vertim_border_radius',
        'Testimonial Border Radius (px)',
        'vertim_border_radius_callback',
        'vertim-settings',
        'vertim_general_section'
    );

    add_settings_field(
        'vertim_show_rating',
        'Show Ratings',
        'vertim_show_rating_callback',
        'vertim-settings',
        'vertim_general_section'
    );

    add_settings_field(
        'vertim_pause_on_hover',
        'Pause on Hover',
        'vertim_pause_on_hover_callback',
        'vertim-settings',
        'vertim_general_section'
    );

    add_settings_field(
        'vertim_single_column',
        'Use Single Column',
        'vertim_single_column_callback',
        'vertim-settings',
        'vertim_general_section'
    );
}
add_action('admin_init', 'vertim_register_settings');

function vertim_animation_speed_callback() {
    $speed = get_option('vertim_animation_speed', 25);
    echo '<input type="number" name="vertim_animation_speed" value="' . esc_attr($speed) . '" min="1" max="60" />';
}

function vertim_background_color_callback() {
    $color = get_option('vertim_background_color', '#ffffff');
    echo '<input type="text" name="vertim_background_color" value="' . esc_attr($color) . '" class="vertim-color-picker" />';
}

function vertim_text_color_callback() {
    $color = get_option('vertim_text_color', '#000000');
    echo '<input type="text" name="vertim_text_color" value="' . esc_attr($color) . '" class="vertim-color-picker" />';
}

function vertim_column_width_callback() {
    $width = get_option('vertim_column_width', 700);
    echo '<input type="number" name="vertim_column_width" value="' . esc_attr($width) . '" min="100" />';
}

function vertim_column_height_callback() {
    $height = get_option('vertim_column_height', 500);
    echo '<input type="number" name="vertim_column_height" value="' . esc_attr($height) . '" min="100" />';
}

function vertim_gap_size_callback() {
    $gap = get_option('vertim_gap_size', 8);
    echo '<input type="number" name="vertim_gap_size" value="' . esc_attr($gap) . '" min="0" />';
}

function vertim_padding_size_callback() {
    $padding = get_option('vertim_padding_size', 20);
    echo '<input type="number" name="vertim_padding_size" value="' . esc_attr($padding) . '" min="0" />';
}

function vertim_border_radius_callback() {
    $radius = get_option('vertim_border_radius', 15);
    echo '<input type="number" name="vertim_border_radius" value="' . esc_attr($radius) . '" min="0" />';
}

function vertim_show_rating_callback() {
    $show = get_option('vertim_show_rating', 1);
    echo '<input type="checkbox" name="vertim_show_rating" value="1" ' . checked(1, $show, false) . ' />';
}

function vertim_pause_on_hover_callback() {
    $pause = get_option('vertim_pause_on_hover', 1);
    echo '<input type="checkbox" name="vertim_pause_on_hover" value="1" ' . checked(1, $pause, false) . ' />';
}

function vertim_single_column_callback() {
    $single = get_option('vertim_single_column', 0);
    echo '<input type="checkbox" name="vertim_single_column" value="1" ' . checked(1, $single, false) . ' />';
}

// Enqueue Admin Assets
function vertim_enqueue_admin_assets($hook) {
    if ($hook !== 'vertim-testimonial_page_vertim-settings' && $hook !== 'vertim-testimonial_page_vertim-documentation') {
        return;
    }
    wp_enqueue_style('wp-color-picker');
    wp_enqueue_script('wp-color-picker');
    wp_add_inline_script('wp-color-picker', 'jQuery(document).ready(function($){ $(".vertim-color-picker").wpColorPicker(); });');

    // Enqueue custom styles for documentation page
    if ($hook === 'vertim-testimonial_page_vertim-documentation') {
        wp_enqueue_style('vertim-docs-css', plugin_dir_url(__FILE__) . 'assets/css/documentation.css', [], VERTIMONIAL_VERSION);
    }
}
add_action('admin_enqueue_scripts', 'vertim_enqueue_admin_assets');

// Enqueue Front-End Assets
function vertim_enqueue_assets() {
    if (!is_admin()) {
        wp_enqueue_style('vertim-slider-css', plugin_dir_url(__FILE__) . 'assets/css/slider.css', [], VERTIMONIAL_VERSION);
        wp_enqueue_script('vertim-slider-js', plugin_dir_url(__FILE__) . 'assets/js/slider.js', ['jquery'], VERTIMONIAL_VERSION, true);

        // Inline dynamic CSS based on settings
        $animation_speed = get_option('vertim_animation_speed', 25);
        $background_color = get_option('vertim_background_color', '#ffffff');
        $text_color = get_option('vertim_text_color', '#000000');
        $column_width = get_option('vertim_column_width', 700);
        $column_height = get_option('vertim_column_height', 500);
        $gap_size = get_option('vertim_gap_size', 8);
        $padding_size = get_option('vertim_padding_size', 20);
        $border_radius = get_option('vertim_border_radius', 15);

        $custom_css = "
            .columns-wrapper {
                gap: {$gap_size}px;
            }
            .column {
                width: {$column_width}px;
                height: {$column_height}px;
            }
            .scroll-inner, .scroll-inner.reverse {
                animation-duration: {$animation_speed}s;
            }
            .testimonial {
                background: {$background_color};
                color: {$text_color};
                padding: {$padding_size}px;
                border-radius: {$border_radius}px;
            }
        ";
        wp_add_inline_style('vertim-slider-css', $custom_css);

        // Localize script for pause on hover
        $pause_on_hover = get_option('vertim_pause_on_hover', 1);
        wp_localize_script('vertim-slider-js', 'vertim_settings', array(
            'pause_on_hover' => $pause_on_hover,
        ));
    }
}
add_action('wp_enqueue_scripts', 'vertim_enqueue_assets');

// Shortcode for VertiMonial Slider
function vertim_testimonials_slider_shortcode($atts) {
    $atts = shortcode_atts(array(
        'limit' => -1,
        'category' => '',
        'speed' => get_option('vertim_animation_speed', 25),
        'bg_color' => get_option('vertim_background_color', '#ffffff'),
        'text_color' => get_option('vertim_text_color', '#000000'),
        'column_width' => get_option('vertim_column_width', 700),
        'column_height' => get_option('vertim_column_height', 500),
        'gap' => get_option('vertim_gap_size', 8),
        'padding' => get_option('vertim_padding_size', 20),
        'border_radius' => get_option('vertim_border_radius', 15),
        'show_rating' => get_option('vertim_show_rating', 1),
        'pause_hover' => get_option('vertim_pause_on_hover', 1),
        'single_column' => get_option('vertim_single_column', 0),
    ), $atts);

    $args = array(
        'post_type' => 'vertim-testimonial',
        'posts_per_page' => intval($atts['limit']),
    );

   if (!empty($atts['category'])) {
    $term = get_term_by('slug', sanitize_text_field($atts['category']), 'vertim-category');
    if ($term && !is_wp_error($term)) {
        // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
        $args['tax_query'] = array(
            array(
                'taxonomy' => 'vertim-category',
                'field'    => 'slug',
                'terms'    => sanitize_text_field($atts['category']),
            ),
        );
        }
}


    $args = apply_filters('vertim_testimonial_query_args', $args);

    $testimonials = new WP_Query($args);
    ob_start();

    if ($testimonials->have_posts()) {
        $custom_css = "
            .columns-wrapper {
                gap: {$atts['gap']}px;
            }
            .column {
                width: {$atts['column_width']}px;
                height: {$atts['column_height']}px;
            }
            .scroll-inner, .scroll-inner.reverse {
                animation-duration: {$atts['speed']}s;
            }
            .testimonial {
                background: {$atts['bg_color']};
                color: {$atts['text_color']};
                padding: {$atts['padding']}px;
                border-radius: {$atts['border_radius']}px;
            }
        ";
        echo '<style>' . $custom_css . '</style>';

        $wrapper_class = 'columns-wrapper';
        if ($atts['single_column']) {
            $wrapper_class .= ' single-column';
        }
        ?>
        <div class="<?php echo esc_attr($wrapper_class); ?>">
            <div class="column">
                <div class="scroll-inner" id="scroll-left">
                    <?php
                    while ($testimonials->have_posts()) {
                        $testimonials->the_post();
                        do_action('vertim_before_testimonial', get_the_ID());

                        $role = get_post_meta(get_the_ID(), '_vertim_testimonial_role', true) ?: 'Client';
                        $company = get_post_meta(get_the_ID(), '_vertim_testimonial_company', true);
                        $rating = get_post_meta(get_the_ID(), '_vertim_testimonial_rating', true);
                        $thumbnail_id = get_post_thumbnail_id(get_the_ID());
                        ?>
                        <div class="testimonial">
                            <?php
                            if ($thumbnail_id) {
                                echo wp_get_attachment_image($thumbnail_id, 'thumbnail', false, ['alt' => get_the_title()]);
                            } else {
                                echo '<img src="' . esc_url(plugin_dir_url(__FILE__) . 'assets/images/vertim-default.png') . '" alt="' . esc_attr(get_the_title()) . '">';
                            }
                            ?>
                            <strong><?php the_title(); ?></strong>
                            <small><?php echo esc_html($role); ?><?php echo $company ? ' at ' . esc_html($company) : ''; ?></small>
                            <?php if ($atts['show_rating'] && $rating) : ?>
                                <div class="rating">
                                    <?php for ($i = 1; $i <= 5; $i++) : ?>
                                        <span class="star <?php echo $i <= $rating ? 'filled' : ''; ?>">&#9733;</span>
                                    <?php endfor; ?>
                                </div>
                            <?php endif; ?>
                            <p><?php the_content(); ?></p>
                        </div>
                        <?php
                    }
                    wp_reset_postdata();
                    ?>
                </div>
            </div>
            <?php if (!$atts['single_column']) : ?>
            <div class="column">
                <div class="scroll-inner reverse" id="scroll-right">
                    <?php
                    $testimonials->rewind_posts();
                    $posts = $testimonials->posts;
                    $posts = array_reverse($posts);
                    foreach ($posts as $post) {
                        do_action('vertim_before_testimonial', $post->ID);

                        $role = get_post_meta($post->ID, '_vertim_testimonial_role', true) ?: 'Client';
                        $company = get_post_meta($post->ID, '_vertim_testimonial_company', true);
                        $rating = get_post_meta($post->ID, '_vertim_testimonial_rating', true);
                        $thumbnail_id = get_post_thumbnail_id($post->ID);
                        $title = $post->post_title;
                        $content = $post->post_content;
                        ?>
                        <div class="testimonial">
                            <?php
                            if ($thumbnail_id) {
                                echo wp_get_attachment_image($thumbnail_id, 'thumbnail', false, ['alt' => esc_attr($title)]);
                            } else {
                                echo '<img src="' . esc_url(plugin_dir_url(__FILE__) . 'assets/images/vertim-default.png') . '" alt="' . esc_attr($title) . '">';
                            }
                            ?>
                            <strong><?php echo esc_html($title); ?></strong>
                            <small><?php echo esc_html($role); ?><?php echo $company ? ' at ' . esc_html($company) : ''; ?></small>
                            <?php if ($atts['show_rating'] && $rating) : ?>
                                <div class="rating">
                                    <?php for ($i = 1; $i <= 5; $i++) : ?>
                                        <span class="star <?php echo $i <= $rating ? 'filled' : ''; ?>">&#9733;</span>
                                    <?php endfor; ?>
                                </div>
                            <?php endif; ?>
                            <p><?php echo wp_kses_post($content); ?></p>
                        </div>
                        <?php
                    }
                    wp_reset_postdata();
                    ?>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    return ob_get_clean();
}
add_shortcode('vertim_testimonials_slider', 'vertim_testimonials_slider_shortcode');
?>