<?php
/**
 * Plugin Name:       Version Locker – Update Control
 * Description:       Securely lock updates for specific plugins to maintain version stability and supply-chain security.
 * Version:           1.2.1
 * Author:            Vishal Paswan
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       version-locker
 * Domain Path:       /languages
 * Requires PHP:      7.4
 * Requires at least: 6.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Constants
 */
define( 'VLOCKER_OPTION_KEY', 'vlocker_locked_plugins' );
define( 'VLOCKER_CAPABILITY', 'update_plugins' );
define( 'VLOCKER_VERSION', '1.2.1' );

/**
 * ------------------------------------------------------------
 * INITIALIZATION & ASSETS
 * ------------------------------------------------------------
 */
add_action( 'admin_enqueue_scripts', function ( $hook ) {
	if ( $hook !== 'plugins.php' && $hook !== 'tools_page_version-locker' ) {
		return;
	}

	wp_enqueue_script(
		'vlocker-admin-js',
		plugin_dir_url( __FILE__ ) . 'assets/admin.js',
		array( 'jquery' ),
		VLOCKER_VERSION,
		true
	);

	wp_localize_script( 'vlocker-admin-js', 'vlocker_vars', array(
		'ajax_url'        => admin_url( 'admin-ajax.php' ),
		'nonce_check'     => wp_create_nonce( 'vlocker_check_locked' ),
		'nonce_cleanup'   => wp_create_nonce( 'vlocker_cleanup_plugin' ),
		'modal_btn_text'  => __( 'Proceed', 'version-locker' ),
		'processing_text' => __( 'Processing...', 'version-locker' ),
		'error_text'      => __( 'An error occurred. Please try again.', 'version-locker' ),
	) );

	wp_add_inline_style( 'wp-admin', '
        .vlocker-locked-notice { color: #d63638; font-weight: 600; }
        .vlocker-modal { display:none; position:fixed; z-index:100000; left:0; top:0; width:100%; height:100%; background:rgba(0,0,0,0.6); }
        .vlocker-modal-box { background:#fff; margin: 15% auto; padding:0; width:500px; max-width:90%; box-shadow:0 5px 15px rgba(0,0,0,0.5); border-radius: 4px; }
        .vlocker-modal-header { padding:15px 20px; border-bottom:1px solid #ddd; background:#fcfcfc; border-radius: 4px 4px 0 0; }
        .vlocker-modal-header h3 { margin:0; font-size: 18px; }
        .vlocker-modal-body { padding:20px; line-height: 1.6; }
        .vlocker-modal-footer { padding:15px 20px; border-top:1px solid #ddd; background:#fcfcfc; text-align:right; border-radius: 0 0 4px 4px; }
        .vlocker-modal-footer .button { margin-left: 8px; }
    ' );
} );

/**
 * ------------------------------------------------------------
 * PERFORMANCE CACHE CLASS
 * ------------------------------------------------------------
 */
class VLOCKER_Cache {
	private static $locked_plugins = null;

	/**
	 * Get locked plugins with caching
	 */
	public static function get_locked_plugins() {
		if ( self::$locked_plugins === null ) {
			self::$locked_plugins = (array) get_option( VLOCKER_OPTION_KEY, array() );
		}
		return self::$locked_plugins;
	}

	/**
	 * Clear the cache
	 */
	public static function clear_cache() {
		self::$locked_plugins = null;
	}
}

/**
 * ------------------------------------------------------------
 * MULTI-ADMIN DETECTION
 * ------------------------------------------------------------
 */
function vlocker_is_multi_admin_site() {
	$cached = get_transient( 'vlocker_is_multi_admin' );
	if ( $cached !== false ) {
		return (bool) $cached;
	}

	$count    = count_users();
	$is_multi = isset( $count['avail_roles']['administrator'] ) && $count['avail_roles']['administrator'] > 1;

	set_transient( 'vlocker_is_multi_admin', $is_multi ? 1 : 0, DAY_IN_SECONDS );

	return $is_multi;
}

add_action( 'user_register', 'vlocker_clear_admin_cache' );
add_action( 'deleted_user', 'vlocker_clear_admin_cache' );
add_action( 'set_user_role', 'vlocker_clear_admin_cache' );

function vlocker_clear_admin_cache() {
	delete_transient( 'vlocker_is_multi_admin' );
}

/**
 * ------------------------------------------------------------
 * AUDIT LOG
 * ------------------------------------------------------------
 */
function vlocker_log_action( $action, $plugins ) {
	if ( ! vlocker_is_multi_admin_site() ) {
		return;
	}

	$log  = get_option( 'vlocker_audit_log', array() );
	$user = wp_get_current_user();

	foreach ( $plugins as $plugin ) {
		$log[] = array(
			'timestamp'  => current_time( 'mysql' ),
			'user_id'    => $user->ID,
			'user_login' => $user->user_login,
			'action'     => sanitize_key( $action ),
			'plugin'     => sanitize_text_field( $plugin ),
		);
	}

	// Keep only last 100 entries
	$log = array_slice( $log, -100 );
	update_option( 'vlocker_audit_log', $log, false );
}

/**
 * ------------------------------------------------------------
 * PLUGIN PATH VALIDATION
 * ------------------------------------------------------------
 */
function vlocker_validate_plugin_path( $plugin_path ) {
	// Remove any whitespace
	$plugin_path = trim( $plugin_path );
	
	// Must end with .php
	if ( substr( $plugin_path, -4 ) !== '.php' ) {
		return false;
	}
	
	// Strict regex: alphanumeric, hyphens, underscores, forward slashes only
	// Format: folder/file.php or file.php
	if ( ! preg_match( '/^[a-zA-Z0-9_-]+(\/[a-zA-Z0-9_-]+)*\.php$/', $plugin_path ) ) {
		return false;
	}
	
	// WordPress native validation (prevents ../, null bytes, etc)
	if ( validate_file( $plugin_path ) !== 0 ) {
		return false;
	}
	
	return true;
}

/**
 * ------------------------------------------------------------
 * 1. ADMIN MENU & HELP
 * ------------------------------------------------------------
 */
add_action( 'admin_menu', function () {
	$hook = add_submenu_page(
		'tools.php',
		__( 'Version Locker', 'version-locker' ),
		__( 'Version Locker', 'version-locker' ),
		VLOCKER_CAPABILITY,
		'version-locker',
		'vlocker_render_admin_page'
	);

	add_action( 'load-' . $hook, function () {
		$screen = get_current_screen();
		$screen->add_help_tab( array(
			'id'      => 'vlocker_overview',
			'title'   => __( 'Overview', 'version-locker' ),
			'content' => '<p>' . __( 'Select the plugins you wish to protect from updates. Locked plugins will not receive automatic or manual updates until unlocked.', 'version-locker' ) . '</p>',
		) );
		$screen->add_help_tab( array(
			'id'      => 'vlocker_security',
			'title'   => __( 'Security', 'version-locker' ),
			'content' => '<p>' . __( 'Locking plugins helps prevent supply-chain attacks and maintains version stability. Only users with plugin update permissions can manage locks.', 'version-locker' ) . '</p>',
		) );
	} );
} );

/**
 * ------------------------------------------------------------
 * 2. ADMIN PAGE RENDER
 * ------------------------------------------------------------
 */
function vlocker_render_admin_page() {
	if ( ! current_user_can( VLOCKER_CAPABILITY ) ) {
		wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'version-locker' ) );
	}

	if ( ! function_exists( 'get_plugins' ) ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}

	$all_plugins = get_plugins();
	$locked      = VLOCKER_Cache::get_locked_plugins();
	$updates     = get_site_transient( 'update_plugins' );

	$active_plugins = array_flip( (array) get_option( 'active_plugins', array() ) );
	$sortable       = array();

	foreach ( $all_plugins as $path => $plugin ) {
		$sortable[ $path ] = array(
			'data'   => $plugin,
			'active' => isset( $active_plugins[ $path ] ),
		);
	}

	// Sort: Active first, then alphabetically
	uasort( $sortable, function ( $a, $b ) {
		if ( $a['active'] !== $b['active'] ) {
			return $b['active'] ? 1 : -1;
		}
		return strcasecmp( $a['data']['Name'], $b['data']['Name'] );
	} );

	if ( isset( $_GET['settings-updated'] ) ) {
		echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Settings saved successfully.', 'version-locker' ) . '</p></div>';
	}
	?>
    <div class="wrap">
        <h1 class="wp-heading-inline"><?php esc_html_e( 'Version Locker', 'version-locker' ); ?></h1>
        <hr class="wp-header-end">

        <div class="tablenav top">
            <div class="alignleft actions bulkactions">
                <button type="button" class="button vlocker-filter-btn active button-primary" data-filter="all"><?php esc_html_e( 'All', 'version-locker' ); ?></button>
                <button type="button" class="button vlocker-filter-btn" data-filter="active"><?php esc_html_e( 'Active', 'version-locker' ); ?></button>
                <button type="button" class="button vlocker-filter-btn" data-filter="locked"><?php esc_html_e( 'Locked', 'version-locker' ); ?></button>
            </div>
            <div class="alignright actions">
                <input type="text" class="regular-text" id="vlocker-search" placeholder="<?php esc_attr_e( 'Search plugins...', 'version-locker' ); ?>">
            </div>
        </div>

        <form method="post" action="">
			<?php wp_nonce_field( 'vlocker_save_settings', 'vlocker_nonce' ); ?>

            <table class="widefat striped table-view-list">
                <thead>
                <tr>
                    <td id="cb" class="manage-column column-cb check-column">
                        <input id="cb-select-all-1" type="checkbox" aria-label="<?php esc_attr_e( 'Select all', 'version-locker' ); ?>">
                    </td>
                    <th class="manage-column column-primary"><?php esc_html_e( 'Plugin', 'version-locker' ); ?></th>
                    <th class="manage-column"><?php esc_html_e( 'Status', 'version-locker' ); ?></th>
                    <th class="manage-column"><?php esc_html_e( 'Version', 'version-locker' ); ?></th>
                    <th class="manage-column"><?php esc_html_e( 'Author', 'version-locker' ); ?></th>
                </tr>
                </thead>
                <tbody id="the-list">
				<?php
				foreach ( $sortable as $path => $item ) :
					$plugin    = $item['data'];
					$is_active = $item['active'];
					$is_locked = in_array( $path, $locked, true );
					?>
                    <tr data-active="<?php echo $is_active ? '1' : '0'; ?>" data-locked="<?php echo $is_locked ? '1' : '0'; ?>">
                        <th scope="row" class="check-column">
                            <input 
                                type="checkbox" 
                                name="vlocker_locked_plugins[]" 
                                value="<?php echo esc_attr( $path ); ?>" 
                                <?php checked( $is_locked ); ?>
                                aria-label="<?php echo esc_attr( sprintf( __( 'Select %s', 'version-locker' ), $plugin['Name'] ) ); ?>"
                            >
                        </th>
                        <td class="plugin-title column-primary">
                            <strong><?php echo esc_html( $plugin['Name'] ); ?></strong>
							<?php if ( $is_locked ) : ?>
                                <span class="dashicons dashicons-lock" style="color: #856404;" title="<?php esc_attr_e( 'Updates Locked', 'version-locker' ); ?>"></span>
								<?php
								if ( is_object( $updates ) && isset( $updates->response[ $path ] ) ) :
									$new_version = $updates->response[ $path ]->new_version;
									?>
                                    <span class="update-message notice inline notice-warning notice-alt">
                                        <?php
                                        /* translators: %s: Plugin version */
                                        printf( esc_html__( 'v%s available', 'version-locker' ), esc_html( $new_version ) );
                                        ?>
                                    </span>
								<?php endif; ?>
							<?php endif; ?>
                        </td>
                        <td class="column-status">
							<?php 
							if ( $is_active ) {
								echo '<span style="color:#46b450;">' . esc_html__( 'Active', 'version-locker' ) . '</span>';
							} else {
								echo '<span style="color:#646970;">' . esc_html__( 'Inactive', 'version-locker' ) . '</span>';
							}
							?>
                        </td>
                        <td class="column-version"><?php echo esc_html( $plugin['Version'] ); ?></td>
                        <td class="column-author"><?php echo wp_kses_post( $plugin['Author'] ); ?></td>
                    </tr>
				<?php endforeach; ?>
                </tbody>
            </table>

            <div class="card" style="margin-top: 20px; padding: 0 0 15px 0; max-width: 100%;">
                <h3 style="padding: 15px; margin: 0; border-bottom: 1px solid #f0f0f1; background: #fafafa;">
					<?php esc_html_e( 'Configuration', 'version-locker' ); ?>
                </h3>
                <div style="padding: 15px;">
                    <label for="vlocker_keep_settings">
                        <input 
                            type="checkbox" 
                            id="vlocker_keep_settings" 
                            name="vlocker_keep_settings_on_delete" 
                            value="1" 
                            <?php checked( get_option( 'vlocker_keep_settings_on_delete', false ) ); ?>
                        >
                        <strong><?php esc_html_e( 'Preserve locks on plugin deletion', 'version-locker' ); ?></strong>
                    </label>
                    <p class="description" style="margin-left: 25px;">
						<?php esc_html_e( 'Enable this to retain lock settings if a plugin is uninstalled. Useful if you plan to reinstall plugins later.', 'version-locker' ); ?>
                    </p>
                </div>
            </div>

            <p class="submit">
                <button type="submit" class="button button-primary button-hero">
                    <span class="dashicons dashicons-lock" style="margin-top: 4px;"></span>
                    <?php esc_html_e( 'Save Changes', 'version-locker' ); ?>
                </button>
            </p>
        </form>

		<?php if ( vlocker_is_multi_admin_site() ) : ?>
            <div style="margin-top: 40px;">
                <h2><?php esc_html_e( 'Audit Log', 'version-locker' ); ?></h2>
                <p class="description"><?php esc_html_e( 'Track who locked or unlocked plugins (displayed for sites with multiple administrators).', 'version-locker' ); ?></p>
                <div class="card" style="margin-top: 10px; padding: 0; max-width: 100%;">
					<?php
					$log = get_option( 'vlocker_audit_log', array() );
					if ( ! empty( $log ) ) :
						$log = array_reverse( $log );
						?>
                        <table class="widefat striped" style="border:none; box-shadow:none;">
                            <thead>
                            <tr>
                                <th><?php esc_html_e( 'Date', 'version-locker' ); ?></th>
                                <th><?php esc_html_e( 'User', 'version-locker' ); ?></th>
                                <th><?php esc_html_e( 'Action', 'version-locker' ); ?></th>
                                <th><?php esc_html_e( 'Plugin', 'version-locker' ); ?></th>
                            </tr>
                            </thead>
                            <tbody>
							<?php foreach ( array_slice( $log, 0, 20 ) as $entry ) : ?>
                                <tr>
                                    <td><?php echo esc_html( wp_date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $entry['timestamp'] ) ) ); ?></td>
                                    <td><?php echo esc_html( $entry['user_login'] ); ?></td>
                                    <td>
										<?php if ( $entry['action'] === 'locked' ) : ?>
                                            <span class="dashicons dashicons-lock" style="color:#856404;"></span> <?php esc_html_e( 'Locked', 'version-locker' ); ?>
										<?php else : ?>
                                            <span class="dashicons dashicons-unlock" style="color:#46b450;"></span> <?php esc_html_e( 'Unlocked', 'version-locker' ); ?>
										<?php endif; ?>
                                    </td>
                                    <td><code><?php echo esc_html( $entry['plugin'] ); ?></code></td>
                                </tr>
							<?php endforeach; ?>
                            </tbody>
                        </table>
					<?php else : ?>
                        <div style="padding: 20px; text-align: center; color: #646970;">
                            <?php esc_html_e( 'No activity recorded yet.', 'version-locker' ); ?>
                        </div>
					<?php endif; ?>
                </div>
            </div>
		<?php endif; ?>
    </div>
	<?php
}

/**
 * ------------------------------------------------------------
 * 3. SAVE SETTINGS
 * ------------------------------------------------------------
 */
add_action( 'admin_init', function () {
	// Early return if not a form submission
	if ( ! isset( $_POST['vlocker_nonce'] ) ) {
		return;
	}

	// Verify nonce
	if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['vlocker_nonce'] ) ), 'vlocker_save_settings' ) ) {
		wp_die( esc_html__( 'Security check failed. Please try again.', 'version-locker' ) );
	}

	// Check permissions
	if ( ! current_user_can( VLOCKER_CAPABILITY ) ) {
		wp_die( esc_html__( 'You do not have sufficient permissions to perform this action.', 'version-locker' ) );
	}

	$old_locked = VLOCKER_Cache::get_locked_plugins();
	$locked     = array();

	// Process submitted plugins
	if ( isset( $_POST['vlocker_locked_plugins'] ) && is_array( $_POST['vlocker_locked_plugins'] ) ) {
		$raw_plugins = array_map( 'sanitize_text_field', wp_unslash( $_POST['vlocker_locked_plugins'] ) );

		foreach ( $raw_plugins as $plugin ) {
			// Use our validation function
			if ( vlocker_validate_plugin_path( $plugin ) ) {
				$locked[] = $plugin;
			} else {
				// Log invalid attempts for security monitoring
				error_log( sprintf( 'Version Locker: Invalid plugin path rejected: %s', $plugin ) );
			}
		}
	}

	// Remove non-existent plugins if setting is disabled
	$keep_settings = get_option( 'vlocker_keep_settings_on_delete', false );
	if ( ! $keep_settings ) {
		$locked = array_filter( $locked, function ( $plugin_path ) {
			return file_exists( WP_PLUGIN_DIR . '/' . $plugin_path );
		} );
	}

	// Save settings
	update_option( VLOCKER_OPTION_KEY, array_values( array_unique( $locked ) ), false );
	
	$checkbox_val = isset( $_POST['vlocker_keep_settings_on_delete'] );
	update_option( 'vlocker_keep_settings_on_delete', (bool) $checkbox_val, false );

	// Clear cache
	VLOCKER_Cache::clear_cache();

	// Log changes for multi-admin sites
	if ( vlocker_is_multi_admin_site() ) {
		$newly_locked   = array_diff( $locked, $old_locked );
		$newly_unlocked = array_diff( $old_locked, $locked );
		
		if ( ! empty( $newly_locked ) ) {
			vlocker_log_action( 'locked', $newly_locked );
		}
		if ( ! empty( $newly_unlocked ) ) {
			vlocker_log_action( 'unlocked', $newly_unlocked );
		}
	}

	// Redirect with success message
	$redirect_url = add_query_arg( 'settings-updated', 'true', wp_get_referer() );
	wp_safe_redirect( $redirect_url );
	exit;
} );

/**
 * ------------------------------------------------------------
 * 4. LOCKING LOGIC (CORE)
 * ------------------------------------------------------------
 */
add_filter( 'site_transient_update_plugins', function ( $transient ) {
	if ( ! is_object( $transient ) || empty( $transient->response ) ) {
		return $transient;
	}
	
	$locked = VLOCKER_Cache::get_locked_plugins();
	foreach ( $locked as $plugin ) {
		unset( $transient->response[ $plugin ] );
	}
	
	return $transient;
}, 99 );

add_filter( 'auto_update_plugin', function ( $update, $item ) {
	if ( ! isset( $item->plugin ) ) {
		return $update;
	}
	
	$locked = VLOCKER_Cache::get_locked_plugins();
	if ( in_array( $item->plugin, $locked, true ) ) {
		return false;
	}
	
	return $update;
}, 99, 2 );

add_filter( 'upgrader_pre_install', function ( $response, $hook_extra ) {
	if ( ! isset( $hook_extra['plugin'] ) ) {
		return $response;
	}
	
	$locked = VLOCKER_Cache::get_locked_plugins();
	if ( in_array( $hook_extra['plugin'], $locked, true ) ) {
		set_transient( 'vlocker_update_blocked', $hook_extra['plugin'], 120 );
		return new WP_Error( 
			'plugin_update_locked', 
			__( 'Update blocked: This plugin is locked by an administrator.', 'version-locker' ) 
		);
	}
	
	return $response;
}, 10, 2 );

add_action( 'admin_notices', function () {
	$blocked_plugin = get_transient( 'vlocker_update_blocked' );
	if ( $blocked_plugin ) {
		printf(
			'<div class="notice notice-error is-dismissible"><p><strong>%s</strong> %s</p></div>',
			esc_html__( 'Update Blocked:', 'version-locker' ),
			esc_html__( 'This plugin is locked and cannot be updated. Please unlock it first from Version Locker settings.', 'version-locker' )
		);
		delete_transient( 'vlocker_update_blocked' );
	}
} );

/**
 * ------------------------------------------------------------
 * 5. UI ENHANCEMENTS & MODAL HTML
 * ------------------------------------------------------------
 */
add_filter( 'plugin_action_links', function ( $actions, $plugin_file ) {
	if ( ! current_user_can( VLOCKER_CAPABILITY ) ) {
		return $actions;
	}
	
	$locked = VLOCKER_Cache::get_locked_plugins();
    
	if ( in_array( $plugin_file, $locked, true ) ) {
		// Remove default actions
		unset( $actions['update'], $actions['delete'] );
        
		// Add locked status
		$updates = get_site_transient( 'update_plugins' );
		$text    = esc_html__( 'Locked', 'version-locker' );
		
		if ( is_object( $updates ) && isset( $updates->response[ $plugin_file ] ) ) {
			$new_version = $updates->response[ $plugin_file ]->new_version;
			$text .= ' <span style="color:#d63638;">(v' . esc_html( $new_version ) . ' ' . esc_html__( 'available', 'version-locker' ) . ')</span>';
		}
		
		$actions['vlocker_locked'] = sprintf(
			'<span class="dashicons dashicons-lock" style="color:#856404;font-size:16px;vertical-align:text-bottom;"></span> %s',
			$text
		);

		// Add secure delete link
		$delete_url = wp_nonce_url( 
			'plugins.php?action=delete-selected&checked[]=' . urlencode( $plugin_file ) . '&plugin_status=all&paged=1&s=', 
			'bulk-plugins' 
		);
        
		$actions['vlocker_delete'] = sprintf(
			'<a href="%s" class="vlocker-secure-delete" data-plugin="%s" style="color:#a00;">%s</a>',
			esc_url( $delete_url ),
			esc_attr( $plugin_file ),
			esc_html__( 'Delete', 'version-locker' )
		);
	}
	
	return $actions;
}, 10, 2 );

add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), function ( $links ) {
	$settings_link = sprintf(
		'<a href="%s">%s</a>',
		esc_url( admin_url( 'tools.php?page=version-locker' ) ),
		esc_html__( 'Settings', 'version-locker' )
	);
	array_unshift( $links, $settings_link );
	return $links;
} );

add_action( 'admin_footer-plugins.php', function () {
	if ( ! current_user_can( VLOCKER_CAPABILITY ) ) {
		return;
	}
	?>
    <div id="vlocker-delete-modal" class="vlocker-modal" role="dialog" aria-labelledby="vlocker-modal-title" aria-modal="true">
        <div class="vlocker-modal-box">
            <div class="vlocker-modal-header">
                <h3 id="vlocker-modal-title"><?php esc_html_e( 'Confirm Deletion', 'version-locker' ); ?></h3>
            </div>
            <div class="vlocker-modal-body">
                <p><strong><?php esc_html_e( 'You are deleting a locked plugin.', 'version-locker' ); ?></strong></p>
                <p><?php esc_html_e( 'Do you want to permanently remove the lock rule for this plugin, or keep it for future re-installation?', 'version-locker' ); ?></p>
                <p>
                    <label>
                        <input type="radio" name="vlocker_delete_action" value="temporary" checked> 
                        <?php esc_html_e( 'Keep Lock Rule (Temporary Deletion)', 'version-locker' ); ?>
                    </label><br>
                    <label>
                        <input type="radio" name="vlocker_delete_action" value="permanent"> 
                        <?php esc_html_e( 'Remove Lock Rule (Permanent)', 'version-locker' ); ?>
                    </label>
                </p>
            </div>
            <div class="vlocker-modal-footer">
                <button type="button" class="button" id="vlocker-cancel-delete"><?php esc_html_e( 'Cancel', 'version-locker' ); ?></button>
                <button type="button" class="button button-primary" id="vlocker-confirm-delete"><?php esc_html_e( 'Proceed', 'version-locker' ); ?></button>
            </div>
        </div>
    </div>
	<?php
} );

/**
 * ------------------------------------------------------------
 * AJAX HANDLERS
 * ------------------------------------------------------------
 */
add_action( 'wp_ajax_vlocker_check_locked', function () {
	check_ajax_referer( 'vlocker_check_locked', 'nonce' );
	
	if ( ! current_user_can( VLOCKER_CAPABILITY ) ) {
		wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'version-locker' ) ) );
	}
	
	$plugin = isset( $_POST['plugin'] ) ? sanitize_text_field( wp_unslash( $_POST['plugin'] ) ) : '';
	
	if ( empty( $plugin ) ) {
		wp_send_json_error( array( 'message' => __( 'Invalid plugin specified.', 'version-locker' ) ) );
	}
	
	$locked = VLOCKER_Cache::get_locked_plugins();
	wp_send_json_success( array( 'is_locked' => in_array( $plugin, $locked, true ) ) );
} );

add_action( 'wp_ajax_vlocker_cleanup_plugin', function () {
	check_ajax_referer( 'vlocker_cleanup_plugin', 'nonce' );
	
	if ( ! current_user_can( VLOCKER_CAPABILITY ) ) {
		wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'version-locker' ) ) );
	}

	$raw_plugin = isset( $_POST['plugin'] ) ? wp_unslash( $_POST['plugin'] ) : '';
	$plugin     = sanitize_text_field( $raw_plugin );
	$plugin     = trim( $plugin );

	if ( empty( $plugin ) || ! vlocker_validate_plugin_path( $plugin ) ) {
		wp_send_json_error( array( 'message' => __( 'Invalid plugin path.', 'version-locker' ) ) );
	}

	$locked = VLOCKER_Cache::get_locked_plugins();
	$key    = array_search( $plugin, $locked, true );
	
	// Fallback: handle URL encoding differences
	if ( $key === false ) {
		foreach ( $locked as $index => $locked_plugin ) {
			if ( $locked_plugin === $plugin || urldecode( $locked_plugin ) === urldecode( $plugin ) ) {
				$key = $index;
				break;
			}
		}
	}

	if ( $key !== false ) {
		unset( $locked[ $key ] );
		update_option( VLOCKER_OPTION_KEY, array_values( $locked ), false );
		VLOCKER_Cache::clear_cache();
		
		// Log the action
		if ( vlocker_is_multi_admin_site() ) {
			vlocker_log_action( 'unlocked', array( $plugin ) );
		}
	}

	wp_send_json_success( array( 'message' => __( 'Lock removed successfully.', 'version-locker' ) ) );
} );

/**
 * ------------------------------------------------------------
 * 6. UNINSTALL CLEANUP
 * ------------------------------------------------------------
 */
register_uninstall_hook( __FILE__, 'vlocker_uninstall_cleanup' );

function vlocker_uninstall_cleanup() {
	// If keep settings is enabled, only remove that option
	if ( get_option( 'vlocker_keep_settings_on_delete', false ) ) {
		delete_option( 'vlocker_keep_settings_on_delete' );
		return;
	}
	
	// Otherwise, clean up everything
	delete_option( VLOCKER_OPTION_KEY );
	delete_option( 'vlocker_audit_log' );
	delete_option( 'vlocker_keep_settings_on_delete' );
	delete_transient( 'vlocker_update_blocked' );
	delete_transient( 'vlocker_is_multi_admin' );
}