<?php

/*
 * Plugin Name:       Venjue Widget
 * Plugin URI:        https://venjue.com
 * Description:       Easily add the Venjue widget to your WordPress website with one-click installation. Customize colors, language, and analytics.
 * Version:           1.1.0
 * Author:            Venjue ApS
 * Author URI:        https://venjue.com
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       venjue-widget
 * Requires at least: 6.7
 * Requires PHP:      7.4
 */

if (!defined('ABSPATH')) exit;

/*
 Settings page and its UI
 */
add_action('admin_menu', function () {
  add_options_page('Venjue Widget Settings','Venjue','manage_options','venjue','venjuewidget_render_page');
});
add_action('admin_init', function () {
  register_setting('venjuewidget_group','venjuewidget_business_vat',[
    'type'=>'string',
    'sanitize_callback'=>fn($v)=>preg_replace('/\D+/','',(string)$v),
    'default'=>'',
  ]);
  register_setting('venjuewidget_group','venjuewidget_accent_color',[
    'type'=>'string',
    'sanitize_callback'=>function($v){
      $v = trim((string)$v);
      return preg_match('/^#([0-9a-fA-F]{6})$/',$v) ? $v : '#8A89D8';
    },
    'default'=>'#8A89D8',
  ]);
  register_setting('venjuewidget_group','venjuewidget_language',[
    'type'=>'string',
    'sanitize_callback'=>function($v){
      $allowed=['auto','da_DK','en_DK'];
      return in_array($v,$allowed,true)?$v:'auto';
    },
    'default'=>'auto',
  ]);
  register_setting('venjuewidget_group','venjuewidget_gtm_id',[
    'type'=>'string',
    'sanitize_callback'=>function($v){
      $v = strtoupper(trim((string)$v));
      return preg_match('/^GTM-[A-Z0-9]+$/',$v) ? $v : '';
    },
    'default'=>'',
  ]);
  register_setting('venjuewidget_group','venjuewidget_custom_code',[
    'type'=>'string',
    'sanitize_callback'=>function($v){
      $v = (string)$v;
      if (current_user_can('unfiltered_html')) return $v;
      return sanitize_textarea_field($v);
    },
    'default'=>'',
  ]);
});
function venjuewidget_render_page(){
  if (!current_user_can('manage_options')) wp_die(__('You do not have permission.','venjue-widget'));
  //get set values for variables
  $vat = get_option('venjuewidget_business_vat','');
  $color = get_option('venjuewidget_accent_color','#8A89D8');
  $lang = get_option('venjuewidget_language','auto');
  $gtm_id = get_option('venjuewidget_gtm_id','');
  $code = get_option('venjuewidget_custom_code','');
  ?>
  <div class="wrap">
    <h1>Venjue Widget Settings</h1>
    <p>Use the options below to configure the Venjue widget on your Wordpress site.</p>
    <p>If any errors occur, you can inspect them in the Developer Console on your Wordpress site.</p>
    <p>See more on <a href="https://venjue.com">Venjue.com</a></p>
    <form method="post" action="options.php">
      <?php settings_fields('venjuewidget_group'); ?>
      <table class="form-table" role="presentation">
        <tr>
          <th scope="row"><label for="venjuewidget_business_vat">VAT (CVR)</label></th>
          <td>
            <input id="venjuewidget_business_vat" name="venjuewidget_business_vat" type="text" class="regular-text" placeholder="12345678" value="<?php echo esc_attr($vat); ?>">
            <p class="description">This <strong>needs</strong> to match your setup on Venjue!</p>
          </td>
        </tr>
        <tr>
          <th scope="row"><label for="venjuewidget_accent_color">Accent color</label></th>
          <td>
            <input id="venjuewidget_accent_color" name="venjuewidget_accent_color" type="text" class="color-field" data-default-color="#8A89D8" value="<?php echo esc_attr($color); ?>">
            <p class="description">Sets the color of the widget's UI on accented elements; call-to-action buttons, links, borders, inputs and more. Pick a color to match your brand identity.</p>
          </td>
        </tr>
        <tr>
          <th scope="row"><label for="venjuewidget_language">Language</label></th>
          <td>
            <select id="venjuewidget_language" name="venjuewidget_language">
              <option value="auto"  <?php selected($lang,'auto');  ?>>Auto (recommended)</option>
              <option value="da_DK" <?php selected($lang,'da_DK'); ?>>Dansk</option>
              <option value="en_DK" <?php selected($lang,'en_DK'); ?>>English</option>
            </select>
            <p class="description">Language can be "Auto" (recommended) for Wordpress to automatically tell the Venjue widget the language of the current page.</p>
          </td>
        </tr>
        <tr>
          <th scope="row"><label for="venjuewidget_gtm_id">Google Tag Manager ID</label></th>
          <td>
            <input id="venjuewidget_gtm_id" name="venjuewidget_gtm_id" type="text" class="regular-text" placeholder="GTM-XXXXXXX" value="<?php echo esc_attr($gtm_id); ?>">
            <p class="description">Optional. The Venjue widget emits events to your Google Analytics, and tries to do so automatically. If you need to overwrite the auto-discovered GTM-ID, you can supply it here. Example: <code>GTM-ABC1234</code>. List of emitted events can be found on Venjue.</p>
          </td>
        </tr>
        <tr>
          <th scope="row"><label for="venjuewidget_custom_code">Custom code</label></th>
          <td>
            <textarea id="venjuewidget_custom_code" name="venjuewidget_custom_code" rows="5" class="large-text code" placeholder="Set custom code to run when the widget runs"><?php echo esc_textarea($code); ?></textarea>
            <p class="description">No &lt;script&gt; tags needed, just use plain JavaScript. The custom code is run after the Venjue widget fires its ready event. List of events, methods and more advanced options can be found on Venjue.</p>
          </td>
        </tr>
      </table>
      <?php submit_button(__('Save Changes','venjue-widget')); ?>
    </form>
  </div>
  <?php
}
add_action('admin_enqueue_scripts', function ($hook) {
  if ($hook !== 'settings_page_venjue') return;
  wp_enqueue_style('wp-color-picker');
  wp_enqueue_script('wp-color-picker');
  wp_add_inline_script(
    'wp-color-picker',
    'jQuery(document).ready(function($){$(".color-field").wpColorPicker();});',
    'after'
  );
});

/*
 Output widget container in footer with resolved data attributes
 */
add_action('wp_footer', function () {
  $vat = get_option('venjuewidget_business_vat','');
  $color = get_option('venjuewidget_accent_color','#8A89D8');
  $lang = get_option('venjuewidget_language','auto');
  $gtm_id = get_option('venjuewidget_gtm_id','');
  if ($lang === 'auto') {
    $site_locale = get_locale();
    $lang = (stripos($site_locale,'da_') === 0) ? 'da_DK' : 'en_DK';
  }
  echo '<div id="venjue-widget-container"'
  .' data-accentcolor="'.esc_attr($color).'"'
  .' data-langcode="'.esc_attr($lang).'"'
  .' data-businessid="'.esc_attr($vat).'"'
  .' data-gtm-id="'.esc_attr($gtm_id).'"'
  .'></div>';
});

/*
 Enqueue Venjue hosted script and attach user's custom code
 */
add_action('wp_enqueue_scripts', function () {
  //load the hosted widget script
  wp_enqueue_script('venjuewidget-frontend','https://venjue.com/widget/script.js',[], '1.1.0', true);
  //inject user custom code (runs on venjueWidgetReady event)
  $user_code = get_option('venjuewidget_custom_code','');
  if (!empty($user_code)) {
    $wrapped = "(function(){try{".$user_code."}catch(e){console&&console.error&&console.error('Venjue custom code error:',e);}})();";
    wp_add_inline_script('venjuewidget-frontend', $wrapped, 'after');
  }
});
?>
