<?php
/**
 * Plugin Name:       Vehizo – Vehicle Management
 * Plugin URI:        https://vehizo.de
 * Description:       Modern and professional vehicle management plugin for WordPress. Perfect for car dealerships and automotive listings, with advanced filtering, contact forms, and comprehensive vehicle inventory management.
 * Version:           4.1.4
 * Author:            Thorsten Glander
 * Author URI:        https://wuemme-media.de
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       vehizo-vehicle-management
 * Domain Path:       /languages
 * Requires at least: 6.0
 * Tested up to:      6.7
 * Requires PHP:      8.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// =================================================================================
// 1. ZENTRALE KONSTANTEN UND LADEVORGÄNGE
// =================================================================================

// Define plugin constants
define( 'VEHIZO_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'VEHIZO_PLUGIN_PATH', plugin_dir_path( __FILE__ ) );
define( 'VEHIZO_PLUGIN_VERSION', '4.1.4' );

// Load all class files centrally from the /includes/ folder
require_once VEHIZO_PLUGIN_PATH . 'includes/class-mobile-importer.php';
require_once VEHIZO_PLUGIN_PATH . 'includes/class-admin-ui.php';
require_once VEHIZO_PLUGIN_PATH . 'includes/class-shortcode-vehicles.php';
require_once VEHIZO_PLUGIN_PATH . 'includes/class-statistics-module.php';
require_once VEHIZO_PLUGIN_PATH . 'includes/class-admin-vehicle-editor.php';
require_once VEHIZO_PLUGIN_PATH . 'includes/class-filter-settings.php';
require_once VEHIZO_PLUGIN_PATH . 'includes/class-wishlist-module.php';
require_once VEHIZO_PLUGIN_PATH . 'includes/class-modules.php';
// Language files available as separate downloads



// =================================================================================
// 2. INITIALISIERUNG DES PLUGINS
// =================================================================================

// Initialize all modules and classes after plugins are loaded
add_action('plugins_loaded', 'vehizo_initialize_plugin');
function vehizo_initialize_plugin() {
    // Central initialization for admin-specific classes
    if ( is_admin() ) {
        Vehizo_Admin::init();
        Vehizo_Admin_Vehicle_Editor::init();
        // Vehizo_Filter_Settings is now included within Vehizo_Admin, so this direct call is not needed.
        // If it were standalone, it would be initialized here.
        // Vehizo_Filter_Settings::init(); 
        Vehizo_Modules::init();
    }

    // Initialize components that can run everywhere
    Vehizo_Mobile_Importer::init();

    // WordPress.org automatically loads translations for plugins hosted there


    // Initialize wishlist module if active
    if (get_option('vehizo_wishlist_module_active', '1') === '1') {
        Vehizo_Wishlist_Module::init();
    }

    // Statistics module needs to track views on frontend, but dashboard part is admin only
    if (get_option('vehizo_statistics_module_active', '0') === '1') {
        Vehizo_Statistics_Module::init();
    }

    // NEU: Zählfunktion für Fahrzeugaufrufe hinzufügen
    add_action('wp_head', 'vehizo_track_vehicle_views');
}


// =================================================================================
// 3. REGISTRIERUNG DES CUSTOM POST TYPE "VEHICLE"
// =================================================================================

add_action('init', 'vehizo_register_cpt_vehicle');
function vehizo_register_cpt_vehicle() {
    register_post_status('reserved', array(
        'label'                     => _x( 'Reserved', 'post status', 'vehizo-vehicle-management' ),
        'public'                    => true,
        'exclude_from_search'       => false,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        /* translators: %s: Number of reserved vehicles */
        'label_count'               => _n_noop( 'Reserved <span class="count">(%s)</span>', 'Reserved <span class="count">(%s)</span>', 'vehizo-vehicle-management' ),
    ));

    register_post_status('sold', array(
        'label'                     => _x( 'Sold', 'post status', 'vehizo-vehicle-management' ),
        'public'                    => true,
        'exclude_from_search'       => true,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        /* translators: %s: Number of sold vehicles */
        'label_count'               => _n_noop( 'Sold <span class="count">(%s)</span>', 'Sold <span class="count">(%s)</span>', 'vehizo-vehicle-management' ),
    ));

    register_post_type('vehizo_vehicle', [
        'label' => __('Vehicles', 'vehizo-vehicle-management'),
        'labels' => [
            'name' => __('Vehicles', 'vehizo-vehicle-management'),
            'singular_name' => __('Vehicle', 'vehizo-vehicle-management'),
            'add_new' => __('Add New Vehicle', 'vehizo-vehicle-management'),
            'add_new_item' => __('Add New Vehicle', 'vehizo-vehicle-management'),
            'edit_item' => __('Edit Vehicle', 'vehizo-vehicle-management'),
            'new_item' => __('New Vehicle', 'vehizo-vehicle-management'),
            'view_item' => __('View Vehicle', 'vehizo-vehicle-management'),
            'view_items' => __('View Vehicles', 'vehizo-vehicle-management'),
            'search_items' => __('Search Vehicles', 'vehizo-vehicle-management'),
            'not_found' => __('No vehicles found', 'vehizo-vehicle-management'),
            'not_found_in_trash' => __('No vehicles found in trash', 'vehizo-vehicle-management'),
            'parent_item_colon' => __('Parent Vehicle:', 'vehizo-vehicle-management'),
            'all_items' => __('All Vehicles', 'vehizo-vehicle-management'),
            'archives' => __('Vehicle Archives', 'vehizo-vehicle-management'),
            'attributes' => __('Vehicle Attributes', 'vehizo-vehicle-management'),
            'insert_into_item' => __('Insert into vehicle', 'vehizo-vehicle-management'),
            'uploaded_to_this_item' => __('Uploaded to this vehicle', 'vehizo-vehicle-management'),
        ],
        'public' => true,
        'has_archive' => true,
        'menu_icon' => 'dashicons-car',
        'supports' => ['title', 'editor', 'thumbnail', 'custom-fields'],
        'show_in_rest' => true,
        'show_ui' => true,
        'capability_type' => 'post',
        'rewrite' => ['slug' => 'vehicle', 'with_front' => false],
        'show_in_menu' => false, // Ihr Wunsch: Keine automatischen Untermenüs.
    ]);

    // NEU: Registrierung der 'vehizo_feature' Taxonomie für Ausstattungsmerkmale
    register_taxonomy('vehizo_feature', 'vehizo_vehicle', [
        'label' => __('Features', 'vehizo-vehicle-management'),
        'labels' => [
            'name' => __('Features', 'vehizo-vehicle-management'),
            'singular_name' => __('Feature', 'vehizo-vehicle-management'),
            'search_items' => __('Search Features', 'vehizo-vehicle-management'),
            'all_items' => __('All Features', 'vehizo-vehicle-management'),
            'parent_item' => __('Parent Feature', 'vehizo-vehicle-management'),
            'parent_item_colon' => __('Parent Feature:', 'vehizo-vehicle-management'),
            'edit_item' => __('Edit Feature', 'vehizo-vehicle-management'),
            'update_item' => __('Update Feature', 'vehizo-vehicle-management'),
            'add_new_item' => __('Add New Feature', 'vehizo-vehicle-management'),
            'new_item_name' => __('New Feature Name', 'vehizo-vehicle-management'),
            'menu_name' => __('Features', 'vehizo-vehicle-management'),
        ],
        'public' => true,
        'hierarchical' => false, // Wie Schlagwörter, nicht wie Kategorien
        'show_ui' => true,
        'show_in_menu' => true, // Zeigt "Features" unter "Vehizo" an
        'show_in_nav_menus' => false,
        'show_admin_column' => true,
        'show_in_rest' => true, // Wichtig für Gutenberg & REST API
    ]);
}

// =================================================================================
// 4. SCRIPTS, STYLES UND ANDERE GLOBALE FUNKTIONEN
// =================================================================================

// WordPress.org automatically loads translations for hosted plugins
// No manual textdomain loading needed

// Enqueue Frontend Assets
add_action( 'wp_enqueue_scripts', 'vehizo_enqueue_frontend_assets' );
function vehizo_enqueue_frontend_assets() {
	// Only enqueue on vehicle pages or when shortcode is present
	if ( is_singular( 'vehizo_vehicle' ) || is_post_type_archive( 'vehizo_vehicle' ) || vehizo_has_shortcode() ) {
		
		// Frontend JavaScript
		wp_enqueue_script(
			'vehizo-frontend-script',
			VEHIZO_PLUGIN_URL . 'assets/js/frontend.js',
			array( 'jquery' ),
			VEHIZO_PLUGIN_VERSION,
			true
		);

		// Localize script for frontend AJAX and translations
		wp_localize_script( 'vehizo-frontend-script', 'vehizo_frontend_vars', array(
			'ajaxurl'               => admin_url( 'admin-ajax.php' ),
			'nonce'                 => wp_create_nonce( 'vehizo_frontend_nonce' ),
			'loading_text'          => __( 'Loading...', 'vehizo-vehicle-management' ),
			'error_text'            => __( 'An error occurred. Please try again.', 'vehizo-vehicle-management' ),
			'contact_success_text'  => __( 'Your message has been sent successfully!', 'vehizo-vehicle-management' ),
			'required_fields_text'  => __( 'Please fill in all required fields.', 'vehizo-vehicle-management' ),
			'invalid_email_text'    => __( 'Please enter a valid email address.', 'vehizo-vehicle-management' ),
		));
	}

	// Single vehicle page specific scripts
	if ( is_singular( 'vehizo_vehicle' ) ) {
        // Single vehicle CSS
        wp_enqueue_style( 'vehizo-single-vehicle', VEHIZO_PLUGIN_URL . 'assets/css/single-vehicle.css', [], VEHIZO_PLUGIN_VERSION );
		// Single vehicle JavaScript
		wp_enqueue_script(
			'vehizo-single-vehicle',
			VEHIZO_PLUGIN_URL . 'assets/js/single-vehicle.js',
			array( 'jquery' ),
			VEHIZO_PLUGIN_VERSION,
			true
		);

		// Localize script for contact form functionality
		wp_localize_script( 'vehizo-single-vehicle', 'vehizoContact', array(
			'ajaxUrl'    => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'vehizo_contact_ajax_nonce' ),
			'strings'    => array(
				'sending'   => __( 'Sending...', 'vehizo-vehicle-management' ),
                'success'   => __( 'Your message has been sent successfully!', 'vehizo-vehicle-management' ),
				'error'     => __( 'Failed to send message. Please try again later.', 'vehizo-vehicle-management' ),
				'required_fields' => __( 'Please fill in all required fields.', 'vehizo-vehicle-management' ),
			)
		));

		// Localize script for wishlist functionality
		wp_localize_script( 'vehizo-single-vehicle', 'vehizoWishlist', array(
			'ajaxUrl'    => admin_url( 'admin-ajax.php' ),
			'nonce' => wp_create_nonce( 'vehizo_wishlist_nonce' ),
			'strings'    => array(
				'addToWishlist' => __( 'Add to Wishlist', 'vehizo-vehicle-management' ),
				'onWishlist'    => __( 'On Wishlist', 'vehizo-vehicle-management' ),
				'removeFromWishlist' => __( 'Remove from Wishlist', 'vehizo-vehicle-management' ),
				'viewWishlist'  => __( 'View Wishlist', 'vehizo-vehicle-management' ),
				'loading'       => __( 'Loading your wishlist...', 'vehizo-vehicle-management' ),
				'empty'         => __( 'Your wishlist is empty', 'vehizo-vehicle-management' ),
				'emptyDesc'     => __( 'Add vehicles to your wishlist to see them here', 'vehizo-vehicle-management' ),
				'error'         => __( 'Error loading wishlist', 'vehizo-vehicle-management' ),
				'errorDesc'     => __( 'Please try again later', 'vehizo-vehicle-management' ),
			)
		));
	}
	
	// Enqueue filter/search assets when needed
	if ( vehizo_has_filter_shortcode() || is_post_type_archive( 'vehizo_vehicle' ) ) {
		wp_enqueue_style(
			'vehizo-filter-style',
			VEHIZO_PLUGIN_URL . 'assets/css/filter.css',
			array(),
			VEHIZO_PLUGIN_VERSION
		);
		
		wp_enqueue_script(
			'vehizo-filter-script',
			VEHIZO_PLUGIN_URL . 'assets/js/filter.js',
			array( 'jquery' ),
			VEHIZO_PLUGIN_VERSION,
			true
		);
		
		wp_localize_script( 'vehizo-filter-script', 'vehizo_filter_vars', array(
			'ajaxurl'           => admin_url( 'admin-ajax.php' ),
			'filter_nonce'      => wp_create_nonce( 'vehizo_filter_nonce' ),
			'no_results_text'   => __( 'No vehicles found matching your criteria.', 'vehizo-vehicle-management' ),
			'loading_text'      => __( 'Searching...', 'vehizo-vehicle-management' ),
		));
	}
}

// KORREKTUR: Die folgende Funktion 'vehizo_enqueue_admin_assets' und der zugehörige Hook wurden entfernt.
// Dies löst den Konflikt, da das Laden der Skripte nun ausschließlich und korrekt
// über die 'enqueue_admin_assets' Methode in der 'class-admin-ui.php' gesteuert wird.

// Helper function to check if vehicle shortcode is present on current page
function vehizo_has_shortcode() {
	global $post;
	if ( ! is_a( $post, 'WP_Post' ) ) {
		return false;
	}
	
	$shortcodes = array( 'vehizo_fahrzeuge', 'vehizo_vehicles', 'vehizo_vehicle_list' );
	foreach ( $shortcodes as $shortcode ) {
		if ( has_shortcode( $post->post_content, $shortcode ) ) {
			return true;
		}
	}
	
	return false;
}

// Helper function to check if filter shortcode is present
function vehizo_has_filter_shortcode() {
	global $post;
	if ( ! is_a( $post, 'WP_Post' ) ) {
		return false;
	}
	
	$filter_shortcodes = array( 'vehizo_filter', 'vehizo_vehicle_filter', 'vehizo_search' );
	foreach ( $filter_shortcodes as $shortcode ) {
		if ( has_shortcode( $post->post_content, $shortcode ) ) {
			return true;
		}
	}
	
	return false;
}

// Add dynamic CSS for customization options
add_action( 'wp_head', 'vehizo_add_dynamic_styles' );
function vehizo_add_dynamic_styles() {
	if ( ! is_singular( 'vehizo_vehicle' ) && ! is_post_type_archive( 'vehizo_vehicle' ) && ! vehizo_has_shortcode() ) {
		return;
	}
	
	$primary_color = get_option( 'vehizo_primary_color', '#0073aa' );
	$secondary_color = get_option( 'vehizo_secondary_color', '#005177' );
	$button_style = get_option( 'vehizo_button_style', 'rounded' );
	
	$custom_css = "
		:root {
			--vehizo-primary-color: " . esc_attr( $primary_color ) . ";
			--vehizo-secondary-color: " . esc_attr( $secondary_color ) . ";
		}
		
		.vehizo-btn-primary {
			background-color: var(--vehizo-primary-color);
			border-color: var(--vehizo-primary-color);
		}
		
		.vehizo-btn-primary:hover {
			background-color: var(--vehizo-secondary-color);
			border-color: var(--vehizo-secondary-color);
		}
		
		.vehizo-price {
			color: var(--vehizo-primary-color);
		}
	";
	
	if ( $button_style === 'square' ) {
		$custom_css .= "
			.vehizo-btn {
				border-radius: 0;
			}
		";
	}
	
	wp_add_inline_style( 'vehizo-frontend-style', $custom_css );
}



// Template Override for Single & Archive Vehicle
add_filter('template_include', function($template) {
    if (is_singular('vehizo_vehicle')) {
        $plugin_template = plugin_dir_path(__FILE__) . 'templates/single-vehizo_vehicle.php';
        if (file_exists($plugin_template)) { return $plugin_template; }
    }
    if (is_post_type_archive('vehizo_vehicle')) {
        $plugin_template = plugin_dir_path(__FILE__) . 'templates/archive-vehizo_vehicle.php';
        if (file_exists($plugin_template)) { return $plugin_template; }
    }
    return $template;
});

// Enqueue archive styles (WordPress.org Guidelines - no hardcoded styles)
add_action('wp_enqueue_scripts', function() {
    if (is_post_type_archive('vehizo_vehicle')) {
        // Enqueue base styles first
        wp_enqueue_style(
            'vehizo-base',
            plugins_url('assets/css/vehizo-base.css', __FILE__),
            array(),
            '4.1.4'
        );

        // Enqueue archive styles
        wp_enqueue_style(
            'vehizo-archive-vehicles',
            plugins_url('assets/css/archive-vehicles.css', __FILE__),
            array('vehizo-base'),
            '4.1.4'
        );

        // Add dynamic CSS from appearance settings
        if (class_exists('Vehizo_Filter_Settings')) {
            $dynamic_css = Vehizo_Filter_Settings::generate_dynamic_css();
            wp_add_inline_style('vehizo-base', $dynamic_css);
        }
    }
});

/**
 * NEU: Zählt die Aufrufe für einzelne Fahrzeuge.
 * Wird nur ausgeführt, wenn das Statistik-Modul aktiv ist.
 */
function vehizo_track_vehicle_views() {
    // Nur auf Fahrzeug-Detailseiten und wenn das Statistik-Modul aktiv ist
    if ( ! is_singular('vehizo_vehicle') || get_option('vehizo_statistics_module_active', '0') !== '1' ) {
        return;
    }

    // Bots und eingeloggte Admins ignorieren, um die Zählung nicht zu verfälschen
    if ( is_user_logged_in() && current_user_can('manage_options') ) {
        return;
    }
    // Einfache Bot-Erkennung anhand des User-Agents
    if ( isset( $_SERVER['HTTP_USER_AGENT'] ) && preg_match( '/bot|crawl|slurp|spider|mediapartners/i', sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) ) ) {
        return;
    }

    $post_id = get_the_ID();
    $count_key = '_vehizo_view_count';
    $count = (int) get_post_meta( $post_id, $count_key, true );
    update_post_meta( $post_id, $count_key, $count + 1 );
}

// =================================================================================
// ANPASSUNG: MODULARE INHALTE (KONTAKTFORMULAR)
// =================================================================================

add_action('vehizo_after_single_vehicle_content', 'vehizo_render_single_contact_form', 10);
add_action('wp_footer', 'vehizo_add_contact_popup_html');

/**
 * Rendert das Kontaktformular, wenn das Modul aktiv ist.
 *
 * @param int $post_id Die ID des aktuellen Fahrzeugs.
 */
function vehizo_render_single_contact_form($post_id) {
    if (get_option('vehizo_contact_module_active', '1') !== '1') {
        return;
    }
    // Hide the original form - it's now in the popup via CSS
    wp_add_inline_style('vehizo-single-vehicle', '.vehizo-single__contact-form { display: none !important; }');
}

/**
 * Add contact popup HTML to footer on vehicle pages
 */
function vehizo_add_contact_popup_html() {
    if (!is_singular('vehizo_vehicle') || get_option('vehizo_contact_module_active', '1') !== '1') {
        return;
    }

    // Sicherstellen, dass wir die echte Vehicle-ID haben
    global $post;
    $post_id = $post->ID;

    // Zusätzliche Prüfung: Ist es wirklich ein Vehicle?
    if (get_post_type($post_id) !== 'vehizo_vehicle') {
        // Fallback: Nach dem aktuellen Vehicle in der URL suchen
        $queried_object = get_queried_object();
        if ($queried_object && $queried_object->post_type === 'vehizo_vehicle') {
            $post_id = $queried_object->ID;
        } else {
            return; // Kein Vehicle gefunden
        }
    }
    ?>
    <div id="vehizo-contact-popup" class="vehizo-popup-overlay" style="display: none;">
        <div class="vehizo-popup-content">
            <div class="vehizo-popup-header">
                <h3><?php esc_html_e('Inquire About This Vehicle', 'vehizo-vehicle-management'); ?></h3>
                <button class="vehizo-popup-close" aria-label="<?php esc_attr_e('Close contact form', 'vehizo-vehicle-management'); ?>"></button>
            </div>
            <div class="vehizo-popup-body">
                <form method="POST" action="" id="vehizo-contact-popup-form">
                        <?php wp_nonce_field('vehizo_send_contact_form_' . $post_id, 'vehizo_contact_nonce'); ?>
                        <input type="hidden" name="vehizo_vehicle_id" value="<?php echo esc_attr($post_id); ?>">
                        <p class="form-row">
                            <label for="vehizo_contact_name_popup"><?php esc_html_e('Name', 'vehizo-vehicle-management'); ?></label>
                            <input type="text" id="vehizo_contact_name_popup" name="vehizo_contact_name" required>
                        </p>
                        <p class="form-row">
                            <label for="vehizo_contact_email_popup"><?php esc_html_e('Email', 'vehizo-vehicle-management'); ?></label>
                            <input type="email" id="vehizo_contact_email_popup" name="vehizo_contact_email" required>
                        </p>
                        <p class="form-row">
                            <label for="vehizo_contact_message_popup"><?php esc_html_e('Message', 'vehizo-vehicle-management'); ?></label>
                            <textarea id="vehizo_contact_message_popup" name="vehizo_contact_message" rows="5" required><?php
                            /* translators: %s: Vehicle title/name */
                            printf(esc_html__('I am interested in this vehicle: %s', 'vehizo-vehicle-management'), esc_html(get_the_title($post_id))); ?></textarea>
                        </p>
                        <p class="form-row">
                            <button type="submit" name="vehizo_contact_submit" class="vehizo-button vehizo-button--accent"><?php esc_html_e('Send Inquiry', 'vehizo-vehicle-management'); ?></button>
                        </p>
                </form>
            </div>
        </div>
    </div>

    <!-- Success Modal -->
    <div id="vehizo-success-modal" class="vehizo-popup-overlay" style="display: none;">
        <div class="vehizo-popup-content vehizo-success-modal">
            <div class="vehizo-popup-header">
                <h3><?php esc_html_e('Message Sent Successfully!', 'vehizo-vehicle-management'); ?></h3>
                <button class="vehizo-popup-close" aria-label="<?php esc_attr_e('Close success message', 'vehizo-vehicle-management'); ?>"></button>
            </div>
            <div class="vehizo-popup-body">
                <div class="vehizo-success-content">
                    <div class="vehizo-success-icon">✅</div>
                    <p><?php esc_html_e('Thank you for your interest! We have received your inquiry about this vehicle and will get back to you shortly.', 'vehizo-vehicle-management'); ?></p>
                    <button class="vehizo-button vehizo-button--primary" id="vehizo-success-close"><?php esc_html_e('Close', 'vehizo-vehicle-management'); ?></button>
                </div>
            </div>
        </div>
    </div>

    <!-- Wishlist View Modal -->
    <div id="vehizo-wishlist-modal" class="vehizo-popup-overlay" style="display: none;">
        <div class="vehizo-popup-content vehizo-wishlist-modal">
            <div class="vehizo-popup-header">
                <h3><?php esc_html_e('Your Vehicle Wishlist', 'vehizo-vehicle-management'); ?></h3>
                <button class="vehizo-popup-close" aria-label="<?php esc_attr_e('Close wishlist', 'vehizo-vehicle-management'); ?>"></button>
            </div>
            <div class="vehizo-popup-body">
                <div id="vehizo-wishlist-content">
                    <div class="vehizo-wishlist-loading">
                        <div class="vehizo-loading-spinner"></div>
                        <p><?php esc_html_e('Loading your wishlist...', 'vehizo-vehicle-management'); ?></p>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php
}

/**
 * AJAX Handler für Contact Form
 */
// AJAX Handlers for Contact Form - moved here for better structure
add_action('wp_ajax_vehizo_send_contact_form', 'vehizo_ajax_send_contact_form');
add_action('wp_ajax_nopriv_vehizo_send_contact_form', 'vehizo_ajax_send_contact_form');

function vehizo_ajax_send_contact_form() {
    // Get vehicle_id and email for later use
    $vehicle_id = isset($_POST['vehizo_vehicle_id']) ? absint($_POST['vehizo_vehicle_id']) : 0;
    $email = isset($_POST['vehizo_contact_email']) ? sanitize_email(wp_unslash($_POST['vehizo_contact_email'])) : '';

    // Check if contact module is active
    if (get_option('vehizo_contact_module_active', '1') !== '1') {
        wp_send_json_error(array('message' => esc_html__('Contact form is currently disabled.', 'vehizo-vehicle-management')));
        return;
    }

    // Vehicle ID already sanitized above
    if ($vehicle_id === 0) {
        wp_send_json_error(array('message' => esc_html__('Invalid vehicle ID.', 'vehizo-vehicle-management')));
        return;
    }

    // Verify Nonce - das Form verwendet die originale Nonce vom Template
    $nonce = isset($_POST['vehizo_contact_nonce']) ? sanitize_text_field(wp_unslash($_POST['vehizo_contact_nonce'])) : '';
    if (!wp_verify_nonce($nonce, 'vehizo_send_contact_form_' . $vehicle_id)) {
        wp_send_json_error(array(
            'message' => esc_html__('Security check failed.', 'vehizo-vehicle-management'),
            'debug' => array(
                'vehicle_id' => $vehicle_id,
                'nonce_action' => 'vehizo_send_contact_form_' . $vehicle_id,
                'nonce_received' => !empty($nonce)
            )
        ));
        return;
    }

    // Sanitize all inputs
    $name = isset($_POST['vehizo_contact_name']) ? sanitize_text_field(wp_unslash($_POST['vehizo_contact_name'])) : '';
    $email = isset($_POST['vehizo_contact_email']) ? sanitize_email(wp_unslash($_POST['vehizo_contact_email'])) : '';
    $message_content = isset($_POST['vehizo_contact_message']) ? sanitize_textarea_field(wp_unslash($_POST['vehizo_contact_message'])) : '';

    // Validation
    if (empty($name)) {
        wp_send_json_error(array('message' => esc_html__('Name is required.', 'vehizo-vehicle-management')));
        return;
    }

    if (empty($email) || !is_email($email)) {
        wp_send_json_error(array('message' => esc_html__('Valid email address is required.', 'vehizo-vehicle-management')));
        return;
    }

    if (empty($message_content)) {
        wp_send_json_error(array('message' => esc_html__('Message is required.', 'vehizo-vehicle-management')));
        return;
    }

    // Check if vehicle exists
    $vehicle_post = get_post($vehicle_id);
    if (!$vehicle_post || $vehicle_post->post_type !== 'vehizo_vehicle') {
        wp_send_json_error(array(
            'message' => esc_html__('Vehicle not found.', 'vehizo-vehicle-management'),
            'debug' => array(
                'vehicle_id' => $vehicle_id,
                'post_exists' => !empty($vehicle_post),
                'post_type' => $vehicle_post ? $vehicle_post->post_type : 'none',
                'expected_type' => 'vehizo_vehicle'
            )
        ));
        return;
    }

    // Send email
    $vehicle_title = get_the_title($vehicle_id);
    $contact_email = get_option('vehizo_contact_email', '');
    $to = !empty($contact_email) ? $contact_email : get_option('admin_email');
    /* translators: %s: Vehicle title/name */
    $subject = sprintf(__('Vehicle Inquiry: %s', 'vehizo-vehicle-management'), $vehicle_title);

    $message = __('New vehicle inquiry received!', 'vehizo-vehicle-management') . "\n\n";
    /* translators: %s: Vehicle title/name */
    $message .= sprintf(__('Vehicle: %s', 'vehizo-vehicle-management'), $vehicle_title) . "\n";
    /* translators: %s: Customer name */
    $message .= sprintf(__('Name: %s', 'vehizo-vehicle-management'), $name) . "\n";
    /* translators: %s: Customer email address */
    $message .= sprintf(__('E-Mail: %s', 'vehizo-vehicle-management'), $email) . "\n\n";
    /* translators: %s: Customer message content */
    $message .= sprintf(__('Message: %s', 'vehizo-vehicle-management'), $message_content) . "\n";

    $headers = ['Reply-To: ' . $email];
    $mail_sent = wp_mail($to, $subject, $message, $headers);

    if ($mail_sent) {
        wp_send_json_success(array(
            'message' => esc_html__('Your message has been sent successfully!', 'vehizo-vehicle-management')
        ));
    } else {
        wp_send_json_error(array(
            'message' => esc_html__('Failed to send message. Please try again later.', 'vehizo-vehicle-management')
        ));
    }
}

// Contact Form E-Mail Handler (DEAKTIVIERT - nur AJAX wird verwendet)
/*
add_action('template_redirect', function() {
    // Only proceed if our specific form has been submitted AND it's not an AJAX request
    if ( ! is_singular('vehizo_vehicle') || ! isset( $_POST['vehizo_contact_submit'] ) || wp_doing_ajax() ) {
        return;
    }

    // Sanitize the vehicle ID first to use it in the nonce check.
    $vehicle_id = isset( $_POST['vehizo_vehicle_id'] ) ? absint( $_POST['vehizo_vehicle_id'] ) : 0;
    if ( $vehicle_id === 0 ) {
        // Optional: Redirect with an error if the ID is missing.
        return;
    }
    
    // **Step 1: Verify Nonce**
    $nonce = isset( $_POST['vehizo_contact_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['vehizo_contact_nonce'] ) ) : '';
    if ( ! wp_verify_nonce( $nonce, 'vehizo_send_contact_form_' . $vehicle_id ) ) {
        wp_die( esc_html__( 'Security check failed.', 'vehizo-vehicle-management' ) );
    }

    // **Step 2: Sanitize all other inputs**
    $name            = isset( $_POST['vehizo_contact_name'] ) ? sanitize_text_field( wp_unslash( $_POST['vehizo_contact_name'] ) ) : '';
    $email           = isset( $_POST['vehizo_contact_email'] ) ? sanitize_email( wp_unslash( $_POST['vehizo_contact_email'] ) ) : '';
    $message_content = isset( $_POST['vehizo_contact_message'] ) ? sanitize_textarea_field( wp_unslash( $_POST['vehizo_contact_message'] ) ) : '';

    // **Step 3: Basic validation**
    if ( ! empty( $name ) && ! empty( $email ) && is_email( $email ) && ! empty( $message_content ) ) {
        $vehicle_title = get_the_title( $vehicle_id );
        $to             = get_option( 'admin_email' );
        $subject        = sprintf( __( 'Vehicle Inquiry: %s', 'vehizo-vehicle-management' ), $vehicle_title );

        $message  = __( 'New vehicle inquiry received!', 'vehizo-vehicle-management' ) . "\n\n";
        $message .= sprintf( __( 'Vehicle: %s', 'vehizo-vehicle-management' ), $vehicle_title ) . "\n";
        $message .= sprintf( __( 'Name: %s', 'vehizo-vehicle-management' ), $name ) . "\n";
        $message .= sprintf( __( 'E-Mail: %s', 'vehizo-vehicle-management' ), $email ) . "\n\n";
        $message .= sprintf( __( 'Message: %s', 'vehizo-vehicle-management' ), $message_content ) . "\n";

        $headers = [ 'Reply-To: ' . $email ];
        wp_mail( $to, $subject, $message, $headers );

        // Legacy support: Only redirect if not AJAX request
        if (!wp_doing_ajax()) {
            wp_redirect( add_query_arg( 'contact_success', '1', get_permalink( $vehicle_id ) ) . '#vehizo-contact-form' );
            exit;
        }
    }
});
*/

// Activation & Deactivation
register_activation_hook(__FILE__, function() {
    // Register CPT to make it available for the rewrite rule flush.
    register_post_type('vehizo_vehicle', [
        'label' => __('Vehicles', 'vehizo-vehicle-management'),
        'public' => true,
        'rewrite' => ['slug' => 'vehicle']
    ]);
    flush_rewrite_rules();
    
    // Set default options
    add_option('vehizo_contact_module_active', '1');
    add_option('vehizo_statistics_module_active', '0');
    add_option('vehizo_wishlist_module_active', '1');
    add_option('vehizo_primary_color', '#0073aa');
    add_option('vehizo_secondary_color', '#005177');
    add_option('vehizo_button_style', 'rounded');

    // Migrate German values to English (one-time conversion)
    if (!get_option('vehizo_german_values_migrated')) {
        vehizo_migrate_german_values_to_english();
        add_option('vehizo_german_values_migrated', '1');
    }
});
register_deactivation_hook(__FILE__, function() {
    // Clear any scheduled cron jobs
    $timestamp = wp_next_scheduled('vehizo_cron_import');
    if ($timestamp) {
        wp_unschedule_event($timestamp, 'vehizo_cron_import');
    }
    flush_rewrite_rules();
});

/**
 * Migrate German meta values to English equivalents
 * One-time conversion for existing data
 */
function vehizo_migrate_german_values_to_english() {
    global $wpdb;

    // Define German to English mappings
    $fuel_mappings = array(
        'Benzin' => 'Gasoline',
        'Elektro' => 'Electric'
    );

    $transmission_mappings = array(
        'Schaltgetriebe' => 'Manual',
        'Automatik' => 'Automatic',
        'Halbautomatik' => 'Semi-Automatic'
    );

    $vehicle_type_mappings = array(
        'Limousine' => 'Sedan',
        'Kombi' => 'Station Wagon',
        'Cabrio' => 'Convertible'
    );

    $total_updates = 0;

    // Update fuel types
    foreach ($fuel_mappings as $german => $english) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration during activation, bulk update more efficient than individual post meta calls
        $result = $wpdb->update(
            $wpdb->postmeta,
            array('meta_value' => $english),
            array('meta_key' => 'vehizo_fuelType', 'meta_value' => $german)
        );
        if ($result !== false) $total_updates += $result;
    }

    // Update transmission types
    foreach ($transmission_mappings as $german => $english) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration during activation, bulk update more efficient than individual post meta calls
        $result = $wpdb->update(
            $wpdb->postmeta,
            array('meta_value' => $english),
            array('meta_key' => 'vehizo_transmission', 'meta_value' => $german)
        );
        if ($result !== false) $total_updates += $result;
    }

    // Update vehicle types
    foreach ($vehicle_type_mappings as $german => $english) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration during activation, bulk update more efficient than individual post meta calls
        $result = $wpdb->update(
            $wpdb->postmeta,
            array('meta_value' => $english),
            array('meta_key' => 'vehizo_vehicleType', 'meta_value' => $german)
        );
        if ($result !== false) $total_updates += $result;
    }

    return $total_updates;
}

// Force migration immediately if not done yet
add_action('admin_init', function() {
    if (!get_option('vehizo_german_values_migrated')) {
        $updated_count = vehizo_migrate_german_values_to_english();
        update_option('vehizo_german_values_migrated', '1');

        // Add admin notice
        add_action('admin_notices', function() use ($updated_count) {
            echo '<div class="notice notice-success is-dismissible">';
            echo '<p><strong>Vehizo:</strong> Migrated ' . esc_html($updated_count) . ' German values to English successfully!</p>';
            echo '</div>';
        });
    }
});



// Save Vehicle Data
add_action('save_post_vehizo_vehicle', function($post_id, $post) {
    // Early bail: If this is an autosave, our form has not been submitted.
    if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
        return;
    }

    // Early bail: Check user permissions first (before processing any POST data)
    if ( ! current_user_can( 'edit_post', $post_id ) ) {
        return;
    }

    // Early bail: Only process if POST data is present
    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce is checked below before processing any data
    if ( empty( $_POST ) ) {
        return;
    }

    // **Nonce verification - REQUIRED before accessing any POST data**
    // Check for custom editor nonce OR standard WordPress editor nonce
    $nonce_verified = false;

    if ( isset( $_POST['vehizo_vehicle_editor_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['vehizo_vehicle_editor_nonce'] ) ), 'vehizo_vehicle_editor' ) ) {
        $nonce_verified = true;
    } elseif ( isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'update-post_' . $post_id ) ) {
        $nonce_verified = true;
    }

    // If nonce verification failed, bail immediately
    if ( ! $nonce_verified ) {
        return;
    }

    // At this point: User is authorized, nonce is verified, we can safely process POST data

    // Text fields that need sanitize_text_field
    $text_fields = ['make', 'model', 'fuelType', 'transmission', 'vehicleType', 'vin', 'color', 'doors', 'emissionClass', 'consumption', 'next_inspection'];
    foreach ($text_fields as $field) {
        $post_key = 'vehizo_' . $field;
        if (isset($_POST[$post_key])) {
            $value = sanitize_text_field(wp_unslash($_POST[$post_key]));
            update_post_meta($post_id, $post_key, $value);
        }
    }

    // Integer fields that need absint
    $int_fields = ['price', 'firstRegistration', 'mileage', 'power', 'engine_displacement', 'seats', 'number_of_owners'];
    foreach ($int_fields as $field) {
        $post_key = 'vehizo_' . $field;
        if (isset($_POST[$post_key])) {
            $value = absint(wp_unslash($_POST[$post_key]));
            update_post_meta($post_id, $post_key, $value);
        }
    }

    // Special handling for vehicle_status with whitelist validation
    if (isset($_POST['vehizo_vehicle_status'])) {
        $value = sanitize_text_field(wp_unslash($_POST['vehizo_vehicle_status']));
        // Only allow specific values
        if (in_array($value, ['available', 'reserved', 'sold'], true)) {
            update_post_meta($post_id, 'vehizo_vehicle_status', $value);
        }
    }
    
    if (isset($_POST['vehizo_gallery'])) {
        $gallery_ids_raw = sanitize_text_field(wp_unslash($_POST['vehizo_gallery']));
        // Sanitize to ensure it's a comma-separated list of positive integers.
        $gallery_ids = implode(',', array_filter(array_map('absint', explode(',', $gallery_ids_raw))));
        update_post_meta($post_id, 'vehizo_gallery', $gallery_ids);
    }
}, 10, 2);

// Admin Columns for Vehicle Overview
add_filter('manage_vehizo_vehicle_posts_columns', function($columns) {
    $columns['vehizo_price'] = __('Price', 'vehizo-vehicle-management');
    return $columns;
});

add_action('manage_vehizo_vehicle_posts_custom_column', function($column, $post_id) {
    if ('vehizo_price' === $column) {
        $price = get_post_meta($post_id, 'vehizo_price', true);
        if (!empty($price) && is_numeric($price)) {
            echo esc_html(number_format_i18n(absint($price), 0) . ' €');
        } else {
            echo '—';
        }
    }
}, 10, 2);

// AJAX Handler for Free Modules
add_action('wp_ajax_vehizo_toggle_module', function() {
    // **Step 1: Verify Nonce**
    $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
    if ( ! wp_verify_nonce( $nonce, 'vehizo_toggle_module' ) ) {
        wp_send_json_error( [ 'message' => __( 'Nonce verification failed.', 'vehizo-vehicle-management' ) ] );
    }

    // **Step 2: Verify Permissions**
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( [ 'message' => __( 'You do not have permission to perform this action.', 'vehizo-vehicle-management' ) ] );
    }
    
    // **Step 3: Sanitize Inputs**
    $option = isset( $_POST['option'] ) ? sanitize_key( $_POST['option'] ) : '';
    $value  = isset( $_POST['value'] ) ? sanitize_key( $_POST['value'] ) : '';
    
    $allowed_options = [
        'vehizo_contact_module_active',
        'vehizo_statistics_module_active',
        'vehizo_wishlist_module_active'
    ];
    
    if ( in_array( $option, $allowed_options, true ) ) {
        update_option( $option, $value );
        wp_send_json_success( [ 'message' => __( 'Setting saved.', 'vehizo-vehicle-management' ) ] );
    } else {
        wp_send_json_error( [ 'message' => __( 'Invalid option specified.', 'vehizo-vehicle-management' ) ] );
    }
});

// Intercepts "Edit" links and redirects them to the integrated dashboard.
// Intercepts "Edit" links and redirects them to the integrated dashboard.
add_filter('get_edit_post_link', function($link, $post_id) {
    // Nur bei unserem Post-Typ im Admin-Bereich handeln
    if (get_post_type($post_id) === 'vehizo_vehicle' && is_admin()) {
        
        // KORREKTUR: Wir prüfen die aktuelle Seite, um eine Endlosschleife sicher zu verhindern.
        // Wir leiten NUR von der Standard-Übersichtsseite ("edit.php") um.
        global $pagenow;
        if ( $pagenow === 'edit.php' ) {
            return admin_url('admin.php?page=vehizo-dashboard&tab=editor&fahrzeug_id=' . absint($post_id));
        }
    }
    
    // In allen anderen Fällen (z.B. im Editor selbst) den normalen Link zurückgeben.
    return $link;
}, 10, 2);