<?php
/**
 * Vehizo Wishlist Module
 *
 * Handles vehicle wishlist functionality with localStorage and comparison table
 *
 * @package      Vehizo
 * @version      4.2.1
 * @author       Thorsten Glander
 * @since        1.0.0
 */

// Sicherheitscheck - Direkter Zugriff verhindern
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Vehizo_Wishlist_Module')) {
    
    /**
     * Vehizo Wishlist Module Class
     *
     * @since 1.0.0
     */
    class Vehizo_Wishlist_Module {

        /**
         * Initialize module actions and filters
         *
         * @since 1.0.0
         */
        public static function init() {
            // Only initialize if module is active
            if (get_option('vehizo_wishlist_module_active', '1') !== '1') {
                return;
            }

            // ONLY enable button rendering, disable scripts to avoid conflicts
            // add_action('wp_enqueue_scripts', array(__CLASS__, 'enqueue_assets')); // DISABLED
            add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_admin_assets'));
            add_shortcode('vehizo_wishlist', array(__CLASS__, 'render_wishlist_shortcode')); // ENABLED for wishlist functionality
            add_action('wp_ajax_nopriv_vehizo_get_wishlist_vehicles', array(__CLASS__, 'ajax_get_wishlist_vehicles')); // ENABLED for wishlist functionality
            add_action('wp_ajax_vehizo_get_wishlist_vehicles', array(__CLASS__, 'ajax_get_wishlist_vehicles')); // ENABLED for wishlist functionality
            // add_action('wp_footer', array(__CLASS__, 'add_floating_button_html')); // DISABLED

            // Hook for adding the button to the single vehicle page - KEEP THIS
            add_action('vehizo_single_vehicle_actions', array(__CLASS__, 'add_wishlist_button_to_single_page'), 10);

            // Register cleanup hook
            add_action('vehizo_cleanup_wishlist', array(__CLASS__, 'cleanup_old_data'));
        }
/**
         * Enqueue necessary CSS and JavaScript for frontend
         *
         * @since 1.0.0
         */
        public static function enqueue_assets() {
            // Only enqueue on relevant pages
            if (!self::should_load_assets()) {
                return;
            }

            self::enqueue_styles();
            self::enqueue_scripts();
        }

        /**
         * Enqueue admin assets
         *
         * @since 1.0.0
         * @param string $hook Current admin page hook
         */
        public static function enqueue_admin_assets($hook) {
            // Only load on relevant admin pages
            if (in_array($hook, array('dashboard', 'plugins.php', 'vehizo_page_vehizo-settings'))) {
                wp_enqueue_style(
                    'vehizo-wishlist-admin',
                    plugin_dir_url(__FILE__) . 'assets/css/wishlist-admin.css',
                    array(),
                    VEHIZO_PLUGIN_VERSION,
                    'all'
                );
            }
        }

        /**
         * Check if assets should be loaded on current page
         *
         * @since 1.0.0
         * @return bool
         */
        private static function should_load_assets() {
            global $post;
            
            // Load on vehicle pages, archive pages, and pages with wishlist shortcode
            return (
                is_singular('vehizo_vehicle') ||
                is_post_type_archive('vehizo_vehicle') ||
                (is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'vehizo_wishlist')) ||
                is_page('merkliste') // Keep slug for user-facing pages
            );
        }

        /**
         * Enqueue CSS styles
         *
         * @since 1.0.0
         */
        private static function enqueue_styles() {
            // Main wishlist stylesheet
            wp_enqueue_style(
                'vehizo-wishlist-frontend',
                VEHIZO_PLUGIN_URL . 'assets/css/wishlist-frontend.css',
                array(),
                VEHIZO_PLUGIN_VERSION,
                'all'
            );

            // Add dynamic CSS for currency formatting
            $dynamic_css = self::get_dynamic_css();
            if (!empty($dynamic_css)) {
                wp_add_inline_style('vehizo-wishlist-frontend', $dynamic_css);
            }
        }

        /**
         * Get dynamic CSS based on plugin settings
         *
         * @since 1.0.0
         * @return string Dynamic CSS
         */
        private static function get_dynamic_css() {
            $css = '';
            
            // Currency symbol positioning
            $currency_position = get_option('vehizo_currency_position', 'after');
            if ($currency_position === 'before') {
                $css .= '.comparison-table .price::before { content: "' . esc_html(get_option('vehizo_currency_symbol', '€')) . ' "; }';
            }
            
            // Brand colors if set
            $primary_color = get_option('vehizo_primary_color');
            if (!empty($primary_color)) {
                $css .= '.floating-park-button { background: linear-gradient(135deg, ' . esc_html($primary_color) . ' 0%, ' . esc_html($primary_color) . ' 100%); }';
            }
            
            return $css;
        }

        /**
         * Enqueue JavaScript
         *
         * @since 1.0.0
         */
        private static function enqueue_scripts() {
            // Main wishlist script
            wp_enqueue_script(
                'vehizo-wishlist-frontend',
                VEHIZO_PLUGIN_URL . 'assets/js/wishlist-frontend.js',
                array('jquery'),
                VEHIZO_PLUGIN_VERSION,
                true
            );

            // Wishlist counter initialization (WordPress.org Guidelines - no hardcoded scripts)
            wp_enqueue_script(
                'vehizo-wishlist-counter-init',
                VEHIZO_PLUGIN_URL . 'assets/js/wishlist-counter-init.js',
                array('vehizo-wishlist-frontend'),
                VEHIZO_PLUGIN_VERSION,
                true
            );

            // Localize script with dynamic data
            wp_localize_script(
                'vehizo-wishlist-frontend',
                'vehizoWishlist',
                array(
                    'ajaxUrl' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('vehizo_wishlist_nonce'),
                    'maxVehicles' => self::get_max_vehicles(),
                    'strings' => array(
                        'save' => __('Save', 'vehizo-vehicle-management'),
                        'saved' => __('Saved', 'vehizo-vehicle-management'),
                        'addToWishlist' => __('Add to Wishlist', 'vehizo-vehicle-management'),
                        'removeFromWishlist' => __('Remove from Wishlist', 'vehizo-vehicle-management'),
                        'viewWishlist' => __('View Wishlist', 'vehizo-vehicle-management'),
                        'onWishlist' => __('On Wishlist', 'vehizo-vehicle-management'),
                        'loadingSavedVehicles' => __('Loading saved vehicles...', 'vehizo-vehicle-management'),
                        'loadingWishlist' => __('Loading wishlist...', 'vehizo-vehicle-management'),
                        'errorOccurred' => __('An error occurred.', 'vehizo-vehicle-management'),
                        'tryAgainLater' => __('Please try again later.', 'vehizo-vehicle-management'),
                        'checkConnection' => __('Please check your internet connection and try again.', 'vehizo-vehicle-management'),
                        'maxVehiclesReached' => __('Maximum number of vehicles reached. Please remove some vehicles first.', 'vehizo-vehicle-management'),
                        /* translators: %s: Vehicle title/name */
                        'removeFromList' => __('Remove %s from wishlist', 'vehizo-vehicle-management'),
                        /* translators: %s: Vehicle title/name */
                        'viewDetails' => __('View details for %s', 'vehizo-vehicle-management')
                    ),
                    'storageKeys' => array(
                        'wishlist' => 'vehizo_wishlist_vehicles',
                        'lastPage' => 'vehizo_last_vehicle_page'
                    )
                )
            );
        }
/**
         * Render wishlist shortcode
         *
         * @since 1.0.0
         * @param array $atts Shortcode attributes
         * @return string HTML output
         */
        public static function render_wishlist_shortcode($atts = array()) {
            $atts = shortcode_atts(array(
                'show_header' => 'true',
                'max_vehicles' => 10
            ), $atts, 'vehizo_wishlist');

            $container_class = $atts['show_header'] === 'false' ? 'no-header' : '';
            
            return sprintf(
                '<div id="vehizo-wishlist-container" class="%s"><div class="wishlist-loading"><div class="wishlist-spinner"></div>%s</div></div>',
                esc_attr($container_class),
                esc_html__('Loading wishlist...', 'vehizo-vehicle-management')
            );
        }

        /**
         * AJAX handler to get wishlist vehicles data
         *
         * @since 1.0.0
         */
        public static function ajax_get_wishlist_vehicles() {
            // Verify nonce
            if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'vehizo_wishlist_nonce')) {
                wp_send_json_error(array('message' => esc_html__('Security check failed', 'vehizo-vehicle-management')));
                return;
            }

            $ids_json = isset($_POST['vehicle_ids']) ? sanitize_text_field(wp_unslash($_POST['vehicle_ids'])) : '[]';
            $ids = array_filter(array_map('absint', json_decode($ids_json, true) ?: array()));

            // Limit number of vehicles
            $ids = array_slice($ids, 0, self::get_max_vehicles()); // Use self::get_max_vehicles()

            // Get actual vehicle data instead of rendering HTML
            $vehicles = self::get_wishlist_vehicles_data($ids);

            wp_send_json_success(array('vehicles' => $vehicles));
        }

        /**
         * Get wishlist vehicles data for AJAX response
         *
         * @since 1.0.0
         * @param array $ids Vehicle IDs
         * @return array Vehicle data
         */
        private static function get_wishlist_vehicles_data($ids) {
            if (empty($ids)) {
                return array();
            }

            $query = new WP_Query(array(
                'post_type' => 'vehizo_vehicle',
                'post__in' => $ids,
                'posts_per_page' => -1,
                'orderby' => 'post__in',
                'post_status' => 'publish',
                'no_found_rows' => true,
                'update_post_meta_cache' => true,
                'update_post_term_cache' => false
            ));

            $vehicles = array();

            while ($query->have_posts()) {
                $query->the_post();
                $post_id = get_the_ID();
                $vehicles[] = self::prepare_vehicle_data($post_id);
            }

            wp_reset_postdata();

            return $vehicles;
        }
/**
         * Render comparison table content
         *
         * @since 1.0.0
         * @param array $ids Vehicle IDs
         */
        private static function render_comparison_content($ids) {
            // Get fallback URL for back button
            $back_url = self::get_fallback_url();
            ?>
            <div class="wishlist-header">
                <h2><?php esc_html_e('Your Vehicle Wishlist', 'vehizo-vehicle-management'); ?></h2>
                <a href="<?php echo esc_url($back_url); ?>" 
                   id="btn-back-to-vehicles" 
                   class="btn-back"
                   aria-label="<?php esc_attr_e('Back to vehicle overview', 'vehizo-vehicle-management'); ?>">
                    ← <?php esc_html_e('Back to Overview', 'vehizo-vehicle-management'); ?>
                </a>
            </div>
            <?php

            if (empty($ids)) {
                ?>
                <div class="no-results">
                    <h3><?php esc_html_e('The wishlist is empty.', 'vehizo-vehicle-management'); ?></h3>
                    <p><?php esc_html_e('Click "Save" on a vehicle to add it here for comparison.', 'vehizo-vehicle-management'); ?></p>
                </div>
                <?php
            } else {
                self::render_comparison_table($ids);
            }
        }

        /**
         * Render vehicle comparison table
         *
         * @since 1.0.0
         * @param array $ids Vehicle IDs
         */
        private static function render_comparison_table($ids) {
            $query = new WP_Query(array(
                'post_type' => 'vehizo_vehicle',
                'post__in' => $ids,
                'posts_per_page' => -1,
                'orderby' => 'post__in',
                'no_found_rows' => true,
                'update_post_meta_cache' => true,
                'update_post_term_cache' => false
            ));
            
            $vehicles = array();
            
            while ($query->have_posts()) {
                $query->the_post();
                $post_id = get_the_ID();
                $vehicles[$post_id] = self::prepare_vehicle_data($post_id);
            }
            
            wp_reset_postdata();
            
            if (empty($vehicles)) {
                ?>
                <div class="no-results">
                    <h3><?php esc_html_e('No vehicles found.', 'vehizo-vehicle-management'); ?></h3>
                    <p><?php esc_html_e('The selected vehicles may have been removed or are no longer available.', 'vehizo-vehicle-management'); ?></p>
                </div>
                <?php
                return;
            }
            ?>
            
            <div class="comparison-table-wrapper">
                <table class="comparison-table" role="table" aria-label="<?php esc_attr_e('Vehicle comparison table', 'vehizo-vehicle-management'); ?>">
                    <thead>
                        <tr>
                            <th scope="col"><?php esc_html_e('Property', 'vehizo-vehicle-management'); ?></th>
                            <?php foreach ($vehicles as $vehicle): ?>
                                <th scope="col" class="vehicle-col">
                                    <a href="<?php echo esc_url($vehicle['permalink']); ?>"
                                       aria-label="<?php
                                       /* translators: %s: Vehicle title/name */
                                       echo esc_attr(sprintf(esc_html__('View details for %s', 'vehizo-vehicle-management'), $vehicle['title'])); ?>">
                                        <img src="<?php echo esc_url($vehicle['image_url']); ?>" 
                                             alt="<?php echo esc_attr($vehicle['title']); ?>"
                                             loading="lazy">
                                        <div class="title"><?php echo esc_html($vehicle['title']); ?></div>
                                    </a>
                                </th>
                            <?php endforeach; ?>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        $comparison_fields = self::get_comparison_fields();
                        foreach ($comparison_fields as $key => $label): 
                        ?>
                            <tr>
                                <th scope="row"><?php echo esc_html($label); ?></th>
                                <?php foreach ($vehicles as $vehicle): ?>
                                    <td><?php echo esc_html($vehicle[$key] ?: '—'); ?></td>
                                <?php endforeach; ?>
                            </tr>
                        <?php endforeach; ?>
                        
                        <tr>
                            <th scope="row"><?php esc_html_e('Action', 'vehizo-vehicle-management'); ?></th>
                            <?php foreach ($vehicles as $vehicle): ?>
                                <td>
                                    <a href="#"
                                       class="remove-from-list"
                                       data-vehicle-id="<?php echo esc_attr($vehicle['id']); ?>"
                                       aria-label="<?php
                                       /* translators: %s: Vehicle title/name */
                                       echo esc_attr(sprintf(esc_html__('Remove %s from wishlist', 'vehizo-vehicle-management'), $vehicle['title'])); ?>">
                                        <?php esc_html_e('Remove', 'vehizo-vehicle-management'); ?>
                                    </a>
                                </td>
                            <?php endforeach; ?>
                        </tr>
                    </tbody>
                </table>
            </div>
            <?php
        }
/**
         * Prepare vehicle data for comparison table
         *
         * @since 1.0.0
         * @param int $post_id Vehicle post ID
         * @return array Vehicle data
         */
        private static function prepare_vehicle_data($post_id) {
            $price = get_post_meta($post_id, 'vehizo_price', true);
            $mileage = get_post_meta($post_id, 'vehizo_mileage', true);
            $power = get_post_meta($post_id, 'vehizo_power', true);
            
            // Get image with fallback
            $image_url = '';
            if (has_post_thumbnail($post_id)) {
                $image_url = get_the_post_thumbnail_url($post_id, 'medium');
            }
            
            if (empty($image_url)) {
                $image_url = apply_filters('vehizo_default_vehicle_image', 
                    plugin_dir_url(__FILE__) . 'assets/images/no-image.png'
                );
            }
            
            return array(
                'id' => $post_id,
                'title' => get_the_title($post_id),
                'permalink' => get_permalink($post_id),
                'image_url' => $image_url,
                'price' => self::format_price($price),
                'year' => get_post_meta($post_id, 'vehizo_firstRegistration', true),
                'mileage' => self::format_mileage($mileage),
                'power' => self::format_power($power),
                'fuel' => get_post_meta($post_id, 'vehizo_fuelType', true),
                'transmission' => get_post_meta($post_id, 'vehizo_transmission', true),
                'color' => get_post_meta($post_id, 'vehizo_color', true),
                'vehicleType' => get_post_meta($post_id, 'vehizo_vehicleType', true)
            );
        }

        /**
         * Get comparison fields with labels
         *
         * @since 1.0.0
         * @return array Field mapping
         */
        private static function get_comparison_fields() {
            return apply_filters('vehizo_wishlist_comparison_fields', array(
                'price' => __('Price', 'vehizo-vehicle-management'),
                'year' => __('First Registration', 'vehizo-vehicle-management'),
                'mileage' => __('Mileage', 'vehizo-vehicle-management'),
                'power' => __('Power', 'vehizo-vehicle-management'),
                'fuel' => __('Fuel Type', 'vehizo-vehicle-management'),
                'transmission' => __('Transmission', 'vehizo-vehicle-management'),
                'color' => __('Color', 'vehizo-vehicle-management'),
                'vehicleType' => __('Vehicle Type', 'vehizo-vehicle-management')
            ));
        }

        /**
         * Format price for display
         *
         * @since 1.0.0
         * @param mixed $price Price value
         * @return string Formatted price
         */
        private static function format_price($price) {
            if (empty($price) || !is_numeric($price)) {
                return __('On request', 'vehizo-vehicle-management');
            }
            
            $currency_symbol = get_option('vehizo_currency_symbol', '€');
            $currency_position = get_option('vehizo_currency_position', 'after');
            $formatted_number = number_format_i18n(intval($price));
            
            if ($currency_position === 'before') {
                return $currency_symbol . ' ' . $formatted_number;
            } else {
                return $formatted_number . ' ' . $currency_symbol;
            }
        }

        /**
         * Format mileage for display
         */
        private static function format_mileage($mileage) {
            if (empty($mileage) || !is_numeric($mileage)) return '';
            return number_format_i18n(intval($mileage)) . ' ' . __('km', 'vehizo-vehicle-management');
        }

        /**
         * Format power for display
         */
        private static function format_power($power) {
            if (empty($power) || !is_numeric($power)) return '';
            return intval($power) . ' ' . __('PS', 'vehizo-vehicle-management');
        }

        /**
         * Get fallback URL for back button
         */
        private static function get_fallback_url() {
            $overview_page_id = get_option('vehizo_overview_page_id');
            if (!empty($overview_page_id) && get_post_status($overview_page_id) === 'publish') {
                return get_permalink($overview_page_id);
            }
            $archive_link = get_post_type_archive_link('vehizo_vehicle');
            if ($archive_link) return $archive_link;
            return home_url('/');
        }

        /**
         * Add floating button HTML to footer
         */
        public static function add_floating_button_html() {
            if (!self::should_load_assets()) return;
            $page_url = self::get_wishlist_page_url();
            printf(
                '<a href="%s" id="floating-park-button" class="floating-park-button" aria-label="%s" role="button">%s <span class="parked-vehicle-count">0</span></a>',
                esc_url($page_url),
                esc_attr__('View wishlist', 'vehizo-vehicle-management'),
                esc_html__('To Wishlist', 'vehizo-vehicle-management')
            );
        }

        /**
         * Get wishlist page URL
         */
        private static function get_wishlist_page_url() {
            $page = get_page_by_path('merkliste');
            if ($page && get_post_status($page) === 'publish') {
                return get_permalink($page->ID);
            }
            $pages_with_shortcode = get_posts(['post_type' => 'page', 'post_status' => 'publish', 'posts_per_page' => 1, 's' => '[vehizo_wishlist', 'fields' => 'ids']);
            if (!empty($pages_with_shortcode)) {
                return get_permalink($pages_with_shortcode[0]);
            }
            return '#';
        }
        
        /**
         * Adds the Contact Form and Wishlist buttons to the single vehicle page.
         */
        public static function add_wishlist_button_to_single_page($post_id) {
            // Contact Form Button (above Wishlist)
            if (get_option('vehizo_contact_module_active', '1') === '1') {
                printf(
                    '<button class="vehizo-button vehizo-button--primary btn-contact-popup" data-vehicle-id="%d" aria-label="%s">%s</button>',
                    esc_attr($post_id),
                    esc_attr__('Inquire about this vehicle', 'vehizo-vehicle-management'),
                    '✉️ ' . esc_html__('Contact', 'vehizo-vehicle-management')
                );
            }

            // Wishlist Button
            printf(
                '<button class="vehizo-button vehizo-button--secondary btn-park" data-vehicle-id="%d" aria-label="%s">%s</button>',
                esc_attr($post_id),
                esc_attr__('Save vehicle to wishlist', 'vehizo-vehicle-management'),
                '⭐ ' . esc_html__('Add to Wishlist', 'vehizo-vehicle-management')
            );

            // View Wishlist Button
            printf(
                '<button class="vehizo-button vehizo-button--outline btn-view-wishlist" aria-label="%s">👁️ %s <span class="wishlist-count">(0)</span></button>',
                esc_attr__('View your wishlist', 'vehizo-vehicle-management'),
                esc_html__('View Wishlist', 'vehizo-vehicle-management')
            );

            // Counter initialization is now handled by wishlist-counter-init.js (WordPress.org Guidelines)
            // See: assets/js/wishlist-counter-init.js
        }
/**
         * Get module statistics
         */
        public static function get_statistics() {
            return array(
                'total_vehicles' => wp_count_posts('vehizo_vehicle')->publish,
                'module_active' => get_option('vehizo_wishlist_module_active', '1') === '1',
                'pages_with_shortcode' => count(get_posts(['post_type' => 'page', 'post_status' => 'publish', 's' => '[vehizo_wishlist', 'fields' => 'ids']))
            );
        }

        /**
         * Clean up old localStorage data
         */
        public static function cleanup_old_data() {
            do_action('vehizo_wishlist_cleanup');
        }

        /**
         * Check if vehicle exists and is published
         */
        public static function vehicle_exists($vehicle_id) {
            if (!is_numeric($vehicle_id) || $vehicle_id <= 0) return false;
            $post = get_post($vehicle_id);
            return ($post && $post->post_type === 'vehizo_vehicle' && $post->post_status === 'publish');
        }

        /**
         * Get maximum allowed vehicles in wishlist
         */
        public static function get_max_vehicles() {
            return apply_filters('vehizo_wishlist_max_vehicles', 10);
        }

        /**
         * Validate vehicle IDs array
         */
        public static function validate_vehicle_ids($vehicle_ids) {
            if (!is_array($vehicle_ids)) return array();
            $max_vehicles = self::get_max_vehicles();
            $validated_ids = array();
            foreach ($vehicle_ids as $id) {
                $id = absint($id);
                if ($id > 0 && self::vehicle_exists($id)) {
                    $validated_ids[] = $id;
                    if (count($validated_ids) >= $max_vehicles) break;
                }
            }
            return array_unique($validated_ids);
        }

        /**
         * Register AJAX endpoints for admin
         */
        public static function register_admin_ajax() {
            add_action('wp_ajax_vehizo_wishlist_stats', array(__CLASS__, 'ajax_get_statistics'));
        }

        /**
         * AJAX handler for getting module statistics (admin only)
         */
        public static function ajax_get_statistics() {
            // Verify nonce
            $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
            if (!wp_verify_nonce($nonce, 'vehizo_admin_nonce')) {
                wp_send_json_error(array('message' => esc_html__('Security check failed.', 'vehizo-vehicle-management')));
                return;
            }

            // Check user permissions
            if (!current_user_can('manage_options')) {
                wp_send_json_error(array('message' => esc_html__('Insufficient permissions', 'vehizo-vehicle-management')));
                return;
            }
            $stats = self::get_statistics();
            wp_send_json_success($stats);
        }

        /**
         * Add module info to admin dashboard
         */
        public static function add_dashboard_info() {
            if (get_option('vehizo_wishlist_module_active', '1') === '1') {
                add_meta_box(
                    'vehizo_wishlist_stats',
                    __('Wishlist Module Status', 'vehizo-vehicle-management'),
                    array(__CLASS__, 'render_dashboard_widget'),
                    'dashboard',
                    'normal',
                    'low'
                );
            }
        }

        /**
         * Render dashboard widget
         */
        public static function render_dashboard_widget() {
            $stats = self::get_statistics();
            ?>
            <div class="vehizo-dashboard-widget">
                <p>
                    <strong><?php esc_html_e('Module Status:', 'vehizo-vehicle-management'); ?></strong> 
                    <?php echo $stats['module_active'] ? 
                        '<span style="color: #10b981;">' . esc_html__('Active', 'vehizo-vehicle-management') . '</span>' : 
                        '<span style="color: #ef4444;">' . esc_html__('Inactive', 'vehizo-vehicle-management') . '</span>'; ?>
                </p>
                <p>
                    <strong><?php esc_html_e('Total Vehicles:', 'vehizo-vehicle-management'); ?></strong> 
                    <?php echo esc_html($stats['total_vehicles']); ?>
                </p>
                <p>
                    <strong><?php esc_html_e('Pages with Wishlist:', 'vehizo-vehicle-management'); ?></strong> 
                    <?php echo esc_html($stats['pages_with_shortcode']); ?>
                </p>
            </div>
            <?php
        }
    }

    // Initialize the module if active
    add_action('init', array('Vehizo_Wishlist_Module', 'init'));
    
    // Admin-specific hooks
    if (is_admin()) {
        add_action('wp_dashboard_setup', array('Vehizo_Wishlist_Module', 'add_dashboard_info'));
        add_action('init', array('Vehizo_Wishlist_Module', 'register_admin_ajax'));
    }
}