<?php
/**
 * Vehizo Statistics Module - Dashboard Tab Integration
 * Comprehensive Vehicle Analytics for Admin Backend
 * Extended with Conversion Rate Tracking
 * @package      Vehizo
 * @version      1.0.0
 * @author       Thorsten Glander
 * @license      GPL-2.0+
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Vehizo_Statistics_Module')) {
    class Vehizo_Statistics_Module {

        public static function init() {
            // Only initialize when module is active
            if (get_option('vehizo_statistics_module_active', '0') === '1') {
                // Hook into the main dashboard to add Statistics tab
                add_filter('vehizo_dashboard_tabs', [self::class, 'add_statistics_tab']);
                add_action('vehizo_render_dashboard_tab_statistics', [self::class, 'render_statistics_tab']);
                
                // Activate view tracking for vehicles
                add_action('wp', [self::class, 'track_vehicle_view']);
                
                // AJAX handlers for inquiry tracking
                add_action('wp_ajax_vehizo_track_inquiry', [self::class, 'track_vehicle_inquiry']);
                add_action('wp_ajax_nopriv_vehizo_track_inquiry', [self::class, 'track_vehicle_inquiry']);
                
                // Include JavaScript for frontend tracking
                add_action('wp_enqueue_scripts', [self::class, 'enqueue_frontend_scripts']);
                
                // Enqueue scripts for statistics page
                add_action('admin_enqueue_scripts', [self::class, 'enqueue_admin_scripts']);
            }
        }

        /**
         * Adds Statistics tab to main dashboard
         */
        public static function add_statistics_tab($tabs) {
            $tabs['statistics'] = __('Statistics', 'vehizo-vehicle-management');
            return $tabs;
        }

        /**
         * Enqueue admin scripts for statistics
         */
        public static function enqueue_admin_scripts($hook) {
            if ($hook === 'toplevel_page_vehizo-dashboard') {
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only tab parameter for admin UI
                $current_tab = isset($_GET['tab']) ? sanitize_key(wp_unslash($_GET['tab'])) : 'dashboard';
                if ($current_tab === 'statistics') {
                    
                    // Chart.js mit korrektem Pfad laden
                    wp_enqueue_script(
                        'vehizo-chartjs',
                        VEHIZO_PLUGIN_URL . 'assets/js/chart.min.js',
                        [],
                        '3.9.1',
                        true
                    );
                    
                    // Statistics JavaScript
                    wp_enqueue_script(
                        'vehizo-statistics',
                        VEHIZO_PLUGIN_URL . 'assets/js/statistics.js',
                        ['vehizo-chartjs'],
                        '1.0.0',
                        true
                    );
                    
                    // Statistics CSS
                    wp_enqueue_style(
                        'vehizo-statistics-style',
                        VEHIZO_PLUGIN_URL . 'assets/css/statistics.css',
                        [],
                        '1.0.0'
                    );

                    // Localize script with data for charts
                    $stats_data = self::collect_vehicle_statistics();
                    wp_localize_script('vehizo-statistics', 'vehizoStatsData', [
                        'brands' => array_slice($stats_data['brands'], 0, 8, true),
                        'fuelTypes' => $stats_data['fuel_types'],
                        'vehicleTypes' => $stats_data['vehicle_types'],
                        'transmissions' => $stats_data['transmissions'],
                        'priceRanges' => array_values($stats_data['price_ranges']),
                        'chartColors' => ['#4285f4', '#34a853', '#fbbc04', '#ea4335', '#9c27b0', '#00bcd4', '#8bc34a', '#ff9800']
                    ]);
                }
            }
        }     

        /**
         * Frontend scripts for inquiry tracking
         */
        public static function enqueue_frontend_scripts() {
            // Inquiry tracker functionality requires Pro version
            // Removed to prevent 404 errors
        }

        /**
         * AJAX handler for inquiry tracking
         */
        public static function track_vehicle_inquiry() {
            $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
            if (!wp_verify_nonce($nonce, 'vehizo_inquiry_nonce')) {
                wp_send_json_error(['message' => __('Security check failed.', 'vehizo-vehicle-management')], 403);
            }
            
            $vehicle_id = isset($_POST['vehicle_id']) ? absint($_POST['vehicle_id']) : 0;
            $inquiry_type = isset($_POST['inquiry_type']) ? sanitize_key($_POST['inquiry_type']) : '';
            
            if (!$vehicle_id || empty($inquiry_type)) {
                wp_send_json_error(['message' => __('Invalid data provided.', 'vehizo-vehicle-management')], 400);
            }
            
            if (session_status() === PHP_SESSION_NONE) {
                session_start();
            }

            $session_key = 'vehizo_inquiry_' . $vehicle_id;
            
            if (!isset($_SESSION[$session_key])) {
                $total_inquiries = absint(get_post_meta($vehicle_id, 'vehizo_total_inquiries', true));
                $total_inquiries++;
                update_post_meta($vehicle_id, 'vehizo_total_inquiries', $total_inquiries);
                
                update_post_meta($vehicle_id, 'vehizo_last_inquiry', current_time('mysql'));
                update_post_meta($vehicle_id, 'vehizo_last_inquiry_type', $inquiry_type);
                
                $_SESSION[$session_key] = time();
                
                wp_send_json_success(['message' => __('Inquiry tracked successfully.', 'vehizo-vehicle-management')]);
            } else {
                wp_send_json_error(['message' => __('Already tracked.', 'vehizo-vehicle-management')]);
            }
        }

        /**
         * Tracks vehicle views
         */
        public static function track_vehicle_view() {
            if (!is_singular('vehizo_vehicle')) {
                return;
            }
            
            $post_id = get_the_ID();
            if (!$post_id) {
                return;
            }
            
            if (self::is_bot() || is_user_logged_in()) {
                return;
            }
            
            if (session_status() === PHP_SESSION_NONE) {
                session_start();
            }
            
            $session_key = 'vehizo_viewed_' . $post_id;
            
            if (!isset($_SESSION[$session_key])) {
                $total_views = absint(get_post_meta($post_id, 'vehizo_total_views', true));
                $total_views++;
                update_post_meta($post_id, 'vehizo_total_views', $total_views);
                
                update_post_meta($post_id, 'vehizo_last_viewed', current_time('mysql'));
                
                $_SESSION[$session_key] = time();
            }
        }
        
        /**
         * Checks if the request comes from a bot
         */
        private static function is_bot() {
            if (empty($_SERVER['HTTP_USER_AGENT'])) {
                return true; // Treat requests with no user agent as bots.
            }
            
            $user_agent = strtolower(sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])));
            $bots = ['googlebot', 'bingbot', 'slurp', 'duckduckbot', 'baiduspider', 'yandexbot', 'ahrefsbot', 'semrushbot', 'mj12bot'];
            
            foreach ($bots as $bot) {
                if (strpos($user_agent, $bot) !== false) {
                    return true;
                }
            }
            
            return false;
        }
        
        /**
         * Renders the statistics tab content
         */
        public static function render_statistics_tab() {
            if (!current_user_can('manage_options')) {
                wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'vehizo-vehicle-management'));
            }

            $stats_data = self::collect_vehicle_statistics();
            ?>
            <div class="vehizo-section">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                    <div>
                        <h2 style="margin: 0 0 8px 0;"><?php esc_html_e('Vehicle Statistics', 'vehizo-vehicle-management'); ?></h2>
                        <p style="margin: 0;"><?php esc_html_e('Track views, inquiries, and conversion rates for your vehicle inventory.', 'vehizo-vehicle-management'); ?></p>
                    </div>
                </div>

                <!-- 3-Column Grid Layout -->
                <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px;">

                    <!-- Column 1: Status Cards (Vertical Stack) -->
                    <div>
                        <div class="stat-card" style="margin-bottom: 15px;">
                            <h3><?php echo esc_html(number_format_i18n($stats_data['total_vehicles'])); ?></h3>
                            <p><?php esc_html_e('Total Vehicles', 'vehizo-vehicle-management'); ?></p>
                        </div>
                        <div class="stat-card" style="margin-bottom: 15px;">
                            <h3 style="color: #10b981;"><?php echo esc_html(number_format_i18n($stats_data['available'])); ?></h3>
                            <p><?php esc_html_e('Available', 'vehizo-vehicle-management'); ?></p>
                        </div>
                        <div class="stat-card" style="margin-bottom: 15px;">
                            <h3 style="color: #f59e0b;"><?php echo esc_html(number_format_i18n($stats_data['reserved'])); ?></h3>
                            <p><?php esc_html_e('Reserved', 'vehizo-vehicle-management'); ?></p>
                        </div>
                        <div class="stat-card">
                            <h3 style="color: #ef4444;"><?php echo esc_html(number_format_i18n($stats_data['sold'])); ?></h3>
                            <p><?php esc_html_e('Sold', 'vehizo-vehicle-management'); ?></p>
                        </div>
                    </div>

                    <!-- Column 2: Top Vehicles & Brands -->
                    <div>
                        <div class="meta-box" style="margin-bottom: 20px;">
                            <h3><?php esc_html_e('Top Viewed Vehicles', 'vehizo-vehicle-management'); ?></h3>
                            <?php if (!empty($stats_data['top_viewed_vehicles'])): ?>
                                <?php foreach (array_slice($stats_data['top_viewed_vehicles'], 0, 5) as $vehicle): ?>
                                    <div class="vehicle-item">
                                        <span class="vehicle-title"><?php echo esc_html($vehicle['title']); ?></span>
                                        <span class="vehicle-count"><?php echo esc_html($vehicle['total_views']); ?></span>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="empty-state">
                                    <p><?php esc_html_e('No views tracked yet', 'vehizo-vehicle-management'); ?></p>
                                </div>
                            <?php endif; ?>
                        </div>

                        <div class="meta-box">
                            <h3><?php esc_html_e('Most Popular Brands', 'vehizo-vehicle-management'); ?></h3>
                            <?php if (!empty($stats_data['brands'])): ?>
                                <?php
                                $top_brands = array_slice($stats_data['brands'], 0, 5, true);
                                foreach ($top_brands as $brand => $count): ?>
                                    <div class="quick-stat-item">
                                        <span class="quick-stat-label"><?php echo esc_html($brand); ?></span>
                                        <span class="quick-stat-value"><?php echo esc_html($count); ?></span>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="empty-state">
                                    <p><?php esc_html_e('No brand data available', 'vehizo-vehicle-management'); ?></p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Column 3: Price Stats -->
                    <div>
                        <div class="meta-box">
                            <h3><?php esc_html_e('Price & Vehicle Stats', 'vehizo-vehicle-management'); ?></h3>
                            <div class="quick-stat-item">
                                <span class="quick-stat-label"><?php esc_html_e('Average Price', 'vehizo-vehicle-management'); ?></span>
                                <span class="quick-stat-value"><?php echo esc_html(number_format_i18n($stats_data['avg_price'])); ?>€</span>
                            </div>
                            <div class="quick-stat-item">
                                <span class="quick-stat-label"><?php esc_html_e('Lowest Price', 'vehizo-vehicle-management'); ?></span>
                                <span class="quick-stat-value"><?php echo esc_html(number_format_i18n($stats_data['min_price'])); ?>€</span>
                            </div>
                            <div class="quick-stat-item">
                                <span class="quick-stat-label"><?php esc_html_e('Highest Price', 'vehizo-vehicle-management'); ?></span>
                                <span class="quick-stat-value"><?php echo esc_html(number_format_i18n($stats_data['max_price'])); ?>€</span>
                            </div>
                            <div class="quick-stat-item">
                                <span class="quick-stat-label"><?php esc_html_e('Avg. Mileage', 'vehizo-vehicle-management'); ?></span>
                                <span class="quick-stat-value"><?php echo esc_html(number_format_i18n($stats_data['avg_mileage'])); ?> km</span>
                            </div>
                        </div>

                        <?php
                        /**
                         * Action hook for Pro version analytics
                         */
                        do_action('vehizo_statistics_sidebar_content', $stats_data);
                        ?>
                    </div>

                </div>

                <?php
                /**
                 * Action hook to add advanced charts (Pro version)
                 */
                do_action('vehizo_statistics_main_content', $stats_data);
                ?>
            </div>
            <?php
        }

        /**
         * Collects all vehicle statistics
         */
        public static function collect_vehicle_statistics() {
            $vehicles = get_posts([
                'post_type' => 'vehizo_vehicle',
                'post_status' => ['publish', 'draft', 'private'],
                'numberposts' => -1,
            ]);

            $total_vehicles = count($vehicles);
            $published_vehicles = count(array_filter($vehicles, function($v) { 
                return $v->post_status === 'publish'; 
            }));

            $prices = [];
            $mileages = [];
            $brands = [];
            $fuel_types = [];
            $vehicle_types = [];
            $transmissions = [];
            $price_ranges = [
                '0-10k' => 0, '10k-25k' => 0, '25k-50k' => 0, '50k-100k' => 0, '100k+' => 0
            ];
            $total_views = 0;
            $total_inquiries = 0;
            $view_stats = [];

            foreach ($vehicles as $vehicle) {
                $price = get_post_meta($vehicle->ID, 'vehizo_price', true);
                if (!empty($price) && is_numeric($price)) {
                    $prices[] = (float)$price;
                    if ($price < 10000) $price_ranges['0-10k']++;
                    elseif ($price < 25000) $price_ranges['10k-25k']++;
                    elseif ($price < 50000) $price_ranges['25k-50k']++;
                    elseif ($price < 100000) $price_ranges['50k-100k']++;
                    else $price_ranges['100k+']++;
                }

                $mileage = get_post_meta($vehicle->ID, 'vehizo_mileage', true);
                if (!empty($mileage) && is_numeric($mileage)) $mileages[] = (int)$mileage;

                $brand = sanitize_text_field(get_post_meta($vehicle->ID, 'vehizo_make', true)) ?: __('Unknown', 'vehizo-vehicle-management');
                $brands[$brand] = ($brands[$brand] ?? 0) + 1;
                
                $fuel_type = sanitize_text_field(get_post_meta($vehicle->ID, 'vehizo_fuelType', true)) ?: __('Unknown', 'vehizo-vehicle-management');
                $fuel_types[$fuel_type] = ($fuel_types[$fuel_type] ?? 0) + 1;
                
                $vehicle_type = sanitize_text_field(get_post_meta($vehicle->ID, 'vehizo_vehicleType', true)) ?: __('Unknown', 'vehizo-vehicle-management');
                $vehicle_types[$vehicle_type] = ($vehicle_types[$vehicle_type] ?? 0) + 1;
                
                $transmission = sanitize_text_field(get_post_meta($vehicle->ID, 'vehizo_transmission', true)) ?: __('Unknown', 'vehizo-vehicle-management');
                $transmissions[$transmission] = ($transmissions[$transmission] ?? 0) + 1;
                
                $vehicle_total_views = absint(get_post_meta($vehicle->ID, 'vehizo_total_views', true));
                $vehicle_total_inquiries = absint(get_post_meta($vehicle->ID, 'vehizo_total_inquiries', true));
                
                $total_views += $vehicle_total_views;
                $total_inquiries += $vehicle_total_inquiries;

                if ($vehicle_total_views > 0) {
                    $view_stats[] = [
                        'id' => $vehicle->ID,
                        'title' => sanitize_text_field($vehicle->post_title),
                        'total_views' => $vehicle_total_views,
                        'total_inquiries' => $vehicle_total_inquiries,
                    ];
                }
            }

            $avg_price = !empty($prices) ? round(array_sum($prices) / count($prices)) : 0;
            $avg_mileage = !empty($mileages) ? round(array_sum($mileages) / count($mileages)) : 0;
            $publish_rate = $total_vehicles > 0 ? round(($published_vehicles / $total_vehicles) * 100, 1) : 0;
            $overall_conversion_rate = $total_views > 0 ? round(($total_inquiries / $total_views) * 100, 1) : 0;
            $avg_views_per_vehicle = $published_vehicles > 0 ? round($total_views / $published_vehicles, 1) : 0;
            $avg_inquiries_per_vehicle = $published_vehicles > 0 ? round($total_inquiries / $published_vehicles, 1) : 0;

            usort($view_stats, fn($a, $b) => $b['total_views'] <=> $a['total_views']);
            arsort($brands); arsort($fuel_types); arsort($vehicle_types); arsort($transmissions);

            return [
                'total_vehicles' => $total_vehicles,
                'published_vehicles' => $published_vehicles,
                'publish_rate' => $publish_rate,
                'avg_price' => $avg_price,
                'min_price' => !empty($prices) ? min($prices) : 0,
                'max_price' => !empty($prices) ? max($prices) : 0,
                'avg_mileage' => $avg_mileage,
                'brands' => $brands,
                'fuel_types' => $fuel_types,
                'vehicle_types' => $vehicle_types,
                'transmissions' => $transmissions,
                'price_ranges' => $price_ranges,
                'total_views' => $total_views,
                'avg_views_per_vehicle' => $avg_views_per_vehicle,
                'total_inquiries' => $total_inquiries,
                'avg_inquiries_per_vehicle' => $avg_inquiries_per_vehicle,
                'overall_conversion_rate' => $overall_conversion_rate,
                'top_viewed_vehicles' => array_slice($view_stats, 0, 10),
            ];
        }
    }
    
    // Initialize the module
    Vehizo_Statistics_Module::init();
}
