<?php
/**
 * Vehizo Vehicle Shortcode - FINAL & COMPLETE
 * Shortcode [vehizo_vehicles] with a stable class-based structure and full feature set.
 *
 * @package Vehizo
 * @version 4.2.1
 * @author Thorsten Glander
 * @since 1.0.0
 */

// Security check
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Vehizo_Shortcode_Vehicles')) {

    class Vehizo_Shortcode_Vehicles {

        /**
         * Initialize the shortcode and all related hooks.
         */
        public static function init() {
            add_shortcode('vehizo_vehicles', array(__CLASS__, 'render_shortcode'));
            add_filter('body_class', array(__CLASS__, 'add_body_class'));
            // AJAX endpoint for dependent dropdowns
            add_action('wp_ajax_vehizo_get_models_for_make', array(__CLASS__, 'ajax_get_models_for_make'));
            add_action('wp_ajax_nopriv_vehizo_get_models_for_make', array(__CLASS__, 'ajax_get_models_for_make'));
            // AJAX endpoint for filtering
            add_action('wp_ajax_vehizo_filter_vehicles', array(__CLASS__, 'ajax_filter_vehicles'));
            add_action('wp_ajax_nopriv_vehizo_filter_vehicles', array(__CLASS__, 'ajax_filter_vehicles'));
        }

        /**
         * Main render function for the [vehizo_vehicles] shortcode
         */
        public static function render_shortcode($atts) {
            self::enqueue_shortcode_assets();

            $atts = shortcode_atts(array(
                'layout' => get_option('vehizo_layout', 'grid'),
                'columns' => get_option('vehizo_grid_columns', 3),
                'show_filter' => get_option('vehizo_show_filter', '1'),
                'posts_per_page' => (int)get_option('vehizo_posts_per_page', 12)
            ), $atts, 'vehizo_vehicles');

            ob_start();

            $layout_class = get_option('vehizo_layout', 'grid');
            echo '<div class="vehizo-main-container layout-' . esc_attr($layout_class) . '">';

            if ($atts['show_filter'] === '1') {
                self::render_vehicle_filter();
            }

            $meta_query = array('relation' => 'AND');
            // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Read-only filter parameters for public vehicle search
            $filters = array(
                'vehizo_make' => isset($_GET['make']) ? sanitize_text_field(wp_unslash($_GET['make'])) : '',
                'vehizo_model' => isset($_GET['model']) ? sanitize_text_field(wp_unslash($_GET['model'])) : '',
                'vehizo_vehicleType' => isset($_GET['vehicleType']) ? sanitize_text_field(wp_unslash($_GET['vehicleType'])) : '',
                'vehizo_transmission' => isset($_GET['transmission']) ? sanitize_text_field(wp_unslash($_GET['transmission'])) : '',
                'vehizo_fuelType' => isset($_GET['fuelType']) ? sanitize_text_field(wp_unslash($_GET['fuelType'])) : '',
                'vehizo_color' => isset($_GET['color']) ? sanitize_text_field(wp_unslash($_GET['color'])) : '',
                'price_min' => isset($_GET['price_min']) ? absint($_GET['price_min']) : 0,
                'price_max' => isset($_GET['price_max']) ? absint($_GET['price_max']) : 0,
                'ez_ab' => isset($_GET['ez_ab']) ? absint($_GET['ez_ab']) : 0,
                'ez_bis' => isset($_GET['ez_bis']) ? absint($_GET['ez_bis']) : 0,
                'mileage_max' => isset($_GET['mileage_max']) ? absint($_GET['mileage_max']) : 0
            );
            // phpcs:enable WordPress.Security.NonceVerification.Recommended

            foreach ($filters as $key => $value) {
                if (!empty($value)) {
                    if ($key === 'price_min') $meta_query[] = array('key' => 'vehizo_price', 'value' => $value, 'compare' => '>=', 'type' => 'NUMERIC');
                    elseif ($key === 'price_max') $meta_query[] = array('key' => 'vehizo_price', 'value' => $value, 'compare' => '<=', 'type' => 'NUMERIC');
                    elseif ($key === 'ez_ab') $meta_query[] = array('key' => 'vehizo_firstRegistration', 'value' => $value, 'compare' => '>=', 'type' => 'NUMERIC');
                    elseif ($key === 'ez_bis') $meta_query[] = array('key' => 'vehizo_firstRegistration', 'value' => $value, 'compare' => '<=', 'type' => 'NUMERIC');
                    elseif ($key === 'mileage_max') $meta_query[] = array('key' => 'vehizo_mileage', 'value' => $value, 'compare' => '<=', 'type' => 'NUMERIC');
                    else $meta_query[] = array('key' => $key, 'value' => $value, 'compare' => '=');
                }
            }

            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only sort parameter for public vehicle listing
            $orderby = isset($_GET['sort']) ? sanitize_text_field(wp_unslash($_GET['sort'])) : 'date';
            $order = 'DESC';
            $meta_key = '';

            switch ($orderby) {
                case 'price_asc': $orderby = 'meta_value_num'; $meta_key = 'vehizo_price'; $order = 'ASC'; break;
                case 'price_desc': $orderby = 'meta_value_num'; $meta_key = 'vehizo_price'; $order = 'DESC'; break;
                case 'year_desc': $orderby = 'meta_value_num'; $meta_key = 'vehizo_firstRegistration'; $order = 'DESC'; break;
                case 'year_asc': $orderby = 'meta_value_num'; $meta_key = 'vehizo_firstRegistration'; $order = 'ASC'; break;
                case 'mileage_asc': $orderby = 'meta_value_num'; $meta_key = 'vehizo_mileage'; $order = 'ASC'; break;
                case 'title': $orderby = 'title'; $order = 'ASC'; break;
                default: $orderby = 'date'; $order = 'DESC';
            }

            $query_args = array(
                'post_type' => 'vehizo_vehicle',
                'post_status' => 'publish',
                'posts_per_page' => $atts['posts_per_page'],
                'paged' => get_query_var('paged') ?: 1,
                'meta_query' => $meta_query,
                'orderby' => $orderby,
                'order' => $order,
                'no_found_rows' => false,
                'update_post_meta_cache' => true,
            );
            if (!empty($meta_key)) {
                $query_args['meta_key'] = $meta_key;
            }

            $vehicles_query = new WP_Query($query_args);
            self::render_results_section($vehicles_query->found_posts);

            if ($vehicles_query->have_posts()) {
                echo '<div class="vehizo-vehicles-grid">';
                while ($vehicles_query->have_posts()) {
                    $vehicles_query->the_post();
                    self::render_vehicle_card(get_the_ID());
                }
                echo '</div>';
                if ($vehicles_query->max_num_pages > 1) {
                    self::render_pagination($vehicles_query);
                }
            } else {
                self::render_no_results();
            }
            wp_reset_postdata();

            echo '</div>';
            return ob_get_clean();
        }

        public static function enqueue_shortcode_assets() {
            // Enqueue base styles first
            wp_enqueue_style('vehizo-base', plugin_dir_url(__FILE__) . '../assets/css/vehizo-base.css', array(), '4.2.2');

            // Enqueue advanced styles
            wp_enqueue_style('vehizo-shortcode-vehicles', plugin_dir_url(__FILE__) . '../assets/css/vehizo-advanced.css', array('vehizo-base'), '4.2.2');

            if (class_exists('Vehizo_Filter_Settings')) {
                $dynamic_css = Vehizo_Filter_Settings::generate_dynamic_css();
                // Add inline CSS to base stylesheet so it loads before advanced
                wp_add_inline_style('vehizo-base', $dynamic_css);
            }

            // Get make-model mapping for dependent dropdowns
            global $wpdb;
            $make_model_map = wp_cache_get('vehizo_make_model_map');
            if (false === $make_model_map) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Complex JOIN query for make-model relationships, cached for performance, WP_Query alternative would be less efficient
                $results = $wpdb->get_results($wpdb->prepare("
                    SELECT make.meta_value AS make, model.meta_value AS model
                    FROM {$wpdb->postmeta} AS make
                    INNER JOIN {$wpdb->postmeta} AS model ON make.post_id = model.post_id
                    WHERE make.meta_key = %s AND model.meta_key = %s
                    AND make.meta_value != '' AND model.meta_value != ''
                    GROUP BY make.meta_value, model.meta_value
                    ORDER BY make.meta_value, model.meta_value
                ", 'vehizo_make', 'vehizo_model'));
                $make_model_map = array();
                foreach ($results as $row) {
                    $make_model_map[$row->make][] = $row->model;
                }
                wp_cache_set('vehizo_make_model_map', $make_model_map, '', 3600);
            }

            wp_enqueue_script('vehizo-shortcode-vehicles', plugin_dir_url(__FILE__) . '../assets/js/vehizo-shortcode-vehicles.js', array('jquery'), '4.2.1', true);
            wp_localize_script('vehizo-shortcode-vehicles', 'vehizoShortcode', array('ajaxurl' => admin_url('admin-ajax.php'), 'nonce' => wp_create_nonce('vehizo_filter_nonce'), 'makeModelMap' => $make_model_map,
                'i18n' => array(
                    'loading' => esc_html__('Loading...', 'vehizo-vehicle-management'),
                    'parked' => esc_html__('Parked', 'vehizo-vehicle-management'),
                    'park' => esc_html__('Park', 'vehizo-vehicle-management'),
                )
            ));
        }
        
        public static function render_vehicle_filter() {
            global $wpdb;
            
            $cache_key = 'vehizo_filter_data_full_v2';
            $filter_data = wp_cache_get($cache_key);
            
            if (false === $filter_data) {
                // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery -- Multiple DISTINCT queries for filter dropdowns, cached for performance, WP_Query alternative would require multiple queries and be less efficient
                $filter_data = array(
                    'marken' => $wpdb->get_col($wpdb->prepare("SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE meta_key=%s AND meta_value != '' ORDER BY meta_value ASC", 'vehizo_make')),
                    'getriebe' => $wpdb->get_col($wpdb->prepare("SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE meta_key=%s AND meta_value != '' ORDER BY meta_value ASC", 'vehizo_transmission')),
                    'kraftstoff' => $wpdb->get_col($wpdb->prepare("SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE meta_key=%s AND meta_value != '' ORDER BY meta_value ASC", 'vehizo_fuelType')),
                    'farben' => $wpdb->get_col($wpdb->prepare("SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE meta_key=%s AND meta_value != '' ORDER BY meta_value ASC", 'vehizo_color')),
                    'typen' => $wpdb->get_col($wpdb->prepare("SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE meta_key=%s AND meta_value != '' ORDER BY meta_value ASC", 'vehizo_vehicleType'))
                );
                // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery
                wp_cache_set($cache_key, $filter_data, '', 3600);
            }
            
            extract($filter_data);

            // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Read-only filter parameters for vehicle filter form
            $get_params = array(
                'make' => isset($_GET['make']) ? sanitize_text_field(wp_unslash($_GET['make'])) : '',
                'model' => isset($_GET['model']) ? sanitize_text_field(wp_unslash($_GET['model'])) : '',
                'vehicleType' => isset($_GET['vehicleType']) ? sanitize_text_field(wp_unslash($_GET['vehicleType'])) : '',
                'transmission' => isset($_GET['transmission']) ? sanitize_text_field(wp_unslash($_GET['transmission'])) : '',
                'fuelType' => isset($_GET['fuelType']) ? sanitize_text_field(wp_unslash($_GET['fuelType'])) : '',
                'color' => isset($_GET['color']) ? sanitize_text_field(wp_unslash($_GET['color'])) : '',
                'price_min' => isset($_GET['price_min']) ? absint($_GET['price_min']) : 0,
                'price_max' => isset($_GET['price_max']) ? absint($_GET['price_max']) : 0,
                'ez_ab' => isset($_GET['ez_ab']) ? absint($_GET['ez_ab']) : 0,
                'ez_bis' => isset($_GET['ez_bis']) ? absint($_GET['ez_bis']) : 0,
                'mileage_max' => isset($_GET['mileage_max']) ? absint($_GET['mileage_max']) : 0
            );
            // phpcs:enable WordPress.Security.NonceVerification.Recommended
            ?>
            <div class="filter-section">
                <div id="filter-header" class="filter-header">
                    <div class="filter-title">
                        <h3 class="filter-title-text"><?php esc_html_e('Vehicle Filter', 'vehizo-vehicle-management'); ?></h3>
                        <span class="filter-toggle-hint"><?php esc_html_e('Click to expand/collapse', 'vehizo-vehicle-management'); ?> <span class="filter-arrow">▼</span></span>
                    </div>
                </div>
                <div id="filter-content" class="filter-content">
                    <form method="get" id="filter-form">
                        <div class="filter-grid">

                            <div class="filter-group">
                                <label class="filter-label-top" for="filter-make"><?php esc_html_e('Make', 'vehizo-vehicle-management'); ?></label>
                                <select name="make" id="filter-make" class="filter-select"><option value=""><?php echo empty($get_params['make']) ? esc_html__('All Makes', 'vehizo-vehicle-management') : esc_html__('Select Make', 'vehizo-vehicle-management'); ?></option><?php foreach ($marken as $m): ?><option value="<?php echo esc_attr($m); ?>" <?php selected($get_params['make'], $m); ?>><?php echo esc_html($m); ?></option><?php endforeach; ?></select>
                            </div>

                            <div class="filter-group">
                                <label class="filter-label-top" for="filter-model"><?php esc_html_e('Model', 'vehizo-vehicle-management'); ?></label>
                                <select name="model" id="filter-model" class="filter-select" <?php if (empty($get_params['make'])) echo 'disabled'; ?>><option value=""><?php echo empty($get_params['model']) ? esc_html__('All Models', 'vehizo-vehicle-management') : esc_html__('Select Model', 'vehizo-vehicle-management'); ?></option></select>
                            </div>

                            <div class="filter-group">
                                <label class="filter-label-top" for="filter-vehicleType"><?php esc_html_e('Vehicle Type', 'vehizo-vehicle-management'); ?></label>
                                <select name="vehicleType" id="filter-vehicleType" class="filter-select"><option value=""><?php echo empty($get_params['vehicleType']) ? esc_html__('All Types', 'vehizo-vehicle-management') : esc_html__('Select Type', 'vehizo-vehicle-management'); ?></option><?php foreach ($typen as $t): ?><option value="<?php echo esc_attr($t); ?>" <?php selected($get_params['vehicleType'], $t); ?>><?php echo esc_html($t); ?></option><?php endforeach; ?></select>
                            </div>

                            <div class="filter-group">
                                <input type="number" name="price_min" id="filter-price-min" class="filter-input" min="0" step="100" placeholder=" " value="<?php echo esc_attr($get_params['price_min']); ?>">
                                <label class="filter-label" for="filter-price-min"><?php
                                /* translators: %s: Currency symbol (e.g. €, $) */
                                printf(esc_html__('Price from (%s)', 'vehizo-vehicle-management'), esc_html(get_option('vehizo_currency_symbol', '€'))); ?></label>
                            </div>

                            <div class="filter-group">
                                <input type="number" name="price_max" id="filter-price-max" class="filter-input" min="0" step="100" placeholder=" " value="<?php echo esc_attr($get_params['price_max']); ?>">
                                <label class="filter-label" for="filter-price-max"><?php
                                /* translators: %s: Currency symbol (e.g. €, $) */
                                printf(esc_html__('Price to (%s)', 'vehizo-vehicle-management'), esc_html(get_option('vehizo_currency_symbol', '€'))); ?></label>
                            </div>

                            <div class="filter-group">
                                <input type="number" name="ez_ab" id="filter-ez-ab" class="filter-input" min="1970" max="<?php echo esc_attr(current_time('Y')); ?>" step="1" placeholder=" " value="<?php echo esc_attr($get_params['ez_ab']); ?>">
                                <label class="filter-label" for="filter-ez-ab"><?php esc_html_e('Year from', 'vehizo-vehicle-management'); ?></label>
                            </div>

                            <div class="filter-group">
                                <input type="number" name="ez_bis" id="filter-ez-bis" class="filter-input" min="1970" max="<?php echo esc_attr(current_time('Y')); ?>" step="1" placeholder=" " value="<?php echo esc_attr($get_params['ez_bis']); ?>">
                                <label class="filter-label" for="filter-ez-bis"><?php esc_html_e('Year to', 'vehizo-vehicle-management'); ?></label>
                            </div>

                            <div class="filter-group">
                                <input type="number" name="mileage_max" id="filter-mileage-max" class="filter-input" min="0" step="1000" placeholder=" " value="<?php echo esc_attr($get_params['mileage_max']); ?>">
                                <label class="filter-label" for="filter-mileage-max"><?php esc_html_e('Max Mileage', 'vehizo-vehicle-management'); ?></label>
                            </div>

                        </div>
                        <div class="filter-actions">
                            <?php
                            // Sanitize GET parameters before using them in remove_query_arg
                            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is read-only for display purposes
                            $get_keys = array_map('sanitize_key', array_keys($_GET));
                            ?>
                            <a href="<?php echo esc_url(remove_query_arg($get_keys)); ?>" class="filter-button secondary"><?php esc_html_e('Reset Filters', 'vehizo-vehicle-management'); ?></a>
                            <button type="submit" class="filter-button primary"><?php esc_html_e('Show Vehicles', 'vehizo-vehicle-management'); ?></button>
                        </div>
                    </form>
                </div>
            </div>
        <?php }
        public static function render_results_section($total_count) { ?>
            <div class="results-section">
                <div class="results-count"><strong><?php echo esc_html(number_format_i18n($total_count)); ?></strong> <?php echo esc_html(_n('vehicle found', 'vehicles found', $total_count, 'vehizo-vehicle-management')); ?></div>
                <div class="sort-container">
                    <label class="sort-label"><?php esc_html_e('Sort by:', 'vehizo-vehicle-management'); ?></label>
                    <select name="sort" class="sort-select" onchange="vehizoUpdateSort(this.value);">
                        <?php
                        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only parameter for sorting public vehicle list
                        $current_sort = isset($_GET['sort']) ? sanitize_text_field(wp_unslash($_GET['sort'])) : 'date';
                        ?>
                        <option value="date" <?php selected($current_sort, 'date'); ?>><?php esc_html_e('Newest first', 'vehizo-vehicle-management'); ?></option>
                        <option value="price_asc" <?php selected($current_sort, 'price_asc'); ?>><?php esc_html_e('Price ascending', 'vehizo-vehicle-management'); ?></option>
                        <option value="price_desc" <?php selected($current_sort, 'price_desc'); ?>><?php esc_html_e('Price descending', 'vehizo-vehicle-management'); ?></option>
                        <option value="year_desc" <?php selected($current_sort, 'year_desc'); ?>><?php esc_html_e('Year (newest)', 'vehizo-vehicle-management'); ?></option>
                        <option value="year_asc" <?php selected($current_sort, 'year_asc'); ?>><?php esc_html_e('Year (oldest)', 'vehizo-vehicle-management'); ?></option>
                        <option value="mileage_asc" <?php selected($current_sort, 'mileage_asc'); ?>><?php esc_html_e('Mileage ascending', 'vehizo-vehicle-management'); ?></option>
                    </select>
                </div>
            </div>
        <?php }

        public static function render_vehicle_card($post_id) {
            static $meta_cache = array();
            if (!isset($meta_cache[$post_id])) {
                $meta_cache[$post_id] = [
                    'make' => get_post_meta($post_id, 'vehizo_make', true), 'model' => get_post_meta($post_id, 'vehizo_model', true),
                    'price' => get_post_meta($post_id, 'vehizo_price', true), 'year' => get_post_meta($post_id, 'vehizo_firstRegistration', true),
                    'mileage' => get_post_meta($post_id, 'vehizo_mileage', true), 'fuel' => get_post_meta($post_id, 'vehizo_fuelType', true),
                    'power' => get_post_meta($post_id, 'vehizo_power', true), 'transmission' => get_post_meta($post_id, 'vehizo_transmission', true),
                    'number_of_owners' => get_post_meta($post_id, 'vehizo_number_of_owners', true),
                    'engine_displacement' => get_post_meta($post_id, 'vehizo_engine_displacement', true),
                    'status' => get_post_meta($post_id, 'vehizo_vehicle_status', true),
                    'gallery' => get_post_meta($post_id, 'vehizo_gallery', true),
                ];
            }
            extract($meta_cache[$post_id], EXTR_SKIP);
            $title = get_the_title();

            // Get image URL - first try featured image, then gallery images
            $image_url = '';
            if (has_post_thumbnail()) {
                $image_url = get_the_post_thumbnail_url($post_id, 'large');
            } elseif (!empty($gallery)) {
                // Get first image from gallery
                $gallery_ids = explode(',', $gallery);
                $first_image_id = !empty($gallery_ids[0]) ? absint($gallery_ids[0]) : 0;
                if ($first_image_id) {
                    $image_url = wp_get_attachment_image_url($first_image_id, 'large');
                }
            }

            $formatted_price = '';
            if (!empty($price) && is_numeric($price)) {
                $currency_symbol = get_option('vehizo_currency_symbol', '€');
                $currency_position = get_option('vehizo_currency_position', 'after');
                if ($currency_position === 'before') {
                    $formatted_price = $currency_symbol . ' ' . number_format_i18n((int)$price);
                } else {
                    $formatted_price = number_format_i18n((int)$price) . ' ' . $currency_symbol;
                }
            } ?>
            <div class="vehizo-vehicle-card">
                <div class="card-image">
                    <?php if ($status && $status !== 'available'): ?>
                        <div class="status-badge status-<?php echo esc_attr($status); ?>"><?php echo esc_html(ucfirst($status)); ?></div>
                    <?php endif; ?>
                    <?php if ($image_url): ?><a href="<?php the_permalink(); ?>"><img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($title); ?>" loading="lazy"></a><?php else: ?><div class="no-image">🚗</div><?php endif; ?>
                    <?php if ($formatted_price): ?><div class="price-badge"><?php echo esc_html($formatted_price); ?></div><?php endif; ?>
                </div>
                <div class="card-content">
                    <h3 class="card-title"><a href="<?php the_permalink(); ?>"><?php echo esc_html($title); ?></a></h3>
                    <div class="specs-list">
                        <?php if ($year): ?><div class="spec-row"><span class="spec-label"><?php esc_html_e('Year', 'vehizo-vehicle-management'); ?></span><span class="spec-value"><?php echo esc_html($year); ?></span></div><?php endif; ?>
                        <?php if ($mileage): ?><div class="spec-row"><span class="spec-label"><?php esc_html_e('Mileage', 'vehizo-vehicle-management'); ?></span><span class="spec-value"><?php echo esc_html(number_format_i18n((int)$mileage)); ?> km</span></div><?php endif; ?>
                        <?php if ($power): ?><div class="spec-row"><span class="spec-label"><?php esc_html_e('Power', 'vehizo-vehicle-management'); ?></span><span class="spec-value"><?php echo esc_html($power); ?> PS</span></div><?php endif; ?>
                        <?php if ($engine_displacement): ?><div class="spec-row"><span class="spec-label"><?php esc_html_e('Displacement', 'vehizo-vehicle-management'); ?></span><span class="spec-value"><?php echo esc_html(number_format_i18n((int)$engine_displacement)); ?> ccm</span></div><?php endif; ?>
                        <?php if ($number_of_owners): ?><div class="spec-row"><span class="spec-label"><?php esc_html_e('Owners', 'vehizo-vehicle-management'); ?></span><span class="spec-value"><?php echo esc_html($number_of_owners); ?></span></div><?php endif; ?>
                        <?php if ($transmission): ?><div class="spec-row"><span class="spec-label"><?php esc_html_e('Transmission', 'vehizo-vehicle-management'); ?></span><span class="spec-value"><?php echo esc_html($transmission); ?></span></div><?php endif; ?>
                    </div>
                    <a href="<?php the_permalink(); ?>" class="cta-button"><span><?php esc_html_e('View Details', 'vehizo-vehicle-management'); ?></span><span class="cta-arrow">→</span></a>
                </div>
            </div>
            <?php
        }

        public static function render_pagination($query) {
            $pagination_args = array(
                'total' => $query->max_num_pages,
                'current' => max(1, get_query_var('paged')),
                'prev_next' => false
            );
            echo '<div class="vehizo-pagination">' . wp_kses_post(paginate_links($pagination_args)) . '</div>';
        }

        public static function render_no_results() { ?>
            <div class="no-results"><h3><?php esc_html_e('No vehicles found', 'vehizo-vehicle-management'); ?></h3></div>
        <?php }

        public static function add_body_class($classes) {
            global $post;
            if (is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'vehizo_vehicles')) {
                $classes[] = 'has-vehizo-vehicles';
            }
            return $classes;
        }

        /**
         * AJAX handler to get models for a specific make.
         */
        public static function ajax_get_models_for_make() {
            // Verify nonce
            check_ajax_referer('vehizo_filter_nonce', 'nonce');

            global $wpdb;
            $make = isset($_POST['make']) ? sanitize_text_field(wp_unslash($_POST['make'])) : '';

            if (empty($make)) {
                wp_send_json_error(['message' => __('Make is required.', 'vehizo-vehicle-management')]);
            }

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- AJAX query for dynamic model dropdown based on selected make, requires JOIN for efficiency
            $models = $wpdb->get_col($wpdb->prepare("
                SELECT DISTINCT pm2.meta_value
                FROM {$wpdb->postmeta} pm1
                INNER JOIN {$wpdb->postmeta} pm2 ON pm1.post_id = pm2.post_id
                WHERE pm1.meta_key = 'vehizo_make' AND pm1.meta_value = %s
                AND pm2.meta_key = 'vehizo_model' AND pm2.meta_value != ''
                ORDER BY pm2.meta_value ASC
            ", $make));

            wp_send_json_success($models);
        }

        /**
         * AJAX handler for filtering vehicles.
         */
        public static function ajax_filter_vehicles() {
            check_ajax_referer('vehizo_filter_nonce', 'nonce');

            $atts = array(
                'posts_per_page' => (int)get_option('vehizo_posts_per_page', 12)
            );

            $meta_query = array('relation' => 'AND');
            $filters = array(
                'vehizo_make' => isset($_POST['make']) ? sanitize_text_field(wp_unslash($_POST['make'])) : '',
                'vehizo_model' => isset($_POST['model']) ? sanitize_text_field(wp_unslash($_POST['model'])) : '',
                'vehizo_vehicleType' => isset($_POST['vehicleType']) ? sanitize_text_field(wp_unslash($_POST['vehicleType'])) : '',
                'vehizo_transmission' => isset($_POST['transmission']) ? sanitize_text_field(wp_unslash($_POST['transmission'])) : '',
                'vehizo_fuelType' => isset($_POST['fuelType']) ? sanitize_text_field(wp_unslash($_POST['fuelType'])) : '',
                'vehizo_color' => isset($_POST['color']) ? sanitize_text_field(wp_unslash($_POST['color'])) : '',
                'price_min' => isset($_POST['price_min']) ? absint(wp_unslash($_POST['price_min'])) : 0,
                'price_max' => isset($_POST['price_max']) ? absint(wp_unslash($_POST['price_max'])) : 0,
                'ez_ab' => isset($_POST['ez_ab']) ? absint(wp_unslash($_POST['ez_ab'])) : 0,
                'ez_bis' => isset($_POST['ez_bis']) ? absint(wp_unslash($_POST['ez_bis'])) : 0,
                'mileage_max' => isset($_POST['mileage_max']) ? absint(wp_unslash($_POST['mileage_max'])) : 0
            );

            foreach ($filters as $key => $value) {
                if (!empty($value)) {
                    if ($key === 'price_min') $meta_query[] = array('key' => 'vehizo_price', 'value' => $value, 'compare' => '>=', 'type' => 'NUMERIC');
                    elseif ($key === 'price_max') $meta_query[] = array('key' => 'vehizo_price', 'value' => $value, 'compare' => '<=', 'type' => 'NUMERIC');
                    elseif ($key === 'ez_ab') $meta_query[] = array('key' => 'vehizo_firstRegistration', 'value' => $value, 'compare' => '>=', 'type' => 'NUMERIC');
                    elseif ($key === 'ez_bis') $meta_query[] = array('key' => 'vehizo_firstRegistration', 'value' => $value, 'compare' => '<=', 'type' => 'NUMERIC');
                    elseif ($key === 'mileage_max') $meta_query[] = array('key' => 'vehizo_mileage', 'value' => $value, 'compare' => '<=', 'type' => 'NUMERIC');
                    else $meta_query[] = array('key' => $key, 'value' => $value, 'compare' => '=');
                }
            }

            $orderby = isset($_POST['sort']) ? sanitize_text_field(wp_unslash($_POST['sort'])) : 'date';
            $order = 'DESC';
            $meta_key = '';

            switch ($orderby) {
                case 'price_asc': $orderby = 'meta_value_num'; $meta_key = 'vehizo_price'; $order = 'ASC'; break;
                case 'price_desc': $orderby = 'meta_value_num'; $meta_key = 'vehizo_price'; $order = 'DESC'; break;
                case 'year_desc': $orderby = 'meta_value_num'; $meta_key = 'vehizo_firstRegistration'; $order = 'DESC'; break;
                case 'year_asc': $orderby = 'meta_value_num'; $meta_key = 'vehizo_firstRegistration'; $order = 'ASC'; break;
                case 'mileage_asc': $orderby = 'meta_value_num'; $meta_key = 'vehizo_mileage'; $order = 'ASC'; break;
                case 'title': $orderby = 'title'; $order = 'ASC'; break;
                default: $orderby = 'date'; $order = 'DESC';
            }

            $query_args = array('post_type' => 'vehizo_vehicle', 'post_status' => 'publish', 'posts_per_page' => $atts['posts_per_page'], 'paged' => absint($_POST['paged'] ?? 1), 'meta_query' => $meta_query, 'orderby' => $orderby, 'order' => $order, 'no_found_rows' => false, 'update_post_meta_cache' => true,);
            if (!empty($meta_key)) { $query_args['meta_key'] = $meta_key; }

            $vehicles_query = new WP_Query($query_args);

            ob_start();
            if ($vehicles_query->have_posts()) {
                while ($vehicles_query->have_posts()) { $vehicles_query->the_post(); self::render_vehicle_card(get_the_ID()); }
            } else { self::render_no_results(); }
            $grid_html = ob_get_clean();

            ob_start();
            self::render_pagination($vehicles_query);
            $pagination_html = ob_get_clean();

            wp_send_json_success(['grid_html' => $grid_html, 'pagination_html' => $pagination_html, 'count' => $vehicles_query->found_posts]);
        }


    } // Ende der Klasse

    add_action('plugins_loaded', array('Vehizo_Shortcode_Vehicles', 'init'));
}
