<?php
/**
 * Vehizo Modules Manager
 *
 * Handles module activation/deactivation and settings
 *
 * @package      Vehizo
 * @version      4.1.0
 * @author       Thorsten Glander
 * @since        1.0.0
 */

// Sicherheitscheck - Direkter Zugriff verhindern
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Vehizo_Modules' ) ) {

	/**
	 * Vehizo Modules Management Class
	 *
	 * @since 1.0.0
	 */
	class Vehizo_Modules {

		/**
		 * Initialize the modules
		 *
		 * @since 1.0.0
		 */
		public static function init() {
			// AJAX Actions für Module
			add_action( 'wp_ajax_vehizo_toggle_module_status', array( __CLASS__, 'handle_toggle_module_status' ) );
			add_action( 'wp_ajax_vehizo_get_module_setting', array( __CLASS__, 'handle_get_module_setting' ) );
			add_action( 'wp_ajax_vehizo_save_module_setting', array( __CLASS__, 'handle_save_module_setting' ) );
			
			// Enqueue Scripts and Styles for modules page
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_modules_assets' ) );
		}

		/**
		 * Enqueue styles and scripts for modules page
		 *
		 * @since 4.1.0
		 */
		public static function enqueue_modules_assets( $hook ) {
			// Only load on Vehizo dashboard pages
			if ( $hook !== 'toplevel_page_vehizo-dashboard' && strpos($hook, '_page_vehizo-') === false ) {
				return;
			}

			// KORRIGIERTER PFAD: Zeigt auf das Plugin-Root-Verzeichnis
			wp_enqueue_style(
				'vehizo-modules-style',
				plugin_dir_url( dirname(__FILE__) ) . 'assets/css/modules.css',
				array(),
				VEHIZO_PLUGIN_VERSION
			);

			// KORRIGIERTER PFAD: JavaScript auch korrigieren
			wp_enqueue_script(
				'vehizo-modules-script',
				plugin_dir_url( dirname(__FILE__) ) . 'assets/js/modules.js',
				array( 'jquery' ),
				VEHIZO_PLUGIN_VERSION,
				true
			);

			// Localize script for AJAX
			wp_localize_script(
				'vehizo-modules-script',
				'vehizoModules',
				array(
					'ajaxurl'      => admin_url( 'admin-ajax.php' ),
					'nonces'       => array(
						'toggle_module' => wp_create_nonce( 'vehizo_toggle_module_nonce' ),
						'get_setting'   => wp_create_nonce( 'vehizo_get_setting_nonce' ),
						'save_setting'  => wp_create_nonce( 'vehizo_save_setting_nonce' ),
					),
					'strings'      => array(
						'saving'           => __( 'Saving...', 'vehizo-vehicle-management' ),
						'active'           => __( 'Active', 'vehizo-vehicle-management' ),
						'inactive'         => __( 'Inactive', 'vehizo-vehicle-management' ),
						'error_save'       => __( 'Error saving: ', 'vehizo-vehicle-management' ),
						'error_unknown'    => __( 'Unknown error', 'vehizo-vehicle-management' ),
						'error_save_module' => __( 'Error saving module setting.', 'vehizo-vehicle-management' ),
					)
				)
			);
		}

		/**
		 * Render the modules page
		 *
		 * @since 1.0.0
		 */
		public static function render() {
			// Optionen abrufen
			$contact_active    = get_option( 'vehizo_contact_module_active', '1' );
			$statistics_active = get_option( 'vehizo_statistics_module_active', '0' );
			$merkliste_active  = get_option( 'vehizo_wishlist_module_active', '1' );
			?>

			<div id="save-indicator" class="save-indicator">
				<div class="spinner" id="save-spinner"></div>
				<span id="save-text"><?php esc_html_e( 'Saving...', 'vehizo-vehicle-management' ); ?></span>
			</div>

			<p class="modules-description">
				<?php esc_html_e( 'Activate or deactivate modules according to your needs. Changes are saved automatically.', 'vehizo-vehicle-management' ); ?>
			</p>

			<div class="modules-grid" id="modules-container">
				
				<div class="module-card contact <?php echo esc_attr($contact_active === '1' ? 'active' : 'inactive'); ?> has-settings" id="module-contact">
					<div class="module-header">
						<div class="module-title">
							<div class="module-icon">💬</div>
							<span><?php esc_html_e( 'Contact Form', 'vehizo-vehicle-management' ); ?></span>
						</div>
						<div class="module-status <?php echo esc_attr($contact_active === '1' ? 'active' : 'inactive'); ?>">
							<?php echo $contact_active === '1' ? esc_html__( 'Active', 'vehizo-vehicle-management' ) : esc_html__( 'Inactive', 'vehizo-vehicle-management' ); ?>
						</div>
					</div>
					<div class="module-description">
						<?php esc_html_e( 'Displays an elegant inquiry form on every vehicle page.', 'vehizo-vehicle-management' ); ?>
					</div>
					<div class="module-actions">
						<?php if ( $contact_active === '1' ) : ?>
							<button type="button"
									class="btn-settings"
									data-option="vehizo_contact_email"
									onclick="vehizoOpenSettings(this)"
									aria-label="<?php esc_attr_e( 'Configure contact form settings', 'vehizo-vehicle-management' ); ?>">
								⚙️ <?php esc_html_e( 'Settings', 'vehizo-vehicle-management' ); ?>
							</button>
						<?php else : ?>
							<span></span>
						<?php endif; ?>
						<div class="module-toggle">
							<label class="toggle-switch">
								<input type="checkbox"
									   data-option="vehizo_contact_module_active"
									   <?php checked( $contact_active, '1' ); ?>
									   onchange="vehizoToggleModule(this)"
									   aria-label="<?php esc_attr_e( 'Toggle contact form module', 'vehizo-vehicle-management' ); ?>">
								<span class="toggle-slider"></span>
							</label>
							<span class="toggle-label"><?php esc_html_e( 'Activate module', 'vehizo-vehicle-management' ); ?></span>
						</div>
					</div>
				</div>

				<div class="module-card statistics <?php echo esc_attr($statistics_active === '1' ? 'active' : 'inactive'); ?> has-settings" id="module-statistics">
					<div class="module-header">
						<div class="module-title">
							<div class="module-icon">📊</div>
							<span><?php esc_html_e( 'Vehicle Statistics', 'vehizo-vehicle-management' ); ?></span>
						</div>
						<div class="module-status <?php echo esc_attr($statistics_active === '1' ? 'active' : 'inactive'); ?>">
							<?php echo $statistics_active === '1' ? esc_html__( 'Active', 'vehizo-vehicle-management' ) : esc_html__( 'Inactive', 'vehizo-vehicle-management' ); ?>
						</div>
					</div>
					<div class="module-description">
						<?php esc_html_e( 'Comprehensive analytics for your vehicle inventory.', 'vehizo-vehicle-management' ); ?>
					</div>
					<div class="module-actions">
						 <?php if ( $statistics_active === '1' ) : ?>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=vehizo-dashboard&tab=statistics' ) ); ?>"
							   class="btn-manage"
							   aria-label="<?php esc_attr_e( 'View statistics dashboard', 'vehizo-vehicle-management' ); ?>">
								📊 <?php esc_html_e( 'View Statistics', 'vehizo-vehicle-management' ); ?>
							</a>
						<?php else : ?>
							<span></span>
						<?php endif; ?>
						<div class="module-toggle">
							<label class="toggle-switch">
								<input type="checkbox"
									   data-option="vehizo_statistics_module_active"
									   <?php checked( $statistics_active, '1' ); ?>
									   onchange="vehizoToggleModule(this)"
									   aria-label="<?php esc_attr_e( 'Toggle statistics module', 'vehizo-vehicle-management' ); ?>">
								<span class="toggle-slider"></span>
							</label>
						</div>
					</div>
				</div>

				<div class="module-card merkliste <?php echo esc_attr($merkliste_active === '1' ? 'active' : 'inactive'); ?>" id="module-merkliste">
					<div class="module-header">
						<div class="module-title">
							<div class="module-icon">⭐</div>
							<span><?php esc_html_e( 'Vehicle Wishlist', 'vehizo-vehicle-management' ); ?></span>
						</div>
						<div class="module-status <?php echo esc_attr($merkliste_active === '1' ? 'active' : 'inactive'); ?>">
							<?php echo $merkliste_active === '1' ? esc_html__( 'Active', 'vehizo-vehicle-management' ) : esc_html__( 'Inactive', 'vehizo-vehicle-management' ); ?>
						</div>
					</div>
					<div class="module-description">
						<?php esc_html_e( 'Allows visitors to "park" vehicles on a personal wishlist.', 'vehizo-vehicle-management' ); ?>
					</div>
					<div class="module-actions">
						<div class="module-toggle">
							<label class="toggle-switch">
								<input type="checkbox"
									   data-option="vehizo_wishlist_module_active"
									   <?php checked( $merkliste_active, '1' ); ?>
									   onchange="vehizoToggleModule(this)"
									   aria-label="<?php esc_attr_e( 'Toggle wishlist module', 'vehizo-vehicle-management' ); ?>">
								<span class="toggle-slider"></span>
							</label>
							<span class="toggle-label"><?php esc_html_e( 'Activate module', 'vehizo-vehicle-management' ); ?></span>
						</div>
					</div>
				</div>

				<!-- PRO FEATURES -->
				<div class="module-card pro-feature locked" id="module-pdf-export">
					<div class="pro-badge"><?php esc_html_e( 'PRO', 'vehizo-vehicle-management' ); ?></div>
					<div class="module-header">
						<div class="module-title">
							<div class="module-icon">📄</div>
							<span><?php esc_html_e( 'PDF Export', 'vehizo-vehicle-management' ); ?></span>
						</div>
						<div class="module-status locked">
							<?php esc_html_e( 'Pro Only', 'vehizo-vehicle-management' ); ?>
						</div>
					</div>
					<div class="module-description">
						<?php esc_html_e( 'Generate professional PDF exposés for your vehicles with custom branding.', 'vehizo-vehicle-management' ); ?>
					</div>
					<div class="module-actions">
						<a href="https://vehizo.de" class="btn-upgrade" target="_blank" rel="noopener noreferrer">
							🚀 <?php esc_html_e( 'Upgrade to Pro', 'vehizo-vehicle-management' ); ?>
						</a>
					</div>
				</div>

				<div class="module-card pro-feature locked" id="module-test-drive">
					<div class="pro-badge"><?php esc_html_e( 'PRO', 'vehizo-vehicle-management' ); ?></div>
					<div class="module-header">
						<div class="module-title">
							<div class="module-icon">🚗</div>
							<span><?php esc_html_e( 'Test Drive Scheduler', 'vehizo-vehicle-management' ); ?></span>
						</div>
						<div class="module-status locked">
							<?php esc_html_e( 'Pro Only', 'vehizo-vehicle-management' ); ?>
						</div>
					</div>
					<div class="module-description">
						<?php esc_html_e( 'Let customers schedule test drives directly from vehicle pages.', 'vehizo-vehicle-management' ); ?>
					</div>
					<div class="module-actions">
						<a href="https://vehizo.de" class="btn-upgrade" target="_blank" rel="noopener noreferrer">
							🚀 <?php esc_html_e( 'Upgrade to Pro', 'vehizo-vehicle-management' ); ?>
						</a>
					</div>
				</div>

				<div class="module-card pro-feature locked" id="module-seo">
					<div class="pro-badge"><?php esc_html_e( 'PRO', 'vehizo-vehicle-management' ); ?></div>
					<div class="module-header">
						<div class="module-title">
							<div class="module-icon">🔍</div>
							<span><?php esc_html_e( 'Advanced SEO', 'vehizo-vehicle-management' ); ?></span>
						</div>
						<div class="module-status locked">
							<?php esc_html_e( 'Pro Only', 'vehizo-vehicle-management' ); ?>
						</div>
					</div>
					<div class="module-description">
						<?php esc_html_e( 'Enhanced schema markup and meta tags for better search engine visibility.', 'vehizo-vehicle-management' ); ?>
					</div>
					<div class="module-actions">
						<a href="https://vehizo.de" class="btn-upgrade" target="_blank" rel="noopener noreferrer">
							🚀 <?php esc_html_e( 'Upgrade to Pro', 'vehizo-vehicle-management' ); ?>
						</a>
					</div>
				</div>

			</div>

			<div class="vehizo-modal-overlay" id="vehizo-settings-modal" role="dialog" aria-modal="true" aria-labelledby="vehizo-modal-title">
				<div class="vehizo-modal">
					<div class="vehizo-modal-header">
						<h4 id="vehizo-modal-title"><?php esc_html_e( 'Settings', 'vehizo-vehicle-management' ); ?></h4>
						<button type="button" class="vehizo-modal-close vehizo-js-close-modal" aria-label="<?php esc_attr_e( 'Close modal', 'vehizo-vehicle-management' ); ?>">&times;</button>
					</div>
					<div class="vehizo-modal-content">
						<label for="vehizo-modal-input-email"><?php esc_html_e( 'Email address for inquiries', 'vehizo-vehicle-management' ); ?></label>
						<input type="email" id="vehizo-modal-input-email" placeholder="<?php esc_attr_e( 'e.g. inquiries@dealership.com', 'vehizo-vehicle-management' ); ?>" aria-describedby="vehizo-modal-email-help">
						<input type="hidden" id="vehizo-modal-option-name">
						<small id="vehizo-modal-email-help">
							<?php esc_html_e( 'This email address will receive contact form submissions.', 'vehizo-vehicle-management' ); ?>
						</small>
					</div>
					<div class="vehizo-modal-footer">
						<button type="button" class="vehizo-btn-save-settings" id="vehizo-modal-save-btn">
							<?php esc_html_e( 'Save', 'vehizo-vehicle-management' ); ?>
						</button>
					</div>
				</div>
			</div>
			<?php
		}

		/**
		 * Handle AJAX request to toggle module status
		 *
		 * @since 1.0.0
		 */
		public static function handle_toggle_module_status() {
			$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
			if ( ! wp_verify_nonce( $nonce, 'vehizo_toggle_module_nonce' ) ) {
				wp_send_json_error( [ 'message' => __( 'Security check failed.', 'vehizo-vehicle-management' ) ], 403 );
			}

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( [ 'message' => __( 'Insufficient permissions.', 'vehizo-vehicle-management' ) ], 403 );
			}

			$module_option = isset( $_POST['module_option'] ) ? sanitize_key( $_POST['module_option'] ) : '';
			$status        = isset( $_POST['status'] ) ? sanitize_key( $_POST['status'] ) : '';

			if ( empty( $module_option ) || ! in_array( $status, array( '0', '1' ), true ) ) {
				wp_send_json_error( [ 'message' => __( 'Invalid parameters.', 'vehizo-vehicle-management' ) ] );
			}

			$allowed_options = array(
				'vehizo_contact_module_active',
				'vehizo_statistics_module_active',
				'vehizo_wishlist_module_active',
			);

			if ( ! in_array( $module_option, $allowed_options, true ) ) {
				wp_send_json_error( [ 'message' => __( 'Invalid module option.', 'vehizo-vehicle-management' ) ] );
			}

			if ( update_option( $module_option, $status ) ) {
				do_action( 'vehizo_module_status_changed', $module_option, $status );
				wp_send_json_success(
					[
						'message' => __( 'Module status updated successfully.', 'vehizo-vehicle-management' ),
					]
				);
			} else {
				wp_send_json_error( [ 'message' => __( 'Failed to update module status.', 'vehizo-vehicle-management' ) ] );
			}
		}

		/**
		 * Handle AJAX request to get module setting
		 *
		 * @since 1.0.0
		 */
		public static function handle_get_module_setting() {
			$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
			if ( ! wp_verify_nonce( $nonce, 'vehizo_get_setting_nonce' ) ) {
				wp_send_json_error( [ 'message' => __( 'Security check failed.', 'vehizo-vehicle-management' ) ], 403 );
			}

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( [ 'message' => __( 'Insufficient permissions.', 'vehizo-vehicle-management' ) ], 403 );
			}

			$option_name = isset( $_POST['option_name'] ) ? sanitize_key( $_POST['option_name'] ) : '';

			if ( empty( $option_name ) ) {
				wp_send_json_error( [ 'message' => __( 'Option name is required.', 'vehizo-vehicle-management' ) ] );
			}

			$value = get_option( $option_name, '' );

			wp_send_json_success(
				[
					'value' => $value,
				]
			);
		}

		/**
		 * Handle AJAX request to save module setting
		 *
		 * @since 1.0.0
		 */
		public static function handle_save_module_setting() {
			$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
			if ( ! wp_verify_nonce( $nonce, 'vehizo_save_setting_nonce' ) ) {
				wp_send_json_error( [ 'message' => __( 'Security check failed.', 'vehizo-vehicle-management' ) ], 403 );
			}

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( [ 'message' => __( 'Insufficient permissions.', 'vehizo-vehicle-management' ) ], 403 );
			}

			$option_name   = isset( $_POST['option_name'] ) ? sanitize_key( $_POST['option_name'] ) : '';
			$setting_value = isset( $_POST['setting_value'] ) ? sanitize_text_field( wp_unslash( $_POST['setting_value'] ) ) : '';

			if ( empty( $option_name ) ) {
				wp_send_json_error( [ 'message' => __( 'Option name is required.', 'vehizo-vehicle-management' ) ] );
			}

			// Whitelist of allowed options to prevent arbitrary option updates
			$allowed_settings = array(
				'vehizo_contact_email',
				'vehizo_contact_module_active',
				'vehizo_statistics_module_active',
				'vehizo_wishlist_module_active',
				'vehizo_primary_color',
				'vehizo_secondary_color',
				'vehizo_button_style',
			);

			if ( ! in_array( $option_name, $allowed_settings, true ) ) {
				wp_send_json_error( [ 'message' => __( 'Invalid option name.', 'vehizo-vehicle-management' ) ], 400 );
			}

			if ( strpos( $option_name, 'email' ) !== false && ! empty( $setting_value ) ) {
				if ( ! is_email( $setting_value ) ) {
					wp_send_json_error( [ 'message' => __( 'Please enter a valid email address.', 'vehizo-vehicle-management' ) ] );
				}
				$setting_value = sanitize_email( $setting_value );
			}

			if ( update_option( $option_name, $setting_value ) ) {
				do_action( 'vehizo_module_setting_saved', $option_name, $setting_value );
				wp_send_json_success(
					[
						'message' => __( 'Setting saved successfully.', 'vehizo-vehicle-management' ),
					]
				);
			} else {
				wp_send_json_error( [ 'message' => __( 'Failed to save setting. The value might be unchanged.', 'vehizo-vehicle-management' ) ] );
			}
		}
	}

	// Initialize the modules class
	add_action( 'init', array( 'Vehizo_Modules', 'init' ) );
}
