<?php
/**
 * Vehizo External API Importer
 *
 * Handles vehicle import from external automotive data APIs
 *
 * @package      Vehizo
 * @version      4.1.0
 * @author       Thorsten Glander
 * @since        1.0.0
 */

// Sicherheitscheck - Direkter Zugriff verhindern
if (!defined('ABSPATH')) {
    exit;
}

/**
 * External API Importer Class
 *
 * @since 1.0.0
 */
class Vehizo_Mobile_Importer
{
    /**
     * Maximum allowed pages per import run
     *
     * @since 1.0.0
     * @var int
     */
    const MAX_PAGES = 10;

    /**
     * Page size for API requests
     *
     * @since 1.0.0
     * @var int
     */
    const PAGE_SIZE = 100;

    /**
     * API request timeout in seconds
     *
     * @since 1.0.0
     * @var int
     */
    const REQUEST_TIMEOUT = 30;

    /**
     * Initialize the importer
     *
     * @since 1.0.0
     */
    public static function init()
    {
        // Register admin scripts and styles
        add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_admin_assets'));
        
        // Register AJAX handlers
        add_action('wp_ajax_vehizo_manual_import', array(__CLASS__, 'handle_ajax_import'));
        add_action('wp_ajax_vehizo_test_api', array(__CLASS__, 'handle_ajax_test_api'));
    }

    /**
     * Enqueue admin assets for the importer
     *
     * @since 1.0.0
     * @param string $hook_suffix Current admin page hook suffix
     */
    public static function enqueue_admin_assets($hook_suffix)
    {
        // Only load on relevant admin pages
        if (!self::is_vehizo_admin_page($hook_suffix)) {
            return;
        }

        // Enqueue admin JavaScript
        wp_enqueue_script(
            'vehizo-mobile-importer',
            plugin_dir_url(dirname(__FILE__)) . 'assets/js/mobile-importer.js',
            array('jquery'),
            defined('VEHIZO_VERSION') ? VEHIZO_VERSION : '1.0.0',
            true
        );

        // Enqueue admin CSS
        wp_enqueue_style(
            'vehizo-mobile-importer',
            plugin_dir_url(dirname(__FILE__)) . 'assets/css/mobile-importer.css',
            array(),
            defined('VEHIZO_VERSION') ? VEHIZO_VERSION : '1.0.0'
        );

        // Localize script with necessary data
        wp_localize_script('vehizo-mobile-importer', 'vehizoMobileImporter', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('vehizo_mobile_nonce'),
            'strings' => array(
                'importing' => __('Importing vehicles...', 'vehizo-vehicle-management'),
                'importComplete' => __('Import completed successfully!', 'vehizo-vehicle-management'),
                'importError' => __('Import failed. Please try again.', 'vehizo-vehicle-management'),
                'testingConnection' => __('Testing API connection...', 'vehizo-vehicle-management'),
                'connectionSuccess' => __('API connection successful!', 'vehizo-vehicle-management'),
                'connectionFailed' => __('API connection failed.', 'vehizo-vehicle-management'),
                'confirmImport' => __('Are you sure you want to start the import? This may take several minutes.', 'vehizo-vehicle-management'),
                'pleaseWait' => __('Please wait...', 'vehizo-vehicle-management')
            ),
            'config' => array(
                'maxPages' => self::MAX_PAGES,
                'pageSize' => self::PAGE_SIZE,
                'timeout' => self::REQUEST_TIMEOUT * 1000 // Convert to milliseconds for JS
            )
        ));
    }

    /**
     * Check if current admin page should load importer assets
     *
     * @since 1.0.0
     * @param string $hook_suffix Current admin page hook suffix
     * @return bool
     */
    private static function is_vehizo_admin_page($hook_suffix)
    {
        // List of admin pages where importer assets should be loaded
        $vehizo_pages = array(
            'vehizo_page_vehizo-import',
            'vehizo_page_vehizo-settings',
            'toplevel_page_vehizo-dashboard'
        );

        return in_array($hook_suffix, $vehizo_pages, true);
    }

    /**
     * Handle AJAX import request
     *
     * @since 1.0.0
     */
    public static function handle_ajax_import()
    {
        // Nonce verification
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'vehizo_mobile_nonce')) {
            wp_send_json_error(array('message' => esc_html__('Security check failed', 'vehizo-vehicle-management')));
            return;
        }

        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions', 'vehizo-vehicle-management')));
            return;
        }

        // Get API credentials
        $credentials = self::get_api_credentials();
        
        if (!$credentials) {
            wp_send_json_error(array('message' => esc_html__('API credentials are missing.', 'vehizo-vehicle-management')));
            return;
        }

        // Run import
        $result = self::import($credentials['username'], $credentials['password'], $credentials['api_url']);

        if ($result['error']) {
            update_option('vehizo_last_cron_status', $result['error']);
            wp_send_json_error(array('message' => $result['error']));
        } else {
            update_option('vehizo_last_cron_status', __('Successful (manual)', 'vehizo-vehicle-management'));
            wp_send_json_success(array(
                'message' => $result['message'],
                'stats' => self::get_import_stats()
            ));
        }

        // Update last import time
        update_option('vehizo_last_cron_import', current_time('timestamp'));
    }

    /**
     * Handle AJAX API test request
     *
     * @since 1.0.0
     */
    public static function handle_ajax_test_api()
    {
        // Nonce verification
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'vehizo_mobile_nonce')) {
            wp_send_json_error(array('message' => esc_html__('Security check failed', 'vehizo-vehicle-management')));
            return;
        }

        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions', 'vehizo-vehicle-management')));
            return;
        }

        // Test API connection
        $test_result = self::test_api_connection();

        if ($test_result['success']) {
            wp_send_json_success(array('message' => $test_result['message']));
        } else {
            wp_send_json_error(array('message' => $test_result['message']));
        }
    }

    /**
     * Legacy handle method for backward compatibility
     *
     * @since 1.0.0
     * @deprecated Use handle_ajax_import() instead
     */
    public static function handle()
    {
        self::handle_ajax_import();
    }

    /**
     * Run scheduled cron import
     * Note: Automatic cron imports are a Pro feature.
     * This method is kept for compatibility and can be used by extensions.
     *
     * @since 1.0.0
     */
    public static function run_cron_import()
    {
        // This functionality can be extended by Pro add-ons
        do_action('vehizo_before_cron_import');

        // Get API credentials
        $credentials = self::get_api_credentials();

        if (!$credentials) {
            update_option('vehizo_last_cron_status', __('API credentials are missing', 'vehizo-vehicle-management'));
            return;
        }

        // Run import
        $result = self::import($credentials['username'], $credentials['password'], $credentials['api_url']);

        // Update status
        $status = $result['error'] ? $result['error'] : __('Successful', 'vehizo-vehicle-management');
        update_option('vehizo_last_cron_status', $status);
        update_option('vehizo_last_cron_import', current_time('timestamp'));

        // Log result for debugging
        if (defined('WP_DEBUG') && WP_DEBUG) {
        }

        do_action('vehizo_after_cron_import', $result);
    }

    /**
     * Get API credentials from options
     *
     * @since 1.0.0
     * @return array|false Array of credentials or false if invalid
     */
    private static function get_api_credentials()
    {
        $username = get_option('vehizo_mobile_username', '');
        $password = get_option('vehizo_mobile_password', '');
        $api_url = get_option('vehizo_mobile_api_url', 'https://services.sandbox.mobile.de/search-api/search');

        // Validate credentials
        if (empty($username) || empty($password) || empty($api_url)) {
            return false;
        }

        // Validate API URL
        if (!filter_var($api_url, FILTER_VALIDATE_URL)) {
            return false;
        }

        return array(
            'username' => sanitize_text_field($username),
            'password' => $password, // Don't sanitize password as it might contain special chars
            'api_url' => esc_url_raw($api_url)
        );
    }

    /**
     * Import vehicles from the API
     *
     * @since 1.0.0
     * @param string $username API username
     * @param string $password API password
     * @param string $api_url API endpoint URL
     * @return array Result array with error status and message
     */
    private static function import($username, $password, $api_url)
    {
        // Set time limit for import process
        if (!defined('WP_CLI') || !WP_CLI) {
            // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged, Squiz.PHP.DiscouragedFunctions.Discouraged -- Necessary for long-running import operations
            @set_time_limit(300); // 5 minutes for web requests
        }
        // Get current vehicle count
        $current_vehicle_count = self::get_current_vehicle_count();

        // DEBUG: Log import start
        if (defined('WP_DEBUG') && WP_DEBUG) {
        }

        // Initialize counters
        $page = 1;
        $has_more = true;
        $processed = 0;
        $newly_imported = 0;
        $updated = 0;
        $errors = 0;

        // Process pages
        while ($has_more && $page <= self::MAX_PAGES) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
            }

            // Build API request URL
            $url = add_query_arg(array(
                'page' => absint($page),
                'pageSize' => self::PAGE_SIZE
            ), $api_url);

            // Make API request
            $api_response = self::make_api_request($url, $username, $password);

            if (is_wp_error($api_response)) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                }
                return array(
                    /* translators: %s: Error message from external API */
                    'error' => sprintf(__('Error retrieving data from external API: %s', 'vehizo-vehicle-management'), $api_response->get_error_message()),
                    'message' => ''
                );
            }

            $data = $api_response;

            if (defined('WP_DEBUG') && WP_DEBUG) {
            }

            // Check if we have vehicles in response
            if (empty($data['ads']) || !is_array($data['ads'])) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                }
                break;
            }

            // Process each vehicle
            foreach ($data['ads'] as $ad) {
                $fahrzeug_id = sanitize_text_field($ad['mobileAdId'] ?? '');
                
                if (empty($fahrzeug_id)) {
                    $errors++;
                    continue;
                }

                // Check for existing vehicle
                $existing_post_id = self::get_existing_vehicle($fahrzeug_id);

                // Prepare vehicle data
                $vehicle_data = self::prepare_vehicle_data($ad);
                
                if (!$vehicle_data) {
                    $errors++;
                    continue; // Skip invalid data
                }

                if ($existing_post_id) {
                    // Update existing vehicle
                    $result = self::update_vehicle($existing_post_id, $vehicle_data);
                    if ($result) {
                        $updated++;
                    } else {
                        $errors++;
                    }
                } else {
                    // Create new vehicle
                    $new_post_id = self::create_vehicle($vehicle_data);
                    if ($new_post_id) {
                        $newly_imported++;
                        $current_vehicle_count++;
                    } else {
                        $errors++;
                    }
                }
                
                $processed++;

                // Prevent memory issues with large imports
                if (($processed % 50) === 0) {
                    wp_cache_flush();
                }
            }

            // Check for next page
            $has_more = isset($data['paging']['next']) && $data['paging']['next'];
            $page++;

            // Small delay between pages to be nice to the API
            if ($has_more && $page <= self::MAX_PAGES) {
                usleep(500000); // 0.5 seconds
            }
        }

        // DEBUG: Log final results
        if (defined('WP_DEBUG') && WP_DEBUG) {
        }

        // Build success message
        $message_parts = array(
            sprintf(
                /* translators: %d: Number of vehicles that were processed during import */
                __('%d vehicles processed', 'vehizo-vehicle-management'),
                $processed
            )
        );

        if ($newly_imported > 0) {
            /* translators: %d: Number of vehicles that were newly imported (first time) */
            $message_parts[] = sprintf(__('%d newly imported', 'vehizo-vehicle-management'), $newly_imported);
        }

        if ($updated > 0) {
            /* translators: %d: Number of vehicles that were updated with new data */
            $message_parts[] = sprintf(__('%d updated', 'vehizo-vehicle-management'), $updated);
        }

        if ($errors > 0) {
            /* translators: %d: Number of errors encountered during import process */
            $message_parts[] = sprintf(__('%d errors', 'vehizo-vehicle-management'), $errors);
        }

        $message = implode(', ', $message_parts) . '.';

        // Fire action hook for extensions
        do_action('vehizo_import_completed', $processed, $newly_imported, $updated, $errors);

        // Update import status and time
        update_option('vehizo_last_cron_status', 'Successful (manual)');
        update_option('vehizo_last_cron_import', current_time('timestamp'));

        // DEBUG: Verify import time was saved
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $saved_time = get_option('vehizo_last_cron_import', 'not found');
            $saved_status = get_option('vehizo_last_cron_status', 'not found');
        }

        return array('error' => false, 'message' => $message);
    }

    /**
     * Make API request to the vehicle portal
     *
     * @since 1.0.0
     * @param string $url Request URL
     * @param string $username API username
     * @param string $password API password
     * @return array|WP_Error Response data or error
     */
    private static function make_api_request($url, $username, $password)
    {
        $response = wp_remote_get($url, array(
            'headers' => array(
                'Accept' => 'application/vnd.de.mobile.api+json',
                'Authorization' => 'Basic ' . base64_encode($username . ':' . $password),
                'User-Agent' => 'Vehizo/' . (defined('VEHIZO_VERSION') ? VEHIZO_VERSION : '1.0.0'),
                'Content-Type' => 'application/json'
            ),
            'timeout' => self::REQUEST_TIMEOUT,
            'sslverify' => true,
            'redirection' => 5
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code !== 200) {
            $error_body = wp_remote_retrieve_body($response);
            return new WP_Error(
                'api_error',
                /* translators: %1$d: HTTP error code, %2$s: Error message from API */
                sprintf(__('API returned error code %1$d: %2$s', 'vehizo-vehicle-management'), $response_code, $error_body)
            );
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error(
                'json_error',
                /* translators: %s: JSON decoding error message */
                sprintf(__('Invalid JSON response from API: %s', 'vehizo-vehicle-management'), json_last_error_msg())
            );
        }

        return $data;
    }

    /**
     * Get current vehicle count
     *
     * @since 1.0.0
     * @return int Number of existing vehicles
     */
    private static function get_current_vehicle_count()
    {
        $count = wp_count_posts('vehizo_vehicle');
        
        return ($count->publish ?? 0) + ($count->draft ?? 0) + ($count->pending ?? 0) + ($count->private ?? 0);
    }

    /**
     * Check if vehicle already exists
     *
     * @since 1.0.0
     * @param string $fahrzeug_id Portal's vehicle ID
     * @return int|false Post ID if exists, false otherwise
     */
    private static function get_existing_vehicle($fahrzeug_id)
    {
        global $wpdb;

        // Check if post_id actually exists as a published post
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Complex JOIN query required to check vehicle existence during import, WP_Query alternative would be less efficient
        $post_id = $wpdb->get_var($wpdb->prepare(
            "SELECT pm.post_id FROM {$wpdb->postmeta} pm
             INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
             WHERE pm.meta_key = 'vehizo_fahrzeug_id'
             AND pm.meta_value = %s
             AND p.post_type = 'vehizo_vehicle'
             AND p.post_status != 'trash'
             LIMIT 1",
            $fahrzeug_id
        ));

        $result = $post_id ? intval($post_id) : false;

        if (defined('WP_DEBUG') && WP_DEBUG) {
        }

        return $result;
    }

    /**
     * Prepare vehicle data from API response
     *
     * @since 1.0.0
     * @param array $ad Vehicle data from API
     * @return array|false Prepared vehicle data or false if invalid
     */
    private static function prepare_vehicle_data($ad)
    {
        if (!is_array($ad)) {
            return false;
        }

        $fahrzeug_id = sanitize_text_field($ad['mobileAdId'] ?? '');
        $make = sanitize_text_field($ad['make'] ?? '');
        $model = sanitize_text_field($ad['model'] ?? '');

        if (empty($fahrzeug_id)) {
            return false;
        }

        // Create title
        $title = trim($make . ' ' . $model);
        if (empty($title)) {
            $title = __('Imported Vehicle', 'vehizo-vehicle-management') . ' ' . $fahrzeug_id;
        }

        // Prepare meta data with vehizo_ prefix
        $meta_data = array(
            'vehizo_fahrzeug_id' => $fahrzeug_id,
            'vehizo_make' => $make,
            'vehizo_model' => $model,
            'vehizo_price' => self::sanitize_price($ad['price']['consumerPriceGross'] ?? ''),
            'vehizo_mileage' => absint($ad['mileage'] ?? 0),
            'vehizo_firstRegistration' => self::sanitize_year($ad['firstRegistration'] ?? ''),
            'vehizo_fuelType' => sanitize_text_field($ad['fuelType'] ?? ''),
            'vehizo_transmission' => sanitize_text_field($ad['transmission'] ?? ''),
            'vehizo_power' => absint($ad['power'] ?? 0),
            'vehizo_vehicleType' => sanitize_text_field($ad['vehicleType'] ?? ''),
            'vehizo_color' => sanitize_text_field($ad['color'] ?? ''),
            'vehizo_doors' => absint($ad['doors'] ?? 0),
            'vehizo_seats' => absint($ad['seats'] ?? 0),
            'vehizo_engine_displacement' => absint($ad['engineDisplacement'] ?? 0),
            'vehizo_imported_at' => current_time('mysql'),
            'vehizo_api_source' => 'mobile.de'
        );

        // NEU: Zusätzliche Felder hinzufügen, falls sie in der API-Antwort vorhanden sind
        if (!empty($ad['vin'])) {
            $meta_data['vehizo_vin'] = sanitize_text_field($ad['vin']);
        }
        if (isset($ad['numberOfPreviousOwners'])) {
            $meta_data['vehizo_number_of_owners'] = absint($ad['numberOfPreviousOwners']);
        }
        if (!empty($ad['nextInspection'])) {
            $meta_data['vehizo_next_inspection'] = sanitize_text_field($ad['nextInspection']);
        }
        // Add optional fields if present
        if (!empty($ad['vin'])) {
            $meta_data['vehizo_vin'] = sanitize_text_field($ad['vin']);
        }

        if (!empty($ad['category'])) {
            $meta_data['vehizo_category'] = sanitize_text_field($ad['category']);
        }

        // Filter empty values but keep zeros
        $meta_data = array_filter($meta_data, function($value) {
            return $value !== '' && $value !== null;
        });

        return array(
            'title' => $title,
            'meta' => $meta_data,
            'features' => !empty($ad['features']) && is_array($ad['features']) ? $ad['features'] : [],
            'content' => wp_kses_post($ad['description'] ?? ''), // Nur noch die Beschreibung
            'excerpt' => self::generate_vehicle_excerpt($ad)
        );
    }

    /**
     * Sanitize price value
     *
     * @since 1.0.0
     * @param mixed $price Price value
     * @return int Sanitized price
     */
    private static function sanitize_price($price)
    {
        if (is_numeric($price)) {
            return max(0, intval($price));
        }
        
        // Try to extract numeric value from string
        $numeric = preg_replace('/[^0-9]/', '', (string)$price);
        return $numeric ? intval($numeric) : 0;
    }

    /**
     * Sanitize year value
     *
     * @since 1.0.0
     * @param mixed $year Year value
     * @return int Sanitized year
     */
    private static function sanitize_year($year)
    {
        $year = intval($year);
        $current_year = intval(gmdate('Y'));
        
        // Valid range: 1900 to current year + 1
        if ($year >= 1900 && $year <= ($current_year + 1)) {
            return $year;
        }
        
        return 0;
    }

    /**
     * Generate vehicle content from API data
     *
     * @since 1.0.0
     * @param array $ad Vehicle data
     * @return string Generated content
     */
    private static function generate_vehicle_content($ad)
    {
        // Diese Funktion wird vereinfacht, da die Features jetzt als Taxonomie behandelt werden.
        // Der Inhalt wird direkt in prepare_vehicle_data gesetzt.
        // Wir behalten die Funktion für mögliche zukünftige Erweiterungen.
        return wp_kses_post($ad['description'] ?? '');
    }

    /**
     * Generate vehicle excerpt from API data
     *
     * @since 1.0.0
     * @param array $ad Vehicle data
     * @return string Generated excerpt
     */
    private static function generate_vehicle_excerpt($ad)
    {
        $excerpt_parts = array();
        
        if (!empty($ad['make']) && !empty($ad['model'])) {
            $excerpt_parts[] = esc_html($ad['make']) . ' ' . esc_html($ad['model']);
        }

        if (!empty($ad['firstRegistration'])) {
            /* translators: %s: Vehicle registration year */
            $excerpt_parts[] = sprintf(esc_html__('Year: %s', 'vehizo-vehicle-management'), esc_html($ad['firstRegistration']));
        }

        if (!empty($ad['mileage'])) {
            /* translators: %s: Vehicle mileage in kilometers */
            $excerpt_parts[] = sprintf(esc_html__('Mileage: %s km', 'vehizo-vehicle-management'), number_format_i18n(absint($ad['mileage'])));
        }

        if (!empty($ad['price']['consumerPriceGross'])) {
            $price = self::sanitize_price($ad['price']['consumerPriceGross']);
            if ($price > 0) {
                /* translators: %s: Vehicle price in Euros */
                $excerpt_parts[] = sprintf(esc_html__('Price: %s €', 'vehizo-vehicle-management'), number_format_i18n($price));
            }
        }

        return implode(' | ', $excerpt_parts);
    }

    /**
     * Create new vehicle post
     *
     * @since 1.0.0
     * @param array $vehicle_data Prepared vehicle data
     * @return int|false Post ID on success, false on failure
     */
    private static function create_vehicle($vehicle_data)
    {
        $post_data = array(
            'post_type' => 'vehizo_vehicle',
            'post_title' => $vehicle_data['title'],
            'post_content' => $vehicle_data['content'] ?? '',
            'post_excerpt' => $vehicle_data['excerpt'] ?? '',
            'post_status' => 'publish',
            'meta_input' => $vehicle_data['meta']
        );

        $post_id = wp_insert_post($post_data, true);

        if (is_wp_error($post_id)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
            }
            return false;
        }

        // NEU: Ausstattungsmerkmale als Taxonomie-Begriffe setzen
        if (!empty($vehicle_data['features'])) {
            wp_set_object_terms($post_id, $vehicle_data['features'], 'vehizo_feature', false);
        }

        // Fire action hook
        do_action('vehizo_vehicle_imported', $post_id, $vehicle_data);

        return $post_id;
    }

    /**
     * Update existing vehicle post
     *
     * @since 1.0.0
     * @param int $post_id Existing post ID
     * @param array $vehicle_data Updated vehicle data
     * @return bool Success status
     */
    private static function update_vehicle($post_id, $vehicle_data)
    {
        $post_data = array(
            'ID' => $post_id,
            'post_title' => $vehicle_data['title'],
            'post_content' => $vehicle_data['content'] ?? '',
            'post_excerpt' => $vehicle_data['excerpt'] ?? ''
        );

        $result = wp_update_post($post_data, true);

        if (is_wp_error($result)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
            }
            return false;
        }

        // Update meta data
        foreach ($vehicle_data['meta'] as $key => $value) {
            update_post_meta($post_id, $key, $value);
        }

        // NEU: Ausstattungsmerkmale als Taxonomie-Begriffe aktualisieren (ersetzt die alten)
        if (isset($vehicle_data['features'])) { // Prüfen, ob 'features' im Array ist
            wp_set_object_terms($post_id, $vehicle_data['features'], 'vehizo_feature', false);
        }

        // Fire action hook
        do_action('vehizo_vehicle_updated', $post_id, $vehicle_data);

        return true;
    }

    /**
     * Get import statistics
     *
     * @since 1.0.0
     * @return array Import statistics
     */
    public static function get_import_stats()
    {
        $stats = array(
            'total_vehicles' => self::get_current_vehicle_count(),
            'last_import' => get_option('vehizo_last_cron_import', ''),
            'last_status' => get_option('vehizo_last_cron_status', ''),
            'api_source' => 'mobile.de'
        );

        // Format last import date
        if (!empty($stats['last_import'])) {
            $stats['last_import_formatted'] = wp_date(
                get_option('date_format') . ' ' . get_option('time_format'),
                $stats['last_import']
            );
        }

        return $stats;
    }

    /**
     * Test API connection
     *
     * @since 1.0.0
     * @return array Test result
     */
    public static function test_api_connection()
    {
        $credentials = self::get_api_credentials();
        
        if (!$credentials) {
            return array(
                'success' => false,
                'message' => __('API credentials are missing', 'vehizo-vehicle-management')
            );
        }

        // Test with minimal request
        $test_url = add_query_arg(array(
            'page' => 1,
            'pageSize' => 1
        ), $credentials['api_url']);

        $result = self::make_api_request($test_url, $credentials['username'], $credentials['password']);

        if (is_wp_error($result)) {
            return array(
                'success' => false,
                /* translators: %s: Error message from connection attempt */
                'message' => sprintf(__('Connection failed: %s', 'vehizo-vehicle-management'), $result->get_error_message())
            );
        }

        return array(
            'success' => true,
            'message' => __('API connection successful', 'vehizo-vehicle-management')
        );
    }

    /**
     * Clean up old imported vehicles
     *
     * @since 1.0.0
     * @param int $days Remove vehicles older than X days
     * @return int Number of deleted vehicles
     */
    public static function cleanup_old_vehicles($days = 30)
    {
        if ($days < 1) {
            return 0;
        }

        $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $old_vehicles = get_posts(array(
            'post_type' => 'vehizo_vehicle',
            'post_status' => 'any',
            'numberposts' => -1,
            'date_query' => array(
                array(
                    'before' => $cutoff_date,
                    'inclusive' => false
                )
            ),
            'meta_query' => array(
                array(
                    'key' => 'vehizo_api_source',
                    'value' => 'mobile.de',
                    'compare' => '='
                )
            ),
            'fields' => 'ids'
        ));

        $deleted_count = 0;
        
        foreach ($old_vehicles as $vehicle_id) {
            if (wp_delete_post($vehicle_id, true)) {
                $deleted_count++;
            }
        }

        if ($deleted_count > 0) {
            do_action('vehizo_vehicles_cleaned_up', $deleted_count, $days);
        }

        return $deleted_count;
    }

    /**
     * Get progress information for ongoing imports
     *
     * @since 1.0.0
     * @return array Progress information
     */
    public static function get_import_progress()
    {
        $progress = get_transient('vehizo_import_progress');
        
        if (!$progress) {
            return array(
                'is_running' => false,
                'current_page' => 0,
                'total_pages' => 0,
                'processed' => 0,
                'imported' => 0,
                'updated' => 0
            );
        }

        return $progress;
    }

    /**
     * Set import progress (for long-running imports)
     *
     * @since 1.0.0
     * @param array $progress Progress data
     */
    private static function set_import_progress($progress)
    {
        set_transient('vehizo_import_progress', $progress, 600); // 10 minutes expiry
    }

    /**
     * Clear import progress
     *
     * @since 1.0.0
     */
    private static function clear_import_progress()
    {
        delete_transient('vehizo_import_progress');
    }

    /**
     * Handle bulk import operations
     *
     * @since 1.0.0
     * @param array $options Import options
     * @return array Result
     */
    public static function bulk_import($options = array())
    {
        $defaults = array(
            'max_pages' => self::MAX_PAGES,
            'page_size' => self::PAGE_SIZE,
            'update_existing' => true,
            'skip_images' => false
        );

        $options = wp_parse_args($options, $defaults);

        // Get credentials
        $credentials = self::get_api_credentials();
        if (!$credentials) {
            return array(
                'error' => true,
                'message' => __('API credentials are missing', 'vehizo-vehicle-management')
            );
        }

        // Set initial progress
        self::set_import_progress(array(
            'is_running' => true,
            'current_page' => 0,
            'total_pages' => $options['max_pages'],
            'processed' => 0,
            'imported' => 0,
            'updated' => 0
        ));

        $result = self::import($credentials['username'], $credentials['password'], $credentials['api_url']);
        
        // Clear progress when done
        self::clear_import_progress();

        return $result;
    }

    /**
     * Validate vehicle data before processing
     *
     * @since 1.0.0
     * @param array $ad Raw vehicle data from API
     * @return bool Whether data is valid
     */
    private static function validate_vehicle_data($ad)
    {
        // Required fields
        $required_fields = array('mobileAdId', 'make', 'model');
        
        foreach ($required_fields as $field) {
            if (empty($ad[$field])) {
                return false;
            }
        }

        // Validate ID format (should be numeric or alphanumeric)
        if (!preg_match('/^[a-zA-Z0-9]+$/', $ad['mobileAdId'])) {
            return false;
        }

        return true;
    }

    /**
     * Process vehicle images from API
     *
     * @since 1.0.0
     * @param array $images Image data from API
     * @param int $post_id Vehicle post ID
     * @return array Processed image IDs
     */
    private static function process_vehicle_images($images, $post_id)
    {
        if (empty($images) || !is_array($images)) {
            return array();
        }

        $processed_images = array();
        $max_images = apply_filters('vehizo_max_images_per_vehicle', 10);

        foreach (array_slice($images, 0, $max_images) as $image_data) {
            if (empty($image_data['url'])) {
                continue;
            }

            $image_url = esc_url_raw($image_data['url']);
            $image_id = self::download_and_attach_image($image_url, $post_id);

            if ($image_id) {
                $processed_images[] = $image_id;
            }
        }

        return $processed_images;
    }

    /**
     * Download and attach image to post
     *
     * @since 1.0.0
     * @param string $url Image URL
     * @param int $post_id Post ID to attach to
     * @return int|false Attachment ID or false on failure
     */
    private static function download_and_attach_image($url, $post_id)
    {
        if (!function_exists('media_handle_sideload')) {
            require_once ABSPATH . 'wp-admin/includes/media.php';
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }

        // Download image
        $tmp = download_url($url);
        
        if (is_wp_error($tmp)) {
            return false;
        }

        // Prepare file array
        $file_array = array(
            'name' => basename($url),
            'tmp_name' => $tmp
        );

        // Handle sideload
        $attachment_id = media_handle_sideload($file_array, $post_id);

        // Clean up temp file
        if (file_exists($tmp)) {
            wp_delete_file($tmp);
        }

        if (is_wp_error($attachment_id)) {
            return false;
        }

        return $attachment_id;
    }

    /**
     * Get vehicles that need updating
     *
     * @since 1.0.0
     * @param int $hours Hours since last update
     * @return array Vehicle IDs that need updating
     */
    public static function get_vehicles_needing_update($hours = 24)
    {
        $cutoff_time = gmdate('Y-m-d H:i:s', strtotime("-{$hours} hours"));

        $vehicles = get_posts(array(
            'post_type' => 'vehizo_vehicle',
            'post_status' => 'publish',
            'numberposts' => -1,
            'meta_query' => array(
                'relation' => 'AND',
                array(
                    'key' => 'vehizo_api_source',
                    'value' => 'mobile.de',
                    'compare' => '='
                ),
                array(
                    'key' => 'vehizo_imported_at',
                    'value' => $cutoff_time,
                    'compare' => '<',
                    'type' => 'DATETIME'
                )
            ),
            'fields' => 'ids'
        ));

        return $vehicles;
    }

    /**
     * Schedule automatic cleanup of old vehicles
     *
     * @since 1.0.0
     */
    public static function schedule_cleanup()
    {
        if (!wp_next_scheduled('vehizo_cleanup_old_vehicles')) {
            wp_schedule_event(time(), 'weekly', 'vehizo_cleanup_old_vehicles');
        }
    }

    /**
     * Unschedule automatic cleanup
     *
     * @since 1.0.0
     */
    public static function unschedule_cleanup()
    {
        $timestamp = wp_next_scheduled('vehizo_cleanup_old_vehicles');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'vehizo_cleanup_old_vehicles');
        }
    }
}