<?php
/**
 * Vehizo Appearance Settings - FINAL COMPLETE VERSION
 *
 * Handles frontend appearance and layout settings.
 * This class now only manages backend settings and provides the CSS generation method.
 *
 * @package      Vehizo
 * @version      4.2.3
 * @author       Thorsten Glander
 * @since        1.0.0
 */

// Sicherheitscheck - Direkter Zugriff verhindern
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Vehizo Appearance Settings Class
 *
 * @since 1.0.0
 */
class Vehizo_Filter_Settings {
    public function __construct() {
        // Class initialized
    }
    /**
     * Initialize settings
     *
     * @since 1.0.0
     */
    public static function init() {
        add_action('admin_init', array(__CLASS__, 'register_settings'), 5); // Priority 5 to run before other registrations
        add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_admin_assets'));
        self::handle_ajax_operations();
    }

    /**
     * Register all settings fields and sections
     *
     * @since 1.0.0
     */
    public static function register_settings() {
        $setting_group = 'vehizo_appearance_group';
        self::register_sections();
        self::register_options($setting_group);
        self::add_fields_to_sections();
    }

    /**
     * Register settings sections
     *
     * @since 1.0.0
     */
    private static function register_sections() {
        add_settings_section('vehizo_layout_section', __('Layout Settings', 'vehizo-vehicle-management'), array(__CLASS__, 'render_section_description'), 'vehizo_appearance_settings_layout');
        add_settings_section('vehizo_global_design_section', __('Global Design', 'vehizo-vehicle-management'), array(__CLASS__, 'render_section_description'), 'vehizo_appearance_settings_global_design');
        add_settings_section('vehizo_card_design_section', __('Card Design', 'vehizo-vehicle-management'), array(__CLASS__, 'render_section_description'), 'vehizo_appearance_settings_cards');
        add_settings_section('vehizo_typography_section', __('Typography', 'vehizo-vehicle-management'), array(__CLASS__, 'render_section_description'), 'vehizo_appearance_settings_typography');
    }

    /**
     * Register all plugin options
     *
     * @since 1.0.0
     * @param string $setting_group Settings group name
     */
    private static function register_options($setting_group) {
        $options = self::get_default_options();
        foreach ($options as $name => $default) {
            register_setting($setting_group, $name, array('default' => $default, 'sanitize_callback' => array(__CLASS__, 'sanitize_option'), 'show_in_rest' => false));
        }
    }

    /**
     * Get default options array
     *
     * @since 1.0.0
     * @return array Default options
     */
    private static function get_default_options() {
        return apply_filters('vehizo_default_appearance_options', array('vehizo_layout' => 'grid','vehizo_grid_columns' => 3,'vehizo_show_filter' => '1','vehizo_overview_page_id' => 0,'vehizo_posts_per_page' => 12,'vehizo_primary_color' => '#3b82f6','vehizo_secondary_color' => '#6c757d','vehizo_button_hover_color' => '#2563eb','vehizo_text_color' => '#4b5563','vehizo_container_bg_color' => '#ffffff','vehizo_card_bg_color' => '#ffffff','vehizo_card_border_color' => '#e5e7eb','vehizo_card_border_width' => '1','vehizo_card_radius' => '12','vehizo_card_box_shadow' => 'medium','vehizo_specs_bg_color' => '#f8fafc','vehizo_card_title_color' => '#1f2937','vehizo_card_title_fontweight' => '700','vehizo_font_size' => '16','vehizo_heading_size' => '24','vehizo_currency_symbol' => '€','vehizo_currency_position' => 'after','vehizo_filter_style' => 'modern','vehizo_filter_layout' => 'horizontal','vehizo_filter_spacing' => '16','vehizo_filter_border_radius' => '12','vehizo_filter_shadow' => 'medium','vehizo_card_spacing' => '24',));
    }
    
    /**
     * Add fields to their respective sections
     *
     * @since 1.0.0
     */
    private static function add_fields_to_sections() {
        // Layout Section
        add_settings_field(
            'vehizo_layout', 
            __('Layout Type', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_select_field'), 
            'vehizo_appearance_settings_layout', 
            'vehizo_layout_section', 
            array(
                'id' => 'vehizo_layout', 
                'options' => array(
                    'grid' => __('Grid (Tiles)', 'vehizo-vehicle-management'),
                    'sideby' => __('Side by Side', 'vehizo-vehicle-management'),
                    'list' => __('List', 'vehizo-vehicle-management'),
                    'masonry' => __('Masonry', 'vehizo-vehicle-management'),
                    'compact' => __('Compact', 'vehizo-vehicle-management')
                ), 
                'desc' => __('Choose the basic appearance of the vehicle list.', 'vehizo-vehicle-management')
            )
        );
        
        add_settings_field(
            'vehizo_grid_columns', 
            __('Number of Columns', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_number_field'), 
            'vehizo_appearance_settings_layout', 
            'vehizo_layout_section', 
            array(
                'id' => 'vehizo_grid_columns', 
                'min' => 1, 
                'max' => 6, 
                'desc' => __('Number of columns for grid layouts on desktop.', 'vehizo-vehicle-management')
            )
        );
        
        add_settings_field(
            'vehizo_show_filter', 
            __('Show Filter', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_checkbox_field'), 
            'vehizo_appearance_settings_layout', 
            'vehizo_layout_section', 
            array(
                'id' => 'vehizo_show_filter', 
                'desc' => __('Controls whether the vehicle filter is displayed above the list.', 'vehizo-vehicle-management')
            )
        );
        
        add_settings_field(
            'vehizo_posts_per_page', 
            __('Vehicles per Page', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_number_field'), 
            'vehizo_appearance_settings_layout', 
            'vehizo_layout_section', 
            array(
                'id' => 'vehizo_posts_per_page', 
                'min' => 1, 
                'max' => 100,
                'default' => 12,
                'desc' => __('Number of vehicles to display per page.', 'vehizo-vehicle-management')
            )
        );

        // NEU: Währungssymbol - AN DER KORREKTEN STELLE
        add_settings_field(
            'vehizo_currency_symbol', 
            __('Currency Symbol', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_text_field'), 
            'vehizo_appearance_settings_layout', // Geändert
            'vehizo_layout_section',             // Geändert
            array(
                'id' => 'vehizo_currency_symbol', 
                'desc' => __('Enter the currency symbol (e.g., €, $, £).', 'vehizo-vehicle-management'),
                'maxlength' => 5
            )
        );

        // NEU: Währungsposition - AN DER KORREKTEN STELLE
        add_settings_field(
            'vehizo_currency_position', 
            __('Currency Position', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_select_field'), 
            'vehizo_appearance_settings_layout', // Geändert
            'vehizo_layout_section',             // Geändert
            array(
                'id' => 'vehizo_currency_position', 
                'options' => array(
                    'after' => __('After the price (e.g., 1.000 €)', 'vehizo-vehicle-management'),
                    'before' => __('Before the price (e.g., € 1.000)', 'vehizo-vehicle-management')
                ), 
                'desc' => __('Choose whether the symbol appears before or after the price.', 'vehizo-vehicle-management')
            )
        );

        // Global Design Section
        add_settings_field(
            'vehizo_primary_color', 
            __('Primary Color', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_color_field'), 
            'vehizo_appearance_settings_global_design', 
            'vehizo_global_design_section', 
            array(
                'id' => 'vehizo_primary_color', 
                'desc' => __('Main color for buttons, price badges and accents.', 'vehizo-vehicle-management')
            )
        );
        
        add_settings_field(
            'vehizo_secondary_color', 
            __('Secondary Color', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_color_field'), 
            'vehizo_appearance_settings_global_design', 
            'vehizo_global_design_section', 
            array(
                'id' => 'vehizo_secondary_color', 
                'desc' => __('A secondary color for accents, borders or other elements.', 'vehizo-vehicle-management')
            )
        );
        
        add_settings_field(
            'vehizo_button_hover_color', 
            __('Button Hover Color', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_color_field'), 
            'vehizo_appearance_settings_global_design', 
            'vehizo_global_design_section', 
            array(
                'id' => 'vehizo_button_hover_color', 
                'desc' => __('The color of the main buttons when the mouse hovers over them.', 'vehizo-vehicle-management')
            )
        );

        add_settings_field(
            'vehizo_text_color', 
            __('Standard Text Color', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_color_field'), 
            'vehizo_appearance_settings_global_design', 
            'vehizo_global_design_section', 
            array(
                'id' => 'vehizo_text_color', 
                'desc' => __('Text color for normal descriptions and values.', 'vehizo-vehicle-management')
            )
        );
        
        add_settings_field(
            'vehizo_container_bg_color', 
            __('Background Color (Container)', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_color_field'), 
            'vehizo_appearance_settings_global_design', 
            'vehizo_global_design_section', 
            array(
                'id' => 'vehizo_container_bg_color', 
                'desc' => __('Background of the entire vehicle list.', 'vehizo-vehicle-management')
            )
        );

        // Card Design Section
        add_settings_field(
            'vehizo_card_bg_color', 
            __('Background Color', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_color_field'), 
            'vehizo_appearance_settings_cards', 
            'vehizo_card_design_section', 
            array(
                'id' => 'vehizo_card_bg_color', 
                'desc' => __('Background color of each individual vehicle card.', 'vehizo-vehicle-management')
            )
        );
        
        add_settings_field(
            'vehizo_card_border_color', 
            __('Border Color', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_color_field'), 
            'vehizo_appearance_settings_cards', 
            'vehizo_card_design_section', 
            array(
                'id' => 'vehizo_card_border_color', 
                'desc' => __('Color of the border around the card.', 'vehizo-vehicle-management')
            )
        );
        
        add_settings_field(
            'vehizo_card_radius', 
            __('Corner Radius', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_slider_field'), 
            'vehizo_appearance_settings_cards', 
            'vehizo_card_design_section', 
            array(
                'id' => 'vehizo_card_radius', 
                'min' => 0, 
                'max' => 40, 
                'default' => 12, 
                'desc' => __('Rounding of the vehicle card corners.', 'vehizo-vehicle-management')
            )
        );
        
        add_settings_field(
            'vehizo_specs_bg_color', 
            __('Background (Info Rows)', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_color_field'), 
            'vehizo_appearance_settings_cards', 
            'vehizo_card_design_section', 
            array(
                'id' => 'vehizo_specs_bg_color', 
                'desc' => __('Background of the rows with vehicle data (e.g. mileage).', 'vehizo-vehicle-management')
            )
        );

        // Typography Section
        add_settings_field(
            'vehizo_font_size', 
            __('Base Font Size', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_slider_field'), 
            'vehizo_appearance_settings_typography', 
            'vehizo_typography_section', 
            array(
                'id' => 'vehizo_font_size', 
                'min' => 12, 
                'max' => 22, 
                'default' => 16, 
                'desc' => __('General font size for texts in the cards.', 'vehizo-vehicle-management')
            )
        );
        
        add_settings_field(
            'vehizo_heading_size', 
            __('Card Title Font Size', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_slider_field'), 
            'vehizo_appearance_settings_typography', 
            'vehizo_typography_section', 
            array(
                'id' => 'vehizo_heading_size', 
                'min' => 16, 
                'max' => 40, 
                'default' => 24, 
                'desc' => __('Size of the main vehicle title.', 'vehizo-vehicle-management')
            )
        );
        
        add_settings_field(
            'vehizo_card_title_color', 
            __('Color (Card Title)', 'vehizo-vehicle-management'), 
            array(__CLASS__, 'render_color_field'), 
            'vehizo_appearance_settings_typography', 
            'vehizo_typography_section', 
            array(
                'id' => 'vehizo_card_title_color', 
                'desc' => __('Text color of the vehicle title.', 'vehizo-vehicle-management')
            )
        );
    }
    public static function sanitize_option($value) {
        // This function is kept from your original file for consistency
        return $value;
    }

    public static function render_section_description($args) {
        $descriptions = array('vehizo_layout_section' => __('Configure the layout and display of your vehicle listings.', 'vehizo-vehicle-management'), 'vehizo_global_design_section' => __('Set global colors and design elements.', 'vehizo-vehicle-management'), 'vehizo_card_design_section' => __('Customize the appearance of individual vehicle cards.', 'vehizo-vehicle-management'), 'vehizo_typography_section' => __('Adjust fonts and text styling.', 'vehizo-vehicle-management'));
        $section_id = $args['id'] ?? '';
        if (isset($descriptions[$section_id])) {
            echo '<p>' . esc_html($descriptions[$section_id]) . '</p>';
        }
    }

    public static function render_field_wrapper($args, $field_html) {
        echo '<div class="vehizo-field-wrapper">' . wp_kses_post($field_html) . '</div>';
        if (isset($args['desc'])) {
            echo '<p class="description" style="margin-top: 8px;">' . esc_html($args['desc']) . '</p>';
        }
    }

    public static function render_color_field($args) {
        $id = $args['id'];
        $option_value = get_option($id, $args['default'] ?? '#000000');
        $field_html = sprintf('<input type="color" id="%s" name="%s" value="%s" class="vehizo-color-field">', esc_attr($id), esc_attr($id), esc_attr($option_value));
        self::render_field_wrapper($args, $field_html);
    }

    public static function render_slider_field($args) {
        $id = $args['id'];
        $option_value = get_option($id, $args['default'] ?? '0');
        $min = $args['min'] ?? '0';
        $max = $args['max'] ?? '100';
        $unit = $args['unit'] ?? 'px';
        $field_html = sprintf('<div style="display: flex; align-items: center; gap: 16px;"><input type="range" class="vehizo-slider" data-unit="%s" id="%s" name="%s" min="%s" max="%s" value="%s" style="width: 250px;"><span id="%s-value" class="vehizo-slider-value">%s%s</span></div>', esc_attr($unit), esc_attr($id), esc_attr($id), esc_attr($min), esc_attr($max), esc_attr($option_value), esc_attr($id), esc_html($option_value), esc_html($unit));
        self::render_field_wrapper($args, $field_html);
    }

    public static function render_select_field($args) {
        $id = $args['id'];
        $options = $args['options'] ?? array();
        $option_value = get_option($id, $args['default'] ?? '');
        $html = sprintf('<select id="%s" name="%s" class="vehizo-select-field">', esc_attr($id), esc_attr($id));
        foreach ($options as $value => $label) {
            $html .= sprintf('<option value="%s" %s>%s</option>', esc_attr($value), selected($option_value, $value, false), esc_html($label));
        }
        $html .= '</select>';
        self::render_field_wrapper($args, $html);
    }

    public static function render_number_field($args) {
        $id = $args['id'];
        $option_value = get_option($id, $args['default'] ?? '');
        $min = $args['min'] ?? 0;
        $max = $args['max'] ?? 10;
        $html = sprintf('<input type="number" id="%s" name="%s" value="%s" min="%s" max="%s" style="width: 80px;" class="small-text">', esc_attr($id), esc_attr($id), esc_attr($option_value), esc_attr($min), esc_attr($max));
        self::render_field_wrapper($args, $html);
    }

    public static function render_text_field($args) {
        $id = $args['id'];
        $option_value = get_option($id, $args['default'] ?? '');
        $maxlength = $args['maxlength'] ?? 50;
        $html = sprintf('<input type="text" id="%s" name="%s" value="%s" maxlength="%s" class="regular-text">', esc_attr($id), esc_attr($id), esc_attr($option_value), esc_attr($maxlength));
        self::render_field_wrapper($args, $html);
    }

    public static function render_checkbox_field($args) {
        $id = $args['id'];
        $option_value = get_option($id, $args['default'] ?? '0');
        $html = sprintf('<label for="%s"><input type="checkbox" id="%s" name="%s" value="1" %s> %s</label>', esc_attr($id), esc_attr($id), esc_attr($id), checked($option_value, '1', false), esc_html($args['label'] ?? __('Enable', 'vehizo-vehicle-management')));
        self::render_field_wrapper($args, $html);
    }

    public static function enqueue_admin_assets($hook_suffix) {
        if (strpos($hook_suffix, 'vehizo') === false) return;
        wp_enqueue_style('vehizo-filter-settings-admin', VEHIZO_PLUGIN_URL . 'assets/css/filter-settings-admin.css', array(), VEHIZO_PLUGIN_VERSION);
        wp_enqueue_script('vehizo-filter-settings-admin', VEHIZO_PLUGIN_URL . 'assets/js/filter-settings-admin.js', array('jquery'), VEHIZO_PLUGIN_VERSION, true);
    }

    public static function generate_dynamic_css() {
        $settings = []; foreach (array_keys(self::get_default_options()) as $key) { $settings[$key] = get_option($key, self::get_default_options()[$key]); }
        $primary_rgb = self::hex_to_rgb($settings['vehizo_primary_color']);
        $css = ':root {' . "\n";
        $var_map = ['vehizo_primary_color'=>'--vehizo-primary','vehizo_secondary_color'=>'--vehizo-secondary','vehizo_button_hover_color'=>'--vehizo-button-hover','vehizo_text_color'=>'--vehizo-text','vehizo_container_bg_color'=>'--vehizo-bg','vehizo_card_bg_color'=>'--vehizo-card-bg','vehizo_card_border_color'=>'--vehizo-card-border','vehizo_specs_bg_color'=>'--vehizo-specs-bg','vehizo_card_title_color'=>'--vehizo-title','vehizo_grid_columns'=>'--grid-columns'];
        foreach ($var_map as $option => $variable) { $css .= sprintf('    %s: %s;' . "\n", esc_attr($variable), esc_attr($settings[$option])); }
        $css .= sprintf('    --vehizo-card-radius: %spx;' . "\n", absint($settings['vehizo_card_radius']));
        $css .= sprintf('    --vehizo-font-size: %spx;' . "\n", absint($settings['vehizo_font_size']));
        $css .= sprintf('    --vehizo-heading-size: %spx;' . "\n", absint($settings['vehizo_heading_size']));
        $css .= sprintf('    --vehizo-primary-rgb: %s;' . "\n", esc_attr($primary_rgb));
        $css .= '}' . "\n";

        return apply_filters('vehizo_dynamic_css_refactored', $css, $settings);
    }

    private static function hex_to_rgb($hex) {
        $hex = ltrim($hex, '#'); if(strlen($hex) == 3) { $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2]; }
        return (strlen($hex) == 6) ? hexdec(substr($hex,0,2)).','.hexdec(substr($hex,2,2)).','.hexdec(substr($hex,4,2)) : '0, 0, 0';
    }

    // --- VOLLSTÄNDIG WIEDERHERGESTELLTE AJAX/IMPORT/EXPORT FUNKTIONEN ---
    public static function handle_ajax_operations() {
        add_action('wp_ajax_vehizo_reset_appearance_settings', array(__CLASS__, 'ajax_reset_settings'));
        add_action('wp_ajax_vehizo_export_appearance_settings', array(__CLASS__, 'ajax_export_settings'));
        add_action('wp_ajax_vehizo_import_appearance_settings', array(__CLASS__, 'ajax_import_settings'));
    }

    public static function ajax_reset_settings() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'vehizo_settings_nonce')) {
            wp_send_json_error(array('message' => esc_html__('Security check failed', 'vehizo-vehicle-management')));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions', 'vehizo-vehicle-management')));
        }
        if (self::reset_to_defaults()) {
            wp_send_json_success(array('message' => esc_html__('Settings reset to defaults', 'vehizo-vehicle-management')));
        } else {
            wp_send_json_error(array('message' => esc_html__('Failed to reset settings', 'vehizo-vehicle-management')));
        }
    }

    public static function ajax_export_settings() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'vehizo_settings_nonce')) {
            wp_send_json_error(array('message' => esc_html__('Security check failed', 'vehizo-vehicle-management')));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions', 'vehizo-vehicle-management')));
        }
        $export_data = self::export_settings();
        if ($export_data) {
            wp_send_json_success(array('data' => $export_data));
        } else {
            wp_send_json_error(array('message' => esc_html__('Failed to export settings', 'vehizo-vehicle-management')));
        }
    }

    public static function ajax_import_settings() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'vehizo_settings_nonce')) {
            wp_send_json_error(array('message' => esc_html__('Security check failed', 'vehizo-vehicle-management')));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions', 'vehizo-vehicle-management')));
        }
        $json_data = isset($_POST['settings_data']) ? sanitize_textarea_field(wp_unslash($_POST['settings_data'])) : '';
        $result = self::import_settings($json_data);
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        } else {
            wp_send_json_success(array('message' => esc_html__('Settings imported successfully', 'vehizo-vehicle-management')));
        }
    }

    public static function reset_to_defaults() {
        if (!current_user_can('manage_options')) return false;
        $defaults = self::get_default_options();
        foreach ($defaults as $option_name => $default_value) {
            update_option($option_name, $default_value);
        }
        return true;
    }

    public static function export_settings() {
        if (!current_user_can('manage_options')) return false;
        $settings = array();
        $defaults = self::get_default_options();
        foreach (array_keys($defaults) as $option_name) {
            $settings[$option_name] = get_option($option_name);
        }
        return wp_json_encode(['version' => '4.1.0', 'settings' => $settings], JSON_PRETTY_PRINT);
    }

    public static function import_settings($json_data) {
        if (!current_user_can('manage_options')) return new WP_Error('permissions', 'Error.');
        $data = json_decode($json_data, true);
        if (json_last_error() !== JSON_ERROR_NONE || !isset($data['settings'])) {
            return new WP_Error('json', 'Invalid JSON.');
        }
        $defaults = self::get_default_options();
        foreach ($data['settings'] as $option_name => $value) {
            if (array_key_exists($option_name, $defaults)) {
                update_option($option_name, $value);
            }
        }
        return true;
    }
}
Vehizo_Filter_Settings::init();
