<?php
/**
 * Vehizo Vehicle Editor Class
 *
 * Provides editor functionality for vehicle management in WordPress admin dashboard.
 * Handles vehicle creation, editing, and media management with modern UI.
 *
 * @package      Vehizo
 * @version      4.1.0
 * @author       Thorsten Glander
 * @since        1.2.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Vehizo_Admin_Vehicle_Editor')) {
    
    /**
     * Admin Vehicle Editor Class
     *
     * Manages the vehicle editor interface and handles all editor-related functionality
     * including form processing, AJAX handlers, and UI rendering.
     */
    class Vehizo_Admin_Vehicle_Editor
    {
        /**
         * Initialize the vehicle editor
         *
         * Hooks into WordPress actions for scripts, AJAX handlers, and form processing
         */
        public static function init()
        {
            // Enqueue scripts and styles for the editor
            add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_scripts'));
            
            // AJAX handler for autosave functionality
            add_action('wp_ajax_vehizo_autosave_vehicle', array(__CLASS__, 'handle_autosave'));

            // AJAX handler for gallery order saving
            add_action('wp_ajax_vehizo_save_gallery_order', array(__CLASS__, 'handle_save_gallery_order'));

            // AJAX handler for tab-specific data saving
            add_action('wp_ajax_vehizo_save_tab_data', array(__CLASS__, 'handle_save_tab_data'));

            // Form handler for manual save
            add_action('admin_post_vehizo_save_vehicle', array(__CLASS__, 'handle_save_vehicle'));

            // Handler for emergency upload
            add_action('admin_post_vehizo_emergency_upload', array(__CLASS__, 'handle_emergency_upload'));
        }

        /**
         * Get maximum number of images allowed per vehicle
         *
         * @return int Maximum images allowed
         */
        private static function get_max_images() {
            return 99; // Effectively unlimited for the free version
        }

        /**
         * Handle vehicle save operation
         *
         * Processes form submission for saving vehicle data with proper validation,
         * sanitization, and error handling. Redirects back to editor with status message.
         */
        public static function handle_save_vehicle()
        {
            $nonce = isset($_POST['vehizo_vehicle_editor_nonce']) ? sanitize_text_field(wp_unslash($_POST['vehizo_vehicle_editor_nonce'])) : '';

            // Verify nonce for security
            if (!wp_verify_nonce($nonce, 'vehizo_vehicle_editor')) {
                wp_die(esc_html__('Security verification failed.', 'vehizo-vehicle-management'));
            }
            
            // Check user permissions
            if (!current_user_can('edit_posts')) {
                wp_die(esc_html__('You do not have permission to save this post.', 'vehizo-vehicle-management'));
            }

            // Get post ID from form (0 for new vehicle)
            $post_id = isset($_POST['post_ID']) ? absint($_POST['post_ID']) : 0;


            // Prepare post data with sanitization
            $post_data = array(
                'post_type'    => 'vehizo_vehicle',
                'post_title'   => sanitize_text_field(wp_unslash($_POST['post_title'] ?? '')),
                'post_content' => wp_kses_post(wp_unslash($_POST['post_content'] ?? '')),
                'post_status'  => 'publish', // Always publish vehicles
            );

            // Update existing or create new vehicle
            if ($post_id > 0) {
                $post_data['ID'] = $post_id;
                $result = wp_update_post($post_data);
            } else {
                $result = wp_insert_post($post_data);
                $post_id = $result;
            }

            // Check for errors
            if (is_wp_error($result)) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                }
                $redirect_url = admin_url('admin.php?page=vehizo-dashboard&tab=editor&message=error');
            } else {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                }
                // Save meta fields manually since we're bypassing WordPress hooks
                $allowed_fields = array(
                    'vehizo_make', 'vehizo_model', 'vehizo_price', 'vehizo_firstRegistration',
                    'vehizo_mileage', 'vehizo_fuelType', 'vehizo_transmission', 'vehizo_power',
                    'vehizo_vehicleType', 'vehizo_color', 'vehizo_doors', 'vehizo_seats', 'vehizo_engine_displacement', 'vehizo_vin',
                    'vehizo_emissionClass', 'vehizo_consumption', 'vehizo_number_of_owners', 'vehizo_next_inspection',
                    'vehizo_vehicle_status'
                );

                foreach ($allowed_fields as $field) {
                    if (isset($_POST[$field])) {
                        $value = sanitize_text_field(wp_unslash($_POST[$field]));
                        update_post_meta($post_id, $field, $value);
                    }
                }

                // Handle Features taxonomy
                if (isset($_POST['tax_input']['vehizo_feature'])) {
                    $features_input = sanitize_textarea_field(wp_unslash($_POST['tax_input']['vehizo_feature']));
                    // Convert comma-separated string to array
                    $features_array = array_map('trim', explode(',', $features_input));
                    $features_array = array_filter($features_array); // Remove empty values
                    // Set the terms for the post
                    wp_set_post_terms($post_id, $features_array, 'vehizo_feature', false);
                }

                // Handle SIMPLE image uploads
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated -- Validated below with proper checks
                if (!empty($_FILES['vehizo_simple_upload']['name'][0])) {
                    $current_gallery = get_post_meta($post_id, 'vehizo_gallery', true);
                    $current_ids = !empty($current_gallery) ? explode(',', $current_gallery) : array();
                    $uploaded_count = 0;

                    // Validate that all required $_FILES arrays exist
                    if (isset($_FILES['vehizo_simple_upload']['name'], $_FILES['vehizo_simple_upload']['type'],
                             $_FILES['vehizo_simple_upload']['tmp_name'], $_FILES['vehizo_simple_upload']['error'],
                             $_FILES['vehizo_simple_upload']['size'])) {

                        $files_count = count($_FILES['vehizo_simple_upload']['name']);

                        for ($key = 0; $key < $files_count; $key++) {
                            // Validate each file component exists at this index
                            if (!isset($_FILES['vehizo_simple_upload']['name'][$key],
                                      $_FILES['vehizo_simple_upload']['type'][$key],
                                      $_FILES['vehizo_simple_upload']['tmp_name'][$key],
                                      $_FILES['vehizo_simple_upload']['error'][$key],
                                      $_FILES['vehizo_simple_upload']['size'][$key])) {
                                continue;
                            }

                            $name = sanitize_file_name($_FILES['vehizo_simple_upload']['name'][$key]);

                            if (!empty($name)) {
                                // Validate and sanitize file data
                                $file = array(
                                    'name'     => $name,
                                    'type'     => sanitize_mime_type($_FILES['vehizo_simple_upload']['type'][$key]),
                                    'tmp_name' => sanitize_text_field($_FILES['vehizo_simple_upload']['tmp_name'][$key]),
                                    'error'    => absint($_FILES['vehizo_simple_upload']['error'][$key]),
                                    'size'     => absint($_FILES['vehizo_simple_upload']['size'][$key])
                                );

                                // Additional validation: Check for upload errors
                                if ($file['error'] !== UPLOAD_ERR_OK) {
                                    continue;
                                }

                                // Validate tmp_name is a valid uploaded file
                                if (!is_uploaded_file($file['tmp_name'])) {
                                    continue;
                                }

                                $upload = wp_handle_upload($file, array('test_form' => false));

                                if (!isset($upload['error'])) {
                                    $attachment = array(
                                        'post_mime_type' => $upload['type'],
                                        'post_title' => sanitize_file_name(pathinfo($upload['file'], PATHINFO_FILENAME)),
                                        'post_content' => '',
                                        'post_status' => 'inherit'
                                    );

                                    $attach_id = wp_insert_attachment($attachment, $upload['file'], $post_id);

                                    if (!is_wp_error($attach_id)) {
                                        require_once(ABSPATH . 'wp-admin/includes/image.php');
                                        $attach_data = wp_generate_attachment_metadata($attach_id, $upload['file']);
                                        wp_update_attachment_metadata($attach_id, $attach_data);
                                        $current_ids[] = $attach_id;
                                        $uploaded_count++;
                                    }
                                }
                            }
                        }
                    }

                    // Force save gallery meta with DIRECT DATABASE WRITE
                    $gallery_value = implode(',', array_filter($current_ids));

                    global $wpdb;
                    // Delete existing entries
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query needed to prevent duplicate meta entries for gallery data
                    $wpdb->delete($wpdb->postmeta, array(
                        'post_id' => $post_id,
                        'meta_key' => 'vehizo_gallery'
                    ));

                    // Insert new entry
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct insert to ensure single gallery meta entry
                    $wpdb->insert($wpdb->postmeta, array(
                        'post_id' => $post_id,
                        'meta_key' => 'vehizo_gallery',
                        'meta_value' => $gallery_value
                    ));

                }

                // Handle gallery order - prioritize the drag & drop field (vehizo_gallery) over manual input
                // Only use manual input if it's different from the current gallery order
                // IMPORTANT: Skip if images were just uploaded to avoid overwriting
                $gallery_saved = isset($uploaded_count) && $uploaded_count > 0;

                if (!$gallery_saved && isset($_POST['vehizo_gallery'])) {
                    $gallery_ids = sanitize_text_field(wp_unslash($_POST['vehizo_gallery']));

                    // Validate that gallery IDs contain only numbers and commas
                    if (preg_match('/^[0-9,]*$/', $gallery_ids)) {
                        // Clean up the input (remove extra commas)
                        $clean_ids = preg_replace('/,+/', ',', $gallery_ids);
                        $clean_ids = trim($clean_ids, ',');

                        // Delete existing entries
                        global $wpdb;
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query needed to prevent duplicate meta entries for gallery data
                        $wpdb->delete($wpdb->postmeta, array(
                            'post_id' => $post_id,
                            'meta_key' => 'vehizo_gallery'
                        ));

                        if (!empty($clean_ids)) {
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct insert to ensure single gallery meta entry
                            $wpdb->insert($wpdb->postmeta, array(
                                'post_id' => $post_id,
                                'meta_key' => 'vehizo_gallery',
                                'meta_value' => $clean_ids
                            ));
                        }

                        $gallery_saved = true;
                    }
                }

                // Only process manual input if no drag & drop field was processed and manual field is different
                if (!$gallery_saved && isset($_POST['vehizo_gallery_manual'])) {
                    $manual_ids = sanitize_text_field(wp_unslash($_POST['vehizo_gallery_manual']));
                    $current_gallery = get_post_meta($post_id, 'vehizo_gallery', true);

                    // Only save if manual input is different from current order
                    if ($manual_ids !== $current_gallery && preg_match('/^[0-9,\s]*$/', $manual_ids)) {
                        // Clean up the input (remove spaces, extra commas)
                        $clean_ids = preg_replace('/[^0-9,]/', '', $manual_ids);
                        $clean_ids = preg_replace('/,+/', ',', $clean_ids);
                        $clean_ids = trim($clean_ids, ',');

                        global $wpdb;
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query needed to prevent duplicate meta entries for gallery data
                        $wpdb->delete($wpdb->postmeta, array(
                            'post_id' => $post_id,
                            'meta_key' => 'vehizo_gallery'
                        ));

                        if (!empty($clean_ids)) {
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct insert to ensure single gallery meta entry
                            $wpdb->insert($wpdb->postmeta, array(
                                'post_id' => $post_id,
                                'meta_key' => 'vehizo_gallery',
                                'meta_value' => $clean_ids
                            ));
                        }
                    }
                }

                $redirect_url = admin_url('admin.php?page=vehizo-dashboard&tab=editor&fahrzeug_id=' . $post_id . '&message=success');
            }

            wp_safe_redirect($redirect_url);
            exit;
        }

        /**
         * Handle emergency upload
         */
        public static function handle_emergency_upload() {
            $nonce = isset($_POST['emergency_nonce']) ? sanitize_text_field(wp_unslash($_POST['emergency_nonce'])) : '';

            // Verify nonce for security
            if (!wp_verify_nonce($nonce, 'emergency_upload')) {
                wp_die(esc_html__('Security verification failed.', 'vehizo-vehicle-management'));
            }

            // Check user permissions
            if (!current_user_can('edit_posts')) {
                wp_die(esc_html__('You do not have permission to upload files.', 'vehizo-vehicle-management'));
            }

            $post_id = isset($_POST['emergency_post_id']) ? absint($_POST['emergency_post_id']) : 0;

            if (!$post_id) {
                wp_die(esc_html__('Invalid post ID.', 'vehizo-vehicle-management'));
            }

            // Check current image count and limit
            $current_gallery = get_post_meta($post_id, 'vehizo_gallery', true);
            $current_ids = !empty($current_gallery) ? explode(',', $current_gallery) : array();
            $current_count = count($current_ids);
            $max_images = self::get_max_images();

            $uploaded_ids = array();

            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated -- Validated below with proper checks
            if (!empty($_FILES['emergency_images']['name'][0])) {
                require_once(ABSPATH . 'wp-admin/includes/file.php');
                require_once(ABSPATH . 'wp-admin/includes/image.php');
                require_once(ABSPATH . 'wp-admin/includes/media.php');

                // Validate that all required $_FILES arrays exist
                if (isset($_FILES['emergency_images']['name'], $_FILES['emergency_images']['type'],
                         $_FILES['emergency_images']['tmp_name'], $_FILES['emergency_images']['error'],
                         $_FILES['emergency_images']['size'])) {

                    $files_count = count($_FILES['emergency_images']['name']);

                    for ($key = 0; $key < $files_count; $key++) {
                        // Validate each file component exists at this index
                        if (!isset($_FILES['emergency_images']['name'][$key],
                                  $_FILES['emergency_images']['type'][$key],
                                  $_FILES['emergency_images']['tmp_name'][$key],
                                  $_FILES['emergency_images']['error'][$key],
                                  $_FILES['emergency_images']['size'][$key])) {
                            continue;
                        }

                        $name = sanitize_file_name($_FILES['emergency_images']['name'][$key]);

                        if (!empty($name)) {
                            // Check if limit is reached
                            if ($current_count >= $max_images) {
                                break;
                            }

                            // Validate and sanitize file data
                            $file = array(
                                'name'     => $name,
                                'type'     => sanitize_mime_type($_FILES['emergency_images']['type'][$key]),
                                'tmp_name' => sanitize_text_field($_FILES['emergency_images']['tmp_name'][$key]),
                                'error'    => absint($_FILES['emergency_images']['error'][$key]),
                                'size'     => absint($_FILES['emergency_images']['size'][$key])
                            );

                            // Additional validation: Check for upload errors
                            if ($file['error'] !== UPLOAD_ERR_OK) {
                                continue;
                            }

                            // Validate tmp_name is a valid uploaded file
                            if (!is_uploaded_file($file['tmp_name'])) {
                                continue;
                            }

                            $upload = wp_handle_upload($file, array('test_form' => false));

                            if (!isset($upload['error'])) {
                                $attachment = array(
                                    'post_mime_type' => $upload['type'],
                                    'post_title' => sanitize_file_name(pathinfo($upload['file'], PATHINFO_FILENAME)),
                                    'post_content' => '',
                                    'post_status' => 'inherit'
                                );

                                $attach_id = wp_insert_attachment($attachment, $upload['file'], $post_id);

                                if (!is_wp_error($attach_id)) {
                                    $attach_data = wp_generate_attachment_metadata($attach_id, $upload['file']);
                                    wp_update_attachment_metadata($attach_id, $attach_data);
                                    $uploaded_ids[] = $attach_id;
                                    $current_count++; // Increment for next iteration
                                }
                            }
                        }
                    }
                }
            }

            if (!empty($uploaded_ids)) {
                // Get existing gallery and add new images
                global $wpdb;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query needed to bypass cache and get fresh gallery data
                $existing_gallery = $wpdb->get_var($wpdb->prepare(
                    "SELECT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'vehizo_gallery'",
                    $post_id
                ));

                $all_ids = array();
                if (!empty($existing_gallery)) {
                    $existing_ids = explode(',', $existing_gallery);
                    $all_ids = array_merge($all_ids, $existing_ids);
                }

                $all_ids = array_merge($all_ids, $uploaded_ids);
                $all_ids = array_unique(array_filter($all_ids));

                // Save to database
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct query needed to prevent duplicate meta entries for gallery data
                $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'vehizo_gallery'", $post_id));
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct insert to ensure single gallery meta entry
                $wpdb->insert($wpdb->postmeta, array(
                    'post_id' => $post_id,
                    'meta_key' => 'vehizo_gallery',
                    'meta_value' => implode(',', $all_ids)
                ));

                // Clear cache
                wp_cache_delete($post_id, 'post_meta');
                clean_post_cache($post_id);
            }

            // Redirect back with success message
            $redirect_url = admin_url('admin.php?page=vehizo-dashboard&tab=editor&fahrzeug_id=' . $post_id . '&message=upload_success');
            wp_safe_redirect($redirect_url);
            exit;
        }

        
        /**
         * Enqueue scripts and styles for the editor
         *
         * Only loads assets on the correct admin page to optimize performance
         *
         * @param string $hook Current admin page hook
         */
        public static function enqueue_scripts($hook)
        {
            // Only load on our specific dashboard page with editor tab
            if ($hook !== 'toplevel_page_vehizo-dashboard') {
                return;
            }
            
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only tab check for admin UI
            $current_tab = isset($_GET['tab']) ? sanitize_key(wp_unslash($_GET['tab'])) : '';
            if ($current_tab !== 'editor') {
                return;
            }

            // Set global post type for WordPress media handling
            global $post_type;
            $post_type = 'vehizo_vehicle';

            // Basic jQuery for tab functionality only
            
            // Enqueue our custom CSS with the correct /assets/ path
            wp_enqueue_style(
                'vehizo-admin-vehicle-editor',
                VEHIZO_PLUGIN_URL . 'assets/css/admin-vehicle-editor.css',
                array(),
                VEHIZO_PLUGIN_VERSION,
                'all'
            );
            
            // Enqueue main vehicle editor script (contains tab functionality, save handlers, etc.)
            wp_enqueue_script(
                'vehizo-admin-vehicle-editor',
                VEHIZO_PLUGIN_URL . 'assets/js/admin-vehicle-editor.js',
                array('jquery', 'jquery-ui-sortable'),
                VEHIZO_PLUGIN_VERSION,
                true
            );

            // Enqueue gallery management script
            wp_enqueue_script(
                'vehizo-gallery-manager',
                VEHIZO_PLUGIN_URL . 'assets/js/gallery-manager.js',
                array('jquery', 'jquery-ui-sortable', 'vehizo-admin-vehicle-editor'),
                VEHIZO_PLUGIN_VERSION,
                true
            );

            // Localize editor script with AJAX settings
            wp_localize_script('vehizo-admin-vehicle-editor', 'vehizoEditor', array(
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('vehizo_vehicle_editor'),
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only vehicle ID parameter for editor
                'postId' => isset($_GET['fahrzeug_id']) ? absint(wp_unslash($_GET['fahrzeug_id'])) : 0,
                'strings' => array(
                    'confirmLeave' => esc_html__('You have unsaved changes. Are you sure you want to leave?', 'vehizo-vehicle-management'),
                    'confirmRemove' => esc_html__('Remove this image from the gallery?', 'vehizo-vehicle-management'),
                    'reorderMessage' => esc_html__('Images reordered! Click "Save Vehicle" below to save the new order.', 'vehizo-vehicle-management'),
                    'savingData' => esc_html__('Saving...', 'vehizo-vehicle-management'),
                    'dataSaved' => esc_html__('Data saved successfully!', 'vehizo-vehicle-management'),
                    'saveError' => esc_html__('Error saving data. Please try again.', 'vehizo-vehicle-management'),
                )
            ));

            // Localize gallery script
            wp_localize_script('vehizo-gallery-manager', 'vehizoGallery', array(
                'confirmRemove' => esc_html__('Remove this image from the gallery?', 'vehizo-vehicle-management'),
                'reorderMessage' => esc_html__('Images reordered! Click "Save Vehicle" below to save the new order.', 'vehizo-vehicle-management'),
                'ajaxUrl' => admin_url('admin-ajax.php')
            ));
            
        }        
        /**
         * Render info notice for editor
         *
         * Shows success/error messages and general editor information
         *
         * @param WP_Post $post Current post object
         */
        public static function render_info_notice($post)
        {
            // Display success message after saving
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only message parameter for admin notices
            if (isset($_GET['message'])) {
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only message parameter for admin notices
                $message = sanitize_key(wp_unslash($_GET['message']));
                if ($message === 'success') {
                    echo '<div class="notice notice-success is-dismissible" style="margin-top: 20px;"><p>' . 
                         esc_html__('Vehicle saved successfully.', 'vehizo-vehicle-management') . '</p></div>';
                } elseif ($message === 'error') {
                    echo '<div class="notice notice-error is-dismissible" style="margin-top: 20px;"><p>' . 
                         esc_html__('Error saving vehicle. Please try again.', 'vehizo-vehicle-management') . '</p></div>';
                }
            }

            $vehicle_title = $post->post_title ?: __('New Vehicle', 'vehizo-vehicle-management');
            ?>
            <div class="vehizo-admin-notice">
                <h4><?php esc_html_e('Vehizo Vehicle Editor', 'vehizo-vehicle-management'); ?></h4>
                <p style="margin: 0;">
                    <?php 
                    echo wp_kses(
                        sprintf(
                            /* translators: %s: vehicle title */
                            __('You are editing the vehicle "%s" with the integrated Vehizo interface. All vehicle data will be saved when you click the save button.', 'vehizo-vehicle-management'),
                            '<strong>' . esc_html($vehicle_title) . '</strong>'
                        ),
                        array('strong' => array())
                    );
                    ?>
                </p>
            </div>
            <?php
        }
        
        /**
         * Render the main editor tab
         *
         * Handles both new vehicle creation and existing vehicle editing.
         * Loads the appropriate post object and renders the complete editor interface.
         */
        public static function render_editor_tab()
        {
            global $post;

            // Load existing vehicle or create new one
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only vehicle ID for loading editor data
            $post_id = isset($_GET['fahrzeug_id']) ? absint(wp_unslash($_GET['fahrzeug_id'])) : 0;
            if ($post_id > 0) {
                $post = get_post($post_id);
                
                if (!$post || $post->post_type !== 'vehizo_vehicle') {
                    echo '<div class="notice notice-error"><p>' . 
                         esc_html__('Vehicle not found.', 'vehizo-vehicle-management') . '</p></div>';
                    return;
                }
            } else {
                $post = get_default_post_to_edit('vehizo_vehicle', true);
            }
        
            // Ensure scripts are loaded
            self::enqueue_scripts('toplevel_page_vehizo-dashboard');
            
            $page_title = $post_id ? 
                __('Edit Vehicle', 'vehizo-vehicle-management') : 
                __('Add New Vehicle', 'vehizo-vehicle-management');
            
            $submit_text = $post_id ? 
                __('Save Changes', 'vehizo-vehicle-management') : 
                __('Create Vehicle', 'vehizo-vehicle-management');
            ?>
            <div class="vehizo-section">
                <h2><?php echo esc_html($page_title); ?></h2>
        
                <form id="post" method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" enctype="multipart/form-data">
                    
                    <input type="hidden" name="action" value="vehizo_save_vehicle" />
                    <input type="hidden" name="post_type" value="vehizo_vehicle" />
                    <input type="hidden" name="post_ID" value="<?php echo absint($post->ID); ?>" />
                    
                    <?php wp_nonce_field('vehizo_vehicle_editor', 'vehizo_vehicle_editor_nonce', false); ?>

                    <!-- Hidden field for gallery order - MUST be inside main form -->
                    <?php
                    // Get gallery data - DIRECT DATABASE QUERY to bypass all WordPress caching
                    global $wpdb;
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query needed to bypass cache and get fresh gallery data for editor
                    $gallery_ids = $wpdb->get_var($wpdb->prepare(
                        "SELECT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'vehizo_gallery'",
                        $post->ID
                    ));
                    ?>
                    <input type="hidden" id="vehizo-gallery-ids" name="vehizo_gallery" value="<?php echo esc_attr($gallery_ids); ?>">

                    <?php
                    self::render_info_notice($post);
                    self::render_modern_editor($post);
                    ?>
        
                </form>
            </div>
            <?php
        }
/**
         * Render description editor section
         *
         * Provides WordPress rich text editor for vehicle descriptions
         *
         * @param WP_Post $post Current post object
         */
        public static function render_description_editor($post)
        {
            ?>
            <div id="vehizo_description_editor" class="postbox" style="margin-top: 20px;">
                <h2 class="hndle">
                    <span><?php esc_html_e('Vehicle Description', 'vehizo-vehicle-management'); ?></span>
                </h2>
                <div class="inside">
                    <div style="margin: -6px -12px -12px -12px;">
                    <?php
                    wp_editor($post->post_content, 'vehizo_description', array(
                        'textarea_name' => 'post_content',
                        'media_buttons' => true,
                        'textarea_rows' => 12,
                        'teeny' => false
                    ));
                    ?>
                    </div>
                </div>
            </div>
            <?php
        }

        /**
         * Render modern editor interface
         *
         * Creates the main tabbed editor interface with all vehicle fields
         *
         * @param WP_Post $post Current post object
         */
        public static function render_modern_editor($post)
        {
            // Generate security nonce for form submission
            wp_nonce_field('vehizo_vehicle_editor', 'vehizo_vehicle_editor_nonce');

            // Get all vehicle meta fields with defaults
            $meta_fields = array(
                'make' => get_post_meta($post->ID, 'vehizo_make', true),
                'model' => get_post_meta($post->ID, 'vehizo_model', true),
                'price' => get_post_meta($post->ID, 'vehizo_price', true),
                'status' => get_post_meta($post->ID, 'vehizo_vehicle_status', true),
                'firstRegistration' => get_post_meta($post->ID, 'vehizo_firstRegistration', true),
                'mileage' => get_post_meta($post->ID, 'vehizo_mileage', true),
                'fuelType' => get_post_meta($post->ID, 'vehizo_fuelType', true),
                'transmission' => get_post_meta($post->ID, 'vehizo_transmission', true),
                'power' => get_post_meta($post->ID, 'vehizo_power', true),
                'vehicleType' => get_post_meta($post->ID, 'vehizo_vehicleType', true),
                'color' => get_post_meta($post->ID, 'vehizo_color', true),
                'doors' => get_post_meta($post->ID, 'vehizo_doors', true),
                'seats' => get_post_meta($post->ID, 'vehizo_seats', true),
                'engine_displacement' => get_post_meta($post->ID, 'vehizo_engine_displacement', true),
                'emissionClass' => get_post_meta($post->ID, 'vehizo_emissionClass', true),
                'consumption' => get_post_meta($post->ID, 'vehizo_consumption', true),
                'vin' => get_post_meta($post->ID, 'vehizo_vin', true),
                'number_of_owners' => get_post_meta($post->ID, 'vehizo_number_of_owners', true),
                'next_inspection' => get_post_meta($post->ID, 'vehizo_next_inspection', true)
            );

            // Get gallery data - DIRECT DATABASE QUERY to bypass all WordPress caching
            global $wpdb;
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query needed to bypass cache and get fresh gallery data for editor
            $gallery_ids = $wpdb->get_var($wpdb->prepare(
                "SELECT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = 'vehizo_gallery'",
                $post->ID
            ));

            $gallery_array = !empty($gallery_ids) ? explode(',', $gallery_ids) : array();

            // Remove empty values from array
            $gallery_array = array_filter($gallery_array, function($id) {
                return !empty($id) && is_numeric($id);
            });

            ?>
            <div id="vehizo_modern_editor">
                <div class="vehizo-editor-container">
                    <div class="vehizo-editor-header">
                        <h2><?php esc_html_e('Edit Vehicle Details', 'vehizo-vehicle-management'); ?></h2>
                        <div class="vehizo-save-status" id="vehizo-save-status">
                            <?php esc_html_e('Ready to edit', 'vehizo-vehicle-management'); ?>
                        </div>
                    </div>

                    <div class="vehizo-editor-wrapper">
                        <div class="vehizo-editor-main">
                            <div class="vehizo-editor-content">
                                <div style="padding: 32px 32px 0 32px;">
                                    <div class="vehizo-title-field">
                                        <label for="vehizo-title">
                                            <?php esc_html_e('Vehicle Title', 'vehizo-vehicle-management'); ?>
                                        </label>
                                        <input type="text" 
                                               name="post_title" 
                                               id="vehizo-title"
                                               value="<?php echo esc_attr($post->post_title); ?>"
                                               placeholder="<?php esc_attr_e('e.g. BMW 3 Series 320d M Sport - complete vehicle name', 'vehizo-vehicle-management'); ?>"
                                               required>
                                    </div>
                                </div>

                                <div class="vehizo-editor-tabs">
                                    <button type="button" class="vehizo-tab-button active" data-tab="basic">
                                        <?php esc_html_e('Basic Data', 'vehizo-vehicle-management'); ?>
                                    </button>
                                    <button type="button" class="vehizo-tab-button" data-tab="technical">
                                        <?php esc_html_e('Technical Data', 'vehizo-vehicle-management'); ?>
                                    </button>
                                    <button type="button" class="vehizo-tab-button" data-tab="description">
                                        <?php esc_html_e('Description', 'vehizo-vehicle-management'); ?>
                                    </button>
                                    <button type="button" class="vehizo-tab-button" data-tab="gallery">
                                        <?php esc_html_e('Image Gallery', 'vehizo-vehicle-management'); ?>
                                    </button>
                                    <button type="button" class="vehizo-tab-button" data-tab="features">
                                        <?php esc_html_e('Features', 'vehizo-vehicle-management'); ?>
                                    </button>
                                </div>

                                <div class="vehizo-tab-content active" id="tab-basic">
                                    <div class="vehizo-form-grid">
                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="status-select">
                                                <?php esc_html_e('Status', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <select id="status-select" name="vehizo_vehicle_status" class="vehizo-form-input vehizo-form-select">
                                                <option value="available" <?php selected($meta_fields['status'], 'available'); ?>><?php esc_html_e('Available', 'vehizo-vehicle-management'); ?></option>
                                                <option value="reserved" <?php selected($meta_fields['status'], 'reserved'); ?>><?php esc_html_e('Reserved', 'vehizo-vehicle-management'); ?></option>
                                                <option value="sold" <?php selected($meta_fields['status'], 'sold'); ?>><?php esc_html_e('Sold', 'vehizo-vehicle-management'); ?></option>
                                            </select>
                                        </div>
                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="make-input">
                                                <?php esc_html_e('Make', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <input type="text" 
                                                   id="make-input"
                                                   name="vehizo_make" 
                                                   class="vehizo-form-input" 
                                                   value="<?php echo esc_attr($meta_fields['make']); ?>"
                                                   placeholder="<?php esc_attr_e('e.g. BMW, Mercedes, Audi', 'vehizo-vehicle-management'); ?>">
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="model-input">
                                                <?php esc_html_e('Model', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <input type="text" 
                                                   id="model-input"
                                                   name="vehizo_model" 
                                                   class="vehizo-form-input" 
                                                   value="<?php echo esc_attr($meta_fields['model']); ?>"
                                                   placeholder="<?php esc_attr_e('e.g. 3 Series, C-Class, A4', 'vehizo-vehicle-management'); ?>">
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="price-input">
                                                <?php 
                                                $currency_symbol = function_exists('get_woocommerce_currency_symbol') ? get_woocommerce_currency_symbol() : '€';
                                                printf(
                                                    /* translators: %s: currency symbol */
                                                    esc_html__('Price (%s)', 'vehizo-vehicle-management'),
                                                    esc_html($currency_symbol)
                                                );
                                                ?>
                                            </label>
                                            <input type="number" 
                                                   id="price-input"
                                                   name="vehizo_price" 
                                                   class="vehizo-form-input" 
                                                   value="<?php echo esc_attr($meta_fields['price']); ?>"
                                                   placeholder="25000"
                                                   min="0">
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="reg-input">
                                                <?php esc_html_e('First Registration', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <input type="text"
                                                   id="reg-input"
                                                   name="vehizo_firstRegistration" 
                                                   class="vehizo-form-input" 
                                                   value="<?php echo esc_attr($meta_fields['firstRegistration']); ?>"
                                                   placeholder="03/2020">
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="mileage-input">
                                                <?php esc_html_e('Mileage', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <input type="number" 
                                                   id="mileage-input"
                                                   name="vehizo_mileage" 
                                                   class="vehizo-form-input" 
                                                   value="<?php echo esc_attr($meta_fields['mileage']); ?>"
                                                   placeholder="45000"
                                                   min="0">
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="power-input">
                                                <?php esc_html_e('Power (HP)', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <input type="number" 
                                                   id="power-input"
                                                   name="vehizo_power" 
                                                   class="vehizo-form-input" 
                                                   value="<?php echo esc_attr($meta_fields['power']); ?>"
                                                   placeholder="150"
                                                   min="0">
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="number-of-owners-input">
                                                <?php esc_html_e('Number of Previous Owners', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <input type="number"
                                                   id="number-of-owners-input"
                                                   name="vehizo_number_of_owners"
                                                   class="vehizo-form-input"
                                                   value="<?php echo esc_attr($meta_fields['number_of_owners']); ?>"
                                                   placeholder="1"
                                                   min="0">
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="next-inspection-input">
                                                <?php esc_html_e('Next Inspection (HU)', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <input type="text"
                                                   id="next-inspection-input"
                                                   name="vehizo_next_inspection"
                                                   class="vehizo-form-input"
                                                   value="<?php echo esc_attr($meta_fields['next_inspection']); ?>"
                                                   placeholder="MM/YYYY">
                                        </div>
                                    </div>

                                    <!-- Save Button for Basic Data -->
                                    <div style="margin-top: 24px; padding-top: 20px; border-top: 1px solid #e5e7eb; text-align: center;">
                                        <button type="button" class="vehizo-save-tab-btn button button-primary button-large" data-tab="basic"
                                                style="padding: 12px 30px; font-size: 14px; font-weight: 600;">
                                            💾 <?php esc_html_e('Save Basic Data', 'vehizo-vehicle-management'); ?>
                                        </button>
                                        <p style="margin: 10px 0 0 0; font-size: 13px; color: #6b7280;">
                                            <?php esc_html_e('Saves title, make, model, price and basic vehicle information', 'vehizo-vehicle-management'); ?>
                                        </p>
                                    </div>

                                    <div class="vehizo-info-box">
                                        <h4><?php esc_html_e('Tips for Basic Data:', 'vehizo-vehicle-management'); ?></h4>
                                        <ul>
                                            <li><strong><?php esc_html_e('Title:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('Complete, descriptive vehicle name', 'vehizo-vehicle-management'); ?></li>
                                            <li><strong><?php esc_html_e('Make & Model:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('Exact designation for better searchability', 'vehizo-vehicle-management'); ?></li>
                                            <li><strong><?php esc_html_e('Price:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('Enter numbers only, without currency symbol', 'vehizo-vehicle-management'); ?></li>
                                        </ul>
                                    </div>
                                </div>
                                <div class="vehizo-tab-content" id="tab-technical">
                                    <div class="vehizo-form-grid">
                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="vin-input">
                                                <?php esc_html_e('VIN (Chassis Number)', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <input type="text"
                                                   id="vin-input"
                                                   name="vehizo_vin"
                                                   class="vehizo-form-input"
                                                   value="<?php echo esc_attr($meta_fields['vin']); ?>"
                                                   placeholder="WP0ZZZ99ZJS123456"
                                                   maxlength="17"
                                                   style="font-family: monospace; text-transform: uppercase;">
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="fuel-select">
                                                <?php esc_html_e('Fuel Type', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <select id="fuel-select" name="vehizo_fuelType" class="vehizo-form-input vehizo-form-select">
                                                <option value=""><?php esc_html_e('Please select', 'vehizo-vehicle-management'); ?></option>
                                                <option value="Gasoline" <?php selected($meta_fields['fuelType'], 'Gasoline'); selected($meta_fields['fuelType'], 'Benzin'); ?>><?php esc_html_e('Gasoline', 'vehizo-vehicle-management'); ?></option>
                                                <option value="Diesel" <?php selected($meta_fields['fuelType'], 'Diesel'); ?>><?php esc_html_e('Diesel', 'vehizo-vehicle-management'); ?></option>
                                                <option value="Electric" <?php selected($meta_fields['fuelType'], 'Electric'); selected($meta_fields['fuelType'], 'Elektro'); ?>><?php esc_html_e('Electric', 'vehizo-vehicle-management'); ?></option>
                                                <option value="Hybrid" <?php selected($meta_fields['fuelType'], 'Hybrid'); ?>><?php esc_html_e('Hybrid', 'vehizo-vehicle-management'); ?></option>
                                            </select>
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="trans-select">
                                                <?php esc_html_e('Transmission', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <select id="trans-select" name="vehizo_transmission" class="vehizo-form-input vehizo-form-select">
                                                <option value=""><?php esc_html_e('Please select', 'vehizo-vehicle-management'); ?></option>
                                                <option value="Manual" <?php selected($meta_fields['transmission'], 'Manual'); selected($meta_fields['transmission'], 'Schaltgetriebe'); ?>><?php esc_html_e('Manual', 'vehizo-vehicle-management'); ?></option>
                                                <option value="Automatic" <?php selected($meta_fields['transmission'], 'Automatic'); selected($meta_fields['transmission'], 'Automatik'); ?>><?php esc_html_e('Automatic', 'vehizo-vehicle-management'); ?></option>
                                                <option value="Semi-Automatic" <?php selected($meta_fields['transmission'], 'Semi-Automatic'); selected($meta_fields['transmission'], 'Halbautomatik'); ?>><?php esc_html_e('Semi-Automatic', 'vehizo-vehicle-management'); ?></option>
                                            </select>
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="type-select">
                                                <?php esc_html_e('Vehicle Type', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <select id="type-select" name="vehizo_vehicleType" class="vehizo-form-input vehizo-form-select">
                                                <option value=""><?php esc_html_e('Please select', 'vehizo-vehicle-management'); ?></option>
                                                <option value="Sedan" <?php selected($meta_fields['vehicleType'], 'Sedan'); selected($meta_fields['vehicleType'], 'Limousine'); ?>><?php esc_html_e('Sedan', 'vehizo-vehicle-management'); ?></option>
                                                <option value="Station Wagon" <?php selected($meta_fields['vehicleType'], 'Station Wagon'); selected($meta_fields['vehicleType'], 'Kombi'); ?>><?php esc_html_e('Station Wagon', 'vehizo-vehicle-management'); ?></option>
                                                <option value="SUV" <?php selected($meta_fields['vehicleType'], 'SUV'); ?>><?php esc_html_e('SUV', 'vehizo-vehicle-management'); ?></option>
                                                <option value="Coupe" <?php selected($meta_fields['vehicleType'], 'Coupe'); ?>><?php esc_html_e('Coupe', 'vehizo-vehicle-management'); ?></option>
                                                <option value="Convertible" <?php selected($meta_fields['vehicleType'], 'Convertible'); selected($meta_fields['vehicleType'], 'Cabrio'); ?>><?php esc_html_e('Convertible', 'vehizo-vehicle-management'); ?></option>
                                            </select>
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="color-input">
                                                <?php esc_html_e('Color', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <input type="text"
                                                   id="color-input"
                                                   name="vehizo_color" 
                                                   class="vehizo-form-input" 
                                                   value="<?php echo esc_attr($meta_fields['color']); ?>"
                                                   placeholder="<?php esc_attr_e('e.g. Black Metallic', 'vehizo-vehicle-management'); ?>">
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="doors-select">
                                                <?php esc_html_e('Number of Doors', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <select id="doors-select" name="vehizo_doors" class="vehizo-form-input vehizo-form-select">
                                                <option value=""><?php esc_html_e('Please select', 'vehizo-vehicle-management'); ?></option>
                                                <option value="2/3" <?php selected($meta_fields['doors'], '2/3'); ?>>2/3</option>
                                                <option value="4/5" <?php selected($meta_fields['doors'], '4/5'); ?>>4/5</option>
                                                <option value="6/7" <?php selected($meta_fields['doors'], '6/7'); ?>>6/7</option>
                                            </select>
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="seats-input">
                                                <?php esc_html_e('Number of Seats', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <input type="number" 
                                                   id="seats-input"
                                                   name="vehizo_seats" 
                                                   class="vehizo-form-input" 
                                                   value="<?php echo esc_attr($meta_fields['seats']); ?>"
                                                   placeholder="5"
                                                   max="9">
                                        </div>

                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="engine-displacement-input">
                                                <?php esc_html_e('Engine Displacement (ccm)', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <input type="number"
                                                   id="engine-displacement-input"
                                                   name="vehizo_engine_displacement"
                                                   class="vehizo-form-input"
                                                   value="<?php echo esc_attr($meta_fields['engine_displacement']); ?>"
                                                   placeholder="1995"
                                                   min="0">
                                        </div>

                                    </div>

                                    <!-- Save Button for Technical Data -->
                                    <div style="margin-top: 24px; padding-top: 20px; border-top: 1px solid #e5e7eb; text-align: center;">
                                        <button type="button" class="vehizo-save-tab-btn button button-primary button-large" data-tab="technical"
                                                style="padding: 12px 30px; font-size: 14px; font-weight: 600;">
                                            💾 <?php esc_html_e('Save Technical Data', 'vehizo-vehicle-management'); ?>
                                        </button>
                                        <p style="margin: 10px 0 0 0; font-size: 13px; color: #6b7280;">
                                            <?php esc_html_e('Saves fuel type, transmission, vehicle type and technical specifications', 'vehizo-vehicle-management'); ?>
                                        </p>
                                    </div>

                                    <div class="vehizo-info-box">
                                        <h4><?php esc_html_e('Technical Data Notes:', 'vehizo-vehicle-management'); ?></h4>
                                        <ul>
                                            <li><strong><?php esc_html_e('Fuel Type:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('Important for search filters and customer interest', 'vehizo-vehicle-management'); ?></li>
                                            <li><strong><?php esc_html_e('Transmission:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('Automatic transmission is often preferred in searches', 'vehizo-vehicle-management'); ?></li>
                                            <li><strong><?php esc_html_e('Vehicle Type:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('Helps with categorization', 'vehizo-vehicle-management'); ?></li>
                                        </ul>
                                    </div>
                                </div>

                                <div class="vehizo-tab-content" id="tab-description">
                                    <div class="vehizo-form-section">
                                        <h3 style="margin: 0 0 16px 0; color: #374151;">
                                            <?php esc_html_e('Vehicle Description', 'vehizo-vehicle-management'); ?>
                                        </h3>
                                        <p style="margin: 0 0 16px 0; color: #6b7280;">
                                            <?php esc_html_e('Provide a detailed description of the vehicle including features, condition, history and special highlights.', 'vehizo-vehicle-management'); ?>
                                        </p>
                                        <?php
                                        wp_editor($post->post_content, 'post_content', array(
                                            'textarea_name' => 'post_content',
                                            'media_buttons' => true,
                                            'textarea_rows' => 15,
                                            'teeny' => false,
                                            'tinymce' => array(
                                                'toolbar1' => 'formatselect,bold,italic,underline,bullist,numlist,link,unlink,blockquote',
                                                'toolbar2' => ''
                                            )
                                        ));
                                        ?>
                                    </div>
                                    <!-- Save Button for Description -->
                                    <div style="margin-top: 24px; padding-top: 20px; border-top: 1px solid #e5e7eb; text-align: center;">
                                        <button type="button" class="vehizo-save-tab-btn button button-primary button-large" data-tab="description"
                                                style="padding: 12px 30px; font-size: 14px; font-weight: 600;">
                                            💾 <?php esc_html_e('Save Description', 'vehizo-vehicle-management'); ?>
                                        </button>
                                    </div>
                                </div>

                                <div class="vehizo-tab-content" id="tab-gallery">
                                    <div class="vehizo-gallery-container <?php echo !empty($gallery_array) ? 'has-images' : ''; ?>" id="vehizo-gallery-container">
                                        <?php if (empty($gallery_array)): ?>
                                            <div style="font-size: 48px; margin-bottom: 16px; opacity: 0.6;">📷</div>
                                            <h3 style="margin: 0 0 8px 0; color: #6b7280;">
                                                <?php esc_html_e('No images uploaded yet', 'vehizo-vehicle-management'); ?>
                                            </h3>
                                            <p style="margin: 0 0 24px 0; color: #9ca3af;">
                                                <?php esc_html_e('Upload professional vehicle images for an appealing presentation', 'vehizo-vehicle-management'); ?>
                                            </p>
                                            <!-- SIMPLE WORKING GALLERY MANAGER -->
                                            <div style="margin-top: 20px; padding: 20px; background: #f8f9fa; border: 2px solid #007cba; border-radius: 8px;">
                                                <h3 style="margin: 0 0 15px 0; color: #007cba;">🖼️ <?php esc_html_e('Gallery Manager', 'vehizo-vehicle-management'); ?></h3>

                                                <!-- Manual Image ID Input -->
                                                <div style="margin-bottom: 20px;">
                                                    <label style="display: block; margin-bottom: 5px; font-weight: bold;">Gallery Image IDs (comma separated):</label>
                                                    <input type="text" name="vehizo_gallery_manual" value="<?php echo esc_attr($gallery_ids); ?>"
                                                           placeholder="Enter image IDs: 123,456,789"
                                                           style="width: 100%; padding: 8px; border: 1px solid #ccc;">
                                                    <p style="margin: 5px 0 0 0; font-size: 12px; color: #666;">
                                                        Find image IDs in <strong>Media → Library</strong>. Click an image, look at URL: "post=<strong>123</strong>"
                                                    </p>
                                                </div>

                                                <!-- OR File Upload -->
                                                <div style="border-top: 1px solid #ddd; padding-top: 15px;">
                                                    <label style="display: block; margin-bottom: 5px; font-weight: bold;">OR Upload New Images:</label>
                                                    <?php
                                                    $max_images = self::get_max_images();
                                                    $current_count = !empty($gallery_array) ? count($gallery_array) : 0;
                                                    $is_pro = false; // Free version
                                                    ?>
                                                    <input type="file" name="vehizo_simple_upload[]" multiple accept="image/*"
                                                           style="width: 100%; padding: 8px; border: 1px solid #ccc;">
                                                    <p style="margin: 5px 0 0 0; font-size: 12px; color: #666;">
                                                        Upload will add to existing gallery. IDs will be added automatically.
                                                    </p>
                                                </div>

                                                <!-- Current Gallery Preview -->
                                                <?php if (!empty($gallery_array)): ?>
                                                <div style="border-top: 1px solid #ddd; padding-top: 15px;">
                                                    <strong>Current Gallery:</strong>
                                                    <div style="display: flex; gap: 10px; margin-top: 10px; flex-wrap: wrap;">
                                                        <?php foreach ($gallery_array as $img_id): ?>
                                                            <?php if (!empty($img_id) && is_numeric($img_id)): ?>
                                                                <?php $img_url = wp_get_attachment_image_url($img_id, 'thumbnail'); ?>
                                                                <?php if ($img_url): ?>
                                                                    <div style="text-align: center;">
                                                                        <img src="<?php echo esc_url($img_url); ?>" style="width: 80px; height: 80px; object-fit: cover; border: 1px solid #ddd;">
                                                                        <br><small>ID: <?php echo esc_html($img_id); ?></small>
                                                                    </div>
                                                                <?php endif; ?>
                                                            <?php endif; ?>
                                                        <?php endforeach; ?>
                                                    </div>
                                                </div>
                                                <?php endif; ?>

                                                <!-- Save Button inside Gallery Manager -->
                                                <div style="margin-top: 20px; text-align: center;">
                                                    <button type="submit" class="button button-primary button-hero" style="font-size: 16px; padding: 12px 40px;">
                                                        💾 <?php esc_html_e('Save Images', 'vehizo-vehicle-management'); ?>
                                                    </button>
                                                    <p style="margin: 10px 0 0 0; color: #666; font-size: 13px;">
                                                        <?php esc_html_e('Save gallery images and all other vehicle data', 'vehizo-vehicle-management'); ?>
                                                    </p>
                                                </div>
                                            </div>

                                            <!-- Tips Section -->
                                            <div style="margin-top: 20px; padding: 15px; background: #e7f3ff; border-left: 4px solid #007cba; border-radius: 4px;">
                                                <h4 style="margin: 0 0 10px 0; color: #007cba;">
                                                    <?php esc_html_e('Tips for Professional Vehicle Images:', 'vehizo-vehicle-management'); ?>
                                                </h4>
                                                <ul style="margin: 0; padding-left: 20px; color: #666;">
                                                    <li><strong><?php esc_html_e('Order:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('The first image will be used as the main image in the overview', 'vehizo-vehicle-management'); ?></li>
                                                    <li><strong><?php esc_html_e('Quality:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('Use high-resolution images (at least 1200x800px)', 'vehizo-vehicle-management'); ?></li>
                                                    <li><strong><?php esc_html_e('Quantity:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('8-12 images for optimal presentation', 'vehizo-vehicle-management'); ?></li>
                                                </ul>
                                            </div>
                                        <?php else: ?>
                                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                                                <h3 style="margin: 0; color: #374151;">
                                                    <?php
                                                    printf(
                                                        /* translators: %d: Number of vehicle images */
                                                        esc_html__('Vehicle Images (%d)', 'vehizo-vehicle-management'),
                                                        count(array_filter($gallery_array))
                                                    );
                                                    ?>
                                                </h3>
                                            </div>

                                            <div class="vehizo-gallery-grid" id="vehizo-gallery-grid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(150px, 1fr)); gap: 15px;">
                                                <?php foreach ($gallery_array as $index => $image_id): ?>
                                                    <?php if (!empty($image_id) && is_numeric($image_id)): ?>
                                                        <?php $image_url = wp_get_attachment_image_url($image_id, 'medium'); ?>
                                                        <?php if ($image_url): ?>
                                                            <div class="vehizo-gallery-item" data-id="<?php echo esc_attr($image_id); ?>"
                                                                 style="position: relative; cursor: grab; border: 2px solid #ddd; border-radius: 8px; overflow: hidden; background: white; transition: transform 0.2s ease, box-shadow 0.2s ease; will-change: transform;"
                                                                 onmousedown="this.style.cursor='grabbing'" onmouseup="this.style.cursor='grab'">
                                                                <div class="position-badge" style="position: absolute; top: 5px; left: 5px; background: #007cba; color: white; padding: 2px 6px; border-radius: 3px; font-size: 11px; font-weight: bold; z-index: 2;">
                                                                    <?php echo (int) $index + 1; ?>
                                                                </div>
                                                                <div class="drag-handle" style="position: absolute; top: 5px; right: 30px; background: rgba(0,0,0,0.7); color: white; padding: 4px 6px; border-radius: 3px; cursor: move; font-size: 12px; z-index: 2;" title="Drag to reorder">⋮⋮</div>
                                                                <img src="<?php echo esc_url($image_url); ?>" alt="<?php esc_attr_e('Vehicle Image', 'vehizo-vehicle-management'); ?>"
                                                                     style="width: 100%; height: 120px; object-fit: cover; display: block;">
                                                                <button type="button" class="remove-btn" onclick="removeGalleryImage(this)"
                                                                        style="position: absolute; top: 5px; right: 5px; background: #dc3545; color: white; border: none; width: 20px; height: 20px; border-radius: 50%; cursor: pointer; font-size: 12px; z-index: 2;"
                                                                        title="<?php esc_attr_e('Remove Image', 'vehizo-vehicle-management'); ?>">×</button>
                                                            </div>
                                                        <?php endif; ?>
                                                    <?php endif; ?>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php endif; ?>

                                    </div>

                                    <div class="vehizo-info-box">
                                        <h4><?php esc_html_e('Tips for Professional Vehicle Images:', 'vehizo-vehicle-management'); ?></h4>
                                        <ul>
                                            <li><strong><?php esc_html_e('Order:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('The first image will be used as the main image in the overview', 'vehizo-vehicle-management'); ?></li>
                                            <li><strong><?php esc_html_e('Quality:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('Use high-resolution images (at least 1200x800px)', 'vehizo-vehicle-management'); ?></li>
                                            <li><strong><?php esc_html_e('Quantity:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('8-12 images for optimal presentation', 'vehizo-vehicle-management'); ?></li>
                                            <li><strong><?php esc_html_e('Perspectives:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('Exterior shots from all sides, interior, engine, details', 'vehizo-vehicle-management'); ?></li>
                                            <li><strong><?php esc_html_e('Sorting:', 'vehizo-vehicle-management'); ?></strong> <?php esc_html_e('Drag and drop images to desired order', 'vehizo-vehicle-management'); ?></li>
                                        </ul>
                                    </div>
                                </div>
                                
                                <div class="vehizo-tab-content" id="tab-features">
                                    <div class="vehizo-form-section">
                                        <h3 style="margin: 0 0 16px 0; color: #374151;">
                                            <?php esc_html_e('Vehicle Features', 'vehizo-vehicle-management'); ?>
                                        </h3>
                                        <p style="margin: 0 0 16px 0; color: #6b7280;">
                                            <?php esc_html_e('Enter features separated by commas. Existing features will be suggested as you type.', 'vehizo-vehicle-management'); ?>
                                        </p>
                                        <div class="vehizo-form-group">
                                            <label class="vehizo-form-label" for="vehizo-features-input">
                                                <?php esc_html_e('Features (comma-separated)', 'vehizo-vehicle-management'); ?>
                                            </label>
                                            <textarea id="vehizo-features-input"
                                                      name="tax_input[vehizo_feature]"
                                                      class="vehizo-form-input"
                                                      rows="4"
                                                      placeholder="<?php esc_attr_e('e.g. Sunroof, Heated Seats, Navigation System', 'vehizo-vehicle-management'); ?>"><?php
                                                $terms = get_the_terms($post->ID, 'vehizo_feature');
                                                if ($terms && !is_wp_error($terms)) {
                                                    echo esc_textarea(implode(', ', wp_list_pluck($terms, 'name')));
                                                }
                                            ?></textarea>
                                        </div>
                                        <div class="vehizo-info-box">
                                            <h4><?php esc_html_e('How to use features:', 'vehizo-vehicle-management'); ?></h4>
                                            <ul>
                                                <li><?php esc_html_e('This structured list allows for future filtering capabilities.', 'vehizo-vehicle-management'); ?></li>
                                                <li><?php esc_html_e('Use consistent naming for features (e.g., always "Heated Seats", not "Seat Heating").', 'vehizo-vehicle-management'); ?></li>
                                            </ul>
                                        </div>
                                    </div>
                                    <!-- Save Button for Features -->
                                    <div style="margin-top: 24px; padding-top: 20px; border-top: 1px solid #e5e7eb; text-align: center;">
                                        <button type="button" class="vehizo-save-tab-btn button button-primary button-large" data-tab="features"
                                                style="padding: 12px 30px; font-size: 14px; font-weight: 600;">
                                            💾 <?php esc_html_e('Save Features', 'vehizo-vehicle-management'); ?>
                                        </button>
                                    </div>
                                </div>

                            </div>
                        </div><!-- .vehizo-editor-main -->
                    </div><!-- .vehizo-editor-wrapper -->
                </div>
            </div>
            <?php
        }

        /**
         * Handle AJAX gallery order saving
         *
         * Immediately saves gallery order when images are reordered via drag & drop
         */
        public static function handle_save_gallery_order()
        {
            // Verify nonce for security
            $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
            if (!wp_verify_nonce($nonce, 'vehizo_vehicle_editor')) {
                wp_send_json_error(array('message' => __('Security error', 'vehizo-vehicle-management')));
            }

            // Check user permissions
            if (!current_user_can('edit_posts')) {
                wp_send_json_error(array('message' => __('No permission', 'vehizo-vehicle-management')));
            }

            // Validate post ID
            $post_id = isset($_POST['post_id']) ? absint($_POST['post_id']) : 0;
            if (!$post_id) {
                wp_send_json_error(array('message' => __('Invalid post ID', 'vehizo-vehicle-management')));
            }

            // Get and validate gallery order
            $gallery_order = isset($_POST['gallery_order']) ? sanitize_text_field(wp_unslash($_POST['gallery_order'])) : '';
            if (!preg_match('/^[0-9,]*$/', $gallery_order)) {
                wp_send_json_error(array('message' => __('Invalid gallery order format', 'vehizo-vehicle-management')));
            }

            // Clean up the input
            $clean_ids = preg_replace('/,+/', ',', $gallery_order);
            $clean_ids = trim($clean_ids, ',');

            // Save to database directly
            global $wpdb;

            // Delete existing entries
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct query needed to prevent duplicate meta entries for gallery data
            $wpdb->delete($wpdb->postmeta, array(
                'post_id' => $post_id,
                'meta_key' => 'vehizo_gallery'
            ));

            // Insert new order
            if (!empty($clean_ids)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct insert to ensure single gallery meta entry
                $result = $wpdb->insert($wpdb->postmeta, array(
                    'post_id' => $post_id,
                    'meta_key' => 'vehizo_gallery',
                    'meta_value' => $clean_ids
                ));

                if ($result === false) {
                    wp_send_json_error(array('message' => __('Database error', 'vehizo-vehicle-management')));
                }
            }

            // Clear cache
            wp_cache_delete($post_id, 'post_meta');
            clean_post_cache($post_id);

            wp_send_json_success(array(
                'message' => __('Gallery order saved successfully', 'vehizo-vehicle-management'),
                'order' => $clean_ids
            ));
        }

        /**
         * Handle AJAX tab data saving
         *
         * Saves specific tab data when user clicks save button
         */
        public static function handle_save_tab_data()
        {
            // Verify nonce for security
            $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
            if (!wp_verify_nonce($nonce, 'vehizo_vehicle_editor')) {
                wp_send_json_error(array('message' => __('Security error', 'vehizo-vehicle-management')));
            }

            // Check user permissions
            if (!current_user_can('edit_posts')) {
                wp_send_json_error(array('message' => __('No permission', 'vehizo-vehicle-management')));
            }

            // Validate post ID
            $post_id = isset($_POST['post_id']) ? absint($_POST['post_id']) : 0;
            if (!$post_id) {
                wp_send_json_error(array('message' => __('Invalid post ID', 'vehizo-vehicle-management')));
            }

            // Get tab type and vehicle data
            $tab_type = isset($_POST['tab_type']) ? sanitize_text_field(wp_unslash($_POST['tab_type'])) : '';
            $vehicle_data = isset($_POST['vehicle_data']) ? map_deep(wp_unslash($_POST['vehicle_data']), 'sanitize_text_field') : array();
            $post_content = isset($_POST['post_content']) ? wp_kses_post(wp_unslash($_POST['post_content'])) : null;
            $features = isset($_POST['features']) ? sanitize_textarea_field(wp_unslash($_POST['features'])) : null;

            // Define allowed fields per tab
            $allowed_fields = array();
            if ($tab_type === 'basic') {
                $allowed_fields = array(
                    'vehizo_make', 'vehizo_model', 'vehizo_price', 'vehizo_vehicle_status',
                    'vehizo_firstRegistration', 'vehizo_mileage', 'vehizo_power', 'vehizo_number_of_owners', 'vehizo_next_inspection'
                );
            } elseif ($tab_type === 'technical') {
                $allowed_fields = array(
                    'vehizo_fuelType', 'vehizo_transmission', 'vehizo_vehicleType',
                    'vehizo_color', 'vehizo_doors', 'vehizo_seats', 'vehizo_engine_displacement', 'vehizo_vin',
                    'vehizo_emissionClass', 'vehizo_consumption'
                );
            }

            // Update post title if provided (only for basic tab)
            if ($tab_type === 'basic' && isset($vehicle_data['post_title'])) {
                $post_title = sanitize_text_field(wp_unslash($vehicle_data['post_title']));
                wp_update_post(array(
                    'ID' => $post_id,
                    'post_title' => $post_title
                ));
            }

            // Update post content for description tab
            if ($tab_type === 'description' && $post_content !== null) {
                wp_update_post(array('ID' => $post_id, 'post_content' => $post_content));
            }

            // Update features for features tab
            if ($tab_type === 'features' && $features !== null) {
                $feature_array = array_map('trim', explode(',', $features));
                wp_set_object_terms($post_id, $feature_array, 'vehizo_feature', false);
            }

            // Save allowed meta fields
            $saved_fields = array();
            foreach ($allowed_fields as $field) {
                if (isset($vehicle_data[$field])) {
                    $value = sanitize_text_field(wp_unslash($vehicle_data[$field]));
                    update_post_meta($post_id, $field, $value);
                    $saved_fields[] = $field;
                }
            }

            wp_send_json_success(array(
                'message' => ucfirst($tab_type) . ' data saved successfully',
                'tab_type' => $tab_type,
                'saved_fields' => $saved_fields
            ));
        }

        /**
         * Handle AJAX autosave functionality
         *
         * Processes AJAX requests for saving vehicle data without page reload.
         * Validates permissions and sanitizes all input data.
         */
        public static function handle_autosave()
        {
            $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
            // Verify nonce for security
            if (!wp_verify_nonce($nonce, 'vehizo_vehicle_editor')) {
                wp_send_json_error(array('message' => esc_html__('Security error', 'vehizo-vehicle-management')));
            }

            // Check user permissions
            if (!current_user_can('edit_posts')) {
                wp_send_json_error(array('message' => esc_html__('No permission', 'vehizo-vehicle-management')));
            }

            // Validate post ID
            $post_id = isset($_POST['post_id']) ? absint($_POST['post_id']) : 0;
            if (!$post_id) {
                wp_send_json_error(array('message' => esc_html__('Invalid post ID', 'vehizo-vehicle-management')));
            }

            // Get vehicle data from new structure
            $vehicle_data = isset($_POST['vehicle_data']) ? map_deep(wp_unslash($_POST['vehicle_data']), 'sanitize_text_field') : array();

            // Define allowed meta fields for security
            $allowed_fields = array(
                'vehizo_make', 'vehizo_model', 'vehizo_price', 'vehizo_vehicle_status', 'vehizo_firstRegistration', 'vehizo_mileage',
                'vehizo_fuelType', 'vehizo_transmission', 'vehizo_power', 'vehizo_vehicleType', 'vehizo_color',
                'vehizo_doors', 'vehizo_seats', 'vehizo_engine_displacement', 'vehizo_vin', 'vehizo_emissionClass',
                'vehizo_consumption', 'vehizo_number_of_owners', 'vehizo_next_inspection'
            );

            // Update post title if provided
            if (isset($vehicle_data['post_title'])) {
                $post_title = sanitize_text_field(wp_unslash($vehicle_data['post_title']));
                wp_update_post(array(
                    'ID' => $post_id,
                    'post_title' => $post_title
                ));
            }

            // Save meta fields from vehicle_data
            foreach ($allowed_fields as $field) {
                if (isset($vehicle_data[$field])) {
                    $value = sanitize_text_field(wp_unslash($vehicle_data[$field]));
                    update_post_meta($post_id, $field, $value);
                }
            }

            // Backward compatibility: Also check direct POST fields
            foreach ($allowed_fields as $field) {
                if (isset($_POST[$field])) {
                    $value = sanitize_text_field(wp_unslash($_POST[$field]));
                    update_post_meta($post_id, $field, $value);
                }
            }

            // Handle new image uploads
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated -- Validated below with proper checks
            if (!empty($_FILES['vehizo_new_images']['name'][0])) {
                $current_gallery = get_post_meta($post_id, 'vehizo_gallery', true);
                $current_ids = !empty($current_gallery) ? explode(',', $current_gallery) : array();

                // Validate that all required $_FILES arrays exist
                if (isset($_FILES['vehizo_new_images']['name'], $_FILES['vehizo_new_images']['type'],
                         $_FILES['vehizo_new_images']['tmp_name'], $_FILES['vehizo_new_images']['error'],
                         $_FILES['vehizo_new_images']['size'])) {

                    $files_count = count($_FILES['vehizo_new_images']['name']);

                    for ($key = 0; $key < $files_count; $key++) {
                        // Validate each file component exists at this index
                        if (!isset($_FILES['vehizo_new_images']['name'][$key],
                                  $_FILES['vehizo_new_images']['type'][$key],
                                  $_FILES['vehizo_new_images']['tmp_name'][$key],
                                  $_FILES['vehizo_new_images']['error'][$key],
                                  $_FILES['vehizo_new_images']['size'][$key])) {
                            continue;
                        }

                        $name = sanitize_file_name($_FILES['vehizo_new_images']['name'][$key]);

                        if (!empty($name)) {
                            // Validate and sanitize file data
                            $file = array(
                                'name'     => $name,
                                'type'     => sanitize_mime_type($_FILES['vehizo_new_images']['type'][$key]),
                                'tmp_name' => sanitize_text_field($_FILES['vehizo_new_images']['tmp_name'][$key]),
                                'error'    => absint($_FILES['vehizo_new_images']['error'][$key]),
                                'size'     => absint($_FILES['vehizo_new_images']['size'][$key])
                            );

                            // Additional validation: Check for upload errors
                            if ($file['error'] !== UPLOAD_ERR_OK) {
                                continue;
                            }

                            // Validate tmp_name is a valid uploaded file
                            if (!is_uploaded_file($file['tmp_name'])) {
                                continue;
                            }

                            $upload = wp_handle_upload($file, array('test_form' => false));

                            if (!isset($upload['error'])) {
                                $attachment = array(
                                    'post_mime_type' => $upload['type'],
                                    'post_title' => sanitize_file_name(pathinfo($upload['file'], PATHINFO_FILENAME)),
                                    'post_content' => '',
                                    'post_status' => 'inherit'
                                );

                                $attach_id = wp_insert_attachment($attachment, $upload['file'], $post_id);

                                if (!is_wp_error($attach_id)) {
                                    require_once(ABSPATH . 'wp-admin/includes/image.php');
                                    $attach_data = wp_generate_attachment_metadata($attach_id, $upload['file']);
                                    wp_update_attachment_metadata($attach_id, $attach_data);
                                    $current_ids[] = $attach_id;
                                }
                            }
                        }
                    }
                }

                update_post_meta($post_id, 'vehizo_gallery', implode(',', array_filter($current_ids)));
            }

            // Save gallery data with validation
            if (isset($_POST['vehizo_gallery'])) {
                $gallery_ids = sanitize_text_field(wp_unslash($_POST['vehizo_gallery']));
                // Validate that gallery IDs contain only numbers and commas
                if (preg_match('/^[0-9,]*$/', $gallery_ids)) {
                    update_post_meta($post_id, 'vehizo_gallery', $gallery_ids);
                }
            }

            wp_send_json_success(array('message' => esc_html__('Successfully saved', 'vehizo-vehicle-management')));
        }
    }
}
