<?php
/**
 * Main UI Class for the Vehizo Admin Interface
 *
 * This file defines the main class that builds and manages the Vehizo admin dashboard,
 * including menu registration, tab navigation, settings pages, and core hooks.
 *
 * @package      Vehizo
 * @version      4.1.0
 * @author       Thorsten Glander
 * @since        1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Vehizo_Admin')) {
    /**
     * Main class for the Vehizo admin interface and core functionalities.
     */
    class Vehizo_Admin
    {
        /**
         * Initializes all hooks and filters.
         */
        public static function init()
        {
            add_action('admin_menu', [self::class, 'register_menu']);
            add_action('admin_enqueue_scripts', [self::class, 'enqueue_admin_assets']);
            add_action('admin_init', [self::class, 'register_settings']);
            add_action('admin_init', [self::class, 'handle_settings_reset']);
            add_action('wp_ajax_vehizo_manual_import', [self::class, 'handle_ajax_import']);

            // Handler for activating/deactivating modules
            add_action('wp_ajax_vehizo_toggle_module_status', [self::class, 'handle_ajax_toggle_module']);

            // ADDED CALLS FOR THE E-MAIL POP-UP
            add_action('wp_ajax_vehizo_save_module_setting', [self::class, 'handle_ajax_save_setting']);
            add_action('wp_ajax_vehizo_get_module_setting', [self::class, 'handle_ajax_get_setting']);

            // NEU: Hook zur Verarbeitung der Seitenerstellung
            add_action('admin_init', [self::class, 'handle_create_wishlist_page']);

            // New AJAX action for updating vehicle status
            add_action('wp_ajax_vehizo_update_vehicle_status', [self::class, 'handle_ajax_update_vehicle_status']);
        }

        /**
         * Handles the AJAX request to update vehicle status.
         */
        public static function handle_ajax_update_vehicle_status() {
            // Verify nonce
            $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
            if (!wp_verify_nonce($nonce, 'vehizo_update_vehicle_status_nonce')) {
                wp_send_json_error(['message' => esc_html__('Security check failed.', 'vehizo-vehicle-management')]);
                return;
            }

            // Check user permissions
            if (!current_user_can('edit_posts')) {
                wp_send_json_error(['message' => esc_html__('Insufficient permissions.', 'vehizo-vehicle-management')]);
                return;
            }

            $post_id = isset($_POST['post_id']) ? absint($_POST['post_id']) : 0;
            $status = isset($_POST['status']) ? sanitize_key($_POST['status']) : '';

            if ($post_id > 0 && in_array($status, ['available', 'reserved', 'sold'], true)) {
                update_post_meta($post_id, 'vehizo_vehicle_status', $status);
                wp_send_json_success(['message' => esc_html__('Status updated successfully.', 'vehizo-vehicle-management')]);
            } else {
                wp_send_json_error(['message' => esc_html__('Invalid data received.', 'vehizo-vehicle-management')]);
            }
        }

/**
         * NEU: Verarbeitet die automatische Erstellung der Merklisten-Seite.
         */
        public static function handle_create_wishlist_page() {
            // Nur ausführen, wenn unsere Aktion ausgelöst wird
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Action check for page creation, nonce verified below
            if ( ! isset( $_GET['action'] ) || sanitize_key( $_GET['action'] ) !== 'vehizo_create_page' ) {
                return;
            }

            // Nonce verifizieren
            if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_key( $_GET['_wpnonce'] ), 'vehizo_create_page_nonce' ) ) {
                wp_die( esc_html__( 'Security check failed.', 'vehizo-vehicle-management' ) );
            }

            // Berechtigungen prüfen
            if ( ! current_user_can( 'publish_pages' ) ) {
                wp_die( esc_html__( 'You do not have permission to publish pages.', 'vehizo-vehicle-management' ) );
            }

            // Seiten-Daten vorbereiten
            $page_data = array(
                'post_title'    => esc_html__( 'Wishlist', 'vehizo-vehicle-management' ),
                'post_content'  => '[vehizo_wishlist]',
                'post_status'   => 'publish',
                'post_author'   => get_current_user_id(),
                'post_type'     => 'page',
                'post_name'     => 'wishlist' // Slug
            );

            // Seite einfügen
            $page_id = wp_insert_post( $page_data );

            if ( $page_id && ! is_wp_error( $page_id ) ) {
                // Die ID der neuen Seite in den Optionen speichern
                update_option( 'vehizo_wishlist_page_id', $page_id );
                // Zurück zum Dashboard mit Erfolgsmeldung weiterleiten
                wp_safe_redirect( admin_url( 'admin.php?page=vehizo-dashboard&wishlist_page_created=true' ) );
                exit;
            } else {
                // Zurück zum Dashboard mit Fehlermeldung weiterleiten
                wp_safe_redirect( admin_url( 'admin.php?page=vehizo-dashboard&wishlist_page_failed=true' ) );
                exit;
            }
        }

        /**
         * Enqueues admin assets (styles and scripts).
         * @param string $hook The current admin page hook.
         */
        public static function enqueue_admin_assets($hook)
        {
            // CSS-Dateien laden - beide Admin-CSS Dateien
            wp_enqueue_style(
                'vehizo-admin-style',
                VEHIZO_PLUGIN_URL . 'assets/css/admin.css',
                [],
                VEHIZO_PLUGIN_VERSION
            );

            // Zusätzliche Admin-Styles für moderne UI
            wp_enqueue_style(
                'vehizo-admin-styles',
                VEHIZO_PLUGIN_URL . 'assets/css/admin-styles.css',
                ['vehizo-admin-style'], // Abhängigkeit von admin.css
                VEHIZO_PLUGIN_VERSION . '-responsive-fix' // Cache busting für responsive fixes
            );

            // Dashboard-spezifische Styles
            wp_enqueue_style(
                'vehizo-admin-dashboard',
                VEHIZO_PLUGIN_URL . 'assets/css/admin-dashboard.css',
                ['vehizo-admin-style'],
                VEHIZO_PLUGIN_VERSION
            );

            // Enqueue the new emergency responsive styles
            wp_enqueue_style(
                'vehizo-admin-responsive-emergency',
                VEHIZO_PLUGIN_URL . 'assets/css/admin-responsive-emergency.css',
                ['vehizo-admin-style'],
                VEHIZO_PLUGIN_VERSION
            );

            // Dynamisches CSS hinzufügen
            $dynamic_css = self::get_dynamic_admin_css();

            // Responsive CSS direkt hinzufügen (um Cache-Probleme zu umgehen)
            $responsive_css = '
            /* MOBILE RESPONSIVE FIXES - INLINE */
            @media (max-width: 768px) {
                .modules-grid { grid-template-columns: 1fr !important; gap: 16px !important; }
                .dashboard-grid { grid-template-columns: 1fr !important; gap: 20px !important; }
                .stats-grid { grid-template-columns: repeat(2, 1fr) !important; gap: 16px !important; }
                .modern-table, .vehizo-table, .vehizo-vehicle-table, table.widefat { display: block !important; overflow-x: auto !important; white-space: nowrap !important; }
                .modern-table th, .modern-table td, .vehizo-table th, .vehizo-table td, .vehizo-vehicle-table th, .vehizo-vehicle-table td, table.widefat th, table.widefat td { padding: 12px 16px !important; font-size: 13px !important; min-width: 100px !important; }
                .vehizo-edit-button, .vehizo-delete-button, .button { font-size: 11px !important; padding: 4px 8px !important; margin: 2px !important; }
                .vehizo-header { padding: 16px 20px 20px 20px !important; flex-direction: column !important; align-items: flex-start !important; gap: 16px !important; }
                .vehizo-nav { padding: 0 20px !important; }
                .vehizo-nav .nav-tab-wrapper { flex-wrap: wrap !important; gap: 4px !important; }
                .vehizo-nav .nav-tab { padding: 12px 16px !important; font-size: 14px !important; }
                .vehizo-content { padding: 20px !important; }
                .module-card { padding: 16px !important; }
                .module-header { flex-direction: column !important; align-items: flex-start !important; gap: 12px !important; }
                .modern-button { width: 100% !important; justify-content: center !important; margin-bottom: 8px !important; }
            }
            @media (max-width: 480px) {
                .stats-grid { grid-template-columns: 1fr !important; }
                .vehizo-dashboard { margin: 10px !important; border-radius: 8px !important; }
                .vehizo-header { padding: 12px 16px 16px 16px !important; }
                .vehizo-content { padding: 16px !important; }
                .module-card { padding: 12px !important; }
                .module-title { font-size: 1rem !important; }
                .modern-table th, .modern-table td, .vehizo-table th, .vehizo-table td, .vehizo-vehicle-table th, .vehizo-vehicle-table td, table.widefat th, table.widefat td { padding: 8px 12px !important; font-size: 12px !important; min-width: 80px !important; }
                .vehizo-edit-button, .vehizo-delete-button, .button { display: block !important; width: 100% !important; margin: 2px 0 !important; text-align: center !important; }
            }';

            wp_add_inline_style('vehizo-admin-style', $dynamic_css . $responsive_css);

            // Modal styles (WordPress.org Guidelines - no hardcoded styles in JS)
            wp_enqueue_style(
                'vehizo-admin-modal',
                VEHIZO_PLUGIN_URL . 'assets/css/admin-modal.css',
                array(),
                VEHIZO_PLUGIN_VERSION
            );

            // JavaScript-Datei laden - korrigierter Pfad mit VEHIZO_PLUGIN_URL
            wp_enqueue_script(
                'vehizo-admin-script',
                VEHIZO_PLUGIN_URL . 'assets/js/admin.js',
                ['jquery'],
                VEHIZO_PLUGIN_VERSION,
                true
            );

            // NEU: JavaScript für Slider im Admin-Bereich
            wp_enqueue_script(
                'vehizo-admin-sliders',
                VEHIZO_PLUGIN_URL . 'assets/js/admin-sliders.js',
                ['jquery'],
                VEHIZO_PLUGIN_VERSION,
                true
            );

            // Chart.js für den Statistik-Tab laden - korrigierter Pfad
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only tab parameter for admin UI
            $current_tab = isset($_GET['tab']) ? sanitize_key(wp_unslash($_GET['tab'])) : 'dashboard';
            if ($current_tab === 'statistics') {
                wp_enqueue_script(
                    'vehizo-chartjs',
                    VEHIZO_PLUGIN_URL . 'assets/js/chart.min.js',
                    [],
                    '3.9.1',
                    true
                );
            }
            
            // KORRIGIERT: Daten für AJAX an das Script übergeben mit korrektem Namen
            wp_localize_script('vehizo-admin-script', 'vehizo_ajax', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce_toggle_module' => wp_create_nonce('vehizo_toggle_module_nonce'),
                'nonce_save_setting' => wp_create_nonce('vehizo_save_setting_nonce'),
                'nonce_get_setting' => wp_create_nonce('vehizo_get_setting_nonce'),
                'nonce_manual_import' => wp_create_nonce('vehizo_mobile_nonce'),
                'nonce_update_vehicle_status' => wp_create_nonce('vehizo_update_vehicle_status_nonce'),
                'nonce' => wp_create_nonce('vehizo_mobile_nonce'), // Legacy support
                'strings' => [
                    'import_now' => esc_html__('Import Now', 'vehizo-vehicle-management'),
                    'importing' => esc_html__('Importing...', 'vehizo-vehicle-management'),
                    'import_running' => esc_html__('Import is running...', 'vehizo-vehicle-management'),
                    'import_successful' => esc_html__('Import successful:', 'vehizo-vehicle-management'),
                    'import_error' => esc_html__('Import Error:', 'vehizo-vehicle-management'),
                    'import_error_general' => esc_html__('Error during import.', 'vehizo-vehicle-management'),
                    'error_module_status' => esc_html__('An error occurred while updating the module status.', 'vehizo-vehicle-management'),
                    'error_retrieving' => esc_html__('Error retrieving setting value.', 'vehizo-vehicle-management'),
                    'saving' => esc_html__('Saving...', 'vehizo-vehicle-management'),
                    'error_saving' => esc_html__('Error saving setting.', 'vehizo-vehicle-management'),
                    'module_config' => esc_html__('Module Configuration', 'vehizo-vehicle-management'),
                    'setting_value' => esc_html__('Setting Value', 'vehizo-vehicle-management'),
                    'enter_email' => esc_html__('Enter email address', 'vehizo-vehicle-management'),
                    'cancel' => esc_html__('Cancel', 'vehizo-vehicle-management'),
                    'save' => esc_html__('Save', 'vehizo-vehicle-management'),
                    'confirm_delete_vehicle' => esc_html__('Do you really want to delete this vehicle? It will be moved to the trash.', 'vehizo-vehicle-management'),
                    'confirm_move_trash' => esc_html__('Do you really want to move this vehicle to the trash?', 'vehizo-vehicle-management'),
                    'confirm_reset_settings' => esc_html__('Do you really want to reset all frontend settings to their default values?', 'vehizo-vehicle-management'),
                    'active' => esc_html__('Active', 'vehizo-vehicle-management'),
                    'inactive' => esc_html__('Inactive', 'vehizo-vehicle-management')
                ]
            ]);
            
            // ZUSÄTZLICH: Legacy-Support für vehizo_ajax_data (falls andere Code-Teile darauf zugreifen)
            wp_localize_script('vehizo-admin-script', 'vehizo_ajax_data', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('vehizo_mobile_nonce')
            ]);
        }
/**
         * Returns dynamic CSS that depends on user settings or conditions
         */
        private static function get_dynamic_admin_css() {
            // Fallback CSS für den Fall, dass die externe CSS-Datei nicht geladen wird
            return '
                .vehizo-dashboard { margin: 20px; }
                .vehizo-header { display: flex; align-items: center; gap: 20px; margin-bottom: 30px; padding: 20px; background: #fff; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); }
                .vehizo-logo { max-height: 40px; }
                .vehizo-nav { margin-bottom: 30px; }
                .vehizo-content { background: #fff; padding: 30px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); }
                .vehizo-section { margin-bottom: 40px; }
                .vehizo-section h2 { margin-bottom: 20px; color: #1f2937; font-size: 24px; font-weight: 600; }
                .stats-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 30px; }
                .stat-card { background: #f8fafc; padding: 20px; border-radius: 8px; text-align: center; border: 1px solid #e2e8f0; }
                .stat-card h3 { font-size: 2em; margin: 0 0 8px 0; color: #0073aa; font-weight: 700; }
                .stat-card p { margin: 0; color: #64748b; font-weight: 500; }
                .meta-box { background: #fff; border: 1px solid #e2e8f0; border-radius: 8px; padding: 20px; margin-bottom: 20px; }
                .meta-box h3 { margin-top: 0; margin-bottom: 15px; font-size: 1.2em; color: #1f2937; font-weight: 600; }
                .modern-button { background: #0073aa; color: #fff; border: none; padding: 12px 24px; border-radius: 6px; cursor: pointer; font-weight: 500; text-decoration: none; display: inline-block; transition: background 0.2s; font-size: 14px; }
                .modern-button:hover { background: #005a87; color: #fff; text-decoration: none; }
                .modern-button.secondary{background:#f0f0f1;color:#50575e;border:1px solid #dcdcde;}
		.modern-button.secondary:hover{background:#e0e0e1;color:#50575e;border-color:#d0d0d2;}
                .nav-tab-wrapper { border-bottom: 1px solid #ccd0d4; margin: 0 0 20px; }
                .nav-tab { background: #f1f1f1; border: 1px solid #ccd0d4; border-bottom: none; color: #555; padding: 8px 12px; text-decoration: none; margin-right: 6px; border-radius: 4px 4px 0 0; font-size: 14px; }
                .nav-tab-active, .nav-tab:hover { background: #fff; color: #000; }
                .nav-tab:focus { box-shadow: none; }
                .alert { padding: 16px; border-radius: 8px; margin: 20px 0; }
                .alert-success { background: #d1fae5; color: #047857; border: 1px solid #10b981; }
                .alert-warning { background: #fef3c7; color: #92400e; border: 1px solid #f59e0b; }
                .alert-error { background: #fee2e2; color: #b91c1c; border: 1px solid #ef4444; }
                .alert-info { background: #dbeafe; color: #1e40af; border: 1px solid #3b82f6; }
                /* Stile für die neuen Slider */
                .vehizo-color-field { width: 60px; height: 40px; border: 1px solid #ddd; border-radius: 4px; cursor: pointer; }
                .vehizo-slider { cursor: pointer; width: 250px; }
                .vehizo-slider-value { user-select: none; font-weight: bold; background: #f0f0f1; padding: 4px 8px; border-radius: 4px; min-width: 60px; text-align: center; margin-left: 16px; }
            ';
        }
/**
         * Checks if the vehicle limit has been reached.
         * @return bool Always returns false (no limit).
         */
        private static function is_vehicle_limit_reached() {
            return false;
        }
        /**
         * Handles the AJAX request to toggle a module's status.
         */
        public static function handle_ajax_toggle_module() {
            // Verify nonce
            $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
            if (!wp_verify_nonce($nonce, 'vehizo_toggle_module_nonce')) {
                wp_send_json_error(['message' => esc_html__('Security check failed.', 'vehizo-vehicle-management')]);
                return;
            }

            // Check user permissions
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => esc_html__('Insufficient permissions.', 'vehizo-vehicle-management')]);
                return;
            }

            $option_name = isset($_POST['module_option']) ? sanitize_text_field(wp_unslash($_POST['module_option'])) : '';
            $status = isset($_POST['status']) ? sanitize_key(wp_unslash($_POST['status'])) : '0';

            // Whitelist of allowed module options to prevent arbitrary option updates
            $allowed_module_options = array(
                'vehizo_contact_module_active',
                'vehizo_statistics_module_active',
                'vehizo_wishlist_module_active',
            );

            if (!in_array($option_name, $allowed_module_options, true)) {
                wp_send_json_error(['message' => esc_html__('Invalid option name.', 'vehizo-vehicle-management')]);
                return;
            }

            if (!empty($option_name) && in_array($status, ['0', '1'], true)) {
                update_option($option_name, $status);
                wp_send_json_success(['message' => esc_html__('Module status successfully updated.', 'vehizo-vehicle-management')]);
            } else {
                wp_send_json_error(['message' => esc_html__('Invalid data received.', 'vehizo-vehicle-management')]);
            }
        }
        
        /**
         * Handles the AJAX request to save a module setting (e.g., email address).
         */
        public static function handle_ajax_save_setting() {
            // Verify nonce
            $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
            if (!wp_verify_nonce($nonce, 'vehizo_save_setting_nonce')) {
                wp_send_json_error(['message' => esc_html__('Security check failed.', 'vehizo-vehicle-management')]);
                return;
            }

            // Check user permissions
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => esc_html__('Insufficient permissions.', 'vehizo-vehicle-management')]);
                return;
            }

            $option_name = isset($_POST['option_name']) ? sanitize_text_field(wp_unslash($_POST['option_name'])) : '';
            $setting_value = isset($_POST['setting_value']) ? sanitize_email(wp_unslash($_POST['setting_value'])) : '';

            // Whitelist of allowed settings to prevent arbitrary option updates
            $allowed_settings = array(
                'vehizo_contact_email',
            );

            if (!in_array($option_name, $allowed_settings, true)) {
                wp_send_json_error(['message' => esc_html__('Invalid option name.', 'vehizo-vehicle-management')]);
                return;
            }

            if (!empty($option_name)) {
                 if (is_email($setting_value) || empty($setting_value)) { // Empty is also allowed
                    update_option($option_name, $setting_value);
                    wp_send_json_success(['message' => esc_html__('Setting saved.', 'vehizo-vehicle-management')]);
                } else {
                    wp_send_json_error(['message' => esc_html__('Invalid E-Mail address.', 'vehizo-vehicle-management')]);
                }
            } else {
                 wp_send_json_error(['message' => esc_html__('Missing option name.', 'vehizo-vehicle-management')]);
            }
        }

        /**
         * Handles the AJAX request to get a saved module setting.
         */
        public static function handle_ajax_get_setting() {
            // Verify nonce
            $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
            if (!wp_verify_nonce($nonce, 'vehizo_get_setting_nonce')) {
                wp_send_json_error(['message' => esc_html__('Security check failed.', 'vehizo-vehicle-management')]);
                return;
            }

            // Check user permissions
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => esc_html__('Insufficient permissions.', 'vehizo-vehicle-management')]);
                return;
            }

            $option_name = isset($_POST['option_name']) ? sanitize_text_field(wp_unslash($_POST['option_name'])) : '';
            if (!empty($option_name)) {
                $value = get_option($option_name, '');
                wp_send_json_success(['value' => $value]);
            } else {
                wp_send_json_error(['message' => esc_html__('Invalid option name.', 'vehizo-vehicle-management')]);
            }
        }
        
        /**
         * Handles the request to reset frontend settings to default.
         */
        public static function handle_settings_reset() {
            // KORREKTUR 1: Prüfen, ob der Reset-Button überhaupt geklickt wurde.
            if (!isset($_POST['vehizo_reset_settings']) || !sanitize_text_field(wp_unslash($_POST['vehizo_reset_settings']))) {
                return; // Wenn nicht, Funktion sofort beenden.
            }

            // KORREKTUR 2: Nonce-Prüfung beibehalten für die Sicherheit.
            $nonce = isset($_POST['vehizo_reset_settings_nonce']) ? sanitize_text_field(wp_unslash($_POST['vehizo_reset_settings_nonce'])) : '';
            if (!wp_verify_nonce($nonce, 'vehizo_reset_settings_action')) {
                // Bei einem Nonce-Fehler eine Fehlermeldung ausgeben, anstatt stillschweigend zu scheitern.
                wp_die(esc_html__('Security check failed. Please try again.', 'vehizo-vehicle-management'), 'Security Check', ['response' => 403]);
            }
            
            // Der restliche Code bleibt gleich.
            $frontend_options = [
                'vehizo_layout', 'vehizo_grid_columns', 'vehizo_show_filter',
                'vehizo_overview_page_id', 'vehizo_primary_color', 'vehizo_text_color',
                'vehizo_container_bg_color', 'vehizo_card_bg_color', 'vehizo_card_border_color',
                'vehizo_card_border_width', 'vehizo_card_radius', 'vehizo_card_box_shadow',
                'vehizo_specs_bg_color', 'vehizo_card_title_color', 'vehizo_card_title_fontweight',
                'vehizo_font_size', 'vehizo_heading_size'
            ];
            foreach ($frontend_options as $option) {
                delete_option($option);
            }
            wp_safe_redirect(admin_url('admin.php?page=vehizo-dashboard&tab=frontend&settings-reset=true'));
            exit;
        }
/**
         * Registers the main admin menu page and its default submenu.
         */
        public static function register_menu() {
            // This call creates the main top-level menu item "Vehizo".
            add_menu_page(
                __('Vehizo Dashboard', 'vehizo-vehicle-management'),
                'Vehizo',
                'manage_options',
                'vehizo-dashboard',
                [self::class, 'render_dashboard'],
                'dashicons-car',
                26
            );

            // This call adds the "Dashboard" submenu item.
            // By using the SAME SLUG as the parent, it becomes the default page.
            add_submenu_page(
                'vehizo-dashboard', 
                __('Dashboard', 'vehizo-vehicle-management'), 
                __('Dashboard', 'vehizo-vehicle-management'),
                'manage_options',
                'vehizo-dashboard', 
                [self::class, 'render_dashboard']
            );
        }
/**
         * Renders the main dashboard container and navigation.
         */
        public static function render_dashboard()
        {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only tab parameter for navigation
            $current_tab = isset($_GET['tab']) ? sanitize_key(wp_unslash($_GET['tab'])) : 'dashboard';
            ?>
            <div class="vehizo-dashboard">
                <div class="vehizo-header">
                    <img src="<?php echo esc_url(VEHIZO_PLUGIN_URL . 'assets/images/vehizo-logo.png'); ?>"
                         alt="<?php esc_attr_e('Vehizo Logo', 'vehizo-vehicle-management'); ?>" class="vehizo-logo">
                    <h1><?php esc_html_e('Dashboard', 'vehizo-vehicle-management'); ?></h1>
                </div>

                <?php
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only success/error messages
                if ( isset( $_GET['wishlist_page_created'] ) ) {
                    echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'The wishlist page was created successfully!', 'vehizo-vehicle-management' ) . '</p></div>';
                }
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only success/error messages
                if ( isset( $_GET['wishlist_page_failed'] ) ) {
                    echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__( 'Error: The wishlist page could not be created.', 'vehizo-vehicle-management' ) . '</p></div>';
                }
                ?>

                <div class="vehizo-nav">
                    <div class="nav-tab-wrapper">
                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=dashboard')); ?>"
                           class="nav-tab <?php echo ($current_tab === 'dashboard') ? 'nav-tab-active' : ''; ?>">
                            <?php esc_html_e('Dashboard', 'vehizo-vehicle-management'); ?>
                        </a>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=fahrzeuge')); ?>"
                           class="nav-tab <?php echo ($current_tab === 'fahrzeuge') ? 'nav-tab-active' : ''; ?>">
                            <?php esc_html_e('All Vehicles', 'vehizo-vehicle-management'); ?>
                        </a>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=settings')); ?>"
                           class="nav-tab <?php echo ($current_tab === 'settings') ? 'nav-tab-active' : ''; ?>">
                            <?php esc_html_e('Import Settings', 'vehizo-vehicle-management'); ?>
                        </a>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=frontend')); ?>"
                           class="nav-tab <?php echo ($current_tab === 'frontend') ? 'nav-tab-active' : ''; ?>">
                            <?php esc_html_e('Frontend', 'vehizo-vehicle-management'); ?>
                        </a>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=module')); ?>"
                           class="nav-tab <?php echo ($current_tab === 'module') ? 'nav-tab-active' : ''; ?>">
                            <?php esc_html_e('Modules', 'vehizo-vehicle-management'); ?>
                        </a>
                        <?php if (get_option('vehizo_statistics_module_active', '0') === '1'): ?>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=statistics')); ?>"
                           class="nav-tab <?php echo ($current_tab === 'statistics') ? 'nav-tab-active' : ''; ?>">
                            <?php esc_html_e('Statistics', 'vehizo-vehicle-management'); ?>
                        </a>
                        <?php endif; ?>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=pro-features')); ?>"
                           class="nav-tab <?php echo ($current_tab === 'pro-features') ? 'nav-tab-active' : ''; ?>"
                           style="color: #667eea;">
                            ⭐ <?php esc_html_e('Pro Features', 'vehizo-vehicle-management'); ?>
                        </a>
                    </div>
                </div>

                <div class="vehizo-content">
                    <div class="vehizo-content-wrapper">
                        <?php
                        switch ($current_tab) {
                            case 'fahrzeuge':
                                self::render_fahrzeuge_list();
                                break;
                            case 'settings':
                                self::render_settings();
                                break;
                            case 'frontend':
                                self::render_frontend_settings();
                                break;
                            case 'module':
                                self::render_module_tab();
                                break;
                            case 'statistics':
                                if (get_option('vehizo_statistics_module_active', '0') === '1' && class_exists('Vehizo_Statistics_Module')) {
                                    Vehizo_Statistics_Module::render_statistics_tab();
                                }
                                break;
                            case 'editor':
                                self::render_editor_tab();
                                break;
                            case 'seo-settings':
                                do_action('vehizo_render_seo_settings_tab');
                                break;
                            case 'pro-features':
                                self::render_pro_features_tab();
                                break;
                            case 'dashboard':
                            default:
                                self::render_fahrzeuge_tab();
                                break;
                        }
                        ?>
                    </div>
                </div>
            </div>
            <?php
        }
/**
         * Renders the main dashboard overview tab.
         */
        private static function render_fahrzeuge_tab()
        {
            // NEU: Logik zur Überprüfung der Merklisten-Seite
            $wishlist_page_id = get_option( 'vehizo_wishlist_page_id' );
            $wishlist_page_exists = ($wishlist_page_id && get_post_status( $wishlist_page_id ) === 'publish');

            if ( get_option('vehizo_wishlist_module_active', '1') === '1' && ! $wishlist_page_exists ) {
                $creation_url = wp_nonce_url( admin_url('admin.php?page=vehizo-dashboard&action=vehizo_create_page'), 'vehizo_create_page_nonce' );
                echo '<div class="notice notice-warning" style="padding: 1rem; margin-bottom: 1.5rem;">';
                echo '<p>' . esc_html__( 'The wishlist module is active, but the page to display the wishlist is missing.', 'vehizo-vehicle-management' ) . '</p>';
                echo '<p><a href="' . esc_url( $creation_url ) . '" class="button button-primary">' . esc_html__( 'Create Wishlist Page Automatically', 'vehizo-vehicle-management' ) . '</a></p>';
                echo '</div>';
            }
			
            $fahrzeuge_query = new WP_Query(['post_type' => 'vehizo_vehicle','post_status' => 'publish','posts_per_page' => -1]);
            $total_vehicles = $fahrzeuge_query->found_posts;
            $last_import_raw = get_option('vehizo_last_cron_import', __('Never', 'vehizo-vehicle-management'));
            $last_import = ($last_import_raw !== __('Never', 'vehizo-vehicle-management') && is_numeric($last_import_raw)) ? date_i18n(get_option('date_format'), (int) $last_import_raw) : __('Never', 'vehizo-vehicle-management');
            $last_import_status = get_option('vehizo_last_cron_status', __('Unknown', 'vehizo-vehicle-management'));
            $recent_vehicles = get_posts(['post_type' => 'vehizo_vehicle','post_status' => 'publish','numberposts' => 15,'orderby' => 'date','order' => 'DESC']);
            
            $active_modules = 0;
            $modules = [
                'contact_form' => get_option('vehizo_contact_module_active', '1'),
                'statistics' => get_option('vehizo_statistics_module_active', '0'),
                'wishlist' => get_option('vehizo_wishlist_module_active', '1'),
            ];
            foreach ($modules as $status) { if ($status === '1') $active_modules++; }
            
            $import_interval = get_option('vehizo_cron_interval', 'hourly');
            $interval_labels = [
                'hourly' => __('Hourly', 'vehizo-vehicle-management'),
                '3hours' => __('Every 3h', 'vehizo-vehicle-management'),
                '6hours' => __('Every 6h', 'vehizo-vehicle-management'),
                'daily' => __('Daily', 'vehizo-vehicle-management')
            ];
            ?>
            <div class="vehizo-section">
                <h2><?php esc_html_e('Dashboard Overview', 'vehizo-vehicle-management'); ?></h2>
                <div class="stats-grid">
                    <div class="stat-card"><h3><?php echo (int) $total_vehicles; ?></h3><p><?php esc_html_e('Total Vehicles', 'vehizo-vehicle-management'); ?></p></div>
                    <div class="stat-card"><h3><?php echo (int) $total_vehicles; ?></h3><p><?php esc_html_e('Published', 'vehizo-vehicle-management'); ?></p></div>
                    <div class="stat-card"><h3><?php echo (int) $active_modules; ?></h3><p><?php esc_html_e('Active Modules', 'vehizo-vehicle-management'); ?></p></div>
                    <div class="stat-card"><h3><?php echo esc_html($interval_labels[$import_interval] ?? $import_interval); ?></h3><p><?php esc_html_e('Import Interval', 'vehizo-vehicle-management'); ?></p></div>
                </div>

                <div style="display: grid; grid-template-columns: 2fr 1fr; gap: 32px; margin-top: 32px;">
                    <div class="main-content">
                        <div class="meta-box">
                            <h3><?php esc_html_e('Latest Vehicles', 'vehizo-vehicle-management'); ?></h3>
                            <?php if (empty($recent_vehicles)): ?>
                                <div style="text-align: center; padding: 40px 20px;">
                                    <div style="font-size: 48px; opacity: 0.6; margin-bottom: 16px;">🚗</div>
                                    <h4 style="margin: 16px 0 8px 0;"><?php esc_html_e('No vehicles available', 'vehizo-vehicle-management'); ?></h4>
                                    <p style="margin: 0 0 20px 0; color: #6b7280;"><?php esc_html_e('Run an import or add a vehicle manually.', 'vehizo-vehicle-management'); ?></p>
                                    <div style="display: flex; gap: 12px; justify-content: center; flex-wrap: wrap;">
                                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=editor')); ?>" class="modern-button"><?php esc_html_e('Add Vehicle', 'vehizo-vehicle-management'); ?></a>
                                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=settings')); ?>" class="modern-button secondary"><?php esc_html_e('Import', 'vehizo-vehicle-management'); ?></a>
                                    </div>
                                </div>
                            <?php else: ?>
                                <table style="width: 100%; border-collapse: collapse;">
                                    <thead><tr><th style="padding: 16px 0; text-align: left; font-weight: 600;"><?php esc_html_e('Vehicle', 'vehizo-vehicle-management'); ?></th><th style="padding: 16px 0; text-align: left; font-weight: 600;"><?php esc_html_e('Created', 'vehizo-vehicle-management'); ?></th><th style="padding: 16px 0; text-align: left; font-weight: 600;"><?php esc_html_e('Status', 'vehizo-vehicle-management'); ?></th><th style="padding: 16px 0; text-align: left; font-weight: 600;"><?php esc_html_e('Action', 'vehizo-vehicle-management'); ?></th></tr></thead>
                                    <tbody>
                                        <?php foreach ($recent_vehicles as $fahrzeug): ?>
                                            <tr><td style="padding: 12px 0;"><strong><?php echo esc_html($fahrzeug->post_title); ?></strong></td><td style="padding: 12px 0;"><?php echo esc_html(get_the_date(get_option('date_format'), $fahrzeug)); ?></td><td style="padding: 12px 0;"><span style="background: #d1fae5; color: #065f46; padding: 4px 8px; border-radius: 4px; font-size: 11px; font-weight: 600;">● <?php esc_html_e('Published', 'vehizo-vehicle-management'); ?></span></td><td style="padding: 12px 0;"><a class="modern-button secondary" href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=editor&fahrzeug_id=' . $fahrzeug->ID)); ?>" style="padding: 8px 16px; font-size: 12px; margin-right: 8px;"><?php esc_html_e('Edit', 'vehizo-vehicle-management'); ?></a><a href="<?php echo esc_url(get_delete_post_link($fahrzeug->ID)); ?>" class="vehizo-delete-vehicle-link" style="padding: 8px 16px; font-size: 12px; background: #fee2e2; color: #b91c1c; text-decoration: none; border-radius: 4px;"><?php esc_html_e('Delete', 'vehizo-vehicle-management'); ?></a></td></tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                                <div style="text-align: center; margin-top: 20px;"><a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=fahrzeuge')); ?>" class="modern-button secondary"><?php esc_html_e('View All Vehicles', 'vehizo-vehicle-management'); ?></a></div>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="sidebar-content">
                        <div class="meta-box">
                            <h3><?php esc_html_e('Quick Access', 'vehizo-vehicle-management'); ?></h3>
                            <div style="display: flex; flex-direction: column; gap: 12px;">
                                <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=editor')); ?>" class="modern-button" style="display: inline-block; padding: 12px 20px; font-size: 14px; text-align: center;"><?php esc_html_e('Add Vehicle', 'vehizo-vehicle-management'); ?></a>
                                <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=frontend')); ?>" class="modern-button secondary" style="display: inline-block; padding: 10px 20px; font-size: 13px; text-align: center;"><?php esc_html_e('Frontend Settings', 'vehizo-vehicle-management'); ?></a>
                                <a href="<?php echo esc_url(admin_url('edit.php?post_type=vehizo_vehicle')); ?>" class="modern-button secondary" style="display: inline-block; padding: 10px 20px; font-size: 13px; text-align: center;"><?php esc_html_e('All Vehicles', 'vehizo-vehicle-management'); ?></a>
                            </div>
                        </div>
                        <div class="meta-box">
                            <h3><?php esc_html_e('Import Status', 'vehizo-vehicle-management'); ?></h3>
                            <?php
                            $status_class = ($last_import_status === __('Successful', 'vehizo-vehicle-management')) ? 'vehizo-status-success' : 'vehizo-status-warning';
                            $last_import_display = ($last_import_raw !== 'Never') ?
                                esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), (int) $last_import_raw)) :
                                esc_html__('Never yet', 'vehizo-vehicle-management');
                            ?>
                            <div class="vehizo-import-status <?php echo esc_attr($status_class); ?>">
                                <strong><?php esc_html_e('Status:', 'vehizo-vehicle-management'); ?></strong> <?php echo esc_html($last_import_status); ?><br>
                                <strong><?php esc_html_e('Last Import:', 'vehizo-vehicle-management'); ?></strong><br>
                                <?php echo esc_html($last_import_display); ?>
                            </div>
                            <div style="margin-top: 16px;"><a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=settings')); ?>" class="modern-button secondary" style="display: inline-block; padding: 10px 20px; font-size: 13px; width: 100%; text-align: center; box-sizing: border-box;"><?php esc_html_e('Import Settings', 'vehizo-vehicle-management'); ?></a></div>
                        </div>
                        <div class="meta-box">
                            <h3><?php esc_html_e('Modules', 'vehizo-vehicle-management'); ?></h3>
                            <div>
                                <?php 
                                $module_names = [ 'contact_form' => __('Contact Form', 'vehizo-vehicle-management'), 'statistics' => __('Vehicle Statistics', 'vehizo-vehicle-management'), 'wishlist' => __('Vehicle Wishlist', 'vehizo-vehicle-management') ];
                                foreach ($modules as $module_key => $status): 
                                    $module_name = $module_names[$module_key] ?? $module_key; ?>
                                    <?php $module_status_class = ($status === '1') ? 'vehizo-module-active' : 'vehizo-module-inactive'; ?>
                                    <div class="vehizo-module-item">
                                        <span class="vehizo-module-name"><?php echo esc_html($module_name); ?></span>
                                        <span class="vehizo-module-status <?php echo esc_attr($module_status_class); ?>">
                                            <?php echo ($status === '1') ? esc_html__('Active', 'vehizo-vehicle-management') : esc_html__('Inactive', 'vehizo-vehicle-management'); ?>
                                        </span>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            <div style="margin-top: 16px;"><a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=module')); ?>" class="modern-button secondary" style="display: inline-block; padding: 10px 20px; font-size: 13px; width: 100%; text-align: center; box-sizing: border-box;"><?php esc_html_e('Manage Modules', 'vehizo-vehicle-management'); ?></a></div>
                        </div>
                        <div class="meta-box">
                            <h3><?php esc_html_e('Support & Contact', 'vehizo-vehicle-management'); ?></h3>
                            <div style="color: #6b7280; line-height: 1.6;">
                                <div style="margin-bottom: 16px;"><strong style="color: #374151;"><?php esc_html_e('Thorsten Glander', 'vehizo-vehicle-management'); ?></strong><br><?php esc_html_e('Professional Vehicle Management Solutions', 'vehizo-vehicle-management'); ?></div>
                                <div style="margin-bottom: 12px;"><strong><?php esc_html_e('E-Mail:', 'vehizo-vehicle-management'); ?></strong><br><a href="mailto:support@vehizo.de" style="color: #2563eb; text-decoration: none;">support@vehizo.de</a></div>
                                <div style="margin-bottom: 16px;"><strong><?php esc_html_e('Website:', 'vehizo-vehicle-management'); ?></strong><br><a href="<?php echo esc_url('https://vehizo.de'); ?>" target="_blank" style="color: #2563eb; text-decoration: none;">vehizo.de</a></div>
                                <div style="padding: 12px; background: #f9fafb; border-radius: 6px; font-size: 13px; margin-top: 16px;"><strong><?php esc_html_e('Need help?', 'vehizo-vehicle-management'); ?></strong><br><?php esc_html_e('We are happy to assist you.', 'vehizo-vehicle-management'); ?></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php
        }		
        /**
         * Renders the vehicle editor tab content.
         */
        private static function render_editor_tab()
        {
            global $post;
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only parameter for admin navigation to select vehicle for editing
            $post_id = isset($_GET['fahrzeug_id']) ? absint($_GET['fahrzeug_id']) : 0;
            if ($post_id > 0) {
                $post = get_post($post_id);
                if ( ! $post || $post->post_type !== 'vehizo_vehicle' ) {
                    echo '<div class="notice notice-error"><p>' . esc_html__( 'Vehicle not found.', 'vehizo-vehicle-management' ) . '</p></div>';
                    return;
                }
            } else {
                $post = get_default_post_to_edit( 'vehizo_vehicle', true );
            }
            if ( class_exists( 'Vehizo_Admin_Vehicle_Editor' ) ) {
                Vehizo_Admin_Vehicle_Editor::render_editor_tab();
            }
        }
/**
         * Renders the frontend display settings page.
         */
        public static function render_frontend_settings()
        {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only parameter for displaying admin success message
            if (isset($_GET['settings-updated'])) {
                echo '<div id="message" class="updated notice is-dismissible"><p><strong>' . esc_html__('Settings saved.', 'vehizo-vehicle-management') . '</strong></p></div>';
            }
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only parameter for displaying admin success message
            else if (isset($_GET['settings-reset'])) {
                echo '<div id="message" class="updated notice is-dismissible"><p><strong>' . esc_html__('Settings successfully reset.', 'vehizo-vehicle-management') . '</strong></p></div>';
            }
            ?>
            <div class="vehizo-section">
                <h2><?php esc_html_e('Frontend Display', 'vehizo-vehicle-management'); ?></h2>
                <p><?php esc_html_e('Customize the appearance of the vehicle lists to match your website design.', 'vehizo-vehicle-management'); ?></p>
                
                <form method="post" action="options.php">
                    <?php settings_fields('vehizo_appearance_group'); ?>
                    
                    <div class="meta-box">
                        <h3><?php esc_html_e('Layout Settings', 'vehizo-vehicle-management'); ?></h3>
                        <table class="form-table">
                            <?php do_settings_fields('vehizo_appearance_settings_layout', 'vehizo_layout_section'); ?>
                        </table>
                    </div>
                    
                    <div class="meta-box">
                        <h3><?php esc_html_e('Global Design', 'vehizo-vehicle-management'); ?></h3>
                        <table class="form-table">
                            <?php do_settings_fields('vehizo_appearance_settings_global_design', 'vehizo_global_design_section'); ?>
                        </table>
                    </div>
                    
                    <div class="meta-box">
                        <h3><?php esc_html_e('Card Design', 'vehizo-vehicle-management'); ?></h3>
                        <table class="form-table">
                            <?php do_settings_fields('vehizo_appearance_settings_cards', 'vehizo_card_design_section'); ?>
                        </table>
                    </div>
                    
                    <div class="meta-box">
                        <h3><?php esc_html_e('Typography', 'vehizo-vehicle-management'); ?></h3>
                        <table class="form-table">
                            <?php do_settings_fields('vehizo_appearance_settings_typography', 'vehizo_typography_section'); ?>
                        </table>
                    </div>
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; padding: 20px; background: #f9fafb; border-radius: 8px;">
                        <div>
                            <?php wp_nonce_field('vehizo_reset_settings_action', 'vehizo_reset_settings_nonce'); ?>
                            <button type="submit" name="vehizo_reset_settings" class="vehizo-reset-settings" style="background: #fee2e2; color: #991b1b; border: 2px solid #fecaca; padding: 10px 20px; border-radius: 8px; cursor: pointer;">
                                <?php esc_html_e('Reset to Default', 'vehizo-vehicle-management'); ?>
                            </button>
                        </div>
                        <?php submit_button(__('Save Settings', 'vehizo-vehicle-management'), 'primary', 'submit', false); ?>
                    </div>
                </form>
            </div>
            <?php
        }
        /**
         * Registers the settings fields for the plugin.
         */
        public static function register_settings()
        {
             // --- Import Settings ---
             register_setting('vehizo_import_group', 'vehizo_mobile_username', 'sanitize_text_field');
             register_setting('vehizo_import_group', 'vehizo_mobile_password', 'sanitize_text_field');
             register_setting('vehizo_import_group', 'vehizo_mobile_api_url', 'esc_url_raw');
             add_settings_section('vehizo_import_section', __('API Credentials', 'vehizo-vehicle-management'), null, 'vehizo_import_settings');

             add_settings_field('vehizo_mobile_username', __('Username', 'vehizo-vehicle-management'), function () { echo '<input type="text" name="vehizo_mobile_username" value="' . esc_attr(get_option('vehizo_mobile_username')) . '" class="regular-text">'; }, 'vehizo_import_settings', 'vehizo_import_section');
             add_settings_field('vehizo_mobile_password', __('Password', 'vehizo-vehicle-management'), function () { echo '<input type="password" name="vehizo_mobile_password" value="' . esc_attr(get_option('vehizo_mobile_password')) . '" class="regular-text">'; }, 'vehizo_import_settings', 'vehizo_import_section');
             add_settings_field('vehizo_mobile_api_url', __('API URL', 'vehizo-vehicle-management'), function () { $default = 'https://services.sandbox.mobile.de/search-api/search'; echo '<input type="text" name="vehizo_mobile_api_url" value="' . esc_attr(get_option('vehizo_mobile_api_url', $default)) . '" class="large-text"><p class="description">' . esc_html__('Enter the API URL of the data source. The plugin is prepared for importing data from automotive platforms like mobile.de, but is not affiliated with them.', 'vehizo-vehicle-management') . '</p>'; }, 'vehizo_import_settings', 'vehizo_import_section');
 
             // --- Frontend Appearance Settings ---
             $appearance_group = 'vehizo_appearance_group';
 
             // Register all settings in a loop for clarity
             $appearance_options = [
                 'vehizo_layout' => ['sanitize_callback' => 'sanitize_key', 'default' => 'grid'],
                 'vehizo_grid_columns' => ['sanitize_callback' => 'absint', 'default' => 3],
                 'vehizo_show_filter' => ['sanitize_callback' => 'sanitize_key', 'default' => '1'],
                 'vehizo_posts_per_page' => ['sanitize_callback' => 'absint', 'default' => 12],
                 'vehizo_primary_color' => ['sanitize_callback' => 'sanitize_hex_color', 'default' => '#0073aa'],
                 'vehizo_text_color' => ['sanitize_callback' => 'sanitize_hex_color', 'default' => '#333333'],
                 'vehizo_container_bg_color' => ['sanitize_callback' => 'sanitize_hex_color', 'default' => '#ffffff'],
                 'vehizo_secondary_color' => ['sanitize_callback' => 'sanitize_hex_color', 'default' => '#6c757d'],
                 'vehizo_button_hover_color' => ['sanitize_callback' => 'sanitize_hex_color', 'default' => '#005a87'],
                 'vehizo_currency_symbol' => ['sanitize_callback' => 'sanitize_text_field', 'default' => '€'],
                 'vehizo_currency_position' => ['sanitize_callback' => 'sanitize_key', 'default' => 'after'],
                 'vehizo_specs_bg_color' => ['sanitize_callback' => 'sanitize_hex_color', 'default' => '#f8fafc'],
                 'vehizo_card_bg_color' => ['sanitize_callback' => 'sanitize_hex_color', 'default' => '#ffffff'],
                 'vehizo_card_border_color' => ['sanitize_callback' => 'sanitize_hex_color', 'default' => '#e2e8f0'],
                 'vehizo_card_border_width' => ['sanitize_callback' => 'absint', 'default' => 1],
                 'vehizo_card_radius' => ['sanitize_callback' => 'absint', 'default' => 8],
                 'vehizo_card_box_shadow' => ['sanitize_callback' => 'sanitize_text_field', 'default' => 'medium'],
                 'vehizo_card_title_color' => ['sanitize_callback' => 'sanitize_hex_color', 'default' => '#1f2937'],
                 'vehizo_card_title_fontweight' => ['sanitize_callback' => 'absint', 'default' => 700],
                 'vehizo_font_size' => ['sanitize_callback' => 'absint', 'default' => 16],
                 'vehizo_heading_size' => ['sanitize_callback' => 'absint', 'default' => 24],
                 'vehizo_hot_vehicle_threshold' => ['sanitize_callback' => 'absint', 'default' => 100],
             ];
 
             foreach ($appearance_options as $option_name => $args) {
                 register_setting($appearance_group, $option_name, $args);
             }
 
             // Add sections
             $layout_page = 'vehizo_appearance_settings_layout';
             $design_page = 'vehizo_appearance_settings_global_design';
             $cards_page = 'vehizo_appearance_settings_cards';
             $typo_page = 'vehizo_appearance_settings_typography';
 
             add_settings_section('vehizo_layout_section', '', null, $layout_page);
             add_settings_section('vehizo_global_design_section', '', null, $design_page);
             add_settings_section('vehizo_card_design_section', '', null, $cards_page);
             add_settings_section('vehizo_typography_section', '', null, $typo_page);
 
             // --- Add fields to sections ---
 
            // Helper function for sliders
            $render_slider = function($args) {
                $id = $args['id'];
                $option_value = get_option($id, $args['default'] ?? '0');
                $min = $args['min'] ?? '0';
                $max = $args['max'] ?? '100';
                $step = $args['step'] ?? '1';
                $unit = $args['unit'] ?? 'px';
                echo sprintf(
                    '<div style="display: flex; align-items: center; gap: 16px;">' .
                    '<input type="range" class="vehizo-slider" data-unit="%s" id="%s" name="%s" min="%s" max="%s" step="%s" value="%s" style="width: 250px;">' .
                    '<span id="%s-value" class="vehizo-slider-value">%s%s</span>' .
                    '</div>',
                    esc_attr($unit), esc_attr($id), esc_attr($id), esc_attr($min), esc_attr($max), esc_attr($step), esc_attr($option_value),
                    esc_attr($id), esc_html($option_value), esc_html($unit)
                );
            };

             // Layout Section
             add_settings_field('vehizo_layout', __('Layout', 'vehizo-vehicle-management'), function() { $layout = get_option('vehizo_layout', 'grid'); echo '<select name="vehizo_layout"><option value="grid"' . selected($layout, 'grid', false) . '>' . esc_html__('Grid', 'vehizo-vehicle-management') . '</option><option value="list"' . selected($layout, 'list', false) . '>' . esc_html__('List', 'vehizo-vehicle-management') . '</option></select>'; }, $layout_page, 'vehizo_layout_section');
             add_settings_field('vehizo_grid_columns', __('Grid Columns', 'vehizo-vehicle-management'), $render_slider, $layout_page, 'vehizo_layout_section', ['id' => 'vehizo_grid_columns', 'min' => 1, 'max' => 6, 'default' => 3, 'unit' => '']);
             add_settings_field('vehizo_show_filter', __('Show Filter', 'vehizo-vehicle-management'), function() { $show = get_option('vehizo_show_filter', '1'); echo '<label><input type="checkbox" name="vehizo_show_filter" value="1"' . checked($show, '1', false) . '> ' . esc_html__('Display filter bar above vehicle list', 'vehizo-vehicle-management') . '</label>'; }, $layout_page, 'vehizo_layout_section');
             add_settings_field('vehizo_posts_per_page', __('Vehicles per Page', 'vehizo-vehicle-management'), function() { $ppp = get_option('vehizo_posts_per_page', 12); echo '<input type="number" name="vehizo_posts_per_page" value="' . esc_attr($ppp) . '" min="1">'; }, $layout_page, 'vehizo_layout_section');
             add_settings_field('vehizo_hot_vehicle_threshold', __('"Popular" Threshold', 'vehizo-vehicle-management'), function () { $value = get_option('vehizo_hot_vehicle_threshold', 100); echo '<input type="number" name="vehizo_hot_vehicle_threshold" value="' . esc_attr($value) . '" class="small-text" min="1"><p class="description">' . esc_html__('Number of views after which a vehicle is marked as "Popular".', 'vehizo-vehicle-management') . '</p>'; }, $layout_page, 'vehizo_layout_section');
             add_settings_field('vehizo_currency_symbol', __('Currency Symbol', 'vehizo-vehicle-management'), function() { $symbol = get_option('vehizo_currency_symbol', '€'); echo '<input type="text" name="vehizo_currency_symbol" value="' . esc_attr($symbol) . '" class="small-text" maxlength="5">'; }, $layout_page, 'vehizo_layout_section');
             add_settings_field('vehizo_currency_position', __('Currency Position', 'vehizo-vehicle-management'), function() { $pos = get_option('vehizo_currency_position', 'after'); echo '<select name="vehizo_currency_position"><option value="before"' . selected($pos, 'before', false) . '>' . esc_html__('Before price', 'vehizo-vehicle-management') . '</option><option value="after"' . selected($pos, 'after', false) . '>' . esc_html__('After price', 'vehizo-vehicle-management') . '</option></select>'; }, $layout_page, 'vehizo_layout_section');
 
             // Global Design Section
             add_settings_field('vehizo_primary_color', __('Primary Color', 'vehizo-vehicle-management'), function() { $color = get_option('vehizo_primary_color', '#0073aa'); echo '<input type="color" name="vehizo_primary_color" value="' . esc_attr($color) . '">'; }, $design_page, 'vehizo_global_design_section');
             add_settings_field('vehizo_secondary_color', __('Secondary Color', 'vehizo-vehicle-management'), function() { $color = get_option('vehizo_secondary_color', '#6c757d'); echo '<input type="color" name="vehizo_secondary_color" value="' . esc_attr($color) . '">'; }, $design_page, 'vehizo_global_design_section');
             add_settings_field('vehizo_button_hover_color', __('Button Hover Color', 'vehizo-vehicle-management'), function() { $color = get_option('vehizo_button_hover_color', '#005a87'); echo '<input type="color" name="vehizo_button_hover_color" value="' . esc_attr($color) . '">'; }, $design_page, 'vehizo_global_design_section');
             add_settings_field('vehizo_text_color', __('Text Color', 'vehizo-vehicle-management'), function() { $color = get_option('vehizo_text_color', '#333333'); echo '<input type="color" name="vehizo_text_color" value="' . esc_attr($color) . '">'; }, $design_page, 'vehizo_global_design_section');
             add_settings_field('vehizo_container_bg_color', __('Container Background', 'vehizo-vehicle-management'), function() { $color = get_option('vehizo_container_bg_color', '#ffffff'); echo '<input type="color" name="vehizo_container_bg_color" value="' . esc_attr($color) . '">'; }, $design_page, 'vehizo_global_design_section');
 
             // Card Design Section
             add_settings_field('vehizo_card_bg_color', __('Card Background Color', 'vehizo-vehicle-management'), function() { $color = get_option('vehizo_card_bg_color', '#ffffff'); echo '<input type="color" name="vehizo_card_bg_color" value="' . esc_attr($color) . '">'; }, $cards_page, 'vehizo_card_design_section');
             add_settings_field('vehizo_card_border_color', __('Card Border Color', 'vehizo-vehicle-management'), function() { $color = get_option('vehizo_card_border_color', '#e2e8f0'); echo '<input type="color" name="vehizo_card_border_color" value="' . esc_attr($color) . '">'; }, $cards_page, 'vehizo_card_design_section');
             add_settings_field('vehizo_card_border_width', __('Card Border Width (px)', 'vehizo-vehicle-management'), $render_slider, $cards_page, 'vehizo_card_design_section', ['id' => 'vehizo_card_border_width', 'min' => 0, 'max' => 10, 'default' => 1, 'unit' => 'px']);
             add_settings_field('vehizo_card_radius', __('Card Border Radius (px)', 'vehizo-vehicle-management'), $render_slider, $cards_page, 'vehizo_card_design_section', ['id' => 'vehizo_card_radius', 'min' => 0, 'max' => 50, 'default' => 8, 'unit' => 'px']);
             add_settings_field('vehizo_card_box_shadow', __('Card Box Shadow', 'vehizo-vehicle-management'), function() { $shadow = get_option('vehizo_card_box_shadow', 'medium'); echo '<select name="vehizo_card_box_shadow"><option value="none"' . selected($shadow, 'none', false) . '>None</option><option value="small"' . selected($shadow, 'small', false) . '>Small</option><option value="medium"' . selected($shadow, 'medium', false) . '>Medium</option><option value="large"' . selected($shadow, 'large', false) . '>Large</option></select>'; }, $cards_page, 'vehizo_card_design_section');
             add_settings_field('vehizo_specs_bg_color', __('Background (Info Rows)', 'vehizo-vehicle-management'), function() { $color = get_option('vehizo_specs_bg_color', '#f8fafc'); echo '<input type="color" name="vehizo_specs_bg_color" value="' . esc_attr($color) . '">'; }, $cards_page, 'vehizo_card_design_section');
 
             // Typography Section
             add_settings_field('vehizo_card_title_color', __('Card Title Color', 'vehizo-vehicle-management'), function() { $color = get_option('vehizo_card_title_color', '#1f2937'); echo '<input type="color" name="vehizo_card_title_color" value="' . esc_attr($color) . '">'; }, $typo_page, 'vehizo_typography_section');
             add_settings_field('vehizo_card_title_fontweight', __('Card Title Font Weight', 'vehizo-vehicle-management'), $render_slider, $typo_page, 'vehizo_typography_section', ['id' => 'vehizo_card_title_fontweight', 'min' => 100, 'max' => 900, 'step' => 100, 'default' => 700, 'unit' => '']);
             add_settings_field('vehizo_font_size', __('Base Font Size (px)', 'vehizo-vehicle-management'), $render_slider, $typo_page, 'vehizo_typography_section', ['id' => 'vehizo_font_size', 'min' => 12, 'max' => 22, 'default' => 16, 'unit' => 'px']);
             add_settings_field('vehizo_heading_size', __('Heading Font Size (px)', 'vehizo-vehicle-management'), $render_slider, $typo_page, 'vehizo_typography_section', ['id' => 'vehizo_heading_size', 'min' => 16, 'max' => 40, 'default' => 24, 'unit' => 'px']);
        }

        /**
         * Renders the import settings page.
         */
        public static function render_settings()
        {
            $last_status = get_option('vehizo_last_cron_status', __('Never run', 'vehizo-vehicle-management'));
            $last_import_raw = get_option('vehizo_last_cron_import', 'Never');

            // DEBUG: Log what we're getting from the database
            if (defined('WP_DEBUG') && WP_DEBUG) {
            }

            $last_import = ($last_import_raw !== 'Never' && !empty($last_import_raw) && is_numeric($last_import_raw)) ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), (int) $last_import_raw) : __('Never', 'vehizo-vehicle-management');

            if (defined('WP_DEBUG') && WP_DEBUG) {
            }
            ?>
            <div class="vehizo-section">
                <h2><?php esc_html_e('Import Settings', 'vehizo-vehicle-management'); ?></h2>
                <div class="meta-box">
                    <h3><?php esc_html_e('Import Status', 'vehizo-vehicle-management'); ?></h3>
                    <?php
                    $status_class = ($last_status === __('Successful', 'vehizo-vehicle-management') ||
                                    $last_status === __('Successful (manual)', 'vehizo-vehicle-management') ||
                                    $last_status === 'Successful (manual)') ? 'vehizo-status-success' : 'vehizo-status-warning';
                    ?>
                    <div class="vehizo-import-status <?php echo esc_attr($status_class); ?>">
                        <strong><?php esc_html_e('Last Import:', 'vehizo-vehicle-management'); ?></strong> <?php echo esc_html($last_status); ?><br>
                        <strong><?php esc_html_e('Time:', 'vehizo-vehicle-management'); ?></strong> <?php echo esc_html($last_import); ?>
                    </div>
                </div>
                <form method="post" action="options.php">
                    <?php settings_fields('vehizo_import_group'); do_settings_sections('vehizo_import_settings'); ?>
                    <?php submit_button(__('Save Settings', 'vehizo-vehicle-management')); ?>
                </form>
                <div class="meta-box">
                    <h3><?php esc_html_e('Manual Import', 'vehizo-vehicle-management'); ?></h3>
                    <p><?php esc_html_e('Start an immediate import.', 'vehizo-vehicle-management'); ?></p>
                    <button id="vehizo-import-button" class="modern-button"><?php esc_html_e('Import Now', 'vehizo-vehicle-management'); ?></button>
                    <div id="vehizo-import-status" style="margin-top: 20px; display: none;"></div>
                </div>
            </div>
            <?php
        }

        /**
         * Passes the AJAX import request to the importer class.
         */
        public static function handle_ajax_import()
        {
            if (class_exists('Vehizo_Mobile_Importer')) {
                Vehizo_Mobile_Importer::handle_ajax_import();
            } else {
                wp_send_json_error(['message' => esc_html__('Importer class not found.', 'vehizo-vehicle-management')]);
            }
        }
/**
         * Renders the module management tab.
         */
        private static function render_module_tab()
        {
            ?>
            <div class="vehizo-section">
                <h2><?php esc_html_e('Manage Modules', 'vehizo-vehicle-management'); ?></h2>
                
                <div style="display: grid; grid-template-columns: 1fr 350px; gap: 32px;">
                    <div class="main-content">
                        <?php
                        if (class_exists('Vehizo_Modules')) {
                            Vehizo_Modules::render();
                        } else {
                            echo '<div style="padding: 20px; background: #fee2e2; color: #b91c1c; border-radius: 8px;"><p>' . esc_html__('Modules class not found.', 'vehizo-vehicle-management') . '</p></div>';
                        }
                        ?>
                    </div>
                    
                    <div class="sidebar-content">
                        <div class="meta-box">
                            <h3><?php esc_html_e('Module Statistics', 'vehizo-vehicle-management'); ?></h3>
                            <div class="stats-grid" style="grid-template-columns: 1fr; gap: 16px;">
                                <?php
                                $active_modules = 0;
                                $free_modules = [
                                    'contact_form' => get_option('vehizo_contact_module_active', '1'),
                                    'statistics' => get_option('vehizo_statistics_module_active', '0'),
                                    'wishlist' => get_option('vehizo_wishlist_module_active', '1'),
                                ];
                                
                                $modules = apply_filters('vehizo_counted_modules', $free_modules);
                                
                                foreach ($modules as $status) {
                                    if ($status === '1') {
                                        $active_modules++;
                                    }
                                }
                                ?>
                                <div class="stat-card">
                                    <h3><?php echo (int) $active_modules; ?></h3>
                                    <p><?php esc_html_e('Active Modules', 'vehizo-vehicle-management'); ?></p>
                                </div>
                            </div>
                        </div>

                        <div class="meta-box">
                            <h3><?php esc_html_e('Tips', 'vehizo-vehicle-management'); ?></h3>
                            <ul style="margin: 0; padding-left: 20px;">
                                <li><?php esc_html_e('Modules can be activated/deactivated at any time', 'vehizo-vehicle-management'); ?></li>
                                <li><?php esc_html_e('Premium modules significantly extend functionality', 'vehizo-vehicle-management'); ?></li>
                                <li><?php esc_html_e('Regular updates bring new features', 'vehizo-vehicle-management'); ?></li>
                                <li><?php esc_html_e('Deactivated modules do not consume resources', 'vehizo-vehicle-management'); ?></li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
            <?php
        }
		
        /**
         * Renders the list of all vehicles.
         */
        private static function render_fahrzeuge_list()
        {
            $fahrzeuge = get_posts(['post_type' => 'vehizo_vehicle', 'post_status' => 'any', 'numberposts' => -1]);
            $total_vehicles = count($fahrzeuge);
            $published_vehicles = count(array_filter($fahrzeuge, function($f) { return $f->post_status === 'publish'; }));
            $last_import_raw = get_option('vehizo_last_cron_import', 'Never');
            $last_import = ($last_import_raw !== 'Never' && is_numeric($last_import_raw)) ? date_i18n(get_option('date_format'), (int) $last_import_raw) : __('Never', 'vehizo-vehicle-management');
            ?>
            <div class="vehizo-section">
                <h2><?php esc_html_e('All Vehicles', 'vehizo-vehicle-management'); ?></h2>
                <div class="stats-grid">
                    <div class="stat-card"><h3><?php echo (int) $total_vehicles; ?></h3><p><?php esc_html_e('Total Vehicles', 'vehizo-vehicle-management'); ?></p></div>
                    <div class="stat-card"><h3><?php echo (int) $published_vehicles; ?></h3><p><?php esc_html_e('Published', 'vehizo-vehicle-management'); ?></p></div>
                    <div class="stat-card"><h3><?php echo (int) ($total_vehicles - $published_vehicles); ?></h3><p><?php esc_html_e('Other (Draft, etc.)', 'vehizo-vehicle-management'); ?></p></div>
                    <div class="stat-card"><h3><?php echo esc_html($last_import); ?></h3><p><?php esc_html_e('Last Import', 'vehizo-vehicle-management'); ?></p></div>
                </div>

                <div class="meta-box">
                    <h3><?php esc_html_e('Quick Actions', 'vehizo-vehicle-management'); ?></h3>
                    <div style="display: flex; gap: 16px; flex-wrap: wrap;">
                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=editor')); ?>" class="modern-button">
                            <?php esc_html_e('Add New Vehicle', 'vehizo-vehicle-management'); ?>
                        </a>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=settings')); ?>" class="modern-button secondary">
                            <?php esc_html_e('Perform Import', 'vehizo-vehicle-management'); ?>
                        </a>
                    </div>
                </div>

                <?php if (empty($fahrzeuge)): ?>
                    <div style="text-align: center; padding: 80px 40px;">
                        <div style="font-size: 64px;">🚗</div>
                        <h3><?php esc_html_e('No Vehicles Found', 'vehizo-vehicle-management'); ?></h3>
                        <p><?php esc_html_e('Run an import to display vehicles or add one manually.', 'vehizo-vehicle-management'); ?></p>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=editor')); ?>" class="modern-button"><?php esc_html_e('Add Vehicle', 'vehizo-vehicle-management'); ?></a>
                    </div>
                <?php else: ?>
                    <table style="width: 100%; border-collapse: collapse; background: white; border-radius: 8px; overflow: hidden; border: 1px solid #e5e7eb;">
                        <thead style="background: #f9fafb;"><tr>
                            <th style="padding: 20px; text-align: left; font-weight: 700;"><?php esc_html_e('Vehicle', 'vehizo-vehicle-management'); ?></th>
                            <th style="padding: 20px; text-align: left; font-weight: 700;"><?php esc_html_e('Created on', 'vehizo-vehicle-management'); ?></th>
                            <th style="padding: 20px; text-align: left; font-weight: 700;"><?php esc_html_e('Status', 'vehizo-vehicle-management'); ?></th>
                            <th style="padding: 20px; text-align: left; font-weight: 700;"><?php esc_html_e('Actions', 'vehizo-vehicle-management'); ?></th>
                        </tr></thead>
                        <tbody>
                            <?php foreach ($fahrzeuge as $fahrzeug): ?>
                                <tr>
                                    <td style="padding: 20px;"><strong><?php echo esc_html($fahrzeug->post_title); ?></strong></td>
                                    <td style="padding: 20px;"><?php echo esc_html(get_the_date(get_option('date_format') . ' ' . get_option('time_format'), $fahrzeug)); ?></td>
                                    <td style="padding: 20px;">
                                        <div class="vehizo-status-updater">
                                            <?php
                                            $current_status = get_post_meta($fahrzeug->ID, 'vehizo_vehicle_status', true);
                                            if (empty($current_status)) {
                                                $current_status = 'available';
                                            }
                                            ?>
                                            <select class="vehizo-status-select" data-vehicle-id="<?php echo (int) $fahrzeug->ID; ?>">
                                                <option value="available" <?php selected($current_status, 'available'); ?>><?php esc_html_e('Available', 'vehizo-vehicle-management'); ?></option>
                                                <option value="reserved" <?php selected($current_status, 'reserved'); ?>><?php esc_html_e('Reserved', 'vehizo-vehicle-management'); ?></option>
                                                <option value="sold" <?php selected($current_status, 'sold'); ?>><?php esc_html_e('Sold', 'vehizo-vehicle-management'); ?></option>
                                            </select>
                                            <button class="button button-primary vehizo-save-status-button" style="display:none;"><?php esc_html_e('Save', 'vehizo-vehicle-management'); ?></button>
                                        </div>
                                    </td>
                                    <td style="padding: 20px;">
                                        <a class="modern-button secondary" href="<?php echo esc_url(admin_url('admin.php?page=vehizo-dashboard&tab=editor&fahrzeug_id=' . $fahrzeug->ID)); ?>" style="padding: 8px 16px; font-size: 12px; margin-right: 8px;"><?php esc_html_e('Edit', 'vehizo-vehicle-management'); ?></a>
                                        <a href="<?php echo esc_url(get_delete_post_link($fahrzeug->ID)); ?>" 
                                           class="vehizo-delete-vehicle-link"
                                           data-vehicle-id="<?php echo (int) $fahrzeug->ID; ?>"
                                           style="padding: 8px 16px; font-size: 12px; background: #fee2e2; color: #b91c1c; border: 1px solid #ef4444; text-decoration: none; border-radius: 4px;">
                                            <?php esc_html_e('Delete', 'vehizo-vehicle-management'); ?>
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
            <?php
        }

        /**
         * Renders the Pro Features information tab (WordPress.org compliant).
         */
        private static function render_pro_features_tab()
        {
            ?>
            <div class="vehizo-section">
                <h2><?php esc_html_e('Pro Features', 'vehizo-vehicle-management'); ?></h2>
                <p><?php esc_html_e('Enhance your vehicle management with additional professional features.', 'vehizo-vehicle-management'); ?></p>

                <div class="modules-grid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 24px; margin-top: 32px;">

                    <!-- Automated Imports -->
                    <div class="module-card" style="background: #fff; border: 2px solid #e2e8f0; border-radius: 12px; padding: 24px;">
                        <div style="display: flex; align-items: center; gap: 16px; margin-bottom: 16px;">
                            <div style="font-size: 32px;">🔄</div>
                            <h3 style="margin: 0; color: #1f2937; font-size: 18px;"><?php esc_html_e('Automated Imports', 'vehizo-vehicle-management'); ?></h3>
                        </div>
                        <p style="color: #6b7280; margin-bottom: 16px;"><?php esc_html_e('Schedule automatic imports to keep your vehicle inventory synchronized with external data sources.', 'vehizo-vehicle-management'); ?></p>
                        <ul style="margin: 0; padding-left: 20px; color: #6b7280;">
                            <li><?php esc_html_e('Hourly, 3h, 6h, or daily import schedules', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('Background processing via WordPress cron', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('Automatic inventory updates', 'vehizo-vehicle-management'); ?></li>
                        </ul>
                    </div>

                    <!-- PDF Export -->
                    <div class="module-card" style="background: #fff; border: 2px solid #e2e8f0; border-radius: 12px; padding: 24px;">
                        <div style="display: flex; align-items: center; gap: 16px; margin-bottom: 16px;">
                            <div style="font-size: 32px;">📄</div>
                            <h3 style="margin: 0; color: #1f2937; font-size: 18px;"><?php esc_html_e('PDF Export', 'vehizo-vehicle-management'); ?></h3>
                        </div>
                        <p style="color: #6b7280; margin-bottom: 16px;"><?php esc_html_e('Generate professional PDF documents with vehicle specifications and QR codes for easy sharing.', 'vehizo-vehicle-management'); ?></p>
                        <ul style="margin: 0; padding-left: 20px; color: #6b7280;">
                            <li><?php esc_html_e('Professional PDF layout with branding', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('QR codes linking to online listings', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('Print-ready vehicle specifications', 'vehizo-vehicle-management'); ?></li>
                        </ul>
                    </div>

                    <!-- Advanced SEO -->
                    <div class="module-card" style="background: #fff; border: 2px solid #e2e8f0; border-radius: 12px; padding: 24px;">
                        <div style="display: flex; align-items: center; gap: 16px; margin-bottom: 16px;">
                            <div style="font-size: 32px;">🔍</div>
                            <h3 style="margin: 0; color: #1f2937; font-size: 18px;"><?php esc_html_e('Advanced SEO', 'vehizo-vehicle-management'); ?></h3>
                        </div>
                        <p style="color: #6b7280; margin-bottom: 16px;"><?php esc_html_e('Enhanced search engine optimization for better visibility in Google and other search engines.', 'vehizo-vehicle-management'); ?></p>
                        <ul style="margin: 0; padding-left: 20px; color: #6b7280;">
                            <li><?php esc_html_e('Enhanced schema.org markup', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('Customizable meta titles and descriptions', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('Automatic sitemap generation', 'vehizo-vehicle-management'); ?></li>
                        </ul>
                    </div>

                    <!-- Test Drive Scheduler -->
                    <div class="module-card" style="background: #fff; border: 2px solid #e2e8f0; border-radius: 12px; padding: 24px;">
                        <div style="display: flex; align-items: center; gap: 16px; margin-bottom: 16px;">
                            <div style="font-size: 32px;">📅</div>
                            <h3 style="margin: 0; color: #1f2937; font-size: 18px;"><?php esc_html_e('Test Drive Scheduler', 'vehizo-vehicle-management'); ?></h3>
                        </div>
                        <p style="color: #6b7280; margin-bottom: 16px;"><?php esc_html_e('Allow customers to schedule test drives directly through your website with calendar integration.', 'vehizo-vehicle-management'); ?></p>
                        <ul style="margin: 0; padding-left: 20px; color: #6b7280;">
                            <li><?php esc_html_e('Booking calendar with time slots', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('Email notifications for dealers and customers', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('GDPR-compliant data handling', 'vehizo-vehicle-management'); ?></li>
                        </ul>
                    </div>

                    <!-- Advanced Statistics -->
                    <div class="module-card" style="background: #fff; border: 2px solid #e2e8f0; border-radius: 12px; padding: 24px;">
                        <div style="display: flex; align-items: center; gap: 16px; margin-bottom: 16px;">
                            <div style="font-size: 32px;">📈</div>
                            <h3 style="margin: 0; color: #1f2937; font-size: 18px;"><?php esc_html_e('Advanced Statistics', 'vehizo-vehicle-management'); ?></h3>
                        </div>
                        <p style="color: #6b7280; margin-bottom: 16px;"><?php esc_html_e('Get deep insights into your vehicle performance with interactive charts and detailed metrics.', 'vehizo-vehicle-management'); ?></p>
                        <ul style="margin: 0; padding-left: 20px; color: #6b7280;">
                            <li><?php esc_html_e('Interactive charts (brand, fuel, etc.)', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('Conversion rate tracking', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('Detailed performance metrics', 'vehizo-vehicle-management'); ?></li>
                        </ul>
                    </div>

                    <!-- Priority Support -->
                    <div class="module-card" style="background: #fff; border: 2px solid #e2e8f0; border-radius: 12px; padding: 24px;">
                        <div style="display: flex; align-items: center; gap: 16px; margin-bottom: 16px;">
                            <div style="font-size: 32px;">💬</div>
                            <h3 style="margin: 0; color: #1f2937; font-size: 18px;"><?php esc_html_e('Priority Support', 'vehizo-vehicle-management'); ?></h3>
                        </div>
                        <p style="color: #6b7280; margin-bottom: 16px;"><?php esc_html_e('Get fast, professional support when you need it with priority access to our support team.', 'vehizo-vehicle-management'); ?></p>
                        <ul style="margin: 0; padding-left: 20px; color: #6b7280;">
                            <li><?php esc_html_e('Direct email support', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('Faster response times', 'vehizo-vehicle-management'); ?></li>
                            <li><?php esc_html_e('Bug fix priority', 'vehizo-vehicle-management'); ?></li>
                        </ul>
                    </div>

                </div>

                <!-- Call to Action -->
                <div style="margin-top: 40px; padding: 32px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 12px; text-align: center; color: #fff;">
                    <h3 style="margin: 0 0 16px 0; font-size: 28px; color: #fff;"><?php esc_html_e('Ready to Upgrade?', 'vehizo-vehicle-management'); ?></h3>
                    <p style="margin: 0 0 24px 0; font-size: 16px; color: rgba(255,255,255,0.9);">
                        <?php esc_html_e('Get access to all Pro features and take your vehicle management to the next level.', 'vehizo-vehicle-management'); ?>
                    </p>
                    <a href="<?php echo esc_url('https://vehizo.de'); ?>"
                       target="_blank"
                       rel="noopener noreferrer"
                       class="modern-button"
                       style="background: #fff; color: #667eea; padding: 14px 32px; font-size: 16px; font-weight: 600; text-decoration: none; display: inline-block; border-radius: 8px;">
                        <?php esc_html_e('Learn More', 'vehizo-vehicle-management'); ?>
                    </a>
                </div>

                <!-- Comparison Section -->
                <div style="margin-top: 40px;">
                    <h3 style="color: #1f2937; margin-bottom: 24px; text-align: center;"><?php esc_html_e('Feature Comparison', 'vehizo-vehicle-management'); ?></h3>
                    <table style="width: 100%; border-collapse: collapse; background: #fff; border-radius: 8px; overflow: hidden;">
                        <thead style="background: #f9fafb;">
                            <tr>
                                <th style="padding: 16px; text-align: left; font-weight: 700; border-bottom: 2px solid #e2e8f0;"><?php esc_html_e('Feature', 'vehizo-vehicle-management'); ?></th>
                                <th style="padding: 16px; text-align: center; font-weight: 700; border-bottom: 2px solid #e2e8f0;"><?php esc_html_e('Free', 'vehizo-vehicle-management'); ?></th>
                                <th style="padding: 16px; text-align: center; font-weight: 700; border-bottom: 2px solid #e2e8f0; background: #f0f4ff;"><?php esc_html_e('Pro', 'vehizo-vehicle-management'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td style="padding: 16px; border-bottom: 1px solid #e2e8f0;"><?php esc_html_e('Vehicle Management', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0;">✅</td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0; background: #f0f4ff;">✅</td>
                            </tr>
                            <tr>
                                <td style="padding: 16px; border-bottom: 1px solid #e2e8f0;"><?php esc_html_e('Frontend Display', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0;">✅</td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0; background: #f0f4ff;">✅</td>
                            </tr>
                            <tr>
                                <td style="padding: 16px; border-bottom: 1px solid #e2e8f0;"><?php esc_html_e('Contact Forms', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0;">✅</td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0; background: #f0f4ff;">✅</td>
                            </tr>
                            <tr>
                                <td style="padding: 16px; border-bottom: 1px solid #e2e8f0;"><?php esc_html_e('Manual Import', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0;">✅</td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0; background: #f0f4ff;">✅</td>
                            </tr>
                            <tr>
                                <td style="padding: 16px; border-bottom: 1px solid #e2e8f0;"><?php esc_html_e('Images per Vehicle', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0;">✅</td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0; background: #f0f4ff;">✅</td>
                            </tr>
                            <tr>
                                <td style="padding: 16px; border-bottom: 1px solid #e2e8f0;"><?php esc_html_e('Advanced Statistics', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0;"><?php esc_html_e('Basic Stats', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0; background: #f0f4ff;"><?php esc_html_e('Full Analytics & Charts', 'vehizo-vehicle-management'); ?></td>
                            </tr>
                            <tr>
                                <td style="padding: 16px; border-bottom: 1px solid #e2e8f0;"><?php esc_html_e('Automated Imports', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0;">—</td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0; background: #f0f4ff;">✅</td>
                            </tr>
                            <tr>
                                <td style="padding: 16px; border-bottom: 1px solid #e2e8f0;"><?php esc_html_e('PDF Export', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0;">—</td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0; background: #f0f4ff;">✅</td>
                            </tr>
                            <tr>
                                <td style="padding: 16px; border-bottom: 1px solid #e2e8f0;"><?php esc_html_e('Test Drive Scheduler', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0;">—</td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0; background: #f0f4ff;">✅</td>
                            </tr>
                            <tr>
                                <td style="padding: 16px; border-bottom: 1px solid #e2e8f0;"><?php esc_html_e('Advanced SEO', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0;">—</td>
                                <td style="padding: 16px; text-align: center; border-bottom: 1px solid #e2e8f0; background: #f0f4ff;">✅</td>
                            </tr>
                            <tr>
                                <td style="padding: 16px;"><?php esc_html_e('Priority Support', 'vehizo-vehicle-management'); ?></td>
                                <td style="padding: 16px; text-align: center;">—</td>
                                <td style="padding: 16px; text-align: center; background: #f0f4ff;">✅</td>
                            </tr>
                        </tbody>
                    </table>
                </div>

                <!-- Future Updates Banner -->
                <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: #fff; padding: 40px; border-radius: 12px; margin-top: 40px; text-align: center; box-shadow: 0 10px 25px rgba(102, 126, 234, 0.3);">
                    <div style="font-size: 32px; margin-bottom: 20px;">✨ <?php esc_html_e('More Features Coming Soon', 'vehizo-vehicle-management'); ?></div>
                    <p style="margin: 0 0 30px 0; font-size: 18px; opacity: 0.95; line-height: 1.7; max-width: 700px; margin-left: auto; margin-right: auto;">
                        <?php esc_html_e('We\'re constantly developing new modules and features for Pro users. Upgrade now to get access to all current and future premium modules – no additional costs!', 'vehizo-vehicle-management'); ?>
                    </p>
                    <div style="display: flex; gap: 20px; justify-content: center; align-items: center; flex-wrap: wrap; margin-bottom: 20px;">
                        <a href="https://vehizo.de" class="button button-hero" target="_blank" rel="noopener noreferrer" style="background: #fff; color: #667eea; border: none; padding: 16px 40px; font-size: 18px; font-weight: 700; text-decoration: none; border-radius: 8px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); transition: all 0.3s; display: inline-block;">
                            🚀 <?php esc_html_e('Upgrade to Pro', 'vehizo-vehicle-management'); ?>
                        </a>
                    </div>
                    <div style="display: flex; gap: 30px; justify-content: center; flex-wrap: wrap; opacity: 0.9; font-size: 14px;">
                        <span>✓ <?php esc_html_e('Lifetime Updates', 'vehizo-vehicle-management'); ?></span>
                        <span>✓ <?php esc_html_e('All Future Modules', 'vehizo-vehicle-management'); ?></span>
                        <span>✓ <?php esc_html_e('Priority Support', 'vehizo-vehicle-management'); ?></span>
                    </div>
                </div>

            </div>
            <?php
        }
    }
}
