/**
 * Vehizo Wishlist Module - Frontend JavaScript
 * 
 * @package Vehizo
 * @version 4.1.0
 */

(function($) {
    'use strict';
    
    // Wait for DOM and localized data
    $(document).ready(function() {
        
        // Check if localized data exists
        if (typeof vehizoWishlist === 'undefined') {
            return;
        }
        
        const WISHLIST_KEY = vehizoWishlist.storageKeys.wishlist;
        const LAST_PAGE_KEY = vehizoWishlist.storageKeys.lastPage;
        const MAX_VEHICLES = parseInt(vehizoWishlist.maxVehicles, 10) || 10;
        
        /**
         * Utility functions for localStorage management
         */
        const Storage = {
            /**
             * Get wishlist vehicles from localStorage
             * @returns {Array} Array of vehicle IDs
             */
            getWishlistVehicles: function() {
                try {
                    const stored = localStorage.getItem(WISHLIST_KEY);
                    const vehicles = JSON.parse(stored || '[]');
                    return Array.isArray(vehicles) ? vehicles.slice(0, MAX_VEHICLES) : [];
                } catch (e) {
                    return [];
                }
            },
            
            /**
             * Save wishlist vehicles to localStorage
             * @param {Array} vehicles Array of vehicle IDs
             * @returns {boolean} Success status
             */
            saveWishlistVehicles: function(vehicles) {
                try {
                    if (!Array.isArray(vehicles)) {
                        return false;
                    }
                    
                    // Clean and validate vehicle IDs
                    const cleanVehicles = vehicles
                        .filter(function(id) {
                            const numId = parseInt(id, 10);
                            return Number.isInteger(numId) && numId > 0;
                        })
                        .map(function(id) {
                            return parseInt(id, 10);
                        })
                        .slice(0, MAX_VEHICLES);
                        
                    localStorage.setItem(WISHLIST_KEY, JSON.stringify(cleanVehicles));
                    UI.updateAllParkButtons();
                    UI.updateWishlistCounter();
                    return true;
                } catch (e) {
                    return false;
                }
            },
            
            /**
             * Save current page URL for back navigation
             */
            saveCurrentPage: function() {
                try {
                    sessionStorage.setItem(LAST_PAGE_KEY, window.location.href);
                } catch (e) {
                }
            },
            
            /**
             * Get saved page URL for back navigation
             * @returns {string|null} Saved URL or null
             */
            getSavedPage: function() {
                try {
                    return sessionStorage.getItem(LAST_PAGE_KEY);
                } catch (e) {
                    return null;
                }
            }
        };
        
        /**
         * Vehicle management functions
         */
        const Vehicle = {
            /**
             * Toggle vehicle in wishlist
             * @param {number} vehicleId Vehicle ID to toggle
             */
            toggle: function(vehicleId) {
                const id = parseInt(vehicleId, 10);
                if (!Number.isInteger(id) || id <= 0) {
                    return;
                }
                
                // Save current URL for back navigation
                Storage.saveCurrentPage();
                
                let wishlist = Storage.getWishlistVehicles();
                
                if (wishlist.includes(id)) {
                    // Remove from wishlist
                    wishlist = wishlist.filter(function(wlId) {
                        return wlId !== id;
                    });
                } else {
                    // Add to wishlist
                    if (wishlist.length >= MAX_VEHICLES) {
                        alert(vehizoWishlist.strings.maxVehiclesReached);
                        return;
                    }
                    wishlist.push(id);
                }
                
                Storage.saveWishlistVehicles(wishlist);
            },
            
            /**
             * Remove vehicle from wishlist
             * @param {number} vehicleId Vehicle ID to remove
             * @param {Event} event Optional event object
             */
            remove: function(vehicleId, event) {
                if (event) {
                    event.preventDefault();
                }
                
                const id = parseInt(vehicleId, 10);
                if (!Number.isInteger(id) || id <= 0) {
                    return;
                }
                
                const wishlist = Storage.getWishlistVehicles().filter(function(wlId) {
                    return wlId !== id;
                });
                
                if (Storage.saveWishlistVehicles(wishlist)) {
                    ComparisonTable.load();
                }
            }
        };
        
        /**
         * UI update functions
         */
        const UI = {
            /**
             * Update all park buttons based on current wishlist
             */
            updateAllParkButtons: function() {
                const wishlist = Storage.getWishlistVehicles();
                
                $('.btn-park').each(function() {
                    const $button = $(this);
                    const id = parseInt($button.data('vehicle-id'), 10);
                    
                    if (!Number.isInteger(id)) {
                        return;
                    }
                    
                    const isParked = wishlist.includes(id);
                    
                    $button
                        .toggleClass('parked', isParked)
                        .html(isParked ? 
                            '⭐ ' + vehizoWishlist.strings.saved : 
                            '⭐ ' + vehizoWishlist.strings.save
                        )
                        .attr('aria-pressed', isParked.toString())
                        .attr('title', isParked ? 
                            vehizoWishlist.strings.removeFromWishlist : 
                            vehizoWishlist.strings.addToWishlist
                        );
                });
            },
            
            /**
             * Update wishlist counter in various elements
             */
            updateWishlistCounter: function() {
                const count = Storage.getWishlistVehicles().length;
                
                $('.parked-vehicle-count').each(function() {
                    const $counter = $(this);
                    $counter.text(count);
                    
                    if (!$counter.parent().hasClass('floating-park-button')) {
                        $counter.toggle(count > 0);
                    }
                });
                
                const $floatingButton = $('#floating-park-button');
                if ($floatingButton.length) {
                    $floatingButton
                        .toggle(count > 0)
                        .attr('aria-label', 
                            vehizoWishlist.strings.viewWishlist + ' (' + count + ')'
                        );
                }
            }
        };
        
        /**
         * Comparison table functionality
         */
        const ComparisonTable = {
            /**
             * Load and display comparison table
             */
            load: function() {
                const $container = $('#vehizo-wishlist-container');
                if (!$container.length) {
                    return;
                }
                
                const wishlistIds = Storage.getWishlistVehicles();
                
                // Show loading state
                $container.html(
                    '<div class="wishlist-loading">' +
                    '<div class="wishlist-spinner"></div>' +
                    vehizoWishlist.strings.loadingSavedVehicles +
                    '</div>'
                );
                
                // Prepare AJAX request
                const requestData = {
                    action: 'vehizo_get_wishlist_vehicles',
                    vehicle_ids: JSON.stringify(wishlistIds),
                    nonce: vehizoWishlist.nonce
                };
                
                $.ajax({
                    url: vehizoWishlist.ajaxUrl,
                    type: 'POST',
                    data: requestData,
                    dataType: 'json',
                    timeout: 10000,
                    success: function(response) {
                        if (response.success && response.data && response.data.html) {
                            $container.html(response.data.html);
                            ComparisonTable.updateBackButton();
                        } else {
                            ComparisonTable.showError(
                                vehizoWishlist.strings.errorOccurred,
                                vehizoWishlist.strings.tryAgainLater
                            );
                        }
                    },
                    error: function(xhr, status, error) {
                        ComparisonTable.showError(
                            vehizoWishlist.strings.errorOccurred,
                            vehizoWishlist.strings.checkConnection
                        );
                    }
                });
            },
            
            /**
             * Update back button with saved URL
             */
            updateBackButton: function() {
                const $backButton = $('#btn-back-to-vehicles');
                if ($backButton.length) {
                    const lastPage = Storage.getSavedPage();
                    if (lastPage) {
                        $backButton.attr('href', lastPage);
                    }
                }
            },
            
            /**
             * Show error message in container
             * @param {string} title Error title
             * @param {string} message Error message
             */
            showError: function(title, message) {
                const $container = $('#vehizo-wishlist-container');
                if ($container.length) {
                    $container.html(
                        '<div class="no-results">' +
                        '<h3>' + $('<div>').text(title).html() + '</h3>' +
                        '<p>' + $('<div>').text(message).html() + '</p>' +
                        '</div>'
                    );
                }
            }
        };
        
        /**
         * Event handlers
         */
        const Events = {
            /**
             * Initialize all event listeners
             */
            init: function() {
                // Park button clicks (event delegation)
                $(document).on('click', '.btn-park', function(e) {
                    e.preventDefault();
                    const vehicleId = $(this).data('vehicle-id');
                    if (vehicleId) {
                        Vehicle.toggle(vehicleId);
                    }
                });
                
                // Remove from list clicks (event delegation)
                $(document).on('click', '.remove-from-list', function(e) {
                    e.preventDefault();
                    const vehicleId = $(this).data('vehicle-id');
                    if (vehicleId) {
                        Vehicle.remove(vehicleId, e);
                    }
                });
                
                // Keyboard accessibility for park buttons
                $(document).on('keydown', '.btn-park', function(e) {
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        $(this).trigger('click');
                    }
                });
                
                // Keyboard accessibility for remove buttons
                $(document).on('keydown', '.remove-from-list', function(e) {
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        $(this).trigger('click');
                    }
                });
            }
        };
        
        /**
         * Initialize the module
         */
        const App = {
            init: function() {
                // Initialize event handlers
                Events.init();
                
                // Update UI based on current wishlist
                UI.updateAllParkButtons();
                UI.updateWishlistCounter();
                
                // Load comparison table if container exists
                if ($('#vehizo-wishlist-container').length) {
                    ComparisonTable.load();
                }
                
                // Expose remove function globally for backward compatibility
                window.removeFromList = Vehicle.remove;
            }
        };
        
        // Initialize the application
        App.init();
        
        /**
         * Handle page visibility changes to refresh data
         */
        if (typeof document.addEventListener === 'function') {
            document.addEventListener('visibilitychange', function() {
                if (!document.hidden) {
                    // Page became visible, refresh counters
                    UI.updateAllParkButtons();
                    UI.updateWishlistCounter();
                }
            });
        }
        
        /**
         * Handle storage events (when wishlist is modified in another tab)
         */
        if (typeof window.addEventListener === 'function') {
            window.addEventListener('storage', function(e) {
                if (e.key === WISHLIST_KEY) {
                    // Wishlist was modified in another tab
                    UI.updateAllParkButtons();
                    UI.updateWishlistCounter();
                    
                    // Reload comparison table if it exists
                    if ($('#vehizo-wishlist-container').length) {
                        ComparisonTable.load();
                    }
                }
            });
        }
        
        /**
         * Cleanup function for potential memory leaks
         */
        $(window).on('beforeunload', function() {
            // Clean up any timers or intervals if added in future
        });
    });
    
})(jQuery);