/**
 * Vehizo Vehicle Shortcode JavaScript
 * @package Vehizo
 * @version 4.0
 * @since 1.0.0
 */

(function($) {
    'use strict';

    /**
     * Initialize shortcode functionality when DOM is ready
     */
    $(document).ready(function() {
        VehizoShortcode.init();
    });

    /**
     * Main Vehizo Shortcode object
     */
    window.VehizoShortcode = {

        /**
         * Initialize all functionality
         */
        init: function() {
            this.initFilterToggle();
            this.initFilterForm();
            this.initParkButtons();
            this.initSortSelect();
            this.initDependentFilters();
            this.initAccessibility();
        },

        /**
         * Initialize filter toggle functionality
         */
        initFilterToggle: function() {
            const filterHeader = $('#filter-header');
            const filterContent = $('#filter-content');
            const filterStatus = $('#filter-status');
            
            if (filterHeader.length && filterContent.length) {
                // Initial state - Filter collapsed
                let isExpanded = false;
                
                filterHeader.on('click', function() {
                    isExpanded = !isExpanded;
                    
                    if (isExpanded) {
                        filterContent.addClass('expanded');
                        filterHeader.removeClass('collapsed');
                        filterStatus.text(vehizoShortcode.i18n.click_hide);
                        
                        // Focus first input for accessibility
                        setTimeout(function() {
                            filterContent.find('.filter-input').first().focus();
                        }, 400);
                        
                    } else {
                        filterContent.removeClass('expanded');
                        filterHeader.addClass('collapsed');
                        filterStatus.text(vehizoShortcode.i18n.click_show);
                    }
                    
                    // Announce state change for screen readers
                    this.announceToScreenReader(isExpanded ? 'Filter expanded' : 'Filter collapsed');
                }.bind(this));

                // Keyboard support
                filterHeader.on('keydown', function(e) {
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        filterHeader.click();
                    }
                });
            }
        },

        /**
         * Initialize filter form functionality
         */
        initFilterForm: function() {
            const filterForm = $('#filter-form');
            const filterInputs = filterForm.find('select, input[type="number"]');
            
            if (filterInputs.length) {
                let timeout;
                
                // Auto-submit form on change with debouncing
                filterInputs.on('change input', function(e) {
                    clearTimeout(timeout);
                    
                    // Add visual feedback for user
                    $(this).addClass('filter-input-changed');
                    
                    timeout = setTimeout(function() {
                        // Remove visual feedback
                        filterInputs.removeClass('filter-input-changed');
                        
                        // Submit form
                        if (filterForm.length) {
                            VehizoShortcode.submitFilterForm(filterForm, e.target);
                        }
                    }, 800);
                });

                // Prevent form submission on Enter key for number inputs
                filterForm.find('input[type="number"]').on('keydown', function(e) {
                    if (e.key === 'Enter') {
                        e.preventDefault();
                        $(this).trigger('change');
                    }
                });
            }
        },

        /**
         * Initialize dependent filters (Make -> Model)
         */
        initDependentFilters: function() {
            const $makeSelect = $('#filter-make');
            const $modelSelect = $('#filter-model');

            if (!$makeSelect.length || !$modelSelect.length) {
                return;
            }

            // Function to update model dropdown
            const updateModels = function(selectedMake) {
                const currentModel = new URLSearchParams(window.location.search).get('model');
                $modelSelect.empty();

                if (!selectedMake) {
                    // If no make is selected, disable model and show "All Models"
                    $modelSelect.prop('disabled', true);
                    $modelSelect.append($('<option>', {
                        value: '',
                        text: vehizoShortcode.i18n.all_models || 'All Models'
                    }));
                    return;
                }

                const models = vehizoShortcode.makeModelMap[selectedMake] || [];
                $modelSelect.prop('disabled', models.length === 0);
                $modelSelect.append($('<option>', {
                    value: '',
                    text: vehizoShortcode.i18n.all_models || 'All Models'
                }));

                $.each(models, function(i, model) {
                    const $option = $('<option>', { value: model, text: model });
                    if (model === currentModel) {
                        $option.prop('selected', true);
                    }
                    $modelSelect.append($option);
                });
            };

            // Initial population if a make is already selected on page load
            if ($makeSelect.val()) {
                updateModels($makeSelect.val());
            }

            // Event listener for make change
            $makeSelect.on('change', function() {
                const selectedMake = $(this).val(); // e.g., "BMW" or ""
                updateModels(selectedMake); // Update model dropdown based on make

                // If a make was selected, the model field is now enabled.
                // If "All Makes" was selected, the model field is now disabled.
                // We reset the model selection, but we don't trigger a form submission here.
                // The main 'change' event handler on filterInputs will handle the submission.
                $modelSelect.val(''); // Reset model selection
            });
        },

        /**
         * Submit filter form with loading state
         */
        submitFilterForm: function(form) {
            const $container = $('.vehizo-main-container');
            const $grid = $('.vehizo-vehicles-grid');
            const $resultsSection = $('.results-section');
            const $pagination = $('.vehizo-pagination');

            // Show loading state
            $container.addClass('vehizo-loading');
            $grid.css('opacity', 0.5);
            $resultsSection.css('opacity', 0.5);

            const formData = form.serializeArray();
            const data = {
                action: 'vehizo_filter_vehicles',
                nonce: vehizoShortcode.nonce
            };
            const urlParams = new URLSearchParams();

            formData.forEach(function(item) {
                data[item.name] = item.value;
                if (item.value) {
                    urlParams.set(item.name, item.value);
                }
            });

            // Add sort parameter
            const sortValue = $('.sort-select').val();
            if (sortValue) {
                data.sort = sortValue;
                urlParams.set('sort', sortValue);
            }

            // Update URL
            const newUrl = window.location.pathname + '?' + urlParams.toString();
            window.history.pushState({ path: newUrl }, '', newUrl);

            $.ajax({
                url: vehizoShortcode.ajaxurl,
                type: 'POST',
                data: data,
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        $grid.html(response.data.grid_html);
                        $pagination.html(response.data.pagination_html);
                        $('.results-section .results-count strong').text(response.data.count);
                        $('.results-section .results-count').contents().filter(function() {
                            return this.nodeType === 3;
                        }).first().replaceWith(' ' + (response.data.count === 1 ? 'vehicle found' : 'vehicles found'));
                    }
                },
                error: function() {
                    $grid.html('<div class="no-results"><h3>' + vehizoShortcode.i18n.error_save + '</h3></div>');
                },
                complete: function() {
                    $container.removeClass('vehizo-loading');
                    $grid.css('opacity', 1);
                    $resultsSection.css('opacity', 1);
                }
            });
        },

        /**
         * Initialize park button functionality
         */
        initParkButtons: function() {
            $(document).on('click', '.btn-park', this.handleParkClick.bind(this));
        },

        /**
         * Handle park button clicks
         */
        handleParkClick: function(e) {
            e.preventDefault();
            
            const button = $(e.currentTarget);
            const vehicleId = button.data('vehicle-id');
            
            if (!vehicleId) {
                return;
            }

            // Check if user is logged in (basic check)
            if (typeof vehizoShortcode.nonce === 'undefined' || !vehizoShortcode.nonce) {
                this.showMessage('Please log in to use the favorites feature.', 'warning');
                return;
            }
            
            // Button state during request
            const originalHtml = button.html();
            button.prop('disabled', true)
                  .html('⏳ ' + vehizoShortcode.i18n.loading);
            
            // AJAX request
            $.ajax({
                url: vehizoShortcode.ajaxurl,
                type: 'POST',
                dataType: 'json',
                data: {
                    action: 'vehizo_toggle_park',
                    vehicle_id: vehicleId,
                    nonce: vehizoShortcode.nonce
                },
                timeout: 10000
            })
            .done(this.handleParkSuccess.bind(this, button, originalHtml))
            .fail(this.handleParkError.bind(this, button, originalHtml))
            .always(function() {
                button.prop('disabled', false);
            });
        },

        /**
         * Handle successful park toggle
         */
        handleParkSuccess: function(button, originalHtml, response) {
            if (response.success && response.data) {
                const data = response.data;
                
                if (data.is_parked) {
                    button.html('✓ ' + vehizoShortcode.i18n.parked)
                          .addClass('parked')
                          .css('background', 'linear-gradient(135deg, #10b981 0%, #059669 100%)');
                } else {
                    button.html('⭐ ' + vehizoShortcode.i18n.park)
                          .removeClass('parked')
                          .css('background', 'linear-gradient(135deg, #fbbf24 0%, #f59e0b 100%)');
                }
                
                // Show success message
                this.showMessage(data.message, 'success');
                
                // Update park counter if exists
                this.updateParkCounter(data.total_parked);
                
                // Trigger custom event
                $(document).trigger('vehizo:park_toggled', [data]);
                
            } else {
                button.html(originalHtml);
                this.showMessage(response.data?.message || vehizoShortcode.i18n.error_save, 'error');
            }
        },

        /**
         * Handle park toggle error
         */
        handleParkError: function(button, originalHtml, xhr) {
            button.html(originalHtml);
            
            let errorMessage = vehizoShortcode.i18n.error_network;
            
            if (xhr.status === 429) {
                errorMessage = 'Too many requests. Please wait a moment.';
            } else if (xhr.status === 403) {
                errorMessage = 'Please log in to use this feature.';
            } else if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                errorMessage = xhr.responseJSON.data.message;
            }
            
            this.showMessage(errorMessage, 'error');
            
        },

        /**
         * Initialize sort select functionality
         */
        initSortSelect: function() {
            // Add global function for sort select onchange
            window.vehizoUpdateSort = function(sortValue) {
                const currentUrl = new URL(window.location.href);
                currentUrl.searchParams.set('sort', sortValue);
                currentUrl.searchParams.delete('paged'); // Reset pagination
                
                // Add loading state
                $('.sort-select').prop('disabled', true);
                $('.vehizo-main-container').addClass('vehizo-loading');
                
                window.location.href = currentUrl.toString();
            };
        },

        /**
         * Show message to user
         */
        showMessage: function(message, type) {
            type = type || 'info';
            
            // Remove existing messages
            $('.vehizo-message').remove();
            
            // Create message element
            const messageEl = $('<div class="vehizo-message vehizo-message-' + type + '">')
                .html('<span class="vehizo-message-icon">' + this.getMessageIcon(type) + '</span>' +
                      '<span class="vehizo-message-text">' + this.escapeHtml(message) + '</span>' +
                      '<button class="vehizo-message-close" aria-label="Close message">&times;</button>')
                .css({
                    position: 'fixed',
                    top: '20px',
                    right: '20px',
                    maxWidth: '400px',
                    padding: '1rem 1.5rem',
                    borderRadius: '8px',
                    boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
                    zIndex: 9999,
                    display: 'flex',
                    alignItems: 'center',
                    gap: '0.75rem',
                    fontSize: '0.95rem',
                    fontWeight: '500',
                    backgroundColor: this.getMessageBgColor(type),
                    color: this.getMessageTextColor(type),
                    border: '1px solid ' + this.getMessageBorderColor(type),
                    opacity: 0,
                    transform: 'translateY(-20px)'
                });
            
            // Add to DOM with animation
            $('body').append(messageEl);
            messageEl.animate({
                opacity: 1,
                transform: 'translateY(0)'
            }, 300);
            
            // Auto-remove after 5 seconds
            setTimeout(function() {
                messageEl.animate({
                    opacity: 0,
                    transform: 'translateY(-20px)'
                }, 300, function() {
                    messageEl.remove();
                });
            }, 5000);
            
            // Manual close
            messageEl.find('.vehizo-message-close').on('click', function() {
                messageEl.animate({
                    opacity: 0,
                    transform: 'translateY(-20px)'
                }, 300, function() {
                    messageEl.remove();
                });
            });

            // Announce to screen readers
            this.announceToScreenReader(message);
        },

        /**
         * Get message icon based on type
         */
        getMessageIcon: function(type) {
            const icons = {
                success: '✓',
                error: '✕',
                warning: '⚠',
                info: 'ℹ'
            };
            return icons[type] || icons.info;
        },

        /**
         * Get message background color based on type
         */
        getMessageBgColor: function(type) {
            const colors = {
                success: '#d1fae5',
                error: '#fee2e2',
                warning: '#fef3c7',
                info: '#dbeafe'
            };
            return colors[type] || colors.info;
        },

        /**
         * Get message text color based on type
         */
        getMessageTextColor: function(type) {
            const colors = {
                success: '#065f46',
                error: '#991b1b',
                warning: '#92400e',
                info: '#1e40af'
            };
            return colors[type] || colors.info;
        },

        /**
         * Get message border color based on type
         */
        getMessageBorderColor: function(type) {
            const colors = {
                success: '#a7f3d0',
                error: '#fca5a5',
                warning: '#fde68a',
                info: '#93c5fd'
            };
            return colors[type] || colors.info;
        },

        /**
         * Update park counter in UI if exists
         */
        updateParkCounter: function(count) {
            const counter = $('.vehizo-park-counter');
            if (counter.length) {
                counter.text(count);
                
                if (count > 0) {
                    counter.show().addClass('animate-bounce');
                    setTimeout(function() {
                        counter.removeClass('animate-bounce');
                    }, 600);
                } else {
                    counter.hide();
                }
            }
        },

        /**
         * Initialize accessibility features
         */
        initAccessibility: function() {
            // Add ARIA labels to dynamically generated content
            this.enhanceAriaLabels();
            
            // Add focus indicators
            this.enhanceFocusIndicators();
            
            // Handle keyboard navigation
            this.initKeyboardNavigation();
        },

        /**
         * Enhance ARIA labels
         */
        enhanceAriaLabels: function() {
            // Add ARIA labels to filter inputs
            $('.filter-input').each(function() {
                const label = $(this).siblings('.filter-label').text();
                if (label && !$(this).attr('aria-label')) {
                    $(this).attr('aria-label', label);
                }
            });
            
            // Add ARIA expanded to filter header
            const filterHeader = $('#filter-header');
            if (filterHeader.length) {
                filterHeader.attr('aria-expanded', 'false')
                           .attr('role', 'button')
                           .attr('tabindex', '0');
            }
        },

        /**
         * Enhance focus indicators
         */
        enhanceFocusIndicators: function() {
            // Add custom focus styles for better visibility
            $('<style>')
                .prop('type', 'text/css')
                .html('.vehizo-focus-visible:focus-visible { outline: 2px solid #3b82f6 !important; outline-offset: 2px !important; }')
                .appendTo('head');
            
            // Add focus-visible class to interactive elements
            $('.filter-input, .filter-button, .cta-button, .btn-park, .sort-select')
                .addClass('vehizo-focus-visible');
        },

        /**
         * Initialize keyboard navigation
         */
        initKeyboardNavigation: function() {
            // Handle Enter and Space for buttons
            $(document).on('keydown', '.filter-button, .btn-park', function(e) {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    $(this).click();
                }
            });
            
            // Handle Escape to close messages
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape') {
                    $('.vehizo-message-close').first().click();
                }
            });
        },

        /**
         * Announce message to screen readers
         */
        announceToScreenReader: function(message) {
            const announcement = $('<div>')
                .attr('aria-live', 'polite')
                .attr('aria-atomic', 'true')
                .addClass('sr-only')
                .css({
                    position: 'absolute',
                    left: '-10000px',
                    width: '1px',
                    height: '1px',
                    overflow: 'hidden'
                })
                .text(message);
            
            $('body').append(announcement);
            
            setTimeout(function() {
                announcement.remove();
            }, 1000);
        },

        /**
         * Escape HTML to prevent XSS
         */
        escapeHtml: function(text) {
            const map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, function(m) { return map[m]; });
        },

        /**
         * Utility function to update URL parameters
         */
        updateUrlParameter: function(url, param, paramVal) {
            const urlObj = new URL(url);
            urlObj.searchParams.set(param, paramVal);
            return urlObj.toString();
        },

        /**
         * Check if user prefers reduced motion
         */
        prefersReducedMotion: function() {
            return window.matchMedia('(prefers-reduced-motion: reduce)').matches;
        }
    };

    /**
     * Handle page visibility changes to pause/resume animations
     */
    document.addEventListener('visibilitychange', function() {
        if (document.hidden) {
            $('.vehizo-main-container').addClass('vehizo-paused');
        } else {
            $('.vehizo-main-container').removeClass('vehizo-paused');
        }
    });

    /**
     * Add loading styles
     */
    $('<style>')
        .prop('type', 'text/css')
        .html(`
            .vehizo-loading {
                opacity: 0.7;
                pointer-events: none;
                transition: opacity 0.3s ease;
            }
            
            .vehizo-paused * {
                animation-play-state: paused !important;
            }
            
            .filter-input-changed {
                border-color: #3b82f6 !important;
                box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1) !important;
            }
            
            .sr-only {
                position: absolute !important;
                left: -10000px !important;
                width: 1px !important;
                height: 1px !important;
                overflow: hidden !important;
            }
        `)
        .appendTo('head');

})(jQuery);