/**
 * EINFACHE VEHIZO TAB FUNKTIONALITÄT
 * Direkt und sofort funktionsfähig
 */

// Warten bis DOM geladen ist
document.addEventListener('DOMContentLoaded', function() {

    // Tab-Funktionalität starten
    initTabs();

    // Galerie-Funktionalität starten
    initGallery();

    // Neue Thumbnail Strip Gallery starten
    initThumbnailGallery();

    // Contact PopUp-Funktionalität starten
    initContactPopup();

    // Wishlist-Funktionalität starten
    initSimpleWishlist();

    // Wishlist View-Funktionalität starten
    initWishlistView();
});

/**
 * TAB SWITCHING - EINFACH UND DIREKT
 */
function initTabs() {
    // Alle Tab-Buttons finden
    const tabButtons = document.querySelectorAll('.vehizo-single__tab');
    const tabContents = document.querySelectorAll('.vehizo-single__tab-content');
    
    
    // Prüfen ob Tabs vorhanden sind
    if (tabButtons.length === 0 || tabContents.length === 0) {
        return;
    }
    
    // Jeden Tab-Button mit Event versehen
    tabButtons.forEach(function(button, index) {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            
            
            // Alle Tabs deaktivieren
            tabButtons.forEach(function(btn) {
                btn.classList.remove('active');
            });
            
            // Alle Tab-Inhalte verstecken
            tabContents.forEach(function(content) {
                content.classList.remove('active');
                content.style.display = 'none';
            });
            
            // Aktuellen Tab aktivieren
            button.classList.add('active');
            
            // Entsprechenden Inhalt anzeigen
            if (tabContents[index]) {
                tabContents[index].classList.add('active');
                tabContents[index].style.display = 'block';
            }
        });
    });
    
    // Synchronisation: Aktiven Tab mit entsprechendem Content abgleichen
    let activeTabIndex = -1;
    
    // Finde welcher Tab aktiv ist
    tabButtons.forEach(function(tab, index) {
        if (tab.classList.contains('active')) {
            activeTabIndex = index;
        }
    });
    
    // Falls kein Tab aktiv ist, ersten Tab aktivieren
    if (activeTabIndex === -1) {
        activeTabIndex = 0;
        tabButtons[0].classList.add('active');
    }
    
    // Alle Contents erstmal verstecken
    tabContents.forEach(function(content) {
        content.classList.remove('active');
        content.style.display = 'none';
    });
    
    // Nur den entsprechenden Content zum aktiven Tab anzeigen
    if (tabContents[activeTabIndex]) {
        tabContents[activeTabIndex].classList.add('active');
        tabContents[activeTabIndex].style.display = 'block';
    }
}

/**
 * GALERIE FUNKTIONALITÄT
 */
function initGallery() {
    const thumbnails = document.querySelectorAll('.vehizo-single__thumbnail');
    const mainImage = document.querySelector('.vehizo-single__main-image');
    
    
    if (thumbnails.length === 0 || !mainImage) {
        return;
    }
    
    thumbnails.forEach(function(thumbnail, index) {
        thumbnail.addEventListener('click', function() {
            
            // Alle Thumbnails deaktivieren
            thumbnails.forEach(function(thumb) {
                thumb.classList.remove('active');
            });
            
            // Aktuelles Thumbnail aktivieren
            thumbnail.classList.add('active');
            
            // Bild-URL aus Thumbnail holen
            const thumbImg = thumbnail.querySelector('img');
            if (thumbImg && mainImage) {
                // Große Version des Bildes laden
                let newSrc = thumbImg.src;
                newSrc = newSrc.replace('-150x150', '');
                newSrc = newSrc.replace('thumbnail', 'large');
                
                mainImage.src = newSrc;
                mainImage.alt = thumbImg.alt;
                
            }
        });
    });
}

/**
 * CONTACT POPUP FUNKTIONALITÄT
 */
function initContactPopup() {
    const contactButton = document.querySelector('.btn-contact-popup');
    const popup = document.getElementById('vehizo-contact-popup');
    const successModal = document.getElementById('vehizo-success-modal');
    const form = popup ? popup.querySelector('#vehizo-contact-popup-form') : null;
    const closeButton = popup ? popup.querySelector('.vehizo-popup-close') : null;


    // Check if already initialized to prevent double initialization
    if (form && form.dataset.initialized === 'true') {
        return;
    }

    if (!contactButton || !popup) {
        return;
    }

    // PopUp öffnen
    contactButton.addEventListener('click', function(e) {
        e.preventDefault();

        popup.style.display = 'flex';
        // Kleine Verzögerung für CSS-Transition
        setTimeout(() => {
            popup.classList.add('show');
        }, 10);

        // Body Scrolling verhindern
        document.body.style.overflow = 'hidden';

        // Focus auf erstes Input-Feld
        const firstInput = popup.querySelector('input, textarea');
        if (firstInput) {
            firstInput.focus();
        }
    });

    // PopUp schließen
    function closePopup() {
        popup.classList.remove('show');

        // Nach Transition das PopUp verstecken
        setTimeout(() => {
            popup.style.display = 'none';
            document.body.style.overflow = '';
        }, 300);
    }

    // Success Modal öffnen
    function showSuccessModal() {

        if (successModal) {
            successModal.style.display = 'flex';
            setTimeout(() => {
                successModal.classList.add('show');
            }, 10);
        }
    }

    // Success Modal schließen
    function closeSuccessModal() {

        if (successModal) {
            successModal.classList.remove('show');
            setTimeout(() => {
                successModal.style.display = 'none';
                document.body.style.overflow = '';
            }, 300);
        }
    }

    // Form Submit Handler
    if (form && typeof vehizoContact !== 'undefined') {
        // Mark form as initialized
        form.dataset.initialized = 'true';

        let isSubmitting = false; // Prevent double submissions

        // Remove any existing event listeners to prevent duplicates
        const newForm = form.cloneNode(true);
        form.parentNode.replaceChild(newForm, form);

        newForm.addEventListener('submit', function(e) {
            e.preventDefault();

            // Prevent double submissions
            if (isSubmitting) {
                return;
            }

            isSubmitting = true;

            const submitButton = newForm.querySelector('button[type="submit"]');
            const originalText = submitButton.textContent;

            // Button deaktivieren und Text ändern
            submitButton.disabled = true;
            submitButton.textContent = vehizoContact.strings.sending;

            // Form-Daten sammeln
            const formData = new FormData(newForm);
            formData.append('action', 'vehizo_send_contact_form');

            // Debug: Log form data
            for (let [key, value] of formData.entries()) {
            }

            // AJAX-Request
            fetch(vehizoContact.ajaxUrl, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {

                if (data.success) {
                    // Form zurücksetzen
                    newForm.reset();

                    // Contact PopUp schließen
                    closePopup();

                    // Success Modal anzeigen
                    setTimeout(() => {
                        showSuccessModal();
                    }, 300);
                } else {

                    // Show debug info if available
                    if (data.data && data.data.debug) {
                        console.table(data.data.debug);
                        for (let key in data.data.debug) {
                        }
                    }

                    alert((data.data && data.data.message) || vehizoContact.strings.error);
                }
            })
            .catch(error => {
                alert(vehizoContact.strings.error);
            })
            .finally(() => {
                // Button wieder aktivieren und Submission-Flag zurücksetzen
                submitButton.disabled = false;
                submitButton.textContent = originalText;
                isSubmitting = false;
            });
        });
    }

    // Close Button
    if (closeButton) {
        closeButton.addEventListener('click', function(e) {
            e.preventDefault();
            closePopup();
        });
    }

    // Success Modal Close Buttons
    if (successModal) {
        const successCloseButton = successModal.querySelector('.vehizo-popup-close');
        const successOkButton = successModal.querySelector('#vehizo-success-close');

        if (successCloseButton) {
            successCloseButton.addEventListener('click', function(e) {
                e.preventDefault();
                closeSuccessModal();
            });
        }

        if (successOkButton) {
            successOkButton.addEventListener('click', function(e) {
                e.preventDefault();
                closeSuccessModal();
            });
        }

        // Overlay Click für Success Modal
        successModal.addEventListener('click', function(e) {
            if (e.target === successModal) {
                closeSuccessModal();
            }
        });
    }

    // Overlay Click (außerhalb des Contents)
    popup.addEventListener('click', function(e) {
        if (e.target === popup) {
            closePopup();
        }
    });

    // ESC-Taste
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            if (successModal && successModal.classList.contains('show')) {
                closeSuccessModal();
            } else if (popup.classList.contains('show')) {
                closePopup();
            }
        }
    });

}

/**
 * ULTRA-SIMPLE WISHLIST - STATE-BASED APPROACH
 */
function initSimpleWishlist() {
    const wishlistButton = document.querySelector('.btn-park');
    const STORAGE_KEY = 'vehizo_wishlist_vehicles';


    if (!wishlistButton) {
        return;
    }

    const vehicleId = wishlistButton.getAttribute('data-vehicle-id');
    if (!vehicleId) {
        return;
    }

    const vehicleIdInt = parseInt(vehicleId);

    // THIS IS OUR SINGLE SOURCE OF TRUTH
    let isInWishlist = false;

    // Initialize from localStorage (with delay to allow unified cleaning)
    function loadInitialState() {
        // Delay to allow unified function to clean localStorage first
        setTimeout(function() {
            try {
                const stored = localStorage.getItem(STORAGE_KEY);
                const wishlist = stored ? JSON.parse(stored) : [];
                isInWishlist = wishlist.includes(vehicleIdInt);
                updateButton();
            } catch (e) {
                isInWishlist = false;
                updateButton();
            }
        }, 800); // Wait even longer to ensure unified function has run
    }

    // Update button based on current state
    function updateButton() {

        if (isInWishlist) {
            wishlistButton.textContent = '❤️ ' + (vehizoWishlist.strings.onWishlist || 'On Wishlist');
            wishlistButton.style.backgroundColor = '#dc2626';
            wishlistButton.style.color = '#ffffff';
        } else {
            wishlistButton.textContent = '⭐ ' + (vehizoWishlist.strings.addToWishlist || 'Add to Wishlist');
            wishlistButton.style.backgroundColor = '';
            wishlistButton.style.color = '';
        }
    }

    // Save current state to localStorage
    function saveState() {
        try {
            const stored = localStorage.getItem(STORAGE_KEY);
            let wishlist = stored ? JSON.parse(stored) : [];

            if (isInWishlist) {
                // Add if not already in list
                if (!wishlist.includes(vehicleIdInt)) {
                    wishlist.push(vehicleIdInt);
                }
            } else {
                // Remove from list
                wishlist = wishlist.filter(id => id !== vehicleIdInt);
            }

            localStorage.setItem(STORAGE_KEY, JSON.stringify(wishlist));
        } catch (e) {
        }
    }

    // Show toast message
    function showToast(message, isAdded) {
        // Remove existing toasts
        document.querySelectorAll('.vehizo-toast').forEach(toast => {
            if (toast.parentNode) toast.parentNode.removeChild(toast);
        });

        const toast = document.createElement('div');
        toast.className = 'vehizo-toast';
        toast.style.cssText = `
            position: fixed; top: 20px; right: 20px;
            background: ${isAdded ? '#059669' : '#374151'};
            color: white; padding: 12px 20px; border-radius: 8px;
            font-size: 14px; font-weight: 500; z-index: 10001;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            transform: translateX(100%); transition: transform 0.3s ease;
        `;
        toast.textContent = message;
        document.body.appendChild(toast);

        setTimeout(() => toast.style.transform = 'translateX(0)', 10);
        setTimeout(() => {
            if (toast.parentNode) {
                toast.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (toast.parentNode) toast.parentNode.removeChild(toast);
                }, 300);
            }
        }, 2500);
    }

    // Click handler - ULTRA SIMPLE
    wishlistButton.addEventListener('click', function(e) {
        e.preventDefault();


        // TOGGLE STATE
        isInWishlist = !isInWishlist;


        // Update everything
        updateButton();
        saveState();
        updateWishlistCounter();

        if (isInWishlist) {
            showToast('Added to wishlist', true);
        } else {
            showToast('Removed from wishlist', false);
        }
    });

    // Update wishlist counter globally (WILL BE OVERRIDDEN BY PHP)
    function updateWishlistCounter() {

        // Use unified version if available, otherwise fallback to basic version
        if (typeof window.updateWishlistCounterUnified === 'function') {
            return window.updateWishlistCounterUnified();
        }

        // Fallback basic version (no validation)
        try {
            const stored = localStorage.getItem(STORAGE_KEY);
            const wishlist = stored ? JSON.parse(stored) : [];
            const count = wishlist.length;

            const counterElements = document.querySelectorAll('.wishlist-count');
            counterElements.forEach(counter => {
                counter.textContent = `(${count})`;
            });

            return count;
        } catch (e) {
            return 0;
        }
    }

    // Make updateWishlistCounter globally available
    window.updateWishlistCounter = updateWishlistCounter;

    // Initialize
    loadInitialState();

    // Delay the initial counter update to allow unified function to override
    setTimeout(function() {
        updateWishlistCounter();
    }, 700); // Wait longer than the unified function (600ms)

}

/**
 * FALLBACK: Falls DOM schon geladen ist
 */
if (document.readyState === 'loading') {
    // DOM wird noch geladen
    document.addEventListener('DOMContentLoaded', function() {
        initTabs();
        initGallery();
        initContactPopup();
        initSimpleWishlist();
        initWishlistView();
    });
} else {
    // DOM ist bereits geladen
    initTabs();
    initGallery();
    initContactPopup();
    initSimpleWishlist();
    initWishlistView();
}

/**
 * DEBUGGING FUNKTIONEN
 */
function debugTabs() {
    
    const tabs = document.querySelectorAll('.vehizo-single__tab');
    const contents = document.querySelectorAll('.vehizo-single__tab-content');
    
    tabs.forEach((tab, i) => {
    });
    
    contents.forEach((content, i) => {
    });
}

/**
 * WISHLIST VIEW FUNKTIONALITÄT
 */
function initWishlistView() {
    const viewWishlistButton = document.querySelector('.btn-view-wishlist');
    const wishlistModal = document.getElementById('vehizo-wishlist-modal');
    const wishlistContent = document.getElementById('vehizo-wishlist-content');


    if (!viewWishlistButton || !wishlistModal || !wishlistContent) {
        return;
    }

    // View Wishlist Button Click
    viewWishlistButton.addEventListener('click', function(e) {
        e.preventDefault();

        openWishlistModal();
    });

    // Modal Close Handlers
    const closeButton = wishlistModal.querySelector('.vehizo-popup-close');
    if (closeButton) {
        closeButton.addEventListener('click', function(e) {
            e.preventDefault();
            closeWishlistModal();
        });
    }

    // Overlay Click
    wishlistModal.addEventListener('click', function(e) {
        if (e.target === wishlistModal) {
            closeWishlistModal();
        }
    });

    // ESC Key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && wishlistModal.classList.contains('show')) {
            closeWishlistModal();
        }
    });

    function openWishlistModal() {

        // Show modal
        wishlistModal.style.display = 'flex';
        setTimeout(() => {
            wishlistModal.classList.add('show');
        }, 10);

        // Prevent body scrolling
        document.body.style.overflow = 'hidden';

        // Load wishlist content
        loadWishlistContent();
    }

    function closeWishlistModal() {

        wishlistModal.classList.remove('show');
        setTimeout(() => {
            wishlistModal.style.display = 'none';
            document.body.style.overflow = '';
        }, 300);
    }

    function loadWishlistContent() {

        // Show loading state
        wishlistContent.innerHTML = `
            <div class="vehizo-wishlist-loading">
                <div class="vehizo-loading-spinner"></div>
                <p>${vehizoWishlist.strings.loading}</p>
            </div>
        `;

        // Get wishlist from localStorage
        const STORAGE_KEY = 'vehizo_wishlist_vehicles';
        let wishlistIds = [];

        try {
            const stored = localStorage.getItem(STORAGE_KEY);
            wishlistIds = stored ? JSON.parse(stored) : [];
        } catch (e) {
        }


        if (wishlistIds.length === 0) {
            showEmptyWishlist();
            return;
        }

        // Load real vehicle data via AJAX
        loadRealVehicleData(wishlistIds);
    }

    function showEmptyWishlist() {
        wishlistContent.innerHTML = `
            <div class="vehizo-wishlist-empty">
                <div class="vehizo-wishlist-empty-icon">🚗</div>
                <h3>${vehizoWishlist.strings.empty}</h3>
                <p>${vehizoWishlist.strings.emptyDesc}</p>
            </div>
        `;
    }

    function loadRealVehicleData(vehicleIds) {

        // Make AJAX request
        fetch(vehizoWishlist.ajaxUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'vehizo_get_wishlist_vehicles',
                nonce: vehizoWishlist.nonce,
                vehicle_ids: JSON.stringify(vehicleIds)
            })
        })
        .then(response => response.json())
        .then(data => {

            if (data.success && data.data.vehicles && data.data.vehicles.length > 0) {
                displayVehicles(data.data.vehicles);
            } else {

                // Clear localStorage if no vehicles found (invalid IDs)
                const STORAGE_KEY = 'vehizo_wishlist_vehicles';
                localStorage.removeItem(STORAGE_KEY);

                // Update counter to 0
                if (typeof updateWishlistCounter === 'function') {
                    updateWishlistCounter();
                }

                // Show empty wishlist
                showEmptyWishlist();
            }
        })
        .catch(error => {
            wishlistContent.innerHTML = `
                <div class="vehizo-wishlist-empty">
                    <div class="vehizo-wishlist-empty-icon">❌</div>
                    <h3>${vehizoWishlist.strings.error}</h3>
                    <p>${vehizoWishlist.strings.errorDesc || 'Please try again later'}</p>
                </div>
            `;
        });
    }

    function displayVehicles(vehicles) {

        const wishlistHtml = vehicles.map(vehicle => {
            return `
                <div class="vehizo-wishlist-item" data-vehicle-id="${vehicle.id}">
                    <div class="vehizo-wishlist-item-image">
                        <img src="${vehicle.image_url}" alt="${vehicle.title}" loading="lazy">
                    </div>
                    <div class="vehizo-wishlist-item-content">
                        <h4 class="vehizo-wishlist-item-title">${vehicle.title}</h4>
                        ${vehicle.subtitle ? `<p class="vehizo-wishlist-item-subtitle">${vehicle.subtitle}</p>` : ''}
                        <div class="vehizo-wishlist-item-price">${vehicle.price}</div>
                        <div class="vehizo-wishlist-item-specs">
                            ${vehicle.year ? `<div>Year: ${vehicle.year}</div>` : ''}
                            ${vehicle.mileage ? `<div>Mileage: ${vehicle.mileage}</div>` : ''}
                            ${vehicle.fuel ? `<div>Fuel: ${vehicle.fuel}</div>` : ''}
                            ${vehicle.power ? `<div>Power: ${vehicle.power}</div>` : ''}
                        </div>
                        <div class="vehizo-wishlist-item-actions">
                            <button class="vehizo-button btn-view-details" onclick="window.location.href='${vehicle.permalink}'">
                                ${vehizoWishlist.strings.viewDetails}
                            </button>
                            <button class="vehizo-button btn-remove-from-wishlist" onclick="removeFromWishlistModalWithConfirm(${vehicle.id})">
                                ${vehizoWishlist.strings.remove}
                            </button>
                        </div>
                    </div>
                </div>
            `;
        }).join('');

        wishlistContent.innerHTML = `
            <div class="vehizo-wishlist-grid">
                ${wishlistHtml}
            </div>
        `;

    }

    // Global function for removing items with confirmation
    window.removeFromWishlistModalWithConfirm = function(vehicleId) {

        if (confirm(vehizoWishlist.strings.removeConfirm)) {
            removeFromWishlistModal(vehicleId);
        }
    };

    // Global function for removing items from wishlist modal
    window.removeFromWishlistModal = function(vehicleId) {

        const STORAGE_KEY = 'vehizo_wishlist_vehicles';
        let wishlist = [];

        try {
            const stored = localStorage.getItem(STORAGE_KEY);
            wishlist = stored ? JSON.parse(stored) : [];
        } catch (e) {
            return;
        }

        // Remove vehicle from wishlist
        const newWishlist = wishlist.filter(id => id !== parseInt(vehicleId));

        try {
            localStorage.setItem(STORAGE_KEY, JSON.stringify(newWishlist));
        } catch (e) {
            return;
        }

        // Update the main wishlist button if we're on that vehicle's page
        const mainWishlistButton = document.querySelector('.btn-park[data-vehicle-id="' + vehicleId + '"]');
        if (mainWishlistButton) {
            updateWishlistButtonFromModal(mainWishlistButton, false);
        }

        // Update counter
        if (typeof updateWishlistCounter === 'function') {
            updateWishlistCounter();
        }

        // Reload wishlist content
        loadWishlistContent();

        // Show toast
        showWishlistToastFromModal('Removed from wishlist', false);
    };

    function updateWishlistButtonFromModal(button, isOnWishlist) {
        if (isOnWishlist) {
            button.textContent = '❤️ ' + (vehizoWishlist.strings.onWishlist || 'On Wishlist');
            button.classList.add('btn-park--active');
            button.style.backgroundColor = '#dc2626';
            button.style.borderColor = '#dc2626';
            button.style.color = '#ffffff';
        } else {
            button.textContent = '⭐ ' + (vehizoWishlist.strings.addToWishlist || 'Add to Wishlist');
            button.classList.remove('btn-park--active');
            button.style.backgroundColor = '';
            button.style.borderColor = '';
            button.style.color = '';
        }
    }

    function showWishlistToastFromModal(message, isAdded) {
        // Reuse the existing toast function
        if (typeof showWishlistToast === 'function') {
            showWishlistToast(message, isAdded);
        } else {
            // Simple fallback toast
            const toast = document.createElement('div');
            toast.className = 'vehizo-toast';
            toast.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: ${isAdded ? '#059669' : '#374151'};
                color: white;
                padding: 12px 20px;
                border-radius: 8px;
                font-size: 14px;
                font-weight: 500;
                z-index: 10001;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                transform: translateX(100%);
                transition: transform 0.3s ease;
                max-width: 300px;
            `;
            toast.textContent = message;

            document.body.appendChild(toast);

            setTimeout(() => {
                toast.style.transform = 'translateX(0)';
            }, 10);

            setTimeout(() => {
                if (toast && toast.parentNode) {
                    toast.style.transform = 'translateX(100%)';
                    setTimeout(() => {
                        if (toast.parentNode) {
                            toast.parentNode.removeChild(toast);
                        }
                    }, 300);
                }
            }, 2500);
        }
    }

}

/**
 * THUMBNAIL STRIP GALLERY FUNKTIONALITÄT
 * Vertikale Thumbnail-Liste mit Hauptbild-Display
 */
function initThumbnailGallery() {

    const galleryContainer = document.querySelector('.vehizo-gallery-container');
    const thumbnailStrip = document.querySelector('.vehizo-thumbnail-strip');
    const mainImage = document.querySelector('.vehizo-main-image');
    const imageCounter = document.querySelector('.vehizo-image-counter');
    const navPrev = document.querySelector('.vehizo-gallery-prev');
    const navNext = document.querySelector('.vehizo-gallery-next');
    const fullscreenBtn = document.querySelector('.vehizo-fullscreen-btn');

    // Lightbox elements
    const lightbox = document.getElementById('vehizo-lightbox');
    const lightboxImage = lightbox ? lightbox.querySelector('.vehizo-lightbox__image') : null;
    const lightboxCounter = lightbox ? lightbox.querySelector('.vehizo-lightbox__counter') : null;


    if (!galleryContainer || !thumbnailStrip || !mainImage) {
        return;
    }

    const thumbnails = document.querySelectorAll('.vehizo-thumbnail');
    let currentIndex = 0;
    let allImages = [];

    // Sammle alle Bild-URLs und initialisiere
    thumbnails.forEach((thumb, index) => {
        const imageUrl = thumb.dataset.image;
        if (imageUrl) {
            allImages.push(imageUrl);

            // Thumbnail Click Event
            thumb.addEventListener('click', function() {
                setActiveImage(index);
            });
        }
    });


    function setActiveImage(index) {
        if (index < 0 || index >= allImages.length) return;

        currentIndex = index;

        // Update main image
        mainImage.src = allImages[currentIndex];

        // Update active thumbnail
        thumbnails.forEach((thumb, i) => {
            thumb.classList.toggle('active', i === currentIndex);
        });

        // Update counter
        if (imageCounter) {
            const currentSpan = imageCounter.querySelector('.current');
            if (currentSpan) {
                currentSpan.textContent = currentIndex + 1;
            }
        }

    }

    function showNext() {
        const nextIndex = currentIndex < allImages.length - 1 ? currentIndex + 1 : 0;
        setActiveImage(nextIndex);
    }

    function showPrev() {
        const prevIndex = currentIndex > 0 ? currentIndex - 1 : allImages.length - 1;
        setActiveImage(prevIndex);
    }

    function openLightbox() {
        if (!lightbox || !lightboxImage) return;


        lightboxImage.src = allImages[currentIndex];

        // Update lightbox counter
        if (lightboxCounter) {
            const currentSpan = lightboxCounter.querySelector('.vehizo-lightbox__current');
            const totalSpan = lightboxCounter.querySelector('.vehizo-lightbox__total');
            if (currentSpan) currentSpan.textContent = currentIndex + 1;
            if (totalSpan) totalSpan.textContent = allImages.length;
        }

        lightbox.classList.add('active');
        document.body.style.overflow = 'hidden';
    }

    function closeLightbox() {
        if (!lightbox) return;

        lightbox.classList.remove('active');
        document.body.style.overflow = '';
    }

    function lightboxNext() {
        const nextIndex = currentIndex < allImages.length - 1 ? currentIndex + 1 : 0;
        currentIndex = nextIndex;
        lightboxImage.src = allImages[currentIndex];

        // Update counter
        if (lightboxCounter) {
            const currentSpan = lightboxCounter.querySelector('.vehizo-lightbox__current');
            if (currentSpan) currentSpan.textContent = currentIndex + 1;
        }
    }

    function lightboxPrev() {
        const prevIndex = currentIndex > 0 ? currentIndex - 1 : allImages.length - 1;
        currentIndex = prevIndex;
        lightboxImage.src = allImages[currentIndex];

        // Update counter
        if (lightboxCounter) {
            const currentSpan = lightboxCounter.querySelector('.vehizo-lightbox__current');
            if (currentSpan) currentSpan.textContent = currentIndex + 1;
        }
    }

    // Event Listeners
    if (navPrev) {
        navPrev.addEventListener('click', showPrev);
    }

    if (navNext) {
        navNext.addEventListener('click', showNext);
    }

    if (fullscreenBtn) {
        fullscreenBtn.addEventListener('click', openLightbox);
    }

    // Main image click opens lightbox
    if (mainImage) {
        mainImage.addEventListener('click', openLightbox);
    }

    // Lightbox event listeners
    if (lightbox) {
        const closeBtn = lightbox.querySelector('.vehizo-lightbox__close');
        const lightboxPrevBtn = lightbox.querySelector('.vehizo-lightbox__prev');
        const lightboxNextBtn = lightbox.querySelector('.vehizo-lightbox__next');
        const overlay = lightbox.querySelector('.vehizo-lightbox__overlay');

        if (closeBtn) {
            closeBtn.addEventListener('click', closeLightbox);
        }

        if (lightboxPrevBtn) {
            lightboxPrevBtn.addEventListener('click', lightboxPrev);
        }

        if (lightboxNextBtn) {
            lightboxNextBtn.addEventListener('click', lightboxNext);
        }

        if (overlay) {
            overlay.addEventListener('click', closeLightbox);
        }
    }

    // Keyboard Navigation
    document.addEventListener('keydown', function(e) {
        if (lightbox && lightbox.classList.contains('active')) {
            // Lightbox navigation
            switch(e.key) {
                case 'Escape':
                    closeLightbox();
                    break;
                case 'ArrowLeft':
                    lightboxPrev();
                    break;
                case 'ArrowRight':
                    lightboxNext();
                    break;
            }
        } else {
            // Main gallery navigation
            switch(e.key) {
                case 'ArrowLeft':
                    showPrev();
                    break;
                case 'ArrowRight':
                    showNext();
                    break;
            }
        }
    });

}

// Debug-Funktion global verfügbar machen
window.debugTabs = debugTabs;
