/**
 * Vehizo PDF Generator JavaScript
 * Handles PDF download functionality and user feedback
 *
 * @package Vehizo
 * @since 4.1.0
 */

(function($) {
    'use strict';

    // Main PDF Generator object
    window.VehizoPDFGenerator = {
        
        // Initialize the PDF generator
        init: function() {
            this.bindEvents();
            this.setupProgressIndicators();
        },

        // Bind event listeners
        bindEvents: function() {
            $(document).on('click', '.vehizo-pdf-btn', this.handlePDFClick.bind(this));
            $(document).on('contextmenu', '.vehizo-pdf-btn', this.handleRightClick.bind(this));
        },

        // Handle PDF button click
        handlePDFClick: function(e) {
            const $button = $(e.currentTarget);
            const vehicleId = $button.data('vehicle-id');
            
            // Prevent multiple clicks
            if ($button.hasClass('loading')) {
                e.preventDefault();
                return false;
            }

            // Add loading state
            this.setLoadingState($button, true);
            
            // Show generating message
            this.showMessage(vehizo_pdf_vars.generating_text, 'info', $button);

            // Track the download attempt
            this.trackPDFDownload(vehicleId);

            // Set timeout to remove loading state (fallback)
            setTimeout(() => {
                this.setLoadingState($button, false);
                this.clearMessages($button);
            }, 5000);

            // The actual download happens via the href, we just provide UX feedback
            return true;
        },

        // Handle right-click (context menu) on PDF button
        handleRightClick: function(e) {
            const $button = $(e.currentTarget);
            
            // Allow right-click to work normally for "Save link as..." functionality
            if (!$button.hasClass('loading')) {
                // Brief visual feedback for right-click
                $button.addClass('right-clicked');
                setTimeout(() => {
                    $button.removeClass('right-clicked');
                }, 200);
            }
        },

        // Set loading state on button
        setLoadingState: function($button, loading) {
            if (loading) {
                $button.addClass('loading')
                       .attr('data-original-text', $button.text())
                       .text(vehizo_pdf_vars.generating_text);
            } else {
                const originalText = $button.attr('data-original-text');
                $button.removeClass('loading')
                       .text(originalText || $button.text());
            }
        },

        // Show status message
        showMessage: function(message, type, $button) {
            this.clearMessages($button);
            
            const $message = $('<span class="vehizo-pdf-status ' + type + '">' + message + '</span>');
            $button.after($message);

            // Auto-hide success/info messages after 3 seconds
            if (type === 'success' || type === 'info') {
                setTimeout(() => {
                    $message.fadeOut(() => {
                        $message.remove();
                    });
                }, 3000);
            }
        },

        // Clear existing messages
        clearMessages: function($button) {
            $button.siblings('.vehizo-pdf-status').remove();
        },

        // Setup progress indicators for PDF generation
        setupProgressIndicators: function() {
            // Monitor for successful downloads (when browser starts downloading)
            let downloadTimer;
            
            $(document).on('click', '.vehizo-pdf-btn', function() {
                const $button = $(this);
                
                // Clear any existing timer
                if (downloadTimer) {
                    clearTimeout(downloadTimer);
                }
                
                // Check if download started (browser specific detection)
                downloadTimer = setTimeout(() => {
                    VehizoPDFGenerator.setLoadingState($button, false);
                    VehizoPDFGenerator.showMessage(vehizo_pdf_vars.success_text, 'success', $button);
                }, 2000);
            });

            // Handle cases where download might fail
            $(window).on('focus', function() {
                // When window regains focus, assume download completed or failed
                $('.vehizo-pdf-btn.loading').each(function() {
                    const $button = $(this);
                    VehizoPDFGenerator.setLoadingState($button, false);
                    VehizoPDFGenerator.clearMessages($button);
                });
            });
        },

        // Track PDF download for analytics (if needed)
        trackPDFDownload: function(vehicleId) {
            // This can be extended to send analytics data
            if (typeof gtag !== 'undefined') {
                gtag('event', 'pdf_download', {
                    'vehicle_id': vehicleId,
                    'event_category': 'engagement'
                });
            }

            // Or track with custom analytics
            if (typeof _vehizo_analytics !== 'undefined') {
                _vehizo_analytics.track('pdf_download', {
                    vehicle_id: vehicleId
                });
            }

            console.log('PDF download initiated for vehicle:', vehicleId);
        },

        // Utility: Check if browser supports PDF downloads
        supportsPDFDownload: function() {
            // Check if browser can handle PDF files
            const userAgent = navigator.userAgent.toLowerCase();
            const isMobile = /android|webos|iphone|ipad|ipod|blackberry|iemobile|opera mini/i.test(userAgent);
            
            // Most modern browsers support PDF downloads
            return !isMobile || 'download' in document.createElement('a');
        },

        // Utility: Format file size for display
        formatFileSize: function(bytes) {
            if (bytes === 0) return '0 Bytes';
            
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        },

        // Handle PDF generation errors (called from PHP if needed)
        handleError: function(message, $button) {
            this.setLoadingState($button, false);
            this.showMessage(message || vehizo_pdf_vars.error_text, 'error', $button);
        }
    };

    // Advanced PDF Features
    window.VehizoPDFGenerator.Advanced = {
        
        // Bulk PDF generation (if multiple vehicles selected)
        generateBulkPDFs: function(vehicleIds) {
            if (!vehicleIds || vehicleIds.length === 0) {
                return;
            }

            const total = vehicleIds.length;
            let completed = 0;

            // Show progress indicator
            const $progress = $('<div class="vehizo-bulk-pdf-progress">' +
                '<div class="progress-bar">' +
                '<div class="progress-fill"></div>' +
                '</div>' +
                '<span class="progress-text">0/' + total + ' PDFs generated</span>' +
                '</div>');

            $('body').append($progress);

            // Generate PDFs one by one
            vehicleIds.forEach((vehicleId, index) => {
                setTimeout(() => {
                    this.generateSinglePDF(vehicleId, () => {
                        completed++;
                        const percent = (completed / total) * 100;
                        
                        $progress.find('.progress-fill').css('width', percent + '%');
                        $progress.find('.progress-text').text(completed + '/' + total + ' PDFs generated');
                        
                        if (completed === total) {
                            setTimeout(() => {
                                $progress.fadeOut(() => {
                                    $progress.remove();
                                });
                            }, 2000);
                        }
                    });
                }, index * 1000); // Stagger requests by 1 second
            });
        },

        // Generate single PDF (for bulk operations)
        generateSinglePDF: function(vehicleId, callback) {
            // Create hidden iframe for download
            const $iframe = $('<iframe style="display:none;"></iframe>');
            $('body').append($iframe);
            
            // Set iframe source to PDF generation URL
            const pdfUrl = this.buildPDFUrl(vehicleId);
            $iframe.attr('src', pdfUrl);
            
            // Clean up iframe after download
            setTimeout(() => {
                $iframe.remove();
                if (callback) callback();
            }, 3000);
        },

        // Build PDF generation URL
        buildPDFUrl: function(vehicleId) {
            const baseUrl = window.location.origin + '/wp-admin/admin-post.php';
            const params = new URLSearchParams({
                action: 'vehizo_generate_pdf',
                vehizo_vehicle_id: vehicleId,
                nonce: this.getPDFNonce()
            });
            
            return baseUrl + '?' + params.toString();
        },

        // Get PDF nonce (should be localized from PHP)
        getPDFNonce: function() {
            // This should be provided via wp_localize_script in the main plugin file
            return vehizo_pdf_vars.pdf_nonce || '';
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        window.VehizoPDFGenerator.init();

        // Check browser support and show warning if needed
        if (!window.VehizoPDFGenerator.supportsPDFDownload()) {
            $('.vehizo-pdf-btn').after(
                '<small class="vehizo-pdf-warning">' +
                'Note: PDF downloads may not work optimally on your device. ' +
                'Consider using a desktop browser for best results.' +
                '</small>'
            );
        }
    });

    // Expose global functions for external use
    window.vehizo_generate_pdf = function(vehicleId) {
        const $button = $('.vehizo-pdf-btn[data-vehicle-id="' + vehicleId + '"]').first();
        if ($button.length) {
            $button.trigger('click');
        }
    };

    window.vehizo_bulk_pdf = function(vehicleIds) {
        window.VehizoPDFGenerator.Advanced.generateBulkPDFs(vehicleIds);
    };

})(jQuery);