/**
 * Vehizo Mobile Importer JavaScript
 *
 * Handles AJAX functionality for external API import
 *
 * @package Vehizo
 * @since 1.0.0
 */

(function($) {
    'use strict';

    var VehizoMobileImporter = {
        
        /**
         * Initialize the importer
         */
        init: function() {
            this.bindEvents();
            this.updateStatus();
        },

        /**
         * Bind event listeners
         */
        bindEvents: function() {
            $(document).on('click', '#vehizo-start-import', this.startImport);
            $(document).on('click', '#vehizo-test-api', this.testApiConnection);
            $(document).on('click', '#vehizo-stop-import', this.stopImport);
            
            // Auto-refresh status every 30 seconds during import
            this.statusInterval = setInterval(this.updateStatus, 30000);
        },

        /**
         * Start the import process
         */
        startImport: function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var $status = $('#vehizo-import-status');
            
            // Confirm action
            if (!confirm(vehizoMobileImporter.strings.confirmImport)) {
                return;
            }
            
            // Update UI
            $button.prop('disabled', true).text(vehizoMobileImporter.strings.importing);
            $status.html('<div class="notice notice-info"><p>' + vehizoMobileImporter.strings.importing + '</p></div>');
            
            // Show progress bar
            VehizoMobileImporter.showProgressBar();
            
            // Make AJAX request
            $.post(vehizoMobileImporter.ajaxUrl, {
                action: 'vehizo_mobile_import',
                nonce: vehizoMobileImporter.nonce
            })
            .done(function(response) {
                if (response.success) {
                    $status.html('<div class="notice notice-success"><p>' + 
                        vehizoMobileImporter.strings.importComplete + '<br>' + 
                        response.data.message + '</p></div>');
                    
                    // Update statistics if provided
                    if (response.data.stats) {
                        VehizoMobileImporter.updateStats(response.data.stats);
                    }
                } else {
                    $status.html('<div class="notice notice-error"><p>' + 
                        vehizoMobileImporter.strings.importError + '<br>' + 
                        (response.data.message || 'Unknown error') + '</p></div>');
                }
            })
            .fail(function(xhr, status, error) {
                $status.html('<div class="notice notice-error"><p>' + 
                    vehizoMobileImporter.strings.importError + '<br>' + 
                    'HTTP Error: ' + error + '</p></div>');
            })
            .always(function() {
                $button.prop('disabled', false).text($button.data('original-text') || 'Start Import');
                VehizoMobileImporter.hideProgressBar();
                VehizoMobileImporter.updateStatus();
            });
        },

        /**
         * Test API connection
         */
        testApiConnection: function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var $status = $('#vehizo-api-test-status');
            
            // Update UI
            $button.prop('disabled', true).text(vehizoMobileImporter.strings.testingConnection);
            $status.html('<div class="notice notice-info"><p>' + vehizoMobileImporter.strings.testingConnection + '</p></div>');
            
            // Make AJAX request
            $.post(vehizoMobileImporter.ajaxUrl, {
                action: 'vehizo_test_api',
                nonce: vehizoMobileImporter.nonce
            })
            .done(function(response) {
                if (response.success) {
                    $status.html('<div class="notice notice-success"><p>' + 
                        vehizoMobileImporter.strings.connectionSuccess + '<br>' + 
                        response.data.message + '</p></div>');
                } else {
                    $status.html('<div class="notice notice-error"><p>' + 
                        vehizoMobileImporter.strings.connectionFailed + '<br>' + 
                        (response.data.message || 'Unknown error') + '</p></div>');
                }
            })
            .fail(function(xhr, status, error) {
                $status.html('<div class="notice notice-error"><p>' + 
                    vehizoMobileImporter.strings.connectionFailed + '<br>' + 
                    'HTTP Error: ' + error + '</p></div>');
            })
            .always(function() {
                $button.prop('disabled', false).text($button.data('original-text') || 'Test Connection');
            });
        },

        /**
         * Stop ongoing import
         */
        stopImport: function(e) {
            e.preventDefault();
            
            // This is a placeholder - actual implementation would depend on
            // server-side support for stopping imports
            alert('Import stop functionality would be implemented here');
        },

        /**
         * Update import status display
         */
        updateStatus: function() {
            var $statusContainer = $('#vehizo-import-stats');
            
            if ($statusContainer.length === 0) {
                return;
            }
            
            // This could fetch current stats via AJAX
            // For now, it's just a placeholder
        },

        /**
         * Show progress bar
         */
        showProgressBar: function() {
            var $progressContainer = $('#vehizo-progress-container');
            
            if ($progressContainer.length === 0) {
                $progressContainer = $('<div id="vehizo-progress-container" class="vehizo-progress-wrapper"></div>');
                $('#vehizo-import-status').after($progressContainer);
            }
            
            var progressHtml = 
                '<div class="vehizo-progress-bar">' +
                    '<div class="vehizo-progress-fill"></div>' +
                    '<div class="vehizo-progress-text">' + vehizoMobileImporter.strings.pleaseWait + '</div>' +
                '</div>';
                
            $progressContainer.html(progressHtml).show();
            
            // Animate progress bar
            this.animateProgressBar();
        },

        /**
         * Hide progress bar
         */
        hideProgressBar: function() {
            $('#vehizo-progress-container').hide();
        },

        /**
         * Animate progress bar
         */
        animateProgressBar: function() {
            var $fill = $('.vehizo-progress-fill');
            var width = 0;
            
            var interval = setInterval(function() {
                width += Math.random() * 10;
                if (width >= 90) {
                    width = 90; // Never show 100% until actually complete
                    clearInterval(interval);
                }
                $fill.css('width', width + '%');
            }, 500);
            
            // Store interval reference for cleanup
            this.progressInterval = interval;
        },

        /**
         * Update statistics display
         */
        updateStats: function(stats) {
            if (stats.total_vehicles !== undefined) {
                $('#vehizo-total-vehicles').text(stats.total_vehicles);
            }
            
            if (stats.last_import_formatted) {
                $('#vehizo-last-import').text(stats.last_import_formatted);
            }
            
            if (stats.last_status) {
                $('#vehizo-last-status').text(stats.last_status);
            }
        },

        /**
         * Cleanup intervals on page unload
         */
        cleanup: function() {
            if (this.statusInterval) {
                clearInterval(this.statusInterval);
            }
            
            if (this.progressInterval) {
                clearInterval(this.progressInterval);
            }
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        VehizoMobileImporter.init();
        
        // Store original button text
        $('#vehizo-start-import, #vehizo-test-api').each(function() {
            $(this).data('original-text', $(this).text());
        });
    });

    // Cleanup on page unload
    $(window).on('beforeunload', function() {
        VehizoMobileImporter.cleanup();
    });

    // Make available globally for debugging
    window.VehizoMobileImporter = VehizoMobileImporter;

})(jQuery);