/**
 * Simple Gallery Uploader for Vehizo Vehicle Editor
 *
 * This is a simplified, reliable version that directly uses WordPress Media API
 */

jQuery(document).ready(function($) {
    'use strict';

    let mediaFrame;

    // Initialize upload buttons when they exist
    function initUploadButtons() {
        // Main upload button (when no images)
        $('#vehizo-upload-images-btn').off('click').on('click', function(e) {
            e.preventDefault();
            openMediaUploader();
        });

        // Add more images button (when images exist)
        $('#vehizo-add-more-images-btn').off('click').on('click', function(e) {
            e.preventDefault();
            openMediaUploader();
        });
    }

    function openMediaUploader() {
        // If media frame already exists, reopen it
        if (mediaFrame) {
            mediaFrame.open();
            return;
        }

        // Create new media frame
        mediaFrame = wp.media({
            title: 'Select Vehicle Images',
            button: {
                text: 'Add Images to Gallery'
            },
            multiple: true,
            library: {
                type: 'image'
            }
        });

        // Handle image selection
        mediaFrame.on('select', function() {
            const selection = mediaFrame.state().get('selection');
            const currentIds = getCurrentGalleryIds();
            let newImages = [];

            selection.map(function(attachment) {
                const data = attachment.toJSON();
                const imageId = data.id.toString();

                // Only add if not already in gallery
                if (currentIds.indexOf(imageId) === -1) {
                    newImages.push({
                        id: imageId,
                        url: data.sizes && data.sizes.medium ? data.sizes.medium.url : data.url,
                        alt: data.alt || 'Vehicle Image'
                    });
                }
            });

            // Add new images to gallery
            if (newImages.length > 0) {
                addImagesToGallery(newImages);
                updateGalleryHiddenField();
                updateGalleryDisplay();
            }
        });

        // Open the media frame
        mediaFrame.open();
    }

    function getCurrentGalleryIds() {
        const hiddenField = $('#vehizo-gallery-ids');
        const value = hiddenField.val();
        return value ? value.split(',').filter(id => id.trim() !== '') : [];
    }

    function addImagesToGallery(newImages) {
        const container = $('#vehizo-gallery-container');
        let grid = $('#vehizo-gallery-grid');

        // Create grid if it doesn't exist
        if (grid.length === 0) {
            container.addClass('has-images');
            const currentCount = getCurrentGalleryIds().length;
            container.html(`
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                    <h3 style="margin: 0; color: #374151;">Vehicle Images (${currentCount})</h3>
                    <button type="button" class="vehizo-gallery-upload" id="vehizo-add-more-images-btn">Add More Images</button>
                </div>
                <div class="vehizo-gallery-grid" id="vehizo-gallery-grid"></div>
            `);
            grid = $('#vehizo-gallery-grid');

            // Re-bind the new "Add More Images" button
            $('#vehizo-add-more-images-btn').on('click', function(e) {
                e.preventDefault();
                openMediaUploader();
            });
        }

        // Add each new image
        newImages.forEach(function(image) {
            const position = grid.find('.vehizo-gallery-item').length + 1;
            const imageHtml = `
                <div class="vehizo-gallery-item" data-id="${image.id}">
                    <div class="position-badge">${position}</div>
                    <img src="${image.url}" alt="${image.alt}" loading="lazy">
                    <button type="button" class="remove-btn" onclick="removeGalleryImage(this)" title="Remove Image">×</button>
                </div>
            `;
            grid.append(imageHtml);
        });
    }

    function updateGalleryHiddenField() {
        const ids = [];
        $('#vehizo-gallery-grid .vehizo-gallery-item').each(function() {
            const id = $(this).data('id');
            if (id) {
                ids.push(id);
            }
        });
        $('#vehizo-gallery-ids').val(ids.join(','));
    }

    function updateGalleryDisplay() {
        const imageCount = $('#vehizo-gallery-grid .vehizo-gallery-item').length;
        const heading = $('#vehizo-gallery-container h3');
        if (heading.length && imageCount > 0) {
            heading.text(`Vehicle Images (${imageCount})`);
        }
    }

    // Make remove function globally available
    window.removeGalleryImage = function(button) {
        if (confirm('Do you really want to remove this image from the gallery?')) {
            $(button).closest('.vehizo-gallery-item').fadeOut(300, function() {
                $(this).remove();
                updateGalleryHiddenField();
                updateImagePositions();
                updateGalleryDisplay();
            });
        }
    };

    function updateImagePositions() {
        $('#vehizo-gallery-grid .vehizo-gallery-item').each(function(index) {
            $(this).find('.position-badge').text(index + 1);
        });
    }

    // Initialize on page load
    initUploadButtons();

    // Re-initialize if content changes (for dynamic content)
    $(document).on('DOMNodeInserted', function(e) {
        if ($(e.target).find('#vehizo-upload-images-btn, #vehizo-add-more-images-btn').length) {
            setTimeout(initUploadButtons, 100);
        }
    });
});