/**
 * Vehizo Gallery Manager
 *
 * Handles drag & drop sorting and gallery management
 * WordPress.org compliant external script
 */

jQuery(document).ready(function($) {
    'use strict';

    // Optimized update functions with throttling
    let updateTimeout;

    // Add drag & drop support for upload areas
    $('.vehizo-upload-area').each(function() {
        const uploadArea = this;
        const fileInput = $(this).find('input[type="file"]');

        // Prevent default drag behaviors
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            uploadArea.addEventListener(eventName, preventDefaults, false);
            document.body.addEventListener(eventName, preventDefaults, false);
        });

        // Highlight drop area when item is dragged over it
        ['dragenter', 'dragover'].forEach(eventName => {
            uploadArea.addEventListener(eventName, highlight, false);
        });

        ['dragleave', 'drop'].forEach(eventName => {
            uploadArea.addEventListener(eventName, unhighlight, false);
        });

        // Handle dropped files
        uploadArea.addEventListener('drop', handleDrop, false);

        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }

        function highlight(e) {
            $(uploadArea).addClass('dragover');
        }

        function unhighlight(e) {
            $(uploadArea).removeClass('dragover');
        }

        function handleDrop(e) {
            const dt = e.dataTransfer;
            const files = dt.files;

            if (files.length > 0) {
                fileInput[0].files = files;
                // Optionally auto-submit the form or show preview
                $(uploadArea).find('form').submit();
            }
        }
    });

    // Make gallery sortable with optimized performance
    if ($('#vehizo-gallery-grid').length && typeof $.fn.sortable !== 'undefined') {
        $('#vehizo-gallery-grid').sortable({
            items: '.vehizo-gallery-item',
            cursor: 'grabbing',
            tolerance: 'intersect',
            distance: 5,
            delay: 100,
            revert: 150,
            opacity: 0.8,
            placeholder: 'vehizo-gallery-placeholder',
            forcePlaceholderSize: true,
            helper: 'clone',
            scroll: true,
            scrollSensitivity: 100,
            scrollSpeed: 20,
            start: function(e, ui) {
                ui.placeholder.css({
                    'background': 'linear-gradient(45deg, #f0f8ff 25%, transparent 25%, transparent 75%, #f0f8ff 75%)',
                    'background-size': '20px 20px',
                    'border': '2px dashed #007cba',
                    'border-radius': '8px',
                    'height': ui.item.outerHeight() + 'px',
                    'width': ui.item.outerWidth() + 'px',
                    'box-sizing': 'border-box'
                });
                ui.helper.css({
                    'transform': 'rotate(3deg)',
                    'box-shadow': '0 8px 25px rgba(0,0,0,0.3)',
                    'z-index': 1000
                });
            },
            stop: function(e, ui) {
                updateGalleryOrder();
                updatePositionBadges();
                // Force immediate save via AJAX to ensure order is preserved
                saveGalleryOrderImmediately();
            }
        });

        // Add hover effects for better UX
        $(document).on('mouseenter', '.vehizo-gallery-item', function() {
            $(this).css('transform', 'scale(1.02)');
        }).on('mouseleave', '.vehizo-gallery-item', function() {
            $(this).css('transform', 'scale(1)');
        });
    }

    function updateGalleryOrder() {
        clearTimeout(updateTimeout);
        updateTimeout = setTimeout(function() {
            var ids = [];
            $('#vehizo-gallery-grid .vehizo-gallery-item').each(function() {
                var id = $(this).data('id');
                if (id) {
                    ids.push(id);
                }
            });

            // Debug: Check if elements exist

            // Update both the visible field and the hidden field
            var hiddenField = $('#vehizo-gallery-ids');
            if (hiddenField.length > 0) {
                hiddenField.val(ids.join(','));
            } else {
            }

            // Also update the manual input field to show current order
            var manualField = $('input[name="vehizo_gallery_manual"]');
            if (manualField.length > 0) {
                manualField.val(ids.join(','));
            } else {
            }


            // Show save reminder
            if (ids.length > 0) {
                showSaveReminder();
            }
        }, 100);
    }

    // Optimized position badge updates
    function updatePositionBadges() {
        requestAnimationFrame(function() {
            $('#vehizo-gallery-grid .vehizo-gallery-item').each(function(index) {
                $(this).find('.position-badge').text(index + 1);
            });
        });
    }

    // Show save reminder
    function showSaveReminder() {
        if ($('#sort-reminder').length === 0) {
            var message = vehizoGallery.reorderMessage || 'Images reordered! Click "Save Vehicle" below to save the new order.';
            $('#vehizo-gallery-grid').after('<div id="sort-reminder" style="background: #fff3cd; border: 1px solid #ffeaa7; color: #856404; padding: 10px; margin: 10px 0; border-radius: 4px; text-align: center;"><strong>📌 ' + message + '</strong></div>');
        }
    }

    // Remove gallery image function
    window.removeGalleryImage = function(button) {
        var confirmMessage = vehizoGallery.confirmRemove || 'Remove this image from the gallery?';
        if (confirm(confirmMessage)) {
            $(button).closest('.vehizo-gallery-item').fadeOut(300, function() {
                $(this).remove();
                updateGalleryOrder();
                updatePositionBadges();

                // Save immediately to database (fix for refresh issue)
                saveGalleryOrderImmediately();
            });
        }
    };

    // Add cursor change handlers
    $(document).on('mousedown', '.vehizo-gallery-item', function() {
        $(this).css('cursor', 'grabbing');
    }).on('mouseup', '.vehizo-gallery-item', function() {
        $(this).css('cursor', 'grab');
    });

    // Function to immediately save gallery order via AJAX
    function saveGalleryOrderImmediately() {
        var ids = [];
        $('#vehizo-gallery-grid .vehizo-gallery-item').each(function() {
            var id = $(this).data('id');
            if (id) {
                ids.push(id);
            }
        });

        // Allow saving even with empty gallery (when all images are deleted)
        var postId = $('input[name="post_ID"]').val();
        if (!postId) {
            return;
        }


        // Show saving indicator
        showOrderSavingIndicator();

        $.ajax({
            url: vehizoGallery.ajaxUrl,
            type: 'POST',
            data: {
                action: 'vehizo_save_gallery_order',
                post_id: postId,
                gallery_order: ids.join(','),
                nonce: $('#vehizo_vehicle_editor_nonce').val()
            },
            success: function(response) {
                showOrderSavedIndicator();
            },
            error: function(xhr, status, error) {
                showOrderErrorIndicator();
            }
        });
    }

    function showOrderSavingIndicator() {
        $('#sort-reminder').remove();
        $('#vehizo-gallery-grid').after('<div id="sort-reminder" style="background: #fff3cd; border: 1px solid #ffeaa7; color: #856404; padding: 10px; margin: 10px 0; border-radius: 4px; text-align: center;"><strong>💾 Speichere Bildrreihenfolge...</strong></div>');
    }

    function showOrderSavedIndicator() {
        $('#sort-reminder').remove();
        $('#vehizo-gallery-grid').after('<div id="sort-reminder" style="background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 10px; margin: 10px 0; border-radius: 4px; text-align: center;"><strong>✅ Bildreihenfolge gespeichert!</strong></div>');
        setTimeout(function() {
            $('#sort-reminder').fadeOut();
        }, 3000);
    }

    function showOrderErrorIndicator() {
        $('#sort-reminder').remove();
        $('#vehizo-gallery-grid').after('<div id="sort-reminder" style="background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 10px; margin: 10px 0; border-radius: 4px; text-align: center;"><strong>❌ Fehler beim Speichern der Reihenfolge!</strong></div>');
    }

    // Save buttons for tabs
    $(document).on('click', '.vehizo-save-tab-btn', function() {
        var $btn = $(this);
        var tabType = $btn.data('tab');

        $btn.prop('disabled', true).text('💾 Speichere...');

        saveTabData(tabType, function(success) {
            if (success) {
                $btn.text('✅ Gespeichert').css('background', '#28a745');
                setTimeout(function() {
                    $btn.prop('disabled', false).text('💾 Speichern').css('background', '');
                }, 2000);
            } else {
                $btn.text('❌ Fehler').css('background', '#dc3545');
                setTimeout(function() {
                    $btn.prop('disabled', false).text('💾 Speichern').css('background', '');
                }, 3000);
            }
        });
    });

    function saveTabData(tabType, callback) {
        var formData = {};
        var postId = $('input[name="post_ID"]').val();

        if (!postId) {
            callback(false);
            return;
        }

        // Collect data based on tab type
        if (tabType === 'basic') {
            // Collect basic data fields
            $('#tab-basic input, #tab-basic select').each(function() {
                var $field = $(this);
                var name = $field.attr('name');
                if (name && (name.startsWith('vehizo_') || name === 'post_title')) {
                    formData[name] = $field.val();
                }
            });
        } else if (tabType === 'technical') {
            // Collect technical data fields
            $('#tab-technical input, #tab-technical select').each(function() {
                var $field = $(this);
                var name = $field.attr('name');
                if (name && name.startsWith('vehizo_')) {
                    formData[name] = $field.val();
                }
            });
        }


        $.ajax({
            url: vehizoGallery.ajaxUrl,
            type: 'POST',
            data: {
                action: 'vehizo_save_tab_data',
                post_id: postId,
                tab_type: tabType,
                vehicle_data: formData,
                nonce: $('#vehizo_vehicle_editor_nonce').val()
            },
            success: function(response) {
                callback(true);
            },
            error: function(xhr, status, error) {
                callback(false);
            }
        });
    }
});