/**
 * Vehizo Frontend JavaScript
 * 
 * @package Vehizo
 * @version 4.1.0
 */

(function($) {
    'use strict';

    // Initialize when DOM is ready
    $(document).ready(function() {
        VehizoFrontend.init();
    });

    /**
     * Main Frontend Object
     */
    var VehizoFrontend = {

        /**
         * Initialize all functionality
         */
        init: function() {
            this.initFilters();
            this.initWishlist();
            this.initLazyLoading();
            this.initImageCarousel();
            this.initSearchFunctionality();
            this.initResponsiveFeatures();
            this.initAccessibility();
        },

        /**
         * Initialize filter functionality
         */
        initFilters: function() {
            var self = this;
            var $filterForm = $('.vehizo-filter-form');
            
            if ($filterForm.length === 0) return;

            // Handle filter form submission
            $filterForm.on('submit', function(e) {
                e.preventDefault();
                self.applyFilters();
            });

            // Handle filter input changes with debounce
            $('.vehizo-filter-input').on('input', self.debounce(function() {
                self.applyFilters();
            }, 500));

            // Handle sort changes
            $('.vehizo-sort-select').on('change', function() {
                self.applyFilters();
            });

            // Handle filter reset
            $('.vehizo-filter-reset').on('click', function(e) {
                e.preventDefault();
                self.resetFilters();
            });

            // Handle advanced filter toggles
            $('.vehizo-advanced-toggle').on('click', function() {
                $('.vehizo-advanced-filters').slideToggle();
                $(this).toggleClass('active');
            });
        },

        /**
         * Apply current filters
         */
        applyFilters: function() {
            var $container = $('.vehizo-vehicles-grid');
            var $form = $('.vehizo-filter-form');
            
            // Show loading state
            $container.addClass('vehizo-loading');
            
            // Collect filter data
            var filterData = {
                action: 'vehizo_filter_vehicles',
                nonce: vehizo_frontend.nonce
            };

            $form.find('input, select').each(function() {
                var $field = $(this);
                var name = $field.attr('name');
                var value = $field.val();
                
                if (name && value !== '') {
                    filterData[name] = value;
                }
            });

            // Make AJAX request
            $.ajax({
                url: vehizo_frontend.ajax_url,
                method: 'POST',
                data: filterData,
                success: function(response) {
                    if (response.success) {
                        $container.html(response.data.html);
                        VehizoFrontend.updateResultsCount(response.data.count);
                        VehizoFrontend.updatePagination(response.data.pagination);
                        
                        // Reinitialize features for new content
                        VehizoFrontend.initWishlist();
                        VehizoFrontend.initLazyLoading();
                    } else {
                        VehizoFrontend.showError(response.data.message || vehizo_frontend.strings.error);
                    }
                },
                error: function() {
                    VehizoFrontend.showError(vehizo_frontend.strings.error);
                },
                complete: function() {
                    $container.removeClass('vehizo-loading');
                }
            });
        },

        /**
         * Reset all filters
         */
        resetFilters: function() {
            var $form = $('.vehizo-filter-form');
            $form[0].reset();
            this.applyFilters();
        },

        /**
         * Initialize wishlist functionality
         */
        initWishlist: function() {
            var self = this;

            // Handle wishlist toggle
            $(document).off('click.vehizo', '.vehizo-wishlist-button')
                      .on('click.vehizo', '.vehizo-wishlist-button', function(e) {
                e.preventDefault();
                var $button = $(this);
                var vehicleId = $button.data('vehicle-id');
                
                if (vehicleId) {
                    self.toggleWishlist(vehicleId, $button);
                }
            });

            // Update wishlist counts on load
            this.updateWishlistCounts();
        },

        /**
         * Toggle wishlist status
         */
        toggleWishlist: function(vehicleId, $button) {
            var isActive = $button.hasClass('active');
            var action = isActive ? 'remove' : 'add';
            
            // Optimistic UI update
            $button.toggleClass('active');
            
            $.ajax({
                url: vehizo_frontend.ajax_url,
                method: 'POST',
                data: {
                    action: 'vehizo_toggle_wishlist',
                    nonce: vehizo_frontend.nonce,
                    vehicle_id: vehicleId,
                    wishlist_action: action
                },
                success: function(response) {
                    if (response.success) {
                        VehizoFrontend.updateWishlistCounts();
                        VehizoFrontend.showMessage(response.data.message, 'success');
                    } else {
                        // Revert optimistic update
                        $button.toggleClass('active');
                        VehizoFrontend.showError(response.data.message || vehizo_frontend.strings.error);
                    }
                },
                error: function() {
                    // Revert optimistic update
                    $button.toggleClass('active');
                    VehizoFrontend.showError(vehizo_frontend.strings.error);
                }
            });
        },

        /**
         * Update wishlist counts
         */
        updateWishlistCounts: function() {
            var $floatingButton = $('.floating-park-button');
            if ($floatingButton.length === 0) return;

            $.ajax({
                url: vehizo_frontend.ajax_url,
                method: 'POST',
                data: {
                    action: 'vehizo_get_wishlist_count',
                    nonce: vehizo_frontend.nonce
                },
                success: function(response) {
                    if (response.success && response.data.count !== undefined) {
                        var count = parseInt(response.data.count);
                        $floatingButton.attr('data-count', count);
                        
                        if (count > 0) {
                            $floatingButton.addClass('has-items').text(count);
                        } else {
                            $floatingButton.removeClass('has-items').text('♡');
                        }
                    }
                }
            });
        },

        /**
         * Initialize lazy loading for images
         */
        initLazyLoading: function() {
            if (!('IntersectionObserver' in window)) {
                // Fallback for older browsers
                $('.vehizo-lazy-image').each(function() {
                    var $img = $(this);
                    $img.attr('src', $img.data('src')).removeClass('vehizo-lazy-image');
                });
                return;
            }

            var imageObserver = new IntersectionObserver(function(entries) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        var $img = $(entry.target);
                        $img.attr('src', $img.data('src'))
                            .removeClass('vehizo-lazy-image')
                            .addClass('vehizo-loaded');
                        imageObserver.unobserve(entry.target);
                    }
                });
            }, {
                rootMargin: '50px'
            });

            $('.vehizo-lazy-image').each(function() {
                imageObserver.observe(this);
            });
        },

        /**
         * Initialize image carousel/gallery
         */
        initImageCarousel: function() {
            $('.vehizo-image-gallery').each(function() {
                var $gallery = $(this);
                var $mainImage = $gallery.find('.main-image img');
                var $thumbnails = $gallery.find('.thumbnail-grid img');

                $thumbnails.on('click', function() {
                    var newSrc = $(this).attr('src');
                    var newAlt = $(this).attr('alt');
                    
                    $mainImage.attr('src', newSrc).attr('alt', newAlt);
                    $thumbnails.removeClass('active');
                    $(this).addClass('active');
                });
            });
        },

        /**
         * Initialize search functionality
         */
        initSearchFunctionality: function() {
            var self = this;
            var $searchInput = $('.vehizo-search-input');
            
            if ($searchInput.length === 0) return;

            // Handle search with debounce
            $searchInput.on('input', self.debounce(function() {
                var query = $(this).val();
                self.performSearch(query);
            }, 300));

            // Handle search clear
            $('.vehizo-search-clear').on('click', function() {
                $searchInput.val('').trigger('input').focus();
            });
        },

        /**
         * Perform search
         */
        performSearch: function(query) {
            var $container = $('.vehizo-vehicles-grid');
            
            if (query.length < 2) {
                // Reset to show all vehicles if query is too short
                this.applyFilters();
                return;
            }
            
            $container.addClass('vehizo-loading');
            
            $.ajax({
                url: vehizo_frontend.ajax_url,
                method: 'POST',
                data: {
                    action: 'vehizo_search_vehicles',
                    nonce: vehizo_frontend.nonce,
                    query: query
                },
                success: function(response) {
                    if (response.success) {
                        $container.html(response.data.html);
                        VehizoFrontend.updateResultsCount(response.data.count);
                    } else {
                        VehizoFrontend.showError(response.data.message || vehizo_frontend.strings.error);
                    }
                },
                error: function() {
                    VehizoFrontend.showError(vehizo_frontend.strings.error);
                },
                complete: function() {
                    $container.removeClass('vehizo-loading');
                }
            });
        },

        /**
         * Initialize responsive features
         */
        initResponsiveFeatures: function() {
            var self = this;
            
            // Handle mobile filter toggle
            $('.vehizo-mobile-filter-toggle').on('click', function() {
                $('.vehizo-filter-container').slideToggle();
                $(this).toggleClass('active');
            });

            // Handle window resize
            $(window).on('resize', self.debounce(function() {
                self.handleResize();
            }, 250));
        },

        /**
         * Handle window resize
         */
        handleResize: function() {
            var windowWidth = $(window).width();
            
            // Adjust grid layout on very small screens
            if (windowWidth < 480) {
                $('.vehizo-vehicles-grid').removeClass('layout-sideby').addClass('layout-list');
            }
            
            // Adjust filter visibility
            if (windowWidth > 768) {
                $('.vehizo-filter-container').show();
            }
        },

        /**
         * Initialize accessibility features
         */
        initAccessibility: function() {
            // Add keyboard navigation for cards
            $('.fahrzeug-card').attr('tabindex', '0').on('keydown', function(e) {
                if (e.which === 13 || e.which === 32) { // Enter or Space
                    e.preventDefault();
                    var $link = $(this).find('.cta-button');
                    if ($link.length) {
                        $link[0].click();
                    }
                }
            });

            // Add ARIA labels
            $('.vehizo-wishlist-button').each(function() {
                var vehicleTitle = $(this).closest('.fahrzeug-card').find('.card-title').text();
                $(this).attr('aria-label', 'Add ' + vehicleTitle + ' to wishlist');
            });

            // Handle focus management for modals
            $(document).on('show.vehizo.modal', function() {
                $('body').addClass('vehizo-modal-open');
            });

            $(document).on('hide.vehizo.modal', function() {
                $('body').removeClass('vehizo-modal-open');
            });
        },

        /**
         * Update results count display
         */
        updateResultsCount: function(count) {
            $('.results-count .count').text(count);
            
            // Update page title if needed
            if (count === 0) {
                const tryAdjustingText = vehizo_frontend.strings.try_adjusting || 'Try adjusting your filters or search terms.';
                $('.vehizo-vehicles-grid').html('<div class="vehizo-no-results">' +
                    '<div class="vehizo-no-results-icon">🚗</div>' +
                    '<h3>' + vehizo_frontend.strings.no_results + '</h3>' +
                    '<p>' + tryAdjustingText + '</p>' +
                '</div>');
            }
        },

        /**
         * Update pagination
         */
        updatePagination: function(paginationHtml) {
            $('.vehizo-pagination').html(paginationHtml);
        },

        /**
         * Show success message
         */
        showMessage: function(message, type) {
            type = type || 'info';
            this.showNotification(message, type);
        },

        /**
         * Show error message
         */
        showError: function(message) {
            this.showNotification(message, 'error');
        },

        /**
         * Show notification
         */
        showNotification: function(message, type) {
            var $notification = $('<div class="vehizo-notification vehizo-notification-' + type + '">' +
                '<span class="vehizo-notification-message">' + this.escapeHtml(message) + '</span>' +
                '<button class="vehizo-notification-close">&times;</button>' +
            '</div>');

            // Remove existing notifications
            $('.vehizo-notification').fadeOut(function() {
                $(this).remove();
            });

            // Add new notification
            $('body').append($notification);
            $notification.fadeIn();

            // Auto-hide after 5 seconds
            setTimeout(function() {
                $notification.fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);

            // Handle close button
            $notification.find('.vehizo-notification-close').on('click', function() {
                $notification.fadeOut(function() {
                    $(this).remove();
                });
            });
        },

        /**
         * Escape HTML for security
         */
        escapeHtml: function(text) {
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.toString().replace(/[&<>"']/g, function(m) { return map[m]; });
        },

        /**
         * Debounce function for performance
         */
        debounce: function(func, wait, immediate) {
            var timeout;
            return function() {
                var context = this, args = arguments;
                var later = function() {
                    timeout = null;
                    if (!immediate) func.apply(context, args);
                };
                var callNow = immediate && !timeout;
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
                if (callNow) func.apply(context, args);
            };
        },

        /**
         * Get URL parameter
         */
        getUrlParameter: function(name) {
            name = name.replace(/[\[]/, '\\[').replace(/[\]]/, '\\]');
            var regex = new RegExp('[\\?&]' + name + '=([^&#]*)');
            var results = regex.exec(location.search);
            return results === null ? '' : decodeURIComponent(results[1].replace(/\+/g, ' '));
        },

        /**
         * Update URL with current filters
         */
        updateUrl: function(params) {
            if (typeof history.pushState === 'undefined') return;
            
            var url = new URL(window.location.href);
            
            Object.keys(params).forEach(function(key) {
                if (params[key]) {
                    url.searchParams.set(key, params[key]);
                } else {
                    url.searchParams.delete(key);
                }
            });
            
            history.pushState(null, '', url.toString());
        }
    };

    // Make VehizoFrontend available globally
    window.VehizoFrontend = VehizoFrontend;

})(jQuery);