/**
 * Vehizo Contact Modal JavaScript
 * Handles elegant contact form modal for vehicle inquiries
 * 
 * @package Vehizo
 * @version 4.2.0
 */

'use strict';

document.addEventListener('DOMContentLoaded', function() {
    
    // Initialize contact modal functionality
    initContactModal();
});

/**
 * Initialize contact modal functionality
 */
function initContactModal() {
    const modalTriggers = document.querySelectorAll('.js-contact-modal-trigger');
    const modalOverlay = document.getElementById('vehizo-contact-modal-overlay');
    const modalCloseButtons = document.querySelectorAll('.js-contact-modal-close');
    const contactForm = document.getElementById('vehizo-contact-form');
    
    if (!modalOverlay) {
        return;
    }
    
    
    // Open modal when contact button is clicked
    modalTriggers.forEach(trigger => {
        trigger.addEventListener('click', function(e) {
            e.preventDefault();
            openContactModal(this);
        });
    });
    
    // Close modal when close buttons are clicked
    modalCloseButtons.forEach(closeBtn => {
        closeBtn.addEventListener('click', function(e) {
            e.preventDefault();
            closeContactModal();
        });
    });
    
    // Close modal when clicking outside
    modalOverlay.addEventListener('click', function(e) {
        if (e.target === modalOverlay) {
            closeContactModal();
        }
    });
    
    // Close modal with Escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && modalOverlay.classList.contains('active')) {
            closeContactModal();
        }
    });
    
    // Handle form submission
    if (contactForm) {
        contactForm.addEventListener('submit', handleContactFormSubmit);
    }
}

/**
 * Open contact modal
 */
function openContactModal(trigger) {
    const modalOverlay = document.getElementById('vehizo-contact-modal-overlay');
    const vehicleId = trigger.getAttribute('data-vehicle-id');
    const vehicleTitle = trigger.getAttribute('data-vehicle-title');
    
    if (!modalOverlay) return;
    
    // Pre-fill vehicle information
    const vehicleIdInput = modalOverlay.querySelector('input[name="vehicle_id"]');
    const vehicleTitleInput = modalOverlay.querySelector('input[name="vehicle_title"]');
    const vehicleInfoHeading = modalOverlay.querySelector('.vehizo-contact-vehicle-info h4');
    
    if (vehicleIdInput) vehicleIdInput.value = vehicleId;
    if (vehicleTitleInput) vehicleTitleInput.value = vehicleTitle;
    if (vehicleInfoHeading) vehicleInfoHeading.textContent = vehicleTitle;
    
    // Show modal with animation
    modalOverlay.classList.add('active');
    document.body.style.overflow = 'hidden';
    
    // Focus first input
    setTimeout(() => {
        const firstInput = modalOverlay.querySelector('input[type="text"], input[type="email"]');
        if (firstInput) firstInput.focus();
    }, 150);
    
}

/**
 * Close contact modal
 */
function closeContactModal() {
    const modalOverlay = document.getElementById('vehizo-contact-modal-overlay');
    
    if (!modalOverlay) return;
    
    modalOverlay.classList.remove('active');
    document.body.style.overflow = '';
    
    // Reset form and result
    const contactForm = document.getElementById('vehizo-contact-form');
    const resultDiv = document.getElementById('vehizo-contact-result');
    
    if (contactForm) {
        contactForm.style.display = 'block';
        contactForm.reset();
    }
    
    if (resultDiv) {
        resultDiv.style.display = 'none';
        resultDiv.innerHTML = '';
    }
    
}

/**
 * Handle contact form submission
 */
function handleContactFormSubmit(e) {
    e.preventDefault();
    
    const form = e.target;
    const submitBtn = form.querySelector('#contact-submit-btn');
    const resultDiv = document.getElementById('vehizo-contact-result');
    
    // Validate required fields
    const requiredFields = form.querySelectorAll('input[required], textarea[required]');
    let isValid = true;
    
    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            isValid = false;
            field.style.borderColor = '#ef4444';
        } else {
            field.style.borderColor = '';
        }
    });
    
    // Validate email
    const emailField = form.querySelector('input[type="email"]');
    if (emailField && emailField.value && !isValidEmail(emailField.value)) {
        isValid = false;
        emailField.style.borderColor = '#ef4444';
    }
    
    if (!isValid) {
        showContactResult('error', vehizoContact.strings.required_fields);
        return;
    }
    
    // Show loading state
    const originalText = submitBtn.textContent;
    submitBtn.textContent = vehizoContact.strings.sending;
    submitBtn.disabled = true;
    
    // Prepare form data
    const formData = new FormData(form);
    
    // Send AJAX request
    fetch(vehizoContact.ajaxurl, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showContactResult('success', vehizoContact.strings.success);
            form.style.display = 'none';
            
            // Auto-close after success
            setTimeout(() => {
                closeContactModal();
            }, 3000);
        } else {
            showContactResult('error', data.data?.message || vehizoContact.strings.error);
        }
    })
    .catch(error => {
        showContactResult('error', vehizoContact.strings.error);
    })
    .finally(() => {
        // Reset button state
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
    });
    
}

/**
 * Show contact form result
 */
function showContactResult(type, message) {
    const resultDiv = document.getElementById('vehizo-contact-result');
    
    if (!resultDiv) return;
    
    const alertClass = type === 'success' ? 'vehizo-alert-success' : 'vehizo-alert-error';
    const icon = type === 'success' ? '✅' : '❌';
    
    resultDiv.innerHTML = `
        <div class="vehizo-alert ${alertClass}">
            ${icon} ${message}
        </div>
    `;
    
    resultDiv.style.display = 'block';
    
    // Scroll result into view
    resultDiv.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
}

/**
 * Validate email address
 */
function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

/**
 * Global function for debugging
 */
function debugContactModal() {
}

// Make debug function globally available
if (window) {
    window.debugContactModal = debugContactModal;