/**
 * Vehizo Admin JavaScript
 * 
 * @package Vehizo
 * @version 4.1.0
 */

(function($) {
    'use strict';

    // FALLBACK: Check for both variable names and create unified reference
    let vehizo_ajax_unified;
    
    if (typeof vehizo_ajax !== 'undefined') {
        vehizo_ajax_unified = vehizo_ajax;
    } else if (typeof vehizo_ajax_data !== 'undefined') {
        vehizo_ajax_unified = vehizo_ajax_data;
    } else {
        // Fallback removed - plugin should always provide localized script data
        vehizo_ajax_unified = {
            ajax_url: '',
            nonce_manual_import: '',
            strings: {}
        };
    }

    /**
     * Main admin object
     */
    const VehizoAdmin = {
        
        /**
         * Initialize all functionality
         */
        init: function() {
            this.bindEvents();
            this.initModules();
            this.initImport();
            this.initDeleteConfirmations();
            this.initSettingsReset();
            this.initVehicleStatusUpdater();
        },

        /**
         * Bind general events
         */
        bindEvents: function() {
            $(document).ready(function() {
                VehizoAdmin.handleResponsiveNav();
            });

            $(window).on('resize', function() {
                VehizoAdmin.handleResponsiveNav();
            });
        },

        /**
         * Handle responsive navigation
         */
        handleResponsiveNav: function() {
            const navTabs = $('.vehizo-nav .nav-tab-wrapper');
            if ($(window).width() < 768 && navTabs.length) {
                navTabs.addClass('responsive');
            } else if (navTabs.length) {
                navTabs.removeClass('responsive');
            }
        },

        /**
         * Initialize module toggle functionality
         */
        initModules: function() {
            $(document).on('change', '.vehizo-module-toggle', function() {
                const $toggle = $(this);
                const moduleOption = $toggle.data('module-option');
                const status = $toggle.is(':checked') ? '1' : '0';

                VehizoAdmin.toggleModuleStatus(moduleOption, status, $toggle);
            });

            $(document).on('click', '.vehizo-module-setting-trigger', function(e) {
                e.preventDefault();
                const optionName = $(this).data('option-name');
                VehizoAdmin.openSettingModal(optionName);
            });
        },

        /**
         * Toggle module status via AJAX
         */
        toggleModuleStatus: function(moduleOption, status, $toggle) {
            const originalState = $toggle.is(':checked');
            
            // Disable toggle during request
            $toggle.prop('disabled', true);

            $.ajax({
                url: vehizo_ajax_unified.ajax_url,
                type: 'POST',
                data: {
                    action: 'vehizo_toggle_module_status',
                    module_option: moduleOption,
                    status: status,
                    nonce: vehizo_ajax_unified.nonce_toggle_module
                },
                success: function(response) {
                    if (response.success) {
                        VehizoAdmin.showNotice(response.data.message, 'success');
                        
                        // Update status indicator if exists
                        const $statusIndicator = $toggle.closest('.vehizo-module-item').find('.module-status');
                        if ($statusIndicator.length) {
                            $statusIndicator.removeClass('status-active status-inactive')
                                           .addClass(status === '1' ? 'status-active' : 'status-inactive')
                                           .text(status === '1' ? vehizo_ajax_unified.strings.active || 'Active' : vehizo_ajax_unified.strings.inactive || 'Inactive');
                        }
                    } else {
                        // Revert toggle state on error
                        $toggle.prop('checked', !originalState);
                        VehizoAdmin.showNotice(response.data.message, 'error');
                    }
                },
                error: function() {
                    // Revert toggle state on error
                    $toggle.prop('checked', !originalState);
                    VehizoAdmin.showNotice(vehizo_ajax.strings.error_module_status || 'An error occurred while updating the module status.', 'error');
                },
                complete: function() {
                    $toggle.prop('disabled', false);
                }
            });
        },

        /**
         * Open setting modal for module configuration
         */
        openSettingModal: function(optionName) {
            // First, get the current setting value
            $.ajax({
                url: vehizo_ajax_unified.ajax_url,
                type: 'POST',
                data: {
                    action: 'vehizo_get_module_setting',
                    option_name: optionName,
                    nonce: vehizo_ajax_unified.nonce_get_setting
                },
                success: function(response) {
                    if (response.success) {
                        VehizoAdmin.showSettingModal(optionName, response.data.value);
                    } else {
                        VehizoAdmin.showNotice(response.data.message, 'error');
                    }
                },
                error: function() {
                    VehizoAdmin.showNotice(vehizo_ajax.strings.error_retrieving || 'Error retrieving setting value.', 'error');
                }
            });
        },

        /**
         * Show setting modal
         */
        showSettingModal: function(optionName, currentValue) {
            const modalHtml = `
                <div class="vehizo-modal-backdrop" id="vehizo-setting-modal">
                    <div class="vehizo-modal-content">
                        <div class="vehizo-modal-header">
                            <h3>${vehizo_ajax.strings.module_config || 'Module Configuration'}</h3>
                            <button class="vehizo-modal-close">&times;</button>
                        </div>
                        <div class="vehizo-modal-body">
                            <label for="vehizo-setting-value">${vehizo_ajax.strings.setting_value || 'Setting Value'}:</label>
                            <input type="email" id="vehizo-setting-value" value="${currentValue || ''}" placeholder="${vehizo_ajax.strings.enter_email || 'Enter email address'}">
                        </div>
                        <div class="vehizo-modal-footer">
                            <button class="modern-button secondary vehizo-modal-cancel">${vehizo_ajax.strings.cancel || 'Cancel'}</button>
                            <button class="modern-button vehizo-modal-save" data-option-name="${optionName}">${vehizo_ajax.strings.save || 'Save'}</button>
                        </div>
                    </div>
                </div>
            `;

            $('body').append(modalHtml);

            // Bind modal events
            $('#vehizo-setting-modal').on('click', '.vehizo-modal-close, .vehizo-modal-cancel, .vehizo-modal-backdrop', function(e) {
                if (e.target === this) {
                    VehizoAdmin.closeSettingModal();
                }
            });

            $('#vehizo-setting-modal').on('click', '.vehizo-modal-save', function() {
                const optionName = $(this).data('option-name');
                const settingValue = $('#vehizo-setting-value').val();
                VehizoAdmin.saveModuleSetting(optionName, settingValue);
            });

            // Focus input
            $('#vehizo-setting-value').focus();
        },

        /**
         * Close setting modal
         */
        closeSettingModal: function() {
            $('#vehizo-setting-modal').remove();
        },

        /**
         * Save module setting via AJAX
         */
        saveModuleSetting: function(optionName, settingValue) {
            const $saveButton = $('.vehizo-modal-save');
            const originalText = $saveButton.text();

            $saveButton.text(vehizo_ajax.strings.saving || 'Saving...').prop('disabled', true);

            $.ajax({
                url: vehizo_ajax_unified.ajax_url,
                type: 'POST',
                data: {
                    action: 'vehizo_save_module_setting',
                    option_name: optionName,
                    setting_value: settingValue,
                    nonce: vehizo_ajax_unified.nonce_save_setting
                },
                success: function(response) {
                    if (response.success) {
                        VehizoAdmin.showNotice(response.data.message, 'success');
                        VehizoAdmin.closeSettingModal();
                    } else {
                        VehizoAdmin.showNotice(response.data.message, 'error');
                    }
                },
                error: function() {
                    VehizoAdmin.showNotice(vehizo_ajax.strings.error_saving || 'Error saving setting.', 'error');
                },
                complete: function() {
                    $saveButton.text(originalText).prop('disabled', false);
                }
            });
        },

        /**
         * Initialize import functionality
         */
        initImport: function() {
            
            // New import button (for dashboard)
            $(document).on('click', '#vehizo-import-button', function(e) {
                e.preventDefault();
                VehizoAdmin.runManualImport();
            });

            // Existing import button (legacy support)
            $(document).on('click', '#vehizo-import-btn', function(e) {
                e.preventDefault();
                VehizoAdmin.runLegacyImport(e);
            });
            
            // Force binding for import button if it exists
            if ($('#vehizo-import-button').length) {
                $('#vehizo-import-button').off('click').on('click', function(e) {
                    e.preventDefault();
                    VehizoAdmin.runManualImport();
                });
            }
        },

        /**
         * Run manual import via AJAX (new version)
         */
        runManualImport: function() {
            
            const $button = $('#vehizo-import-button');
            const $statusEl = $('#vehizo-import-status');
            const defaultText = vehizo_ajax_unified.strings.import_now || 'Import Now';
            const loadingText = vehizo_ajax_unified.strings.importing || 'Importing...';


            $button.html(loadingText).prop('disabled', true);
            $statusEl.show().removeClass('alert-success alert-error').addClass('alert-info')
                    .html(vehizo_ajax_unified.strings.import_running || 'Import is running...');


            $.ajax({
                url: vehizo_ajax_unified.ajax_url,
                type: 'POST',
                data: {
                    action: 'vehizo_manual_import',
                    nonce: vehizo_ajax_unified.nonce_manual_import || vehizo_ajax_unified.nonce
                },
                success: function(response) {
                    $button.html(defaultText).prop('disabled', false);
                    
                    if (response.success) {
                        $statusEl.removeClass('alert-info alert-error').addClass('alert-success')
                                .html((vehizo_ajax_unified.strings.import_successful || 'Import successful:') + ' ' + response.data.message);
                        
                        // Reload page after 2 seconds to show updated data
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        $statusEl.removeClass('alert-info alert-success').addClass('alert-error')
                                .html((vehizo_ajax_unified.strings.import_error || 'Import Error:') + ' ' + response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    $button.html(defaultText).prop('disabled', false);
                    $statusEl.removeClass('alert-info alert-success').addClass('alert-error')
                            .html(vehizo_ajax_unified.strings.import_error_general || 'Error during import.');
                }
            });
        },

        /**
         * Run legacy import (compatibility with existing code)
         */
        runLegacyImport: function(e) {
            e.preventDefault();
            
            const $resultEl = $('#vehizo-import-result');

            // Use localized loading text
            const loadingText = vehizo_ajax.strings.importing || 'Loading vehicles...';
            $resultEl.empty().append($('<p>').text(loadingText));

            $.post(vehizo_ajax_unified.ajax_url, {
                action: 'vehizo_mobile_import',
                nonce: vehizo_ajax_unified.nonce || vehizo_ajax_unified.nonce_manual_import
            }, function(response) {
                var $message = $('<p>');

                if (response.success && response.data.message) {
                    $message.text(response.data.message);
                    $resultEl.empty().append($message);
                } else {
                    // Use localized error prefix
                    var errorMessage = response.data.message || (vehizo_ajax.strings.import_error_general || 'An unknown error occurred.');
                    var errorPrefix = vehizo_ajax.strings.import_error || 'Error: ';
                    $message.text(errorPrefix + errorMessage);
                    $resultEl.empty().append($message);
                }
            });
        },

        /**
         * Initialize delete confirmations
         */
        initDeleteConfirmations: function() {
            $(document).on('click', '.vehizo-delete-vehicle', function(e) {
                e.preventDefault();
                const confirmText = vehizo_ajax_unified.strings.confirm_delete_vehicle || 'Do you really want to delete this vehicle? It will be moved to the trash.';
                if (confirm(confirmText)) {
                    window.location.href = $(this).attr('href');
                }
            });

            $(document).on('click', '.vehizo-delete-vehicle-link', function(e) {
                e.preventDefault();
                const confirmText = vehizo_ajax_unified.strings.confirm_move_trash || 'Do you really want to move this vehicle to the trash?';
                if (confirm(confirmText)) {
                    window.location.href = $(this).attr('href');
                }
            });
        },

        /**
         * Initialize settings reset functionality
         */
        initSettingsReset: function() {
            $(document).on('click', '.vehizo-reset-settings', function(e) {
                const confirmText = vehizo_ajax_unified.strings.confirm_reset_settings || 'Do you really want to reset all frontend settings to their default values?';
                if (!confirm(confirmText)) {
                    e.preventDefault();
                }
            });
        },

        /**
         * Initialize vehicle status updater
         */
        initVehicleStatusUpdater: function() {
            $(document).on('change', '.vehizo-status-select', function() {
                $(this).siblings('.vehizo-save-status-button').show();
            });

            $(document).on('click', '.vehizo-save-status-button', function() {
                const $button = $(this);
                const $select = $button.siblings('.vehizo-status-select');
                const vehicleId = $select.data('vehicle-id');
                const newStatus = $select.val();

                $button.prop('disabled', true);
                $select.prop('disabled', true);

                $.ajax({
                    url: vehizo_ajax_unified.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'vehizo_update_vehicle_status',
                        post_id: vehicleId,
                        status: newStatus,
                        nonce: vehizo_ajax_unified.nonce_update_vehicle_status
                    },
                    success: function(response) {
                        if (response.success) {
                            VehizoAdmin.showNotice(response.data.message, 'success');
                            $button.hide();
                        } else {
                            VehizoAdmin.showNotice(response.data.message, 'error');
                        }
                    },
                    error: function() {
                        const errorMsg = vehizo_ajax.strings.status_update_error || vehizo_ajax.strings.error_module_status || 'An error occurred while updating the status.';
                        VehizoAdmin.showNotice(errorMsg, 'error');
                    },
                    complete: function() {
                        $button.prop('disabled', false);
                        $select.prop('disabled', false);
                    }
                });
            });
        },

        /**
         * Show admin notice
         */
        showNotice: function(message, type) {
            type = type || 'info';

            // Secure way: Create elements and use .text() to prevent XSS
            const notice = $('<div class="notice notice-' + type + ' is-dismissible"><p></p></div>')
                .find('p').text(message).end();
            
            // Remove existing notices
            $('.notice').remove();
            
            // Add new notice
            if ($('.vehizo-content-wrapper').length) {
                $('.vehizo-content-wrapper').prepend(notice);
            } else {
                $('.wrap').prepend(notice);
            }

            // Auto-remove after 5 seconds
            setTimeout(function() {
                notice.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);

            // Make notice dismissible
            notice.on('click', '.notice-dismiss', function() {
                notice.fadeOut(300, function() {
                    $(this).remove();
                });
            });
        },

        /**
         * Utility function to debounce function calls
         */
        debounce: function(func, wait, immediate) {
            let timeout;
            return function executedFunction() {
                const context = this;
                const args = arguments;
                const later = function() {
                    timeout = null;
                    if (!immediate) func.apply(context, args);
                };
                const callNow = immediate && !timeout;
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
                if (callNow) func.apply(context, args);
            };
        },

        /**
         * Utility function to format numbers
         */
        formatNumber: function(num) {
            return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ',');
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {

        // Modal styles are now loaded via wp_enqueue_style (WordPress.org Guidelines)
        // See: assets/css/admin-modal.css

        // Initialize admin functionality
        VehizoAdmin.init();
        
        // Debug info
    });

    // Make VehizoAdmin globally accessible for debugging
    window.VehizoAdmin = VehizoAdmin;

})(jQuery);