/**
 * Vehizo Vehicle Editor JavaScript
 *
 * Handles all editor functionality including tab navigation,
 * media library integration, and form management
 *
 * @package      Vehizo
 * @version      4.1.0
 * @author       Thorsten Glander
 */

(function($) {
    'use strict';

    // Global variables
    let galleryFrame;
    let isFormDirty = false;
    
    /**
     * Initialize the vehicle editor
     */
    function initVehizoEditor() {
        setupTabNavigation();
        setupFormChangeDetection();
        setupAutoTitle();
        setupMediaLibrary();
        setupGalleryManagement();
        setupFormValidation();
        setupTabSaveButtons(); // NEU: Initialisierung der Speicher-Buttons
    }

    /**
     * Setup tab navigation functionality
     */
    function setupTabNavigation() {
        $('.vehizo-tab-button').on('click', function(e) {
            e.preventDefault();
            
            const $button = $(this);
            const tab = $button.data('tab');
            
            if (!tab) return;
            
            // Update button states
            $('.vehizo-tab-button').removeClass('active');
            $button.addClass('active');
            
            // Update tab content
            $('.vehizo-tab-content').removeClass('active');
            $('#tab-' + tab).addClass('active').addClass('vehizo-fade-in');
            
            // Trigger custom event
            $(document).trigger('vehizo:tab-changed', [tab]);
        });
    }

    /**
     * Setup form change detection for dirty state management
     */
    function setupFormChangeDetection() {
        const $statusElement = $('#vehizo-save-status');
        
        $('.vehizo-form-input, #vehizo-title, #vehizo_description').on('input change', function() {
            if (!isFormDirty) {
                isFormDirty = true;
                updateSaveStatus('unsaved', vehizoEditor.strings.confirmLeave);
            }
        });

        // Reset dirty state on form submission
        $('form#post').on('submit', function() {
            isFormDirty = false;
            updateSaveStatus('saving', vehizoEditor.strings.savingData || 'Saving...');
        });
    }

    /**
     * Setup automatic title generation from make and model
     */
    function setupAutoTitle() {
        $('input[name="vehizo_make"], input[name="vehizo_model"]').on('input', debounce(function() {
            const make = $('input[name="vehizo_make"]').val().trim();
            const model = $('input[name="vehizo_model"]').val().trim();
            const $titleField = $('#vehizo-title');
            const currentTitle = $titleField.val().trim();
            
            // Only auto-update if title is empty or is "Auto Draft"
            if (make && model && (!currentTitle || currentTitle === 'Auto Draft')) {
                $titleField.val(make + ' ' + model);
                $titleField.trigger('input'); // Trigger change detection
            }
        }, 500));
    }

    /**
     * Setup WordPress Media Library integration
     */
    function setupMediaLibrary() {
        $(document).on('click', '.vehizo-gallery-upload', function(e) {
            e.preventDefault();

            if (galleryFrame) {
                galleryFrame.open();
                return;
            }

            galleryFrame = wp.media({
                title: vehizoEditor.strings.selectImages || 'Select Images',
                button: { text: vehizoEditor.strings.addToGallery || 'Add to Gallery' },
                multiple: true,
                library: { type: 'image' }
            });

            galleryFrame.on('select', function() {
                const selection = galleryFrame.state().get('selection');
                const currentIds = getCurrentGalleryIds();

                selection.map(function(attachment) {
                    const attachmentData = attachment.toJSON();
                    if (currentIds.indexOf(attachmentData.id.toString()) === -1) {
                        const imageUrl = attachmentData.sizes && attachmentData.sizes.medium 
                            ? attachmentData.sizes.medium.url 
                            : attachmentData.url;
                        addImageToGallery(attachmentData.id, imageUrl);
                    }
                });

                updateGalleryIds();
                updateGalleryContainer();
                markFormDirty();
            });

            galleryFrame.open();
        });

        $(document).on('click', '.remove-btn', function(e) {
            e.preventDefault();
            if (confirm(vehizoEditor.strings.confirmRemove)) {
                $(this).closest('.vehizo-gallery-item').fadeOut(300, function() {
                    $(this).remove();
                    updateGalleryIds();
                    updateImagePositions();
                    updateGalleryContainer();
                    markFormDirty();
                });
            }
        });

        const dropzone = $('#vehizo-gallery-container');
        dropzone.on('dragover', function(e) {
            e.preventDefault();
            e.stopPropagation();
            dropzone.addClass('dragover');
        });

        dropzone.on('dragleave', function(e) {
            e.preventDefault();
            e.stopPropagation();
            dropzone.removeClass('dragover');
        });

        dropzone.on('drop', function(e) {
            e.preventDefault();
            e.stopPropagation();
            dropzone.removeClass('dragover');

            const files = e.originalEvent.dataTransfer.files;
            handleFileUpload(files);
        });
    }

    function handleFileUpload(files) {
        const uploader = createUploader();

        $.each(files, function(i, file) {
            uploader.add(file);
        });
    }

    function createUploader() {
        const uploader = new wp.Uploader({
            runtimes: 'html5',
            browse_button: 'vehizo-gallery-upload',
            container: 'vehizo-gallery-container',
            drop_element: 'vehizo-gallery-container',
            file_data_name: 'async-upload',
            url: wp.Uploader.defaults.url,
            filters: {
                max_file_size: '10mb',
                mime_types: [
                    { title: 'Images', extensions: 'jpg,jpeg,gif,png' }
                ]
            },
            multipart_params: {
                action: 'upload-attachment'
            },
            init: {
                FilesAdded: function(up, files) {
                    $.each(files, function(i, file) {
                        addImageToGallery(file.id, '');
                    });

                    up.refresh();
                    up.start();
                },
                UploadProgress: function(up, file) {
                    $('#' + file.id).find('.progress-bar').css('width', file.percent + '%');
                },
                FileUploaded: function(up, file, response) {
                    const result = JSON.parse(response.response);

                    if (result.success) {
                        const attachment = result.data;
                        const $item = $('#' + file.id);
                        $item.data('id', attachment.id);
                        $item.find('img').attr('src', attachment.sizes.medium.url);
                        updateGalleryIds();
                        markFormDirty();
                    }
                }
            }
        });

        uploader.init();
        return uploader;
    }

    /**
     * Setup gallery management functionality
     */
    function setupGalleryManagement() {
        initSortable();
        
        // Handle gallery container updates
        $(document).on('vehizo:gallery-updated', function() {
            updateGalleryContainer();
        });
    }

    /**
     * Setup form validation
     */
    function setupFormValidation() {
        $('form#post').on('submit', function(e) {
            const title = $('#vehizo-title').val().trim();
            
            if (!title) {
                e.preventDefault();
                alert(vehizoEditor.strings.titleRequired || 'Please enter a vehicle title.');
                $('#vehizo-title').focus();
                return false;
            }
            
            // Additional validation can be added here
            return true;
        });
    }

    /**
     * Add image to gallery grid
     * @param {number} imageId - WordPress attachment ID
     * @param {string} thumbnailUrl - Image thumbnail URL
     */
    function addImageToGallery(imageId, thumbnailUrl) {
        const $container = $('#vehizo-gallery-grid');
        let $grid = $container;
        
        if ($container.length === 0) {
            const $galleryContainer = $('#vehizo-gallery-container');
            $galleryContainer.addClass('has-images');
            $galleryContainer.html(`
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                    <h3 style="margin: 0; color: #374151;">${(vehizoEditor.strings.vehicleImagesCount || 'Vehicle Images (%d)').replace('%d', '1')}</h3>
                    <button type="button" class="vehizo-gallery-upload">${vehizoEditor.strings.addMoreImages || 'Add More'}</button>
                </div>
                <div class="vehizo-gallery-grid" id="vehizo-gallery-grid"></div>
            `);
            $grid = $('#vehizo-gallery-grid');
            initSortable(); // Re-initialize sortable after creating grid
        }

        const nextPosition = $grid.find('.vehizo-gallery-item').length + 1;
        
        const $imageHtml = $(`
            <div class="vehizo-gallery-item vehizo-fade-in" id="${imageId}" data-id="${imageId}">
                <div class="position-badge">${nextPosition}</div>
                <img src="${thumbnailUrl}" alt="${vehizoEditor.strings.vehicleImage || 'Vehicle Image'}" loading="lazy">
                <button type="button" class="remove-btn" title="${vehizoEditor.strings.removeImage || 'Remove'}">×</button>
                <div class="progress-bar-container"><div class="progress-bar"></div></div>
            </div>
        `);
        
        $grid.append($imageHtml);
    }

    /**
     * Get current gallery image IDs
     * @returns {Array} Array of image ID strings
     */
    function getCurrentGalleryIds() {
        const ids = [];
        $('#vehizo-gallery-grid .vehizo-gallery-item').each(function() {
            const id = $(this).data('id');
            if (id) {
                ids.push(id.toString());
            }
        });
        return ids;
    }

    /**
     * Update hidden input with gallery IDs
     */
    function updateGalleryIds() {
        const ids = getCurrentGalleryIds();
        $('#vehizo-gallery-ids').val(ids.join(','));
    }

    /**
     * Update position badges on gallery items
     */
    function updateImagePositions() {
        $('#vehizo-gallery-grid .vehizo-gallery-item').each(function(index) {
            $(this).find('.position-badge').text(index + 1);
        });
    }

    /**
     * Update gallery container display based on image count
     */
    function updateGalleryContainer() {
        const $container = $('#vehizo-gallery-container');
        const $images = $('#vehizo-gallery-grid .vehizo-gallery-item');
        
        if ($images.length === 0) {
            $container.removeClass('has-images');
            $container.html(`
                <div style="font-size: 48px; margin-bottom: 16px; opacity: 0.6;">📷</div>
                <h3 style="margin: 0 0 8px 0; color: #6b7280;">${vehizoEditor.strings.noImagesUploaded || 'No images'}</h3>
                <p style="margin: 0 0 24px 0; color: #9ca3af;">${vehizoEditor.strings.uploadProfessional || 'Upload images'}</p>
                <button type="button" class="vehizo-gallery-upload">${vehizoEditor.strings.uploadImages || 'Upload'}</button>
            `);
        } else {
            const $heading = $container.find('h3');
            if ($heading.length) {
                const imageCount = $images.length;
                $heading.text((vehizoEditor.strings.vehicleImagesCount || 'Vehicle Images (%d)').replace('%d', imageCount));
            }
        }
    }

    /**
     * Initialize jQuery UI Sortable for gallery
     */
    function initSortable() {
        const $grid = $('#vehizo-gallery-grid');
        if ($grid.length === 0) return;

        $grid.sortable({
            items: '.vehizo-gallery-item',
            cursor: 'move',
            opacity: 0.8,
            tolerance: 'pointer',
            placeholder: 'vehizo-gallery-item ui-sortable-placeholder',
            helper: 'clone',
            start: function(event, ui) {
                ui.helper.addClass('ui-sortable-helper');
            },
            stop: function(event, ui) {
                ui.item.removeClass('ui-sortable-helper');
            },
            update: function(event, ui) {
                updateGalleryIds();
                updateImagePositions();
                updateSaveStatus('unsaved', vehizoEditor.strings.reorderMessage);
                markFormDirty();
            }
        });
    }

    /**
     * Update save status indicator
     * @param {string} type - Status type (unsaved, saving, saved, error)
     * @param {string} message - Status message
     */
    function updateSaveStatus(type, message) {
        const $status = $('#vehizo-save-status');
        
        $status.removeClass('saving saved error')
               .addClass(type)
               .text(message);

        // Auto-hide success messages after delay
        if (type === 'saved') {
            setTimeout(function() {
                if ($status.hasClass('saved')) {
                    $status.removeClass('saved')
                           .text(vehizoEditor.strings.readyToEdit || 'Ready');
                }
            }, 3000);
        }
    }

    /**
     * Mark form as dirty (has unsaved changes)
     */
    function markFormDirty() {
        if (!isFormDirty) {
            isFormDirty = true;
            updateSaveStatus('unsaved', vehizoEditor.strings.confirmLeave);
        }
    }

    /**
     * Handle browser back/forward navigation warnings
     */
    function setupNavigationWarning() {
        $(window).on('beforeunload', function(e) {
            if (isFormDirty) {
                const message = vehizoEditor.strings.confirmLeave || 'You have unsaved changes. Are you sure you want to leave?';
                e.returnValue = message;
                return message;
            }
        });
    }

    /**
     * Keyboard shortcuts
     */
    function setupKeyboardShortcuts() {
        $(document).on('keydown', function(e) {
            // Ctrl/Cmd + S to save
            if ((e.ctrlKey || e.metaKey) && e.which === 83) {
                e.preventDefault();
                $('form#post').submit();
            }
        });
    }

    /**
     * NEU: Bindet die Klick-Events an die Speicher-Buttons der Tabs.
     */
    function setupTabSaveButtons() {
        $(document).on('click', '.vehizo-save-tab-btn', function() {
            const $btn = $(this);
            const tabType = $btn.data('tab');
            const originalText = $btn.html();

            $btn.prop('disabled', true).html('💾 ' + (vehizoEditor.strings.savingData || 'Saving...'));

            saveTabData(tabType, function(success, message) {
                if (success) {
                    $btn.html('✅ ' + (vehizoEditor.strings.dataSaved || 'Saved!')).css('background-color', '#28a745');
                    setTimeout(function() {
                        $btn.prop('disabled', false).html(originalText).css('background-color', '');
                    }, 2000);
                } else {
                    $btn.html('❌ ' + (vehizoEditor.strings.saveError || 'Error!')).css('background-color', '#dc3545');
                    alert(message); // Show detailed error
                    setTimeout(function() {
                        $btn.prop('disabled', false).html(originalText).css('background-color', '');
                    }, 4000);
                }
            });
        });
    }

    /**
     * NEU: Speichert die Daten eines spezifischen Tabs via AJAX.
     */
    function saveTabData(tabType, callback) {
        const postId = vehizoEditor.postId || $('input[name="post_ID"]').val();
        if (!postId || postId === '0') {
            callback(false, 'Post ID not found. Please save the vehicle first.');
            return;
        }

        const postData = {
            action: 'vehizo_save_tab_data',
            nonce: vehizoEditor.nonce,
            post_id: postId,
            tab_type: tabType,
            vehicle_data: {},
        };

        if (tabType === 'basic' || tabType === 'technical') {
            const tabSelector = '#tab-' + tabType;
            $(tabSelector + ' input, ' + tabSelector + ' select').each(function() {
                const $field = $(this);
                const name = $field.attr('name');
                if (name) {
                    postData.vehicle_data[name] = $field.val();
                }
            });
            // Also include the main title for the basic tab
            if (tabType === 'basic') {
                postData.vehicle_data['post_title'] = $('#vehizo-title').val();
            }
        } else if (tabType === 'description') {
            // Handle TinyMCE editor for description
            if (typeof tinymce !== 'undefined' && tinymce.get('post_content')) {
                postData.post_content = tinymce.get('post_content').getContent();
            } else {
                postData.post_content = $('#post_content').val();
            }
        } else if (tabType === 'features') {
            // Handle features textarea
            const featuresValue = $('#vehizo-features-input').val();
            if (featuresValue !== undefined) {
                postData.features = featuresValue;
            }
        }

        $.ajax({
            url: vehizoEditor.ajaxUrl,
            type: 'POST',
            data: postData,
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    callback(true, response.data.message);
                } else {
                    callback(false, response.data.message || 'Unknown error');
                }
            },
            error: function(xhr) {
                callback(false, 'AJAX request failed. Status: ' + xhr.status);
            }
        });
    }

    /**
     * Utility function: Debounce
     */
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = function() {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    // Initialize when document is ready
    $(document).ready(function() {
        if ($('#vehizo_modern_editor').length > 0) {
            initVehizoEditor();
            setupNavigationWarning();
            setupKeyboardShortcuts();
        }
    });

    // Expose public API
    window.VehizoEditor = {
        updateSaveStatus: updateSaveStatus,
        markFormDirty: markFormDirty,
        getCurrentGalleryIds: getCurrentGalleryIds,
        updateGalleryIds: updateGalleryIds,
        addImageToGallery: addImageToGallery
    };

})(jQuery);
