/**
 * Vehizo Admin Settings JavaScript
 * 
 * @package Vehizo
 * @version 4.1.0
 */

(function($) {
    'use strict';

    // Initialize when DOM is ready
    $(document).ready(function() {
        VehizoAdminSettings.init();
    });

    /**
     * Main Admin Settings Object
     */
    var VehizoAdminSettings = {

        /**
         * Initialize all functionality
         */
        init: function() {
            this.initSliders();
            this.initColorPickers();
            this.initImportExport();
            this.initResetSettings();
            this.initFormValidation();
            this.initLivePreview();
        },

        /**
         * Initialize slider functionality
         */
        initSliders: function() {
            $('.vehizo-slider').each(function() {
                var $slider = $(this);
                var $valueDisplay = $slider.siblings('.vehizo-slider-value');
                var unit = $slider.data('unit') || 'px';

                // Set initial value
                $valueDisplay.text($slider.val() + unit);

                // Update value display on input
                $slider.on('input', function() {
                    var value = $(this).val();
                    $valueDisplay.text(value + unit);
                    
                    // Trigger change event for live preview
                    $(this).trigger('vehizo-setting-changed', {
                        setting: $(this).attr('name'),
                        value: value,
                        unit: unit
                    });
                });
            });
        },

        /**
         * Initialize color picker functionality
         */
        initColorPickers: function() {
            $('.vehizo-color-field').each(function() {
                var $colorField = $(this);
                
                // Add color preview if it doesn't exist
                if (!$colorField.siblings('.vehizo-color-preview').length) {
                    $colorField.after('<span class="vehizo-color-preview"></span>');
                }
                
                var $preview = $colorField.siblings('.vehizo-color-preview');
                $preview.css('background-color', $colorField.val());

                // Update preview on color change
                $colorField.on('input change', function() {
                    var color = $(this).val();
                    $preview.css('background-color', color);
                    
                    // Trigger change event for live preview
                    $(this).trigger('vehizo-setting-changed', {
                        setting: $(this).attr('name'),
                        value: color
                    });
                });
            });
        },

        /**
         * Initialize import/export functionality
         */
        initImportExport: function() {
            var self = this;

            // Export button
            $(document).on('click', '.vehizo-export-button', function(e) {
                e.preventDefault();
                self.exportSettings();
            });

            // Import button
            $(document).on('click', '.vehizo-import-button', function(e) {
                e.preventDefault();
                self.showImportModal();
            });

            // Handle import modal
            $(document).on('click', '.vehizo-import-confirm', function() {
                var settingsData = $('#vehizo-import-data').val();
                if (settingsData.trim()) {
                    self.importSettings(settingsData);
                } else {
                    self.showMessage(vehizo_admin.strings.invalid_file, 'error');
                }
            });

            // Close modal events
            $(document).on('click', '.vehizo-modal-close, .vehizo-modal-cancel', function() {
                $('.vehizo-modal').fadeOut();
            });

            // Close modal on background click
            $(document).on('click', '.vehizo-modal', function(e) {
                if (e.target === this) {
                    $(this).fadeOut();
                }
            });
        },

        /**
         * Initialize reset settings functionality
         */
        initResetSettings: function() {
            var self = this;

            $(document).on('click', '.vehizo-reset-button', function(e) {
                e.preventDefault();
                
                if (confirm(vehizo_admin.strings.confirm_reset)) {
                    self.resetSettings();
                }
            });
        },

        /**
         * Initialize form validation
         */
        initFormValidation: function() {
            // Validate color fields
            $('.vehizo-color-field').on('blur', function() {
                var color = $(this).val();
                if (color && !color.match(/^#[0-9A-F]{6}$/i)) {
                    $(this).addClass('error');
                    this.setCustomValidity('Please enter a valid hex color code');
                } else {
                    $(this).removeClass('error');
                    this.setCustomValidity('');
                }
            });

            // Validate number fields
            $('.vehizo-slider, input[type="number"]').on('change', function() {
                var $input = $(this);
                var min = parseInt($input.attr('min')) || 0;
                var max = parseInt($input.attr('max')) || 100;
                var value = parseInt($input.val());

                if (value < min) {
                    $input.val(min);
                } else if (value > max) {
                    $input.val(max);
                }
            });
        },

        /**
         * Initialize live preview functionality
         */
        initLivePreview: function() {
            // Listen for setting changes
            $(document).on('vehizo-setting-changed', function(e, data) {
                VehizoAdminSettings.updatePreview(data);
            });

            // Update preview for select fields
            $('.vehizo-select-field').on('change', function() {
                $(this).trigger('vehizo-setting-changed', {
                    setting: $(this).attr('name'),
                    value: $(this).val()
                });
            });
        },

        /**
         * Update live preview
         */
        updatePreview: function(data) {
            // This would update a preview area if one exists
            // Add visual feedback
            var $field = $('[name="' + data.setting + '"]');
            $field.closest('.vehizo-field-wrapper').addClass('changed');
            
            setTimeout(function() {
                $field.closest('.vehizo-field-wrapper').removeClass('changed');
            }, 1000);
        },

        /**
         * Export settings
         */
        exportSettings: function() {
            var $button = $('.vehizo-export-button');
            var originalText = $button.text();
            
            $button.addClass('vehizo-loading').prop('disabled', true);

            $.ajax({
                url: vehizo_admin.ajax_url,
                method: 'POST',
                data: {
                    action: 'vehizo_export_appearance_settings',
                    nonce: vehizo_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        VehizoAdminSettings.downloadJSON(response.data.data, 'vehizo-settings-export.json');
                        VehizoAdminSettings.showMessage(vehizo_admin.strings.export_success, 'success');
                    } else {
                        VehizoAdminSettings.showMessage(response.data.message || 'Export failed', 'error');
                    }
                },
                error: function() {
                    VehizoAdminSettings.showMessage('Network error during export', 'error');
                },
                complete: function() {
                    $button.removeClass('vehizo-loading').prop('disabled', false);
                }
            });
        },

        /**
         * Show import modal
         */
        showImportModal: function() {
            var modalHtml = `
                <div class="vehizo-modal" id="vehizo-import-modal">
                    <div class="vehizo-modal-content">
                        <div class="vehizo-modal-header">
                            <h3 class="vehizo-modal-title">${this.escapeHtml('Import Settings')}</h3>
                            <button class="vehizo-modal-close">&times;</button>
                        </div>
                        <div class="vehizo-modal-body">
                            <p>Paste your exported settings JSON data below:</p>
                            <textarea id="vehizo-import-data" placeholder="Paste JSON data here..." rows="10"></textarea>
                        </div>
                        <div class="vehizo-modal-actions">
                            <button class="button vehizo-modal-cancel">Cancel</button>
                            <button class="button button-primary vehizo-import-confirm">Import Settings</button>
                        </div>
                    </div>
                </div>
            `;

            // Remove existing modal
            $('#vehizo-import-modal').remove();
            
            // Add new modal
            $('body').append(modalHtml);
            $('#vehizo-import-modal').fadeIn();
        },

        /**
         * Import settings
         */
        importSettings: function(settingsData) {
            var $button = $('.vehizo-import-confirm');
            var originalText = $button.text();
            
            $button.addClass('vehizo-loading').prop('disabled', true);

            $.ajax({
                url: vehizo_admin.ajax_url,
                method: 'POST',
                data: {
                    action: 'vehizo_import_appearance_settings',
                    nonce: vehizo_admin.nonce,
                    settings_data: settingsData
                },
                success: function(response) {
                    if (response.success) {
                        VehizoAdminSettings.showMessage(vehizo_admin.strings.import_success, 'success');
                        $('.vehizo-modal').fadeOut();
                        
                        // Reload page to show new settings
                        setTimeout(function() {
                            window.location.reload();
                        }, 1500);
                    } else {
                        VehizoAdminSettings.showMessage(response.data.message || vehizo_admin.strings.import_error, 'error');
                    }
                },
                error: function() {
                    VehizoAdminSettings.showMessage('Network error during import', 'error');
                },
                complete: function() {
                    $button.removeClass('vehizo-loading').prop('disabled', false);
                }
            });
        },

        /**
         * Reset settings to defaults
         */
        resetSettings: function() {
            var $button = $('.vehizo-reset-button');
            var originalText = $button.text();
            
            $button.addClass('vehizo-loading').prop('disabled', true);

            $.ajax({
                url: vehizo_admin.ajax_url,
                method: 'POST',
                data: {
                    action: 'vehizo_reset_appearance_settings',
                    nonce: vehizo_admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        VehizoAdminSettings.showMessage(vehizo_admin.strings.reset_success, 'success');
                        
                        // Reload page to show reset settings
                        setTimeout(function() {
                            window.location.reload();
                        }, 1500);
                    } else {
                        VehizoAdminSettings.showMessage(response.data.message || vehizo_admin.strings.reset_error, 'error');
                    }
                },
                error: function() {
                    VehizoAdminSettings.showMessage('Network error during reset', 'error');
                },
                complete: function() {
                    $button.removeClass('vehizo-loading').prop('disabled', false);
                }
            });
        },

        /**
         * Download JSON data as file
         */
        downloadJSON: function(data, filename) {
            var blob = new Blob([data], { type: 'application/json' });
            var url = window.URL.createObjectURL(blob);
            var a = document.createElement('a');
            a.href = url;
            a.download = filename;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            window.URL.revokeObjectURL(url);
        },

        /**
         * Show message to user
         */
        showMessage: function(message, type) {
            type = type || 'info';
            
            var messageHtml = '<div class="vehizo-message ' + this.escapeHtml(type) + '">' + this.escapeHtml(message) + '</div>';
            
            // Remove existing messages
            $('.vehizo-message').fadeOut(function() {
                $(this).remove();
            });
            
            // Add new message
            $('.vehizo-settings-actions').after(messageHtml);
            
            // Auto-hide after 5 seconds
            setTimeout(function() {
                $('.vehizo-message').fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);
        },

        /**
         * Escape HTML for security
         */
        escapeHtml: function(text) {
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.toString().replace(/[&<>"']/g, function(m) { return map[m]; });
        },

        /**
         * Debounce function for performance
         */
        debounce: function(func, wait, immediate) {
            var timeout;
            return function() {
                var context = this, args = arguments;
                var later = function() {
                    timeout = null;
                    if (!immediate) func.apply(context, args);
                };
                var callNow = immediate && !timeout;
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
                if (callNow) func.apply(context, args);
            };
        }
    };

    // Make VehizoAdminSettings available globally
    window.VehizoAdminSettings = VehizoAdminSettings;

})(jQuery);