<?php
/*
Plugin Name: Globe Visitors Counter (Lite)
Plugin URI: https://vectormap.net/product-category/globe_visual_visitors/
Description: 3D globe visitors counter with Land Cover skin; shortcode + visit logging.
Version: 1.5.1
Author: VectorMap
Author URI: https://vectormap.net/
Text Domain: vectormap-visitors-globe-lite
Domain Path: /languages
Requires at least: 5.4
Tested up to: 6.8
Requires PHP: 7.2
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/old-licenses/gpl-2.0.html
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'VGVL_VER', '1.5.1' );
define( 'VGVL_FILE', __FILE__ );
define( 'VGVL_DIR', plugin_dir_path( __FILE__ ) );
define( 'VGVL_URL', plugin_dir_url( __FILE__ ) );

/**
 * При активации создаём папку uploads/visitors-lite-landcover.
 */
register_activation_hook( __FILE__, 'vgvl_activate' );
function vgvl_activate() {
    $upload = wp_upload_dir();
    $dir    = trailingslashit( $upload['basedir'] ) . 'visitors-lite-landcover/';

    if ( ! is_dir( $dir ) ) {
        wp_mkdir_p( $dir );
    }
}

/**
 * Загрузка текстдомена.
 */
add_action(
    'plugins_loaded',
    function() {
        load_plugin_textdomain(
            'vectormap-visitors-globe-lite',
            false,
            dirname( plugin_basename( __FILE__ ) ) . '/languages'
        );
    }
);

/**
 * Ссылки "Settings" и доп. ссылки на странице плагинов.
 */
if ( is_admin() ) {
    add_filter(
        'plugin_action_links_' . plugin_basename( __FILE__ ),
        function( $links ) {
            $settings_link = '<a href="' . esc_url( admin_url( 'options-general.php?page=vgvl-lite' ) ) . '">' . esc_html__( 'Settings', 'vectormap-visitors-globe-lite' ) . '</a>';
            array_unshift( $links, $settings_link );
            return $links;
        }
    );

    add_filter(
        'plugin_row_meta',
        function( $links, $file ) {
            if ( $file === plugin_basename( __FILE__ ) ) {
                $links[] = sprintf(
                    '<a href="%s" target="_blank" rel="noopener">%s</a>',
                    esc_url( 'https://youtu.be/1UniARlkZ6c' ),
                    esc_html__( 'Lite install video', 'vectormap-visitors-globe-lite' )
                );
                $links[] = sprintf(
                    '<a href="%s" target="_blank" rel="noopener">%s</a>',
                    esc_url( 'https://youtu.be/FYQvHUvIjYw' ),
                    esc_html__( 'Premium overview', 'vectormap-visitors-globe-lite' )
                );
                $links[] = sprintf(
                    '<a href="%s" target="_blank" rel="noopener"><strong>%s</strong></a>',
                    esc_url( 'https://vectormap.net/product-category/globe_visual_visitors/' ),
                    esc_html__( 'Get Full (Premium)', 'vectormap-visitors-globe-lite' )
                );
                // Жёлтые звёздочки с ссылкой на отзывы.
                $links[] = sprintf(
                    '<a href="%s" target="_blank" rel="noopener" title="%s"><span style="color:#ffcc00;">★★★★★</span></a>',
                    esc_url( 'https://wordpress.org/support/plugin/vectormap-visitors-globe-lite/reviews/#new-post' ),
                    esc_attr__( 'Rate Globe Visitors Counter (Lite) on WordPress.org', 'vectormap-visitors-globe-lite' )
                );
            }
            return $links;
        },
        10,
        2
    );
}

/**
 * Copy viewer files from assets/seed into uploads (Lite).
 * Files are not overwritten if already present.
 */
function vgvl_seed_viewer_files() {
    $src = trailingslashit( VGVL_DIR . 'assets/seed' );
    if ( ! is_dir( $src ) ) {
        return;
    }

    $upload = wp_upload_dir();

    $dst_lc = trailingslashit( $upload['basedir'] ) . 'visitors-lite-landcover/';
    if ( ! is_dir( $dst_lc ) ) {
        wp_mkdir_p( $dst_lc );
    }

    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator( $src, RecursiveDirectoryIterator::SKIP_DOTS ),
        RecursiveIteratorIterator::SELF_FIRST
    );

    foreach ( $iterator as $item ) {
        $rel_path = substr( $item->getPathname(), strlen( $src ) );
        $target   = $dst_lc . ltrim( $rel_path, '/\\' );

        if ( $item->isDir() ) {
            if ( ! is_dir( $target ) ) {
                wp_mkdir_p( $target );
            }
        } else {
            if ( ! file_exists( $target ) ) {
                if ( ! is_dir( dirname( $target ) ) ) {
                    wp_mkdir_p( dirname( $target ) );
                }
                @copy( $item->getPathname(), $target );
            }
        }
    }
}

/**
 * Шорткод [visitors_globe_lite].
 */
add_shortcode( 'visitors_globe_lite', 'vgvl_shortcode' );

function vgvl_shortcode( $atts ) {
    $atts = shortcode_atts(
        array(
            'title'  => 'VISITORS',
            'width'  => '100%',
            'height' => '600',
            'skin'   => '', // kept for backward compatibility, ignored
        ),
        $atts,
        'visitors_globe_lite'
    );

    $title  = sanitize_text_field( $atts['title'] );
    $width  = preg_replace( '/[^0-9.%]/', '', (string) $atts['width'] );
    $height = absint( $atts['height'] );
    $skin   = 'landcover';

    if ( $height <= 0 ) {
        $height = 600;
    }
    if ( '' === $width ) {
        $width = '100%';
    }

    // Убеждаемся, что viewer-файлы есть в uploads.
    vgvl_seed_viewer_files();

    $upload = wp_upload_dir();
    if ( empty( $upload['baseurl'] ) ) {
        return '';
    }

    $baseurl = trailingslashit( $upload['baseurl'] );
    $iframe  = $baseurl . 'visitors-lite-landcover/index.html';

    // Лёгкий JS для AJAX-логгера (как раньше).
    wp_enqueue_script(
        'vgvl-lite',
        VGVL_URL . 'assets/js/visitors-lite.js',
        array( 'jquery' ),
        VGVL_VER,
        true
    );

    wp_localize_script(
        'vgvl-lite',
        'VGVL',
        array(
            'ajaxUrl' => esc_url_raw( admin_url( 'admin-ajax.php' ) ),
            'nonce'   => wp_create_nonce( 'vgvl-lite' ),
        )
    );

    ob_start();
    ?>
    <div class="vgvl-wrap" style="border-radius:12px;overflow:hidden;box-shadow:0 10px 30px rgba(0,0,0,.15);background:#000;">
        <div class="vgvl-title" style="padding:8px 12px;border-bottom:1px solid rgba(255,255,255,.15);color:#fff;font-size:13px;font-weight:600;font-family:-apple-system,BlinkMacSystemFont,Segoe UI,Roboto,Arial;letter-spacing:.02em;">
            <?php
            echo esc_html( $title );
            echo ' · ' . esc_html__( 'Land Cover', 'vectormap-visitors-globe-lite' );
            ?>
        </div>
        <iframe
            src="<?php echo esc_url( $iframe ); ?>"
            style="display:block;width:<?php echo esc_attr( (string) $width ); ?>;height:<?php echo esc_attr( (string) $height ); ?>px;border:0;max-width:100%;"
            loading="lazy"
            referrerpolicy="no-referrer"
            sandbox="allow-scripts allow-same-origin allow-popups"
        ></iframe>
    </div>
    <?php
    return ob_get_clean();
}

/**
 * AJAX-логгер (опционально; пишет в uploads/visitors-lite-landcover/visitors.json).
 */
add_action( 'wp_ajax_vgvl_hit', 'vgvl_ajax_hit' );
add_action( 'wp_ajax_nopriv_vgvl_hit', 'vgvl_ajax_hit' );

function vgvl_ajax_hit() {
    check_ajax_referer( 'vgvl-lite', 'nonce' );

    $upload = wp_upload_dir();
    $dir    = trailingslashit( $upload['basedir'] ) . 'visitors-lite-landcover/';
    $file   = $dir . 'visitors.json';

    if ( ! is_dir( $dir ) ) {
        wp_mkdir_p( $dir );
    }
    if ( ! file_exists( $file ) ) {
        @file_put_contents( $file, '[]', LOCK_EX );
    }

    $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
    $ua = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
    $ts = time();

    $json = @file_get_contents( $file );
    $list = array();

    if ( is_string( $json ) && '' !== $json ) {
        $tmp = json_decode( $json, true );
        if ( is_array( $tmp ) ) {
            $list = $tmp;
        }
    }

    $list[] = array(
        'ip'        => $ip,
        'ua'        => $ua,
        'timestamp' => $ts,
    );

    $ok = @file_put_contents(
        $file,
        wp_json_encode( $list, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ),
        LOCK_EX
    );

    if ( ! $ok ) {
        wp_send_json_error( array( 'saved' => false ) );
    }

    wp_send_json_success( array( 'saved' => true ) );
}

/**
 * Страница настроек.
 */
add_action( 'admin_menu', 'vgvl_add_options_page' );

function vgvl_add_options_page() {
    add_options_page(
        __( 'Globe Visitors Counter (Lite)', 'vectormap-visitors-globe-lite' ),
        __( 'Globe Visitors Counter (Lite)', 'vectormap-visitors-globe-lite' ),
        'manage_options',
        'vgvl-lite',
        'vgvl_render_settings_page'
    );
}

function vgvl_render_settings_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'Globe Visitors Counter (Lite) v1.5.1', 'vectormap-visitors-globe-lite' ); ?></h1>

        <!-- Tester Program notice -->
        <div class="notice notice-info" style="margin-top:8px;">
            <p>
                <strong><?php esc_html_e( 'Globe Visitors Counter — Tester Program', 'vectormap-visitors-globe-lite' ); ?></strong><br />
                <?php esc_html_e( 'We are polishing defaults and performance based on real-world sites. If you share brief feedback from your setup, we can enable some advanced features during testing at no cost.', 'vectormap-visitors-globe-lite' ); ?>
            </p>
            <p>
                <a class="button button-primary" href="<?php echo esc_url( 'https://wordpress.org/support/topic/looking-for-testers-free-advanced-features-for-feedback/' ); ?>" target="_blank" rel="noopener">
                    <?php esc_html_e( 'Open Support thread', 'vectormap-visitors-globe-lite' ); ?>
                </a>
                <a class="button" href="<?php echo esc_url( 'https://vectormap.net/contacts/' ); ?>" target="_blank" rel="noopener">
                    <?php esc_html_e( 'Contact (Tester Program)', 'vectormap-visitors-globe-lite' ); ?>
                </a>
            </p>
        </div>

        <!-- Upgrade teaser -->
        <div class="notice notice-info" style="margin-top:12px;display:flex;align-items:center;gap:12px;flex-wrap:wrap;">
            <span style="font-weight:600;">
                <?php esc_html_e( 'Need automatic header injection, advanced logging & fine-tuning (all pages)?', 'vectormap-visitors-globe-lite' ); ?>
            </span>
            <a class="button button-primary" href="<?php echo esc_url( 'https://vectormap.net/visitors/index.html' ); ?>" target="_blank" rel="noopener">
                <?php esc_html_e( 'See Premium versions in action', 'vectormap-visitors-globe-lite' ); ?>
            </a>
        </div>

        <h2 style="margin-top:24px;"><?php esc_html_e( 'Usage', 'vectormap-visitors-globe-lite' ); ?></h2>
        <p><?php esc_html_e( '1. Create or edit any page or post where you want to display the globe.', 'vectormap-visitors-globe-lite' ); ?></p>
        <p><?php esc_html_e( '2. Insert the shortcode below into the content editor (Classic or Block editor).', 'vectormap-visitors-globe-lite' ); ?></p>
        <p><?php esc_html_e( '3. Update the page and open it on the front-end — the 3D globe should appear inside the content area.', 'vectormap-visitors-globe-lite' ); ?></p>

        <p><strong><?php esc_html_e( 'Shortcode:', 'vectormap-visitors-globe-lite' ); ?></strong></p>
        <p><code>[visitors_globe_lite title="VISITORS" width="100%" height="600"]</code></p>

        <p><?php esc_html_e( 'You can adjust the title and the height (in pixels). The width can be a percentage or a fixed value (for example: 100% or 600px).', 'vectormap-visitors-globe-lite' ); ?></p>

        <h3 style="margin-top:32px;"><?php esc_html_e( 'Upgrade to Premium', 'vectormap-visitors-globe-lite' ); ?></h3>
        <p>
            <a class="button button-primary" href="<?php echo esc_url( 'https://vectormap.net/product-category/globe_visual_visitors/' ); ?>" target="_blank" rel="noopener">
                <?php esc_html_e( 'See Premium versions on VectorMap.net', 'vectormap-visitors-globe-lite' ); ?>
            </a>
        </p>

        <h3 style="margin-top:24px;"><?php esc_html_e( 'Rate &amp; feedback', 'vectormap-visitors-globe-lite' ); ?></h3>
        <p>
            <span style="color:#ffcc00;font-size:18px;vertical-align:middle;margin-right:6px;">★★★★★</span>
            <?php esc_html_e( 'If Globe Visitors Counter (Lite) works well for you, please consider leaving a rating or a short review — it really helps.', 'vectormap-visitors-globe-lite' ); ?>
        </p>
        <p>
            <a href="<?php echo esc_url( 'https://wordpress.org/support/plugin/vectormap-visitors-globe-lite/reviews/#new-post' ); ?>" target="_blank" rel="noopener">
                <?php esc_html_e( 'Leave a rating or review on WordPress.org', 'vectormap-visitors-globe-lite' ); ?>
                <span style="color:#ffcc00;font-size:18px;vertical-align:middle;margin-left:6px;">★★★★★</span>
            </a>
        </p>
    </div>
    <?php
}

/**
 * Тестер-программа (админ-нотис на дашборде).
 */
add_action(
    'admin_notices',
    function() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $screen = get_current_screen();
        if ( ! $screen || 'dashboard' !== $screen->id ) {
            return;
        }

        if ( isset( $_GET['gvc_tester_dismiss'] ) && check_admin_referer( 'gvc_tester_dismiss' ) ) {
            update_option( 'gvc_tester_dismissed', 1 );
            return;
        }

        if ( get_option( 'gvc_tester_dismissed' ) ) {
            return;
        }

        $dismiss_url = wp_nonce_url(
            add_query_arg( 'gvc_tester_dismiss', '1' ),
            'gvc_tester_dismiss'
        );
        $support_url = 'https://wordpress.org/support/topic/looking-for-testers-free-advanced-features-for-feedback/';
        $contact_url = 'https://vectormap.net/contacts/';

        echo '<div class="notice notice-info is-dismissible" style="border-left:4px solid #2271b1">';
        echo '<p><strong>Globe Visitors Counter — Tester Program</strong>: help us polish defaults &amp; performance. ';
        echo 'If you share brief feedback from your setup, we can tweak defaults for real-world sites and keep some advanced options free for testers. ';
        echo 'Early testers help us a lot, so we\'re happy to enable <em>advanced features during testing at no cost</em>.</p>';
        echo '<p>';
        echo '<a class="button button-primary" href="' . esc_url( $support_url ) . '" target="_blank" rel="noopener">Open Support thread</a> ';
        echo '<a class="button" href="' . esc_url( $contact_url ) . '" target="_blank" rel="noopener">Contact (Tester Program)</a> ';
        echo '<a class="button-link" href="' . esc_url( $dismiss_url ) . '">Dismiss</a>';
        echo '</p></div>';
    }
);
