<?php
// vis_json_writer.php (tuned)
// Safe JSON helpers + append with atomic writes and BACKUP ROTATION.
//
// Defaults (can be edited):
if (!defined('VG_BAK_INTERVAL_SEC')) define('VG_BAK_INTERVAL_SEC', 21600); // every 6 hours (set 0 to disable)
if (!defined('VG_BAK_KEEP')) define('VG_BAK_KEEP', 8); // keep last 8 backups
if (!defined('VG_MAX_POINTS')) define('VG_MAX_POINTS', 10000); // tail length for visitors.json

function vg_vis_safe_json_encode($data){
    return json_encode($data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
}

function vg_vis_atomic_write($path, $content){
    $dir = dirname($path);
    if (!is_dir($dir)) @mkdir($dir, 0755, true);
    $tmp = $path.'.tmp.'.bin2hex(random_bytes(4));
    if (file_put_contents($tmp, $content, LOCK_EX) === false) return false;
    return @rename($tmp, $path);
}

function vg_vis_read_existing($path){
    if (!is_file($path)) return [];
    $raw = @file_get_contents($path);
    $arr = json_decode($raw, true);
    return is_array($arr) ? $arr : [];
}

function vg_vis_rotate_backups($path, $interval_sec=0, $keep=5){
    if ($interval_sec <= 0) return; // disabled
    $dir = dirname($path);
    $name = basename($path); // visitors.json
    $pattern = $dir.'/.'.$name.'.bak.*'; // .visitors.json.bak.YYYYMMDD_HHMMSS
    $list = glob($pattern);
    usort($list, function($a,$b){ return strcmp($a,$b); }); // by name (timestamp in name)
    $now = time();
    // create new if last older than interval
    $last = end($list);
    $need = false;
    if (!$last) { $need = true; }
    else {
        // parse timestamp from ...bak.YYYYMMDD_HHMMSS
        if (preg_match('~\.bak\.(\d{8})_(\d{6})$~', $last, $m)){
            $ts = strtotime($m[1].' '.$m[2].' UTC');
            if (!$ts || ($now - $ts) >= $interval_sec) $need = true;
        } else {
            $need = true;
        }
    }
    if ($need && is_file($path)){
        $stamp = gmdate('Ymd_His');
        @copy($path, $dir.'/.'.$name.'.bak.'.$stamp);
        $list = glob($pattern);
        usort($list, function($a,$b){ return strcmp($a,$b); });
        // trim old
        if ($keep > 0 && count($list) > $keep){
            $toDel = array_slice($list, 0, count($list)-$keep);
            foreach ($toDel as $f){ @unlink($f); }
        }
    }
}

function vg_vis_write_json_append($json_path, $row, $max_points=null){
    if ($max_points === null) $max_points = (defined('VG_MAX_POINTS') ? VG_MAX_POINTS : 10000);
    // Rotate backups on schedule
    vg_vis_rotate_backups($json_path, (defined('VG_BAK_INTERVAL_SEC')?VG_BAK_INTERVAL_SEC:0), (defined('VG_BAK_KEEP')?VG_BAK_KEEP:5));

    $arr = vg_vis_read_existing($json_path);
    $arr[] = $row;
    if ($max_points > 0 && count($arr) > $max_points){
        $arr = array_slice($arr, -$max_points);
    }
    return vg_vis_atomic_write($json_path, vg_vis_safe_json_encode($arr));
}
