<?php
// save_visitor.php (hardened v4, compatible with server-side IP geolocation)
require_once __DIR__ . '/vis_json_writer.php';
date_default_timezone_set('UTC');
header('Content-Type: application/json; charset=utf-8');

// --- helpers ---
function getUserIP() {
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) return $_SERVER['HTTP_CLIENT_IP'];
    if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ipList = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
        return trim($ipList[0]);
    }
    return $_SERVER['REMOTE_ADDR'] ?? '';
}
function _gv($keys){
    foreach ($keys as $k){
        if (isset($_POST[$k]) && $_POST[$k] !== '') return $_POST[$k];
    }
    return null;
}

// 1) try to get lat/lon from POST; else fallback to ip-api.com by client IP
$lat = _gv(['lat','latitude','y','y_coord','lat_dd']);
$lng = _gv(['lng','lon','longitude','x','x_coord','lon_dd']);
if ($lat !== null) $lat = floatval($lat);
if ($lng !== null) $lng = floatval($lng);

if ($lat === null || $lng === null){
    $ip = getUserIP();
    if ($ip){
        $geoJson = @file_get_contents("http://ip-api.com/json/{$ip}?fields=status,message,country,regionName,city,lat,lon,query");
        $geo = $geoJson ? json_decode($geoJson, true) : null;
        if (is_array($geo) && isset($geo['status']) && $geo['status']==='success'){
            $lat = floatval($geo['lat']);
            $lng = floatval($geo['lon']);
        }
    }
}

if ($lat === null || $lng === null){
    http_response_code(400);
    echo json_encode(['status'=>'fail','error'=>'lat/lng unavailable']);
    exit;
}

$ts = _gv(['timestamp','time','ts']);
$ts = ($ts!==null) ? intval($ts) : time();

// 2) paths (for this distributive visitors.json is in the same folder)
$json_main   = __DIR__ . '/visitors.json';
$json_active = __DIR__ . '/visitors_active.json';
$counterFile = __DIR__ . '/counter.txt';

// 3) load existing data
$existing = vg_vis_read_existing($json_main);

// 4) compose row (preserve original POST + standardized fields)
$row = $_POST;
$row['ip'] = $row['ip'] ?? (getUserIP());
$row['lat'] = $lat;
$row['lng'] = $lng;
$row['lon'] = $lng; // compatibility with frontend check v.lon
$row['latitude'] = $lat;
$row['longitude'] = $lng;
$row['timestamp'] = $ts;
$row['country'] = $row['country'] ?? ($geo['country'] ?? null);
$row['region']  = $row['region']  ?? ($geo['regionName'] ?? null);
$row['city']    = $row['city']    ?? ($geo['city'] ?? null);
$row['ua'] = $row['ua'] ?? ($_SERVER['HTTP_USER_AGENT'] ?? '');

// 5) write main visitors.json (tail 10k)
$existing[] = $row;
if (count($existing) > 10000) $existing = array_slice($existing, -10000);
$ok1 = vg_vis_atomic_write($json_main, vg_vis_safe_json_encode($existing));

// 6) update visitors_active.json (15 min window, tail 1000)
$active = vg_vis_read_existing($json_active);
$now = time(); $window = 15*60;
$active[] = $row;
$filtered = [];
foreach ($active as $a){
    $t = isset($a['timestamp']) ? intval($a['timestamp']) : $now;
    if ($now - $t <= $window) $filtered[] = $a;
}
if (count($filtered) > 1000) $filtered = array_slice($filtered, -1000);
$ok2 = vg_vis_atomic_write($json_active, vg_vis_safe_json_encode($filtered));

// 7) eternal counter (counter.txt)
$total = 0;
$fp = @fopen($counterFile, 'c+');
if ($fp){
    if (@flock($fp, LOCK_EX)){
        rewind($fp);
        $prev = stream_get_contents($fp);
        $prev = ($prev === false || trim($prev) === '' || !ctype_digit(trim($prev))) ? (string)count($existing) : (string)((int)trim($prev) + 1);
        ftruncate($fp, 0);
        rewind($fp);
        fwrite($fp, $prev);
        fflush($fp);
        @flock($fp, LOCK_UN);
        $total = (int)$prev;
    }
    fclose($fp);
} else {
    $total = count($existing);
    @file_put_contents($counterFile, (string)$total, LOCK_EX);
}

// 8) response compatible with old frontend
if ($ok1 && $ok2){
    echo json_encode(['status'=>'ok', 'total'=>$total]);
} else {
    http_response_code(500);
    echo json_encode(['status'=>'fail', 'error'=>'write failed']);
}
