// Токен Mapbox
mapboxgl.accessToken = 'pk.eyJ1Ijoid2FyYW5kd29ybGQiLCJhIjoiY21lZDY1aXptMDY5NDJycTBnNW0zZ2g2aSJ9._BkcvscbDTEAfASuY8vZKQ';

// Настройки
const ACTIVE_WINDOW_SEC = 3600; // окно "активных" посетителей = 1 час

// Создаем карту
const map = new mapboxgl.Map({
    container: 'map',
    style: 'mapbox://styles/mapbox/satellite-streets-v12',
    zoom: 2,
    center: [20, 30],
    projection: 'globe'
});

map.on('style.load', () => {
    map.setFog({});

    // Пульсирующая точка для активных посетителей
    map.addImage('active-visitor-dot', createPulsingDot('245, 62, 24'), { pixelRatio: 2 });

    // Статичная точка для старых посетителей с черной обводкой и тенями
    map.addImage('old-visitor-dot', staticDotOld, { pixelRatio: 2 });

    // Источник с кластеризацией для старых посетителей
    map.addSource('oldVisitors', {
        type: 'geojson',
        data: { type: 'FeatureCollection', features: [] },
        cluster: true,
        clusterMaxZoom: 14,
        clusterRadius: 50
    });

    // Источник для активных посетителей (без кластеризации)
    map.addSource('activeVisitors', {
        type: 'geojson',
        data: { type: 'FeatureCollection', features: [] }
    });

    // Кластеры старых посетителей
    map.addLayer({
        id: 'clusters',
        type: 'circle',
        source: 'oldVisitors',
        filter: ['has', 'point_count'],
        paint: {
            'circle-color': [
                'step',
                ['get', 'point_count'],
                '#f9c6c9',      // <10 — светло-розовый
                10, '#f08a5d',  // 10-49 — оранжевый
                50, '#e63a17',  // 50-99 — насыщенный красный
                100, '#b32c07'  // >=100 — темно-красный
            ],
            'circle-radius': [
                'step',
                ['get', 'point_count'],
                10,      // <10 — радиус 10
                10, 15,  // 10-49 — радиус 15
                50, 20,  // 50-99 — радиус 20
                100, 25  // >=100 — радиус 25
            ],
            'circle-stroke-color': 'black',
            'circle-stroke-width': 1,
            'circle-blur': 0.2
        }
    });

    // Количество в кластерах
    map.addLayer({
        id: 'cluster-count',
        type: 'symbol',
        source: 'oldVisitors',
        filter: ['has', 'point_count'],
        layout: {
            'text-field': '{point_count_abbreviated}',
            'text-font': ['DIN Offc Pro Medium', 'Arial Unicode MS Bold'],
            'text-size': 14,
            'text-anchor': 'center',
            'text-allow-overlap': true
        },
        paint: {
            'text-color': 'white',
            'text-halo-color': 'black',
            'text-halo-width': 2
        }
    });

    // Одиночные старые точки
    map.addLayer({
        id: 'oldVisitors-unclustered',
        type: 'symbol',
        source: 'oldVisitors',
        filter: ['!', ['has', 'point_count']],
        layout: {
            'icon-image': 'old-visitor-dot',
            'icon-size': 1,
            'text-field': ['to-string', ['get', 'count']],
            'text-font': ['DIN Offc Pro Medium', 'Arial Unicode MS Bold'],
            'text-size': 14,
            'text-offset': [0, 0],
            'text-anchor': 'center',
            'text-allow-overlap': true
        },
        paint: {
            'text-color': 'white',
            'text-halo-color': 'black',
            'text-halo-width': 2
        }
    });

    // Активные посетители (пульсирующие точки)
    map.addLayer({
        id: 'activeVisitors',
        type: 'symbol',
        source: 'activeVisitors',
        layout: {
            'icon-image': 'active-visitor-dot',
            'icon-size': 0.7,
            'icon-allow-overlap': true
        },
        minzoom: 0,
        maxzoom: 24
    });

    // === ЛАКОНИЧНЫЙ POPUP (город, страна, регион) + ФОЛБЭК К БЛИЖАЙШЕМУ ГОРОДУ ===
    let VM_CITIES = null; // кэш городов
    function vmLoadCities() {
        if (!VM_CITIES) {
            // без начального слэша — важно для WordPress-подпапок
            VM_CITIES = fetch('cities_1000.json?t=' + Date.now())
                .then(r => r.json())
                .catch(() => []);
        }
        return VM_CITIES;
    }
    function vmDistKm(lat1, lon1, lat2, lon2) {
        const R = 6371, toRad = Math.PI/180;
        const dLat = (lat2 - lat1) * toRad, dLon = (lon2 - lon1) * toRad;
        const a = Math.sin(dLat/2)**2 + Math.cos(lat1*toRad)*Math.cos(lat2*toRad)*Math.sin(dLon/2)**2;
        return 2 * R * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
    }
    async function vmResolveLocation(props, lngLat) {
        const city    = (props.city    || props.town || props.village || '').trim();
        const region  = (props.region  || props.state || props.province || props.county || '').trim();
        const country = (props.country || props.country_name || props.cc || '').trim();
        if (city || region || country) return { city, region, country };

        // Фолбэк: ближайший город из cities_1000.json
        const cities = await vmLoadCities();
        if (!Array.isArray(cities) || !cities.length) return { city:'', region:'', country:'' };

        let best = null, bestD = Infinity;
        for (let i = 0; i < cities.length; i++) {
            const c = cities[i];
            const d = vmDistKm(lngLat.lat, lngLat.lng, c.lat, c.lon);
            if (d < bestD) { bestD = d; best = c; }
        }
        if (best && bestD <= 300) {
            return {
                city: (best.name || '').trim(),
                region: (best.region || best.admin || best.admin1 || '').trim(),
                country: (best.country || best.cc || '').trim()
            };
        }
        return { city:'', region:'', country:'' };
    }

    const popup = new mapboxgl.Popup({
        closeButton: false,
        closeOnClick: false,
        className: 'vm-active-popup', // стили — в index.html
        maxWidth: '280px'
    });

    function esc(s){
        return String(s||'').replace(/[&<>"']/g, m => (
          {'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[m]
        ));
    }
    function popupHTML(loc){
        const title = (loc.city && loc.country) ? `${loc.city}, ${loc.country}` :
                      (loc.city || loc.country || 'Unknown location');
        const sub   = loc.region ? `<div style="opacity:.8">${esc(loc.region)}</div>` : '';
        return `<div style="min-width:180px;line-height:1.25;color:#111">
                  <div style="font-weight:700">${esc(title)}</div>
                  ${sub}
                </div>`;
    }

    // Наведение — показать (с фолбэком)
    map.on('mouseenter','activeVisitors', async e => {
        map.getCanvas().style.cursor = 'pointer';
        const f = e.features && e.features[0]; if(!f) return;
        const loc = await vmResolveLocation(f.properties || {}, e.lngLat);
        popup.setLngLat(e.lngLat).setHTML(popupHTML(loc)).addTo(map);
    });
    // Движение — следовать
    map.on('mousemove','activeVisitors', e => {
        if(!e.features || !e.features[0]) return;
        popup.setLngLat(e.lngLat);
    });
    // Уход курсора — скрыть
    map.on('mouseleave','activeVisitors', () => {
        map.getCanvas().style.cursor = '';
        popup.remove();
    });
    // Клик/тап — зафиксировать
    async function openAt(e){
        const f = e.features && e.features[0]; if(!f) return;
        const loc = await vmResolveLocation(f.properties || {}, e.lngLat);
        popup.setLngLat(e.lngLat).setHTML(popupHTML(loc)).addTo(map);
    }
    map.on('click','activeVisitors', openAt);
    map.on('touchend','activeVisitors', openAt);
    // Клик по пустому месту — закрыть
    map.on('click', (e)=>{
        const feats = map.queryRenderedFeatures(e.point, { layers: ['activeVisitors'] });
        if(!feats.length) popup.remove();
    });

    // Загрузка данных и обновление слоев
    loadVisitors();
    setInterval(loadVisitors, 30000);
});

// Пульсирующая точка для активных посетителей
function createPulsingDot(color) {
    const size = 100;
    return {
        width: size,
        height: size,
        data: new Uint8Array(size * size * 4),
        onAdd: function () {
            const canvas = document.createElement('canvas');
            canvas.width = this.width;
            canvas.height = this.height;
            this.context = canvas.getContext('2d');
        },
        render: function () {
            const duration = 1000;
            const t = (performance.now() % duration) / duration;
            const radius = (size / 2) * 0.3;
            const outerRadius = (size / 2) * 0.7 * t + radius;
            const ctx = this.context;
            ctx.clearRect(0, 0, this.width, this.height);

            // Внешний круг (пульс)
            ctx.beginPath();
            ctx.arc(size / 2, size / 2, outerRadius, 0, Math.PI * 2);
            ctx.fillStyle = `rgba(${color}, ${1 - t})`;
            ctx.fill();

            // Внутренний круг
            ctx.beginPath();
            ctx.arc(size / 2, size / 2, radius, 0, Math.PI * 2);
            ctx.fillStyle = `rgba(${color}, 1)`;
            ctx.strokeStyle = 'white';
            ctx.lineWidth = 2 + 4 * (1 - t);
            ctx.fill();
            ctx.stroke();

            this.data = ctx.getImageData(0, 0, this.width, this.height).data;
            map.triggerRepaint();
            return true;
        }
    };
}

// Статичная точка для старых посетителей с черной обводкой и тенью
const staticDotOld = {
    width: 40,
    height: 40,
    data: new Uint8Array(40 * 40 * 4),
    onAdd: function () {
        const canvas = document.createElement('canvas');
        canvas.width = this.width;
        canvas.height = this.height;
        this.context = canvas.getContext('2d');
        this.drawStaticDot();
    },
    drawStaticDot: function () {
        const ctx = this.context;
        const size = this.width;
        ctx.clearRect(0, 0, size, size);

        // Тень
        ctx.shadowColor = 'rgba(0,0,0,0.7)';
        ctx.shadowBlur = 6;
        ctx.shadowOffsetX = 0;
        ctx.shadowOffsetY = 0;

        // Черная обводка
        ctx.beginPath();
        ctx.arc(size / 2, size / 2, size / 3, 0, Math.PI * 2);
        ctx.fillStyle = '#E63A17'; // цвет точки
        ctx.fill();
        ctx.lineWidth = 1;
        ctx.strokeStyle = 'black';
        ctx.stroke();

        this.data = ctx.getImageData(0, 0, size, size).data;
    },
    render: function () {
        return true; // не анимируется
    }
};

// Загрузка и разделение посетителей
function loadVisitors() {
    // берём visitors.json из той же папки, где index.html
    fetch('visitors.json?ts=' + Date.now(), { cache: 'no-store' })
        .then(res => res.json())
        .then(data => {
            const now = Date.now() / 1000;
            const activeFeatures = [];
            const oldFeaturesMap = new Map();

            (Array.isArray(data) ? data : []).forEach(v => {
                if (typeof v.lat !== 'number' || typeof v.lon !== 'number' || typeof v.timestamp !== 'number') return;

                const coords = [v.lon, v.lat];
                const key = coords.join(',');

                if (now - v.timestamp <= ACTIVE_WINDOW_SEC) {
                    activeFeatures.push({
                        type: 'Feature',
                        geometry: { type: 'Point', coordinates: coords },
                        properties: v.properties || {} // совместимость, если поля есть
                    });
                } else {
                    if (!oldFeaturesMap.has(key)) {
                        oldFeaturesMap.set(key, {
                            type: 'Feature',
                            geometry: { type: 'Point', coordinates: coords },
                            properties: { count: 0 }
                        });
                    }
                    oldFeaturesMap.get(key).properties.count++;
                }
            });

            const oldFeatures = Array.from(oldFeaturesMap.values());

            const activeGeoJSON = { type: 'FeatureCollection', features: activeFeatures };
            const oldGeoJSON = { type: 'FeatureCollection', features: oldFeatures };

            if (map.getSource('activeVisitors')) {
                map.getSource('activeVisitors').setData(activeGeoJSON);
            }
            if (map.getSource('oldVisitors')) {
                map.getSource('oldVisitors').setData(oldGeoJSON);
            }
        })
        .catch(console.error);
}

// ===== Доп. сигнатура/seed (как было в предыдущем билде; при желании можно убрать) =====
(function(){
    try{
        var sig="DE|2025-08-18|COLORS|S001"; // подпись под релиз
        var enc={'0':'\u200B','1':'\u200C','2':'\u200D','3':'\u2060'};
        var bytes=[...new TextEncoder().encode(sig)];
        var zw=bytes.map(x=>enc[(x>>6)&3]+enc[(x>>4)&3]+enc[(x>>2)&3]+enc[x&3]).join('');
        var meta=document.createElement('meta');
        meta.name='v-check';
        meta.content=zw;
        (document.head||document.documentElement).appendChild(meta);
    }catch(_){}
}());

const VSEED=(function(){
    const S=0xA17C3F2D;
    function mulberry32(a){
        return function(){
            a|=0;a=(a+0x6D2B79F5)|0;
            let t=Math.imul(a^a>>>15,1|a);
            t^=t+Math.imul(t^t>>>7,61|t);
            return((t^t>>>14)>>>0)/4294967296;
        };
    }
    return { rng: mulberry32(S), _sig: S };
}());
