<?php
/**
 * Public class.
 *
 * @package V7_Product_Alternatives
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class V7PA_Public {

	private $plugin_name;
	private $version;

	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	public function enqueue_styles() {
		if ( ! is_product() ) {
			return;
		}

		wp_enqueue_style(
			$this->plugin_name,
			V7PA_URL . 'public/css/public.css',
			array(),
			$this->version
		);

		$accent_color  = sanitize_hex_color( get_option( 'v7pa_accent_color', '#2271b1' ) );
		$text_color    = sanitize_hex_color( get_option( 'v7pa_text_color', '#333333' ) );
		$bg_color      = sanitize_hex_color( get_option( 'v7pa_bg_color', '#ffffff' ) );
		$border_radius = absint( get_option( 'v7pa_border_radius', '6' ) );

		$custom_css = "
			.v7-alt-card {
				background: {$accent_color};
				border-left-color: {$accent_color};
				border-radius: {$border_radius}px;
			}
			.v7-alt-name a:hover,
			.v7-alt-price {
				color: {$accent_color};
			}
			.v7-alt-info,
			.v7-alt-name a {
				color: {$text_color};
			}
			.v7-alt-cta {
				background: {$accent_color};
				border-radius: {$border_radius}px;
			}
			.v7-alt-cta:hover {
				background: {$accent_color};
				filter: brightness(0.85);
			}
		";
		wp_add_inline_style( $this->plugin_name, $custom_css );
	}

	public function display_alternative_block() {
		global $product;

		if ( ! $product ) {
			return;
		}

		if ( ! $this->is_product_out_of_stock( $product ) ) {
			return;
		}

		$alternative_id = get_post_meta( $product->get_id(), '_v7pa_alt_product', true );

		if ( empty( $alternative_id ) ) {
			return;
		}

		$alternative = wc_get_product( $alternative_id );

		if ( ! $alternative || 'publish' !== $alternative->get_status() || ! $alternative->is_in_stock() ) {
			return;
		}

		$title_text  = get_option( 'v7pa_title_text', __( 'We recommend a similar alternative:', 'v7-product-alternatives' ) );
		$button_text = get_option( 'v7pa_button_text', __( 'View Product', 'v7-product-alternatives' ) );
		$show_brand  = get_option( 'v7pa_show_brand', '1' );
		$show_price  = get_option( 'v7pa_show_price', '1' );

		$image = $alternative->get_image( 'woocommerce_thumbnail' );
		$title = $alternative->get_name();
		$price = $alternative->get_price_html();
		$link  = $alternative->get_permalink();
		$brand = $show_brand ? $this->get_product_brand( $alternative, $alternative_id ) : '';
		?>
		<div class="v7-alt-block">
			<p class="v7-alt-title"><?php echo esc_html( $title_text ); ?></p>
			<div class="v7-alt-card">
				<div class="v7-alt-image">
					<a href="<?php echo esc_url( $link ); ?>" aria-label="<?php echo esc_attr( $title ); ?>">
						<?php echo wp_kses_post( $image ); ?>
					</a>
				</div>
				<div class="v7-alt-info">
					<?php if ( $brand ) : ?>
						<span class="v7-alt-brand"><?php echo esc_html( $brand ); ?></span>
					<?php endif; ?>
					<h4 class="v7-alt-name">
						<a href="<?php echo esc_url( $link ); ?>"><?php echo esc_html( $title ); ?></a>
					</h4>
					<?php if ( $show_price ) : ?>
						<p class="v7-alt-price"><?php echo wp_kses_post( $price ); ?></p>
					<?php endif; ?>
					<a href="<?php echo esc_url( $link ); ?>" class="button v7-alt-cta">
						<?php echo esc_html( $button_text ); ?>
					</a>
				</div>
			</div>
		</div>
		<?php
	}

	private function is_product_out_of_stock( $product ) {
		if ( ! $product->is_type( 'variable' ) ) {
			return ! $product->is_in_stock();
		}

		$variations = $product->get_available_variations();
		foreach ( $variations as $variation ) {
			$var_product = wc_get_product( $variation['variation_id'] );
			if ( $var_product && $var_product->is_in_stock() ) {
				return false;
			}
		}
		return true;
	}

	private function get_product_brand( $product, $product_id ) {
		$brand = $product->get_attribute( 'pa_brand' );
		if ( ! $brand ) {
			$brand = $product->get_attribute( 'pa_manufacturer' );
		}
		if ( ! $brand ) {
			$brand = $product->get_attribute( 'brand' );
		}
		if ( ! $brand ) {
			$terms = wp_get_post_terms( $product_id, 'product_brand', array( 'fields' => 'names' ) );
			if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
				$brand = $terms[0];
			}
		}
		if ( ! $brand ) {
			$brand = get_post_meta( $product_id, '_brand', true );
		}
		if ( ! $brand ) {
			$brand = get_post_meta( $product_id, 'brand', true );
		}
		return $brand;
	}
}
