<?php
/**
 * Settings class.
 *
 * @package V7_Product_Alternatives
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class V7PA_Settings {

	private $plugin_name;
	private $version;

	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	public function add_plugin_action_links( $links ) {
		$settings_link = '<a href="' . admin_url( 'admin.php?page=v7-product-alternatives' ) . '">' . esc_html__( 'Settings', 'v7-product-alternatives' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}

	public function add_settings_page() {
		add_submenu_page(
			'woocommerce',
			__( 'V7 Product Alternatives', 'v7-product-alternatives' ),
			__( 'Product Alternatives', 'v7-product-alternatives' ),
			'manage_woocommerce',
			'v7-product-alternatives',
			array( $this, 'render_settings_page' )
		);
	}

	public function enqueue_color_picker( $hook ) {
		if ( 'woocommerce_page_v7-product-alternatives' !== $hook ) {
			return;
		}
		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_script( 'wp-color-picker' );
		wp_enqueue_style(
			$this->plugin_name . '-settings',
			V7PA_URL . 'admin/css/settings.css',
			array(),
			$this->version
		);
		wp_add_inline_script( 'wp-color-picker', 'jQuery(document).ready(function($){$(".v7-color-picker").wpColorPicker();});' );
	}

	public function register_settings() {
		register_setting( 'v7pa_settings', 'v7pa_title_text', array( 'sanitize_callback' => 'sanitize_text_field' ) );
		register_setting( 'v7pa_settings', 'v7pa_button_text', array( 'sanitize_callback' => 'sanitize_text_field' ) );
		register_setting( 'v7pa_settings', 'v7pa_accent_color', array( 'sanitize_callback' => 'sanitize_hex_color' ) );
		register_setting( 'v7pa_settings', 'v7pa_text_color', array( 'sanitize_callback' => 'sanitize_hex_color' ) );
		register_setting( 'v7pa_settings', 'v7pa_bg_color', array( 'sanitize_callback' => 'sanitize_hex_color' ) );
		register_setting( 'v7pa_settings', 'v7pa_border_radius', array( 'sanitize_callback' => 'absint' ) );
		register_setting( 'v7pa_settings', 'v7pa_show_brand', array( 'sanitize_callback' => 'absint' ) );
		register_setting( 'v7pa_settings', 'v7pa_show_price', array( 'sanitize_callback' => 'absint' ) );

		add_settings_section( 'v7pa_text_section', __( 'Text Settings', 'v7-product-alternatives' ), null, 'v7-product-alternatives' );
		add_settings_section( 'v7pa_style_section', __( 'Style Settings', 'v7-product-alternatives' ), null, 'v7-product-alternatives' );
		add_settings_section( 'v7pa_display_section', __( 'Display Settings', 'v7-product-alternatives' ), null, 'v7-product-alternatives' );

		add_settings_field( 'v7pa_title_text', __( 'Title Text', 'v7-product-alternatives' ), array( $this, 'render_title_text_field' ), 'v7-product-alternatives', 'v7pa_text_section' );
		add_settings_field( 'v7pa_button_text', __( 'Button Text', 'v7-product-alternatives' ), array( $this, 'render_button_text_field' ), 'v7-product-alternatives', 'v7pa_text_section' );
		add_settings_field( 'v7pa_accent_color', __( 'Accent Color', 'v7-product-alternatives' ), array( $this, 'render_accent_color_field' ), 'v7-product-alternatives', 'v7pa_style_section' );
		add_settings_field( 'v7pa_text_color', __( 'Text Color', 'v7-product-alternatives' ), array( $this, 'render_text_color_field' ), 'v7-product-alternatives', 'v7pa_style_section' );
		add_settings_field( 'v7pa_bg_color', __( 'Background Color', 'v7-product-alternatives' ), array( $this, 'render_bg_color_field' ), 'v7-product-alternatives', 'v7pa_style_section' );
		add_settings_field( 'v7pa_border_radius', __( 'Border Radius (px)', 'v7-product-alternatives' ), array( $this, 'render_border_radius_field' ), 'v7-product-alternatives', 'v7pa_style_section' );
		add_settings_field( 'v7pa_show_brand', __( 'Show Brand', 'v7-product-alternatives' ), array( $this, 'render_show_brand_field' ), 'v7-product-alternatives', 'v7pa_display_section' );
		add_settings_field( 'v7pa_show_price', __( 'Show Price', 'v7-product-alternatives' ), array( $this, 'render_show_price_field' ), 'v7-product-alternatives', 'v7pa_display_section' );
	}

	public function render_title_text_field() {
		$value = get_option( 'v7pa_title_text', __( 'We recommend a similar alternative:', 'v7-product-alternatives' ) );
		echo '<input type="text" name="v7pa_title_text" value="' . esc_attr( $value ) . '" class="regular-text">';
	}

	public function render_button_text_field() {
		$value = get_option( 'v7pa_button_text', __( 'View Product', 'v7-product-alternatives' ) );
		echo '<input type="text" name="v7pa_button_text" value="' . esc_attr( $value ) . '" class="regular-text">';
	}

	public function render_accent_color_field() {
		$value = get_option( 'v7pa_accent_color', '#2271b1' );
		echo '<input type="text" name="v7pa_accent_color" value="' . esc_attr( $value ) . '" class="v7-color-picker">';
	}

	public function render_text_color_field() {
		$value = get_option( 'v7pa_text_color', '#333333' );
		echo '<input type="text" name="v7pa_text_color" value="' . esc_attr( $value ) . '" class="v7-color-picker">';
	}

	public function render_bg_color_field() {
		$value = get_option( 'v7pa_bg_color', '#ffffff' );
		echo '<input type="text" name="v7pa_bg_color" value="' . esc_attr( $value ) . '" class="v7-color-picker">';
	}

	public function render_border_radius_field() {
		$value = get_option( 'v7pa_border_radius', '6' );
		echo '<input type="number" name="v7pa_border_radius" value="' . esc_attr( $value ) . '" min="0" max="50" class="small-text"> px';
	}

	public function render_show_brand_field() {
		$value = get_option( 'v7pa_show_brand', '1' );
		echo '<input type="checkbox" name="v7pa_show_brand" value="1" ' . checked( $value, '1', false ) . '>';
	}

	public function render_show_price_field() {
		$value = get_option( 'v7pa_show_price', '1' );
		echo '<input type="checkbox" name="v7pa_show_price" value="1" ' . checked( $value, '1', false ) . '>';
	}

	public function render_settings_page() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'V7 Product Alternatives Settings', 'v7-product-alternatives' ); ?></h1>
			<form method="post" action="options.php">
				<?php
				settings_fields( 'v7pa_settings' );
				do_settings_sections( 'v7-product-alternatives' );
				submit_button();
				?>
			</form>
		</div>
		<?php
	}
}
