<?php
/**
 * User Registration Membership Table List
 *
 * @version 1.0.0
 */

namespace WPEverest\URMembership\Payment\Admin;

use UR_Base_Layout;
use WPEverest\URMembership\Admin\Repositories\MembersSubscriptionRepository;
use WPEverest\URMembership\Admin\Repositories\OrdersRepository;
use WPEverest\URMembership\TableList;

if ( ! class_exists( 'UR_List_Table' ) ) {
	include_once dirname( UR_PLUGIN_FILE ) . '/includes/abstracts/abstract-ur-list-table.php';
}
if ( ! class_exists( 'UR_Base_Layout' ) ) {
	include_once dirname( UR_PLUGIN_FILE ) . '/includes/admin/class-ur-admin-base-layout.php';
}

/**
 * Orders table list class.
 */
class OrdersListTable extends \UR_List_Table {

	protected $orders_repository;

	protected $is_membership_active;

	/**
	 * Initialize the Orders table list.
	 */
	public function __construct() {

		$this->page                 = 'member-payment-history';
		$this->per_page_option      = 'user_registration_membership_per_page';
		$this->addnew_action        = 'add_new_orders';
		$this->sort_by              = array(
			'title' => array( 'title', false ),
		);
		$this->is_membership_active = ur_check_module_activation( 'membership' );
		$this->orders_repository    = ( $this->is_membership_active ) ? new OrdersRepository() : '';

		parent::__construct(
			array(
				'singular' => 'order',
				'plural'   => 'orders',
				'ajax'     => true,
			)
		);
	}


	/**
	 * Prepares the items for display in the list table.
	 *
	 * This function prepares the column headers, sets the items per page, gets the current page number,
	 * determines the payment for parameter, prepares the query arguments, retrieves the total items,
	 * sets the items, and sets the pagination arguments.
	 *
	 * @return void
	 */
	public function prepare_items() {
		$this->prepare_column_headers();
		$per_page     = $this->get_items_per_page( $this->per_page_option );
		$current_page = $this->get_pagenum();
		$payment_for  = isset( $_REQUEST['payment_for'] ) && ! empty( $_REQUEST['payment_for'] ) ? $_REQUEST['payment_for'] : ( $this->is_membership_active ? 'memberships' : 'forms' );

		$args = $this->prepare_query_args( $payment_for, $per_page, $current_page );

		$total_items = array();
		if ( $payment_for === 'memberships' ) {
			$total_items = $this->orders_repository->get_all( $args );
		} else {
			$total_items = $this->get_user_payments( $args );
		}

		$this->items = $total_items;

		$this->set_pagination_args(
			array(
				'total_items' => count( $total_items ),
				'per_page'    => $per_page,
				'total_pages' => ceil( count( $total_items ) / $per_page ),
			)
		);
	}

	/**
	 * @param $payment_for
	 * @param $per_page
	 * @param $current_page
	 *
	 * @return array
	 */
	private function prepare_query_args( $payment_for, $per_page, $current_page ) {
		$args = array(
			'posts_per_page' => $per_page,
			'paged'          => $current_page,
		);

		$params_mapping = array(
			'status'         => 'ur_payment_status',
			'membership_id'  => 'membership_id',
			'payment_method' => 'ur_payment_method',
			'form_id'        => 'ur_form_id',
			's'              => 'search',
		);

		foreach ( $params_mapping as $request_param => $query_param ) {
			if ( ! empty( $_REQUEST[ $request_param ] ) ) {
				$value = sanitize_text_field( wp_unslash( $_REQUEST[ $request_param ] ) );
				if ( $payment_for === 'memberships' ) {
					$args[ $request_param ] = $value;
				} elseif ( $query_param === 'search' ) {
						$args['search']         = '*' . esc_attr( $value ) . '*';
						$args['search_columns'] = array( 'display_name', 'user_email' );
				} else {
					$args['meta_query'][] = array(
						'key'     => $query_param,
						'value'   => $value,
						'compare' => 'LIKE',
					);
				}
			}
		}

		$orderby         = isset( $_REQUEST['orderby'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['orderby'] ) ) : 'created_at';
		$args['orderby'] = in_array( $orderby, array( 'created_at', 'status' ) ) ? $orderby : 'created_at';

		$order         = isset( $_REQUEST['order'] ) ? strtoupper( sanitize_text_field( wp_unslash( $_REQUEST['order'] ) ) ) : 'DESC';
		$args['order'] = in_array( $order, array( 'ASC', 'DESC' ) ) ? $order : 'DESC';

		return $args;
	}

	/**
	 * @param $args
	 *
	 * @return array
	 */
	private function get_user_payments( $args ) {
		$args['meta_key']               = 'ur_payment_status';
		$args['meta_compare']           = 'EXISTS';
		$args['meta_query']['relation'] = 'AND';

		$user_query = new \WP_User_Query( $args );
		$users      = $user_query->get_results();

		$total_items = array();
		if ( ! empty( $users ) ) {
			foreach ( $users as $user ) {
				$meta_value    = get_user_meta( $user->ID, 'ur_payment_invoices', true );
				$total_items[] = array(
					'user_id'        => $user->ID,
					'display_name'   => $user->user_login,
					'user_email'     => $user->user_email,
					'transaction_id' => $meta_value[0]['invoice_no'] ?? '',
					'post_title'     => $meta_value[0]['invoice_plan'] ?? '',
					'status'         => get_user_meta( $user->ID, 'ur_payment_status', true ),
					'created_at'     => $meta_value[0]['invoice_date'] ?? '',
					'type'           => get_user_meta( $user->ID, 'ur_payment_type', true ),
					'payment_method' => str_replace( '_', ' ', get_user_meta( $user->ID, 'ur_payment_method', true ) ),
					'total_amount'   => $meta_value[0]['invoice_amount'] ?? 0,
					'currency'       => $meta_value[0]['invoice_currency'] ?? '',
				);
			}
		}

		return $total_items;
	}

	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'transaction_id':
				return isset( $item['transaction_id'] ) && ! empty( $item['transaction_id'] ) ? $item['transaction_id'] : ( $item['order_id'] ?? '' );
			case 'username':
				return esc_html( $item['display_name'] );
			case 'membership_type':
				return $this->show_column_membership_type( $item );
			case 'status':
				return $this->show_column_status( $item );
			case 'created_at':
				return date_i18n( get_option( 'date_format' ), strtotime( $item[ $column_name ] ) );
			case 'post_title':
			case 'payment_method':
				return esc_html( ucfirst( $item[ $column_name ] ) );
			case 'payer_email':
				return esc_html( $item['user_email'] );
			default:
				return print_r( $item, true ); // Fallback output for unknown columns
		}
	}

	public function column_transaction_id( $row ) {
		return sprintf(
			'<strong><div class="ur-edit-title"><a href="%s" class="row-title">%s</a></div></strong>%s',
			esc_url( isset($row['order_id']) ? admin_url( "admin.php?page=member-payment-history&action=edit&id={$row['order_id']}" ) : '' ),
			esc_html( isset( $row['transaction_id'] ) && ! empty( $row['transaction_id'] ) ? $row['transaction_id'] : ( $row['order_id'] ?? '' ) ),
			$this->row_actions( $this->get_row_actions( $row ) )
		);
	}

	/**
	 * No items found text.
	 */
	public function no_items() {
		UR_Base_Layout::no_items( 'Payments' );
	}

	/**
	 * Get list columns.
	 *
	 * @return array
	 */
	public function get_columns() {
		return array(
			'cb'              => '<input type="checkbox" />',
			'transaction_id'  => __( 'Transaction ID', 'user-registration' ),
			'username'        => __( 'Username', 'user-registration' ),
			'post_title'      => __( 'Product', 'user-registration' ),
			'membership_type' => __( 'Type', 'user-registration' ),
			'payment_method'  => __( 'Gateway', 'user-registration' ),
			'payer_email'     => __( 'Payer Email', 'user-registration' ),
			'status'          => __( 'Status', 'user-registration' ),
			'total'           => __( 'Total', 'user-registration' ),
			'created_at'      => __( 'Payment Date', 'user-registration' ),
		);
	}


	/**
	 * Post Edit Link.
	 *
	 * @param object $row
	 *
	 * @return string
	 */
	public function get_edit_links( $row ) {
		return admin_url( 'admin.php?post_id=' . $row['order_id'] ?? 0 . '&action=' . $this->addnew_action . '&page=' . $this->page );
	}

	/**
	 * Post Duplicate Link.
	 *
	 * @param object $row
	 *
	 * @return string
	 */
	public function get_duplicate_link( $row ) {
		return admin_url( 'post.php?post=' . $row->ID . '&action=edit' );
	}

	/**
	 * @param $order
	 *
	 * @return array
	 */
	public function get_row_actions( $order ) {
		$order_id  = $order['order_id'] ?? 0;
		$user_id   = $order['user_id'] ?? 0;
		$edit_id   = $order_id ? $order_id : $user_id;
		$edit_type = $order_id ? 'order' : 'form';

		return array(
			'id'     => sprintf(
				/* translators: %d: Item id */
				__( 'ID: %d', 'user-registration-file-downloads' ),
				$order_id ?: $user_id
			),
			'edit'   => sprintf(
				'<a href="%s">%s</a>',
				esc_url( admin_url( 'admin.php?page=member-payment-history&action=edit&id=' . $edit_id . '&type=' . $edit_type ) ),
				esc_html__( 'Edit', 'user-registration-file-downloads' )
			),
			'delete' => '<a data-user-id=' . esc_attr( $user_id ) . ' data-order-id = ' . esc_attr( $order_id ) . ' class="single-delete-order" style="cursor:pointer" >' . esc_html__( 'Trash', 'user-registration' ) . '</a>',
		);
	}

	public function get_delete_links( $row ) {
		return wp_nonce_url(
			add_query_arg(
				array(
					'page'   => $this->page,
					'screen' => $this->get_screen(),
					'action' => 'delete',
					'id'     => $row['order_id'] ?? 0,
				),
				admin_url( 'admin.php' )
			),
			'delete'
		);
	}

	/**
	 * @param $order
	 *
	 * @return string
	 */
	public function show_column_status( $order ) {

		$order_id = $order['order_id'] ?? $order['user_id'];

		return sprintf( '<span id="ur-order-%d"  class="payment-status-btn %s">%s</span>', absint( $order_id ), $order['status'], esc_html( ucfirst( $order['status'] ) ) );
	}

	/**
	 * @param $orders
	 *
	 * @return string
	 */
	public function show_column_membership_type( $orders ) {

		if ( isset( $orders['order_id'] ) ) {
			$data = json_decode( wp_unslash( $orders['post_content'] ), true );
			$type = $data['type'];
		} else {
			$type = $orders['type'];
		}
		$status_class = ( $type == 'free' ? 'user-registration-badge user-registration-badge--success-subtle' : ( $type == 'paid' ? 'user-registration-badge user-registration-badge--secondary-subtle' : 'user-registration-badge user-registration-badge--danger-subtle' ) );

		return sprintf( '<span class="%s">%s</span>', $status_class, esc_html( ucfirst( $type ) ) );
	}

	public function get_sortable_columns() {
		return array(
			'created_at' => array( 'created_at' ),
			'status'     => array( 'status' ),
		);
	}

	public function column_created_at( $item ) {
		global $wpdb;
		$orders_meta_table = TableList::order_meta_table();
		$payment_date      = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT meta_value FROM {$orders_meta_table} WHERE meta_key=%s AND order_id=%d LIMIT 1",
				'payment_date',
				$item['order_id'] ?? 0
			)
		);
		$payment_date      = ! empty( $payment_date ) ? $payment_date : $item['created_at'];
		return ( new \DateTime( $payment_date ) )->format( 'F j, Y' );
	}

	public function column_total( $item ) {
		$total_amount        = $item['total_amount'] ?? 0;
		$currency            = get_option( 'user_registration_payment_currency', 'USD' );
		$currencies          = ur_payment_integration_get_currencies();
		$currency_info       = isset( $currencies[ $currency ] ) ? $currencies[ $currency ] : $currencies['USD'];
		$symbol              = isset( $currency_info['symbol'] ) ? $currency_info['symbol'] : '$';
		$symbol_pos          = isset( $currency_info['symbol_pos'] ) ? $currency_info['symbol_pos'] : 'left';
		$thousands_separator = isset( $currency_info['thousands_separator'] ) ? $currency_info['thousands_separator'] : ',';
		$decimal_separator   = isset( $currency_info['decimal_separator'] ) ? $currency_info['decimal_separator'] : '.';
		$decimals            = isset( $currency_info['decimals'] ) ? (int) $currency_info['decimals'] : 2;
		$coupon_discount     = 0;

		if ( isset( $item['subscription_id'] ) ) {
			$subscription = ( new MembersSubscriptionRepository() )->get_subscription_by_subscription_id( absint( $item['subscription_id'] ) );
			if ( ! empty( $subscription ) && ! empty( $subscription['coupon'] ) ) {
				$coupon = ur_get_coupon_details( $subscription['coupon'] );
				if ( ! empty( $coupon ) ) {
					$discount_value = null;
					$discount_type  = 'fixed';

					if ( isset( $coupon['coupon_discount'] ) && isset( $coupon['coupon_discount_type'] ) ) {
						$discount_value = (float) $coupon['coupon_discount'];
						$discount_type  = $coupon['coupon_discount_type'];
					} elseif ( isset( $coupon['discount'] ) ) {
						$discount_value = (float) $coupon['discount'];
						$discount_type  = isset( $coupon['discount_type'] ) ? $coupon['discount_type'] : ( isset( $coupon['coupon_discount_type'] ) ? $coupon['coupon_discount_type'] : 'fixed' );
					}

					if ( null !== $discount_value && $total_amount ) {
						if ( 'percent' === $discount_type ) {
							$coupon_discount = $total_amount * ( $discount_value / 100 );
						} else {
							$coupon_discount = $discount_value;
						}
					}
				}
			}
		}

		$total_amount     = max( $total_amount - $coupon_discount, 0 );
		$formatted_amount = number_format( $total_amount, $decimals, $decimal_separator, $thousands_separator );
		return 'right' === $symbol_pos ? $formatted_amount . ' ' . $symbol : $symbol . $formatted_amount;
	}

	/**
	 * Render the list table page, including header, notices, status filters and table.
	 */
	public function display_page() {
		UR_Base_Layout::render_layout(
			$this,
			array(
				'page'           => $this->page,
				'title'          => esc_html__( 'Payment History', 'user-registration' ),
				'add_new_action' => 'add_new_payment',
				'search_id'      => 'user-registration-payment-history-search',
				'skip_query_key' => 'add-new-membership',
				'form_id'        => 'ur-membership-payment-history-form',
			)
		);
	}

	public function display() {
		$this->display_tablenav( 'top' );
		$this->screen->render_screen_reader_content( 'heading_list' );
		?>
		<table class="wp-list-table <?php echo implode( ' ', array_map( 'esc_attr', $this->get_table_classes() ) ); ?>">
			<thead>
				<tr>
					<?php $this->print_column_headers(); ?>
				</tr>
			</thead>
			<tbody id="the-list">
				<?php $this->display_rows_or_placeholder(); ?>
			</tbody>
		</table>
		<?php
		$this->display_tablenav( 'bottom' );
	}

	/**
		 * Generates the table navigation above or below the table
		 *
		 * @since 4.1
		 *
		 * @param string $which
		 */
	protected function display_tablenav( $which ) {
		if ( 'top' === $which ) {
			wp_nonce_field( 'bulk-' . $this->_args['plural'] );
		}
		?>
			<div class="tablenav <?php echo esc_attr( $which ); ?>">

			<?php if ( $this->has_items() && 'top' === $which ) : ?>
					<div>
						<div class="alignleft actions bulkactions">
							<?php $this->bulk_actions( $which ); ?>
						</div>
						<?php $this->extra_tablenav( $which ); ?>
					</div>
					<?php
				endif;
			if ( 'top' === $which ) :
				?>
				<div class="user-registration-payments-filters">
				<?php $this->display_advance_filter(); ?>
				</div>
					<?php
				endif;
			if ( 'bottom' === $which ) :
				?>
				<div class="alignleft">
				<?php $this->footer_text(); ?>
				</div>
					<?php
					$this->pagination( $which );
			endif;
			?>
		</div>
		<?php
	}

	/**
	 * Displays the search box.
	 *
	 * @since 4.1
	 */
	public function display_search_box( $input_id ) {
		?>
			<div id="user-registration-list-search-form">
				<input type="search" id="<?php echo esc_attr( $input_id ); ?>" name="s" value="<?php _admin_search_query(); ?>" placeholder="<?php esc_html_e( 'Search Members ...', 'user-registration' ); ?>" />
				<button type="submit" id="search-submit">
					<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
						<path fill="#000" fill-rule="evenodd" d="M4 11a7 7 0 1 1 12.042 4.856 1.012 1.012 0 0 0-.186.186A7 7 0 0 1 4 11Zm12.618 7.032a9 9 0 1 1 1.414-1.414l3.675 3.675a1 1 0 0 1-1.414 1.414l-3.675-3.675Z" clip-rule="evenodd"/>
					</svg>
				</button>
			</div>
		<?php
	}


	/**
	 * Displays advance filter.
	 *
	 * @since 4.1
	 */
	public function display_advance_filter() {
		$is_membership_active = $this->is_membership_active;
		$show_membership      = ( isset( $_REQUEST['payment_for'] ) && 'memberships' == $_REQUEST['payment_for'] || ( ! isset( $_REQUEST['payment_for'] ) && $this->is_membership_active !== null ) );

		?>

		<div class="module" style="display: flex; gap: 10px">
			<select name="payment_for" id="user-registration-pro-payment-type-filters">
				<option value="" selected><?php echo esc_html__( 'Select Module', 'user-registration' ); ?></option>
				<option value="forms"
					<?php echo isset( $_REQUEST['payment_for'] ) && 'forms' == $_REQUEST['payment_for'] ? 'selected=selected' : ''; ?>>
					<?php echo esc_html__( 'Forms', 'user-registration' ); ?></option>
				<?php
				if ( $this->is_membership_active ) :
					?>
				<option value="memberships"
					<?php echo ( isset( $_REQUEST['payment_for'] ) && 'memberships' == $_REQUEST['payment_for'] ) || ! isset( $_REQUEST['payment_for'] ) ? 'selected=selected' : ''; ?>>
					<?php echo esc_html__( 'Membership', 'user-registration' ); ?></option>
					<?php
								endif;
				?>
			</select>
		</div>
		<!--		membership dropdown-->
		<div class="module-box" id="user-registration-pro-memberships-filters-container"
			style="display:<?php echo $show_membership ? 'flex' : 'none'; ?>; gap: 10px;">
			<select name="membership_id" id="user-registration-pro-memberships-filter">
				<option value=""><?php echo esc_html__( 'All Membership', 'user-registration' ); ?></option>
				<?php
				foreach ( $this->get_all_memberships() as $id => $form ) {
					$selected = isset( $_REQUEST['membership_id'] ) && $id == $_REQUEST['membership_id'] ? 'selected=selected' : '';
					?>
				<option value='<?php echo esc_attr( $id ); ?>' <?php echo esc_attr( $selected ); ?>>
					<?php echo esc_html( $form ); ?></option>
					<?php
				}
				?>
			</select>
		</div>

		<div class="module-box" id="user-registration-pro-forms-filters-container"
			style="display:<?php echo ! $show_membership && $_REQUEST['payment_for'] == 'forms' ? 'flex' : 'none'; ?>; gap: 10px;">
			<select name="form_id" id="user-registration-pro-forms-filter">
				<option value=""><?php echo esc_html__( 'All Forms', 'user-registration' ); ?></option>
				<?php
				foreach ( ur_get_all_user_registration_form() as $id => $form ) {
					$selected = isset( $_REQUEST['form_id'] ) && $id == $_REQUEST['form_id'] ? 'selected=selected' : '';
					?>
				<option value='<?php echo esc_attr( $id ); ?>' <?php echo esc_attr( $selected ); ?>>
					<?php echo esc_html( $form ); ?></option>
					<?php
				}
				?>
			</select>
		</div>

		<div class="" id="user-registration-pro-members-filters" style="display: flex; gap: 10px">
			<select name="payment_method" id="user_registration_pro_users_form_filter">
				<option value=""><?php echo esc_html__( 'All Gateway', 'user-registration' ); ?></option>
				<?php
								$options = ( isset( $_REQUEST['payment_for'] ) && 'membership' == $_REQUEST['payment_for'] ) ? get_option( 'ur_membership_payment_gateways' ) : get_option( 'ur_payment_gateways' );

				foreach ( $options as $id => $form ) {
					$selected = isset( $_REQUEST['payment_method'] ) && $id == $_REQUEST['payment_method'] ? 'selected=selected' : '';
					?>
				<option value='<?php echo esc_attr( $id ); ?>' <?php echo esc_attr( $selected ); ?>>
					<?php echo esc_html( $form ); ?></option>
					<?php
				}
				?>
			</select>
		</div>
		<div class="payment-status" id="user-registration-pro-members-filters" style="display: flex; gap: 10px">
			<select name="status" id="user_registration_pro_users_form_filter">
				<option value=""><?php echo esc_html__( 'All Status', 'user-registration' ); ?></option>
				<?php
				foreach ( array( 'completed', 'pending', 'failed', 'refunded' ) as $id => $status ) {
					$selected = isset( $_REQUEST['status'] ) && $status == $_REQUEST['status'] ? 'selected=selected' : '';
					?>
				<option value='<?php echo esc_attr( $status ); ?>' <?php echo esc_attr( $selected ); ?>>
					<?php echo esc_html( ucfirst( $status ) ); ?></option>
					<?php
				}
				?>
			</select>
		</div>
		<button type="submit" name="ur_users_filter" id="user-registration-users-filter-btn" class="button ur-button-primary">
				<?php esc_html_e( 'Filter', 'user-registration' ); ?>
		</button>
		<?php
	}

	/**
	 * @return array
	 */
	public function get_all_memberships() {
		$posts        = get_posts(
			array(
				'post_type'   => 'ur_membership',
				'numberposts' => - 1,
			)
		);
		$active_posts = array_filter(
			json_decode( json_encode( $posts ), true ),
			function ( $item ) {
				$content = json_decode( wp_unslash( $item['post_content'] ), true );

				return $content['status'];
			}
		);

		return wp_list_pluck( $active_posts, 'post_title', 'ID' );
	}

	public function get_all_forms() {
		$posts        = get_posts(
			array(
				'post_type'   => 'ur_membership',
				'numberposts' => - 1,
			)
		);
		$active_posts = array_filter(
			json_decode( json_encode( $posts ), true ),
			function ( $item ) {
				$content = json_decode( wp_unslash( $item['post_content'] ), true );

				return $content['status'];
			}
		);

		return wp_list_pluck( $active_posts, 'post_title', 'ID' );
	}

	/**
	 * Generates the table navigation above or below the table
	 *
	 * @param string $which
	 *
	 * @since 3.1.0
	 */
	protected function bulk_actions( $which = '' ) {

		if ( is_null( $this->_actions ) ) {
			$this->_actions = $this->get_bulk_actions();

			/**
			 * Filters the items in the bulk actions menu of the list table.
			 *
			 * The dynamic portion of the hook name, `$this->screen->id`, refers
			 * to the ID of the current screen.
			 *
			 * @param array $actions An array of the available bulk actions.
			 *
			 * @since 5.6.0 A bulk action can now contain an array of options in order to create an optgroup.
			 *
			 * @since 3.1.0
			 */
			$this->_actions = apply_filters( "bulk_actions-{$this->screen->id}", $this->_actions ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores

			$two = '';
		} else {
			$two = '2';
		}

		if ( empty( $this->_actions ) ) {
			return;
		}
		echo '<select name="action' . esc_attr( $two ) . '" id="bulk-action-selector-' . esc_attr( $which ) . '">' . "\n";
		echo '<option value="-1">' . esc_html__( 'Bulk actions' ) . "</option>\n";

		foreach ( $this->_actions as $key => $value ) {
			if ( is_array( $value ) ) {
				echo "\t" . '<optgroup label="' . esc_attr( $key ) . '">' . "\n";

				foreach ( $value as $name => $title ) {
					$class = ( 'edit' === $name ) ? ' class="hide-if-no-js"' : '';

					echo "\t\t" . '<option value="' . esc_attr( $name ) . '"' . $class . '>' . esc_html( $title ) . "</option>\n";
				}
				echo "\t</optgroup>\n";
			} else {
				$class = ( 'edit' === $key ) ? ' class="hide-if-no-js"' : '';

				echo "\t" . '<option value="' . esc_attr( $key ) . '"' . $class . '>' . esc_html( $value ) . "</option>\n";
			}
		}

		echo "</select>\n";

		submit_button( __( 'Apply' ), 'action', '', false, array( 'id' => "doaction-orders$two" ) );
		echo "\n";
	}

	/**
	 * @return array
	 */
	protected function get_bulk_actions() {
		$actions = array();
		if ( is_multisite() ) {
			if ( current_user_can( 'remove_users' ) ) {
				$actions['remove'] = __( 'Remove', 'user-registration' );
			}
		} elseif ( current_user_can( 'delete_users' ) ) {
			$actions['delete'] = __( 'Delete', 'user-registration' );
		}

		return $actions;
	}

	public function column_cb( $item ) {
		return sprintf( '<input type="checkbox" name="order_id[]" value="%1$s" data-user-id="%2$s" /><span class="spinner"></span>', esc_attr( $item['order_id'] ?? '' ), esc_attr( $item['user_id'] ?? '' ) );
	}
}
