<?php

/*
Plugin Name: User List Exporter
Description: A plugin for exporting WordPress user data in CSV or TXT format with the ability to select desired fields.
Version: 1.0
Author: Amir Tohidlo
Author URI: https://tohidlo.com
License: GPL2
Text Domain: user-list-exporter
Domain Path: /languages
*/
defined('ABSPATH') || exit;

add_action('admin_menu', 'user_list_exporter_add_admin_menu');
add_action('plugins_loaded', 'user_list_exporter_load_textdomain');
add_action('admin_init', 'user_list_exporter_handle_export');
add_action('admin_enqueue_scripts', 'user_list_exporter_enqueue_styles');

function user_list_exporter_add_admin_menu() {
    add_submenu_page(
        'tools.php',
        __('Export User Data', 'user-list-exporter'),
        __('User Exporter', 'user-list-exporter'),
        'manage_options',
        'user-list-exporter',
        'user_list_exporter_admin_page'
    );
}

function user_list_exporter_load_textdomain() {
    load_plugin_textdomain('user-list-exporter', false, dirname(plugin_basename(__FILE__)) . '/languages/');
}

function user_list_exporter_admin_page() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to access this page.', 'user-list-exporter'));
    }

    global $wpdb;

    $user_fields = [
        'ID' => __('User ID', 'user-list-exporter'),
        'user_login' => __('Username', 'user-list-exporter'),
        'user_email' => __('Email', 'user-list-exporter'),
        'user_registered' => __('Registration Date', 'user-list-exporter'),
        'display_name' => __('Display Name', 'user-list-exporter'),
    ];

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $usermeta_fields = $wpdb->get_col("SELECT DISTINCT meta_key FROM {$wpdb->usermeta}");
    $usermeta_options = array_combine($usermeta_fields, array_map('esc_html', $usermeta_fields));

    $error = '';
    if (
        isset($_GET['error'], $_GET['_wpnonce'])
    ) {
        $nonce = sanitize_text_field(wp_unslash($_GET['_wpnonce']));
        if (wp_verify_nonce($nonce, 'user_list_exporter_error_nonce')) {
            $error = sanitize_text_field(wp_unslash($_GET['error']));
        }
    }

    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Export User Data', 'user-list-exporter'); ?></h1>
        <?php if ($error) : ?>
            <div class="notice notice-error is-dismissible">
                <p><?php echo esc_html($error); ?></p>
            </div>
        <?php endif; ?>
        <p><?php esc_html_e('Select the fields you want to export and choose the output file type.', 'user-list-exporter'); ?></p>
        <form method="post" action="">
            <?php wp_nonce_field('user_list_exporter_export_action', 'user_list_exporter_export_nonce'); ?>
            <h2><?php esc_html_e('User Table Fields', 'user-list-exporter'); ?></h2>
            <?php foreach ($user_fields as $field => $label) : ?>
                <label><input type="checkbox" name="fields[]" value="<?php echo esc_attr($field); ?>"> <?php echo esc_html($label); ?></label><br>
            <?php endforeach; ?>
            <h2><?php esc_html_e('User Meta Fields', 'user-list-exporter'); ?></h2>
            <?php if (empty($usermeta_options)) : ?>
                <p><?php esc_html_e('No user meta fields found.', 'user-list-exporter'); ?></p>
            <?php else : ?>
                <?php foreach ($usermeta_options as $meta => $label) : ?>
                    <label><input type="checkbox" name="meta_fields[]" value="<?php echo esc_attr($meta); ?>"> <?php echo esc_html($label); ?></label><br>
                <?php endforeach; ?>
            <?php endif; ?>
            <h2><?php esc_html_e('Output File Type', 'user-list-exporter'); ?></h2>
            <label><input type="radio" name="file_type" value="csv" checked> <?php esc_html_e('CSV', 'user-list-exporter'); ?></label>
            <label><input type="radio" name="file_type" value="txt"> <?php esc_html_e('TXT', 'user-list-exporter'); ?></label><br>
            <p class="submit">
                <input type="submit" name="user_list_exporter_export" class="button button-primary" value="<?php esc_html_e('Download File', 'user-list-exporter'); ?>">
            </p>
        </form>
    </div>
    <?php
}

function user_list_exporter_handle_export() {
    if (!isset($_POST['user_list_exporter_export']) || !check_admin_referer('user_list_exporter_export_action', 'user_list_exporter_export_nonce')) {
        return;
    }

    user_list_exporter_export_user_data($_POST);
}

function user_list_exporter_export_user_data($post_data) {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to access this page.', 'user-list-exporter'));
    }

    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem();
    global $wp_filesystem;

    $user_fields = [
        'ID' => __('User ID', 'user-list-exporter'),
        'user_login' => __('Username', 'user-list-exporter'),
        'user_email' => __('Email', 'user-list-exporter'),
        'user_registered' => __('Registration Date', 'user-list-exporter'),
        'display_name' => __('Display Name', 'user-list-exporter'),
    ];

    $selected_fields = !empty($post_data['fields']) ? array_map('sanitize_text_field', (array) $post_data['fields']) : [];
    $selected_meta_fields = !empty($post_data['meta_fields']) ? array_map('sanitize_text_field', (array) $post_data['meta_fields']) : [];
    $file_type = isset($post_data['file_type']) ? sanitize_text_field($post_data['file_type']) : 'csv';

    if (empty($selected_fields) && empty($selected_meta_fields)) {
        $error = esc_html__('Please select at least one field.', 'user-list-exporter');
        wp_safe_redirect(
            add_query_arg([
                'error'    => urlencode($error),
                '_wpnonce' => wp_create_nonce('user_list_exporter_error_nonce')
            ], admin_url('tools.php?page=user-list-exporter'))
        );
        exit;
    }

    $users = get_users([
        'fields' => array_merge($selected_fields, ['ID']),
    ]);

    if (empty($users)) {
        $error = esc_html__('No users found. Please ensure users are registered in the system.', 'user-list-exporter');
        wp_safe_redirect(
            add_query_arg([
                'error'    => urlencode($error),
                '_wpnonce' => wp_create_nonce('user_list_exporter_error_nonce')
            ], admin_url('tools.php?page=user-list-exporter'))
        );
        exit;
    }

    $meta_data = [];
    if (!empty($selected_meta_fields)) {
        foreach ($users as $user) {
            $meta_data[$user->ID] = [];
            foreach ($selected_meta_fields as $meta_key) {
                $meta_value = get_user_meta($user->ID, $meta_key, true);
                $meta_data[$user->ID][$meta_key] = is_array($meta_value) ? wp_json_encode($meta_value, JSON_UNESCAPED_UNICODE) : (string) ($meta_value ?: '');
            }
        }
    }

    $headers = array_merge(
        array_map(function($field) use ($user_fields) {
            return $user_fields[$field] ?? $field;
        }, $selected_fields),
        $selected_meta_fields
    );

    $lines = [];
    $delimiter = $file_type === 'csv' ? ',' : "\t";
    $lines[] = implode($delimiter, $headers);

    foreach ($users as $user) {
        $row = [];
        foreach ($selected_fields as $field) {
            $row[] = isset($user->$field) ? (string) $user->$field : '';
        }
        foreach ($selected_meta_fields as $meta_key) {
            $row[] = isset($meta_data[$user->ID][$meta_key]) ? $meta_data[$user->ID][$meta_key] : '';
        }
        $lines[] = implode($delimiter, $row);
    }

    $content = implode("\n", $lines);
    if ($file_type === 'csv') {
        $content = "\xEF\xBB\xBF" . $content;
    }

    $filename = 'user-data-' . gmdate('Y-m-d-H-i-s') . '.' . $file_type;
    $tmp_file = wp_tempnam($filename);

    if ( ! $wp_filesystem->put_contents($tmp_file, $content, FS_CHMOD_FILE) ) {
        wp_die(esc_html__('Failed to write export file.', 'user-list-exporter'));
    }

    $file_content = $wp_filesystem->get_contents($tmp_file);

    if ($file_content === false) {
        wp_die(esc_html__('Failed to read export file.', 'user-list-exporter'));
    }

    header('Content-Description: File Transfer');
    header('Content-Type: ' . ($file_type === 'csv' ? 'text/csv' : 'text/plain'));
    header('Content-Disposition: attachment; filename=' . $filename);
    header('Content-Transfer-Encoding: binary');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Expires: 0');

    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- safe for file download
    echo $file_content;

    wp_delete_file($tmp_file);
    exit;
}

function user_list_exporter_enqueue_styles($hook) {
    if ($hook !== 'tools_page_user-list-exporter') {
        return;
    }
    wp_enqueue_style('user-list-exporter-admin-style', plugins_url('css/admin-style.css', __FILE__), [], '1.0');
}
