/* js/admin-create-from-url.js */
(function () {
  'use strict';

  // ==== Localized config / i18n ====
  const CFG  = (window.urlifywriterCreateFromUrl || {});
  const I18N = (CFG.i18n || {});
  const AJAX = String(CFG.ajaxurl || (window.ajaxurl || ''));
  const NONCE = String(CFG.ajax_nonce || '');
  const preferUserApi = !!CFG.preferUserApi;
  const REMOTE_TIMEOUT = parseInt(CFG.remoteTimeout || 45, 10) || 45;

  const DEF = Object.assign({
    images_enabled: 0,
    image_source: 'pixabay', // original|pixabay|ai
    image_original_ack: 0,
    pixabay_n: 1,
    ai_n: 0,
    image_insert: 'both',    // featured|inline|both
    ai_style_hint: '',
    pixabay_max: 1,
    ai_max: 0,
    license_includes_ai: false,
    license_ai_per_art: 0,
  }, CFG.defaults || {});

  // ==== DOM ====
  const form        = document.getElementById('urlifywriter-form-url');
  const submitBtn   = document.getElementById('urlifywriter-submit');
  const submitLbl   = submitBtn ? submitBtn.querySelector('.urlifywriter-btn-label') : null;

  const progressCard = document.getElementById('urlifywriter-progress-card');
  const stepsUl      = document.getElementById('urlifywriter-steps');
  const fill         = document.getElementById('urlifywriter-progress-fill');
  const plabel       = document.getElementById('urlifywriter-progress-label');
  const resultBox    = document.getElementById('urlifywriter-result');
  const bar          = document.querySelector('.urlifywriter-progress');

  const batchCard    = document.getElementById('urlifywriter-batch-card');
  const batchList    = document.getElementById('urlifywriter-batch-list');
  const batchSummary = document.getElementById('urlifywriter-batch-summary');

  // Mode / Type
  const modeRadios = document.querySelectorAll('input[name="urlifywritersource_mode"]');
  const typeRadios = document.querySelectorAll('input[name="urlifywriterbatch_type"]');
  const itemsArea  = document.getElementById('urlifywriteritems');
  const itemsLabel = document.getElementById('urlifywriteritems_label');
  const instrArea  = document.getElementById('urlifywriterinstructions_list');
  const instrAllCk = document.getElementById('urlifywriterinstr_apply_all');

  // Length sliders
  const sMin = document.getElementById('urlifywritertarget_min');
  const sMax = document.getElementById('urlifywritertarget_max');
  const vMin = document.getElementById('urlifywritertarget_min_val');
  const vMax = document.getElementById('urlifywritertarget_max_val');

  // Images UI
  const imgEnabled   = document.getElementById('urlifywriterimages_enabled_run');
  const advToggleBtn = document.getElementById('urlifywriter-images-advanced-toggle');
  const advPanel     = document.getElementById('urlifywriter-images-advanced');
  const srcRadios    = document.querySelectorAll('input[name="urlifywriterimage_source_run"]');
  const elOrigAck    = document.getElementById('urlifywriterimage_original_ack_run');
  const elPixCount   = document.getElementById('urlifywriterpixabay_images_per_article_run');
  const elAiCount    = document.getElementById('urlifywriterai_images_per_article_run');
  const elInsert     = document.getElementById('urlifywriterimage_insert_run');
  const elAiStyle    = document.getElementById('urlifywriterai_style_hint_run');

  // Labels dinámicos
  const instrLabel = document.getElementById('urlifywriterinstr_label');
  const instrHelp  = document.getElementById('urlifywriterinstr_help');

  // Scheduling UI
  const publishModeSel     = document.getElementById('urlifywriterpublish_mode');
  const scheduleWrap       = document.getElementById('urlifywriter-schedule-options');
  const countInput         = document.getElementById('urlifywriterschedule_count');            // Number per interval
  const hoursWrap          = document.getElementById('urlifywriter-hours-between-wrap');
  const freqRadiosName     = 'urlifywriterschedule_freq'; // radios
  const dateInput          = document.getElementById('urlifywriterschedule_start_date');
  const timeInput          = document.getElementById('urlifywriterschedule_start_time');
  const hoursBetweenInput  = document.getElementById('urlifywriterschedule_hours_between');

  // State
  let originImageUrl = '';
	
	const postTypeRadios = document.querySelectorAll('input[name="urlifywriterpost_type"]');
	function currentPostType(){
	  const r = document.querySelector('input[name="urlifywriterpost_type"]:checked');
	  return r ? r.value : (DEF.post_type_default || 'post');
	}

  // Scheduling toggles
  function toggleScheduleUI(){
    if (!publishModeSel || !scheduleWrap) return;
    scheduleWrap.style.display = (publishModeSel.value === 'schedule') ? 'block' : 'none';
  }
  if (publishModeSel){
    publishModeSel.addEventListener('change', toggleScheduleUI);
    toggleScheduleUI(); // init
  }
  if (countInput){
    countInput.addEventListener('input', () => {
      const val = parseInt(countInput.value||'1',10);
      if (hoursWrap) hoursWrap.style.display = (val > 1) ? 'block' : 'none';
    });
  }

  /* ========== Helpers generales ========== */
  function currentMode(){
    const r = document.querySelector('input[name="urlifywritersource_mode"]:checked');
    return r ? r.value : 'url';
  }
  function currentType(){
    const r = document.querySelector('input[name="urlifywriterbatch_type"]:checked');
    return r ? r.value : 'single';
  }
  function stripTags(html){
    const div = document.createElement('div');
    div.innerHTML = html || '';
    return (div.textContent || div.innerText || '').trim();
  }
  function makeMiniPreview(html, fallback){
    let text = stripTags(html || '');
    if (!text) text = fallback || '';
    const sentences = text.split(/(?<=[.!?])\s+/).filter(Boolean);
    let preview = sentences.slice(0, 2).join(' ');
    if (!preview) preview = text.slice(0, 180);
    if (preview.length > 220) preview = preview.slice(0,220).replace(/\s+\S*$/, '') + '…';
    else if (text.length > preview.length) preview += '…';
    return preview || '—';
  }
  function parseLines(val){
    return (val || '')
      .split(/\r?\n/)
      .map(s => s.trim())
      .filter(Boolean);
  }
  function sprintf(fmt){ const args = Array.prototype.slice.call(arguments,1); let i=0; return String(fmt).replace(/%[sd]/g, ()=> args[i++]); }
  function escapeHtml(s){ return (s||'').replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c])); }

  function ajax(action, data) {
    data = data || {};
    data.action = action;
    data.nonce  = NONCE;
    return fetch(AJAX, {
      method: 'POST',
      headers: {'Content-Type':'application/x-www-form-urlencoded; charset=UTF-8'},
      body: new URLSearchParams(data)
    }).then(r => r.json());
  }

  function injectNotice(html){
    const holder = document.getElementById('urlifywriter-notices-by-step');
    if (holder && html) holder.insertAdjacentHTML('afterend', html);
  }

  /* ===== Helpers de programación ===== */
  function pad(n){ n = parseInt(n,10); return (n<10?'0':'') + n; }
  function parseLocalDateTime(dStr, tStr){
    if (!dStr) return null;
    const [y,m,d] = dStr.split('-').map(x=>parseInt(x,10));
    let hh=0, mm=0, ss=0;
    if (tStr){
      const parts = tStr.split(':').map(x=>parseInt(x,10));
      hh = parts[0]||0; mm = parts[1]||0; ss = parts[2]||0;
    }
    return new Date(y, (m-1), d, hh, mm, ss);
  }
  function fmtDate(dt){ return dt ? (dt.getFullYear() + '-' + pad(dt.getMonth()+1) + '-' + pad(dt.getDate())) : ''; }
  function fmtTime(dt){ return dt ? (pad(dt.getHours()) + ':' + pad(dt.getMinutes()) + ':00') : ''; }
  function addInterval(dt, freq, n){
    const d = new Date(dt.getTime());
    if (freq === 'daily') d.setDate(d.getDate() + n);
    else if (freq === 'weekly') d.setDate(d.getDate() + 7*n);
    else if (freq === 'monthly') d.setMonth(d.getMonth() + n);
    else d.setDate(d.getDate() + n);
    return d;
  }
  function getScheduleConfig(){
    const mode = publishModeSel ? publishModeSel.value : 'draft';
    if (mode !== 'schedule') return { mode: 'draft' };
    const freq = (document.querySelector(`input[name="${freqRadiosName}"]:checked`)?.value) || 'weekly';
    const date = (dateInput && dateInput.value) || '';
    const time = (timeInput && timeInput.value) || '00:00';
    const nPer = Math.max(1, parseInt((countInput && countInput.value) || '1', 10));
    let hoursBetween = parseFloat((hoursBetweenInput && hoursBetweenInput.value) || '0') || 0;
    if (nPer > 1 && hoursBetween <= 0) hoursBetween = 1; // evita colisiones si ponen 0
    const startDT = parseLocalDateTime(date, time);
    return { mode:'schedule', freq, startDT, nPer, hoursBetween };
  }
  function makeScheduler(cfg){
    if (!cfg || cfg.mode !== 'schedule' || !cfg.startDT) return null;
    let cursorIntervalStart = new Date(cfg.startDT.getTime());
    let slotIdxInInterval = 0;
    return {
      nextSlot(){
        if (cfg.nPer <= 1){
          const dt = new Date(cursorIntervalStart.getTime());
          cursorIntervalStart = addInterval(cursorIntervalStart, cfg.freq, 1);
          return dt;
        }
        // separa dentro del mismo intervalo
        const dt = new Date(cursorIntervalStart.getTime());
        const addMs = Math.round(cfg.hoursBetween * 60 * 60 * 1000 * slotIdxInInterval);
        dt.setTime(dt.getTime() + addMs);
        slotIdxInInterval++;
        if (slotIdxInInterval >= cfg.nPer){
          cursorIntervalStart = addInterval(cursorIntervalStart, cfg.freq, 1);
          slotIdxInInterval = 0;
        }
        return dt;
      },
      metaFor(dt){
        return {
          publish_mode: 'schedule',
          schedule_freq: cfg.freq,
          schedule_start_date: fmtDate(dt),
          schedule_start_time: fmtTime(dt),
          schedule_count: cfg.nPer,
          schedule_hours_between: cfg.hoursBetween
        };
      }
    };
  }

  /* ===== Sliders ===== */
  function sync(){ if (sMin && vMin) vMin.textContent = sMin.value; if (sMax && vMax) vMax.textContent = sMax.value; }
  function enforce(){
    if (!sMin || !sMax) return;
    let minVal = Math.max(100, Math.min(3000, parseInt(sMin.value||0,10)));
    let maxVal = Math.max(200, Math.min(3000, parseInt(sMax.value||0,10)));
    if (minVal > maxVal) { sMax.value = String(minVal); }
    sync();
  }
  if (sMin) sMin.addEventListener('input', enforce);
  if (sMax) sMax.addEventListener('input', enforce);
  enforce();

  /* ===== Labels dinámicos y placeholders ===== */
  function setItemsPlaceholder(){
    const m = currentMode();
    const t = currentType();
    const isBatch = (t === 'batch');

    if (m === 'url') {
      itemsArea.placeholder = (t==='single')
        ? 'https://example.com/article'
        : 'https://example.com/article-1\nhttps://example.com/article-2\nhttps://example.com/article-3';

      itemsLabel.textContent = isBatch ? I18N.urls_batch : I18N.urls_single;
    } else {
      itemsArea.placeholder = (t==='single')
        ? 'Roman aqueducts'
        : 'Roman aqueducts\nKyoto in autumn\nStreet food in Bangkok';

      itemsLabel.textContent = isBatch ? I18N.kw_batch : I18N.kw_single;
    }
  }

  function updateInstructionLabel(){
    const isBatch = (currentType() === 'batch');
    const m = currentMode();
    if (!instrLabel) return;

    if (m === 'url') {
      instrLabel.textContent = isBatch ? I18N.instr_opt_1perline : I18N.instr_opt_single;
    } else {
      instrLabel.textContent = isBatch ? I18N.instr_req_1perline : I18N.instr_req_single;
    }
  }

  function updateInstructionHelp(){
    if (!instrHelp) return;
    const m = currentMode();
    const isBatch = (currentType() === 'batch');

    if (m === 'url') {
      instrHelp.textContent = isBatch ? I18N.help_url_batch : I18N.help_url_single;
    } else {
      instrHelp.textContent = isBatch ? I18N.help_kw_batch : I18N.help_kw_single;
    }
  }

  function resizeTextareas(){
    const isBatch = (currentType() === 'batch');
    if (itemsArea) {
      itemsArea.rows = isBatch ? 10 : 1;
      itemsArea.style.minHeight = isBatch ? '180px' : '36px';
    }
    if (instrArea) {
      const m = currentMode();
      instrArea.rows = isBatch ? (m === 'url' ? 6 : 8) : 1;
      instrArea.style.minHeight = isBatch ? '140px' : '72px';
    }
  }

  // Init labels/UI
  setItemsPlaceholder();
  updateInstructionLabel();
  updateInstructionHelp();
  resizeTextareas();

  typeRadios.forEach(r => r.addEventListener('change', () => {
    setItemsPlaceholder();
    updateInstructionLabel();
    updateInstructionHelp();
    resizeTextareas();
    toggleApplyAllVisibility();
  }));
  modeRadios.forEach(r => r.addEventListener('change', () => {
    setItemsPlaceholder();
    updateInstructionLabel();
    updateInstructionHelp();
    resizeTextareas();
  }));

  // Mostrar/ocultar "Apply to all"
  const instrAllWrap = document.getElementById('urlifywriterinstr_apply_all_wrap');
  function toggleApplyAllVisibility(){
    const isBatch = (currentType() === 'batch');
    if (instrAllWrap) instrAllWrap.style.display = isBatch ? '' : 'none';
    const ck = document.getElementById('urlifywriterinstr_apply_all');
    if (ck && !isBatch) ck.checked = false;
  }
  toggleApplyAllVisibility();

  /* ===== Progress + steps ===== */
  function setProgress(p){
    if (fill)   fill.style.width = p + '%';
    if (plabel) plabel.textContent = p + '%';
    if (bar) {
      try {
        bar.setAttribute('role','progressbar');
        bar.setAttribute('aria-valuemin','0');
        bar.setAttribute('aria-valuemax','100');
        bar.setAttribute('aria-valuenow', String(p));
      } catch(e){}
    }
  }
  function stepEl(key){
    return stepsUl ? stepsUl.querySelector('li[data-key="'+key+'"]') : null;
  }
  function markRun(li){ if(!li) return; li.style.display='list-item'; li.className='run'; }
  function markDone(li, note){
    if(!li) return;
    li.className='done';
    const n = li.querySelector('.urlifywriter-meta-mini');
    if (n) { n.textContent = note ? (' — ' + note) : ''; n.style.display = note ? 'inline' : 'none'; }
  }
  function markErr(li, note){
    if(!li) return;
    li.className='err';
    const n = li.querySelector('.urlifywriter-meta-mini');
    if (n) { n.textContent = note ? (' — ' + note) : ''; n.style.display = note ? 'inline' : 'none'; }
    if (bar) bar.classList.add('error');
    if (plabel) {
      const stepLabel = (li.childNodes[0]?.textContent || 'Step').trim();
      plabel.classList.add('error');
      plabel.textContent = 'Failed at: ' + stepLabel;
    }
  }

  /* ================= Indeterminate progress (Generate step) ================ */
  let genTimer = null, genProg = 50;
  const GEN_CAP = 72, GEN_TICK_MS = 1100;
  function rand(min,max){ return Math.random()*(max-min)+min; }
  function genStepSize(){ const remaining = Math.max(0, GEN_CAP - genProg); const base = 0.10 + (remaining/28)*0.28; return Math.max(0.06, base + rand(-0.06,0.06)); }
  function startGenerateIndeterminate(){ stopGenerateIndeterminate(); genProg = 50; genTimer = setInterval(()=>{ if (genProg>=GEN_CAP) return; genProg = Math.min(GEN_CAP, genProg + genStepSize()); setProgress(Math.round(genProg)); }, GEN_TICK_MS); }
  function stopGenerateIndeterminate(){ if (genTimer){ clearInterval(genTimer); genTimer=null; } }

  /* ================= Indeterminate progress (Images step) ================== */
  let imgTimer = null, imgProg = 75;
  const IMG_START=75, IMG_CAP=95, IMG_TICK_MS=900;
  function imgStepSize(){ const remaining=Math.max(0,IMG_CAP-imgProg); const base=0.18+(remaining/20)*0.22; const jitter=(Math.random()*0.12)-0.06; return Math.max(0.12, base + jitter); }
  function startImagesIndeterminate(){ stopImagesIndeterminate(); const now = parseInt((document.getElementById('urlifywriter-progress-label')?.textContent || '80'),10)||80; imgProg = Math.max(now, IMG_START); imgTimer = setInterval(()=>{ if (imgProg>=IMG_CAP) return; imgProg = Math.min(IMG_CAP, imgProg + imgStepSize()); setProgress(Math.round(imgProg)); }, IMG_TICK_MS); }
  function stopImagesIndeterminate(){ if (imgTimer){ clearInterval(imgTimer); imgTimer=null; } }

  /* ====== Flow steps ====== */
  function extractStep(url){
    const li = stepEl('extract'); markRun(li);
    const srcLink = document.getElementById('urlifywriter-src-link');
    if (srcLink) { srcLink.href = url; srcLink.textContent = url; }

    return ajax('urlifywriterextract', { url }).then(res => {
      if (!res || res.success !== true) throw res;
      markDone(li, I18N.downloaded_parsed);
      setProgress(50);

      const box = li.querySelector('[data-inline-for="extract"]');
      if (box) {
        box.style.display = 'block';
        const ddTitle = document.getElementById('urlifywriter-detected-title');
        const ddPrev  = document.getElementById('urlifywriter-mini-preview');
        if (ddTitle) ddTitle.textContent = res.data?.title_detected || '—';
        if (ddPrev)  ddPrev.textContent  = res.data?.mini_preview   || '—';
      }
      originImageUrl = res?.data?.origin_image_url || '';
      return res.data || {};
    }).catch(err => {
      const msg = (err && err.data && err.data.message) ? err.data.message : I18N.extraction_failed;
      markErr(li, msg);
      setProgress(50);
      injectNotice('<div class="notice notice-error"><p>'+ msg +'</p></div>');
      throw new Error('extract_failed');
    });
  }

  function generateStep(payload){
    const li = stepEl('create'); markRun(li);
    startGenerateIndeterminate();
    return ajax('urlifywritergenerate', payload).then(res => {
      stopGenerateIndeterminate();
      if (!res || res.success !== true) throw res;

      const article = res.data || {};
      if (!article.excerpt || !article.excerpt.length) {
        article.excerpt = makeMiniPreview(article.content || '', '');
      }
      article.seo_title       = article.seo_title || '';
      article.seo_description = article.seo_description || '';
      article.category_id     = parseInt(article.category_id || 0, 10) || 0;
      if (!Array.isArray(article.tags)) article.tags = [];

      markDone(li, I18N.gen_created);
      setProgress(75);
      return article;
    }).catch(err => {
      stopGenerateIndeterminate();
      const code = err?.data?.code || '';
      let msg    = err?.data?.message || I18N.gen_failed;

      if (code === 'license_invalid') {
        const link = String(CFG.licensePageUrl || '#');
        injectNotice('<div class="notice notice-error"><p>'+ I18N.license_invalid +' <a href="'+link+'">'+ I18N.license_menu +'</a> '+ I18N.license_check +'</p></div>');
      } else if (code === 'remote_timeout') {
        msg = I18N.remote_timeout_msg || ('The remote generation exceeded the timeout ('+REMOTE_TIMEOUT+'s). Try again or increase the timeout.');
        injectNotice('<div class="notice notice-error"><p>'+ msg +'</p></div>');
      } else {
        injectNotice('<div class="notice notice-error"><p>'+ msg +'</p></div>');
      }

      markErr(li, msg);
      setProgress(Math.max(55, Math.round(genProg)));
      throw new Error('generate_failed');
    });
  }

  function computeInsertionSectionsForHTML(html, inlineN){
    inlineN = Math.max(1, parseInt(inlineN || 1, 10));
    if (!html) return [];
    const div = document.createElement('div'); div.innerHTML = html;
    const nodes = div.querySelectorAll('h2,h3,p');
    let currentSection = '';
    const sectionByParaIndex = [];
    nodes.forEach(el => {
      const tag = el.tagName;
      if (tag === 'H2' || tag === 'H3') {
        const t = (el.textContent || '').trim();
        if (t) currentSection = t;
      } else if (tag === 'P') {
        sectionByParaIndex.push(currentSection);
      }
    });
    const numParas = sectionByParaIndex.length;
    if (!numParas) return [];
    const step = Math.max(2, Math.floor(numParas / (inlineN + 1)));
    let nextPos = step;
    const picked = [];
    for (let i = 0; i < numParas && picked.length < inlineN; i++) {
      if ((i + 1) >= nextPos) {
        const idx = Math.min(i + 1, numParas - 1);
        picked.push(sectionByParaIndex[idx] || '');
        nextPos += step;
      }
    }
    if (picked.length === 0) {
      const firstH = div.querySelector('h2,h3');
      const t = (firstH?.textContent || '').trim();
      if (t) picked.push(t);
    }
    return picked;
  }

  function imagesStep(article){
    const li = stepEl('images');
    const enabled = imgEnabled && imgEnabled.checked;
    if (!enabled) {
      if (li) li.style.display = 'none';
      return Promise.resolve({ urls: [], used: 'none' });
    }
    const srcChecked   = document.querySelector('input[name="urlifywriterimage_source_run"]:checked');
    const image_source = srcChecked ? srcChecked.value : 'pixabay';
    const image_insert = elInsert ? (elInsert.value || 'featured') : 'featured';

    if (image_source === 'original') {
      markRun(li);
      if (!elOrigAck || !elOrigAck.checked) {
        markDone(li, I18N.img_original_skip);
        setProgress(85);
        return Promise.resolve({ urls: [], used: 'skip' });
      }
      markDone(li, 'Requesting original image from server…');
      setProgress(88);
      return Promise.resolve({ urls: [], used: 'original', insert: image_insert });
    }

    markRun(li);
    startImagesIndeterminate();

    const n = (image_source === 'pixabay')
      ? Math.max(1, Math.min(5, parseInt(elPixCount?.value || String(DEF.pixabay_n || 1), 10)))
      : Math.max(1, Math.min(4, parseInt(elAiCount?.value  || String(DEF.ai_n || 1), 10)));

    let q = (article.seo_title || article.title || '').trim();
    if (Array.isArray(article.tags) && article.tags.length) {
      q += ' ' + String(article.tags[0] || '');
    }
    q = q.trim();

    let style = (image_source === 'ai') ? (elAiStyle?.value || '').trim() : '';
    if (image_source === 'ai' && !style) {
      const def = String(DEF.ai_style_hint || '');
      if (def) style = def.trim();
    }

    let inlineN = (image_insert === 'inline') ? n : Math.max(0, n - 1);
    if (inlineN <= 0 && n > 0) inlineN = 1;

    const contentHTML = article.content || '';
    let sectionsToSend = computeInsertionSectionsForHTML(contentHTML, inlineN);
    if ((!sectionsToSend || sectionsToSend.length === 0) && (article.title || article.seo_title)) {
      sectionsToSend = [ (article.title || article.seo_title || '').trim() ];
    }

    return ajax('urlifywritergenerate_images', {
      mode: image_source === 'ai' ? 'ai' : 'pixabay',
      query: q,
      n: n,
      prefer_user_api: preferUserApi ? 1 : 0,
      style: style,
      lang: (document.documentElement.lang || '').slice(0,2),
      sections_json: JSON.stringify(sectionsToSend)
    })
    .then(res => {
      if (!res || res.success !== true) throw res;

      const urls = Array.isArray(res.data?.urls) ? res.data.urls : [];
      if (!urls.length) {
        setProgress(Math.max(90, Math.round(imgProg)));
        markDone(li, I18N.img_none);
        return { urls: [], used: 'none' };
      }
      const prov = res.data?.provider || (image_source === 'ai' ? 'openai' : 'pixabay');

      setProgress(95);
      markDone(li, sprintf(I18N.img_from_provider, urls.length, prov));
      return { urls, used: prov, insert: image_insert };
    })
    .catch(err => {
      const ecode = err?.data?.code || '';
      if (ecode === 'pixabay_no_results' || ecode === 'no_urls') {
        setProgress(Math.max(90, Math.round(imgProg)));
        markDone(li, I18N.img_none);
        return { urls: [], used: 'none' };
      }
      const msg = err?.data?.message || I18N.img_failed;
      setProgress(Math.max(90, Math.round(imgProg)));
      markErr(li, msg);
      injectNotice('<div class="notice notice-warning"><p>'+ msg +'</p></div>');
      return { urls: [], used: 'none' };
    })
    .finally(() => { stopImagesIndeterminate(); });
  }

  // draftStep ahora acepta schedMeta opcional
  function draftStep(url, article, imgData, schedMeta){
    const li = stepEl('draft'); markRun(li);

    const images_enabled = imgEnabled && imgEnabled.checked ? 1 : 0;
    const srcChecked = document.querySelector('input[name="urlifywriterimage_source_run"]:checked');
    const image_source = srcChecked ? srcChecked.value : 'pixabay';
    const image_original_ack = elOrigAck?.checked ? 1 : 0;

    const pixabay_images_per_article = elPixCount ? parseInt(elPixCount.value||'0',10) : 0;
    const ai_images_per_article      = elAiCount  ? parseInt(elAiCount.value ||'0',10) : 0;
    const image_insert = elInsert ? (elInsert.value || 'featured') : 'featured';

    const imgPayload = {
      images_enabled,
      image_source,
      image_original_ack,
      origin_image_url: (imgData?.origin || originImageUrl || ''),
      image_insert
    };

    const image_urls = Array.isArray(imgData?.urls) ? imgData.urls : [];
    const urlsPayload = image_urls.reduce((acc,u,i)=>{ acc['image_urls['+i+']'] = u; return acc; }, {});

    // Programación (si no hay schedMeta, usamos campos del formulario tal cual)
    const defaultSchedule = {
      publish_mode: publishModeSel ? publishModeSel.value : 'draft',
      schedule_freq: document.querySelector(`input[name="${freqRadiosName}"]:checked`)?.value || '',
      schedule_start_date: dateInput?.value || '',
      schedule_start_time: timeInput?.value || '',
      schedule_count: parseInt(countInput?.value || '1', 10),
      schedule_hours_between: parseInt(hoursBetweenInput?.value || '0', 10)
    };
    const schedulePayload = schedMeta || defaultSchedule;

    return ajax('urlifywritercreate_draft', {
      url,
      title:   article.title || '',
      content: article.content || '',
      excerpt: article.excerpt || '',
      category_id: article.category_id || 0,
      seo_title: article.seo_title || '',
      seo_description: article.seo_description || '',
	  post_type: currentPostType(),
      ...(Array.isArray(article.tags) ? article.tags.reduce((acc,t,i)=>{ acc['tags['+i+']']=t; return acc; }, {}) : {}),
      ...imgPayload,
      ...urlsPayload,
      pixabay_images_per_article,
      ai_images_per_article,
      // programación
      ...schedulePayload
    }).then(res => {
      if (!res || res.success !== true) throw res;
      markDone(li, I18N.draft_saved);
      setProgress(100);
      if (bar) { bar.classList.remove('error'); bar.classList.add('success'); }

      if (res.data?.images?.source === 'original') {
        const liImg = stepEl('images');
        if (res.data.images.found) {
          markDone(liImg, I18N.img_original_set);
        } else {
          markDone(liImg, I18N.img_original_none);
        }
      }

      if (resultBox) {
        const rTitle = document.getElementById('urlifywriter-result-title');
        const rEx    = document.getElementById('urlifywriter-result-excerpt');
        if (rTitle) rTitle.textContent = article.title || '—';
        if (rEx)    rEx.textContent    = makeMiniPreview(article.content || '', article.excerpt || '');

        const edit = res.data?.edit || '';
        const view = res.data?.view || '';
        const linksWrap = document.getElementById('urlifywriter-result-links');
        const editA = document.getElementById('urlifywriter-result-edit');
        const viewA = document.getElementById('urlifywriter-result-view');

        if (linksWrap && (edit || view)) {
          if (editA) { if (edit) { editA.href = edit; editA.style.removeProperty('display'); } else { editA.style.display = 'none'; } }
          if (viewA) { if (view) { viewA.href = view; viewA.style.removeProperty('display'); } else { viewA.style.display = 'none'; } }
          linksWrap.hidden = false;
        }
        resultBox.style.display = 'block';
      }

      return res.data || {};
    }).catch(err => {
      const msg = err?.data?.message || I18N.draft_failed;
      markErr(li, msg);
      setProgress(100);
      injectNotice('<div class="notice notice-error"><p>'+ msg +'</p></div>');
      throw new Error('draft_failed');
    });
  }

  /* ================= Batch helpers ================= */
  function addBatchRow(idx, title, badge){
    const li = document.createElement('li');
    li.className = 'urlifywriter-batch-item';
    li.dataset.index = String(idx);
    li.innerHTML = `
      <div class="title">${title ? escapeHtml(title) : '—'} ${badge ? `<span class="urlifywriter-badge-mini">${escapeHtml(badge)}</span>` : ''}</div>
      <div class="meta">
        <span class="state">${escapeHtml(I18N.queued)}</span>
        <span class="step" style="margin-left:8px;color:#64748b;"></span>
      </div>
      <div class="progress-wrap" style="margin-top:6px;">
        <div class="urlifywriter-progress" style="height:8px;">
          <div class="urlifywriter-progress-fill" style="width:0%;"></div>
        </div>
        <div class="urlifywriter-progress-label" style="font-size:12px;margin-top:2px;">0%</div>
      </div>
      <div class="links" style="display:none;margin-top:6px;"></div>
    `;
    batchList.appendChild(li);
    return li;
  }
  function setBatchState(li, text, cls){
    const s = li.querySelector('.state');
    if (s) s.textContent = text || '';
    li.classList.remove('success','error');
    if (cls) li.classList.add(cls);
  }
  function setBatchLinks(li, edit, view){
    const box = li.querySelector('.links');
    if (!box) return;
    if (edit || view) {
      box.innerHTML = `
        ${edit ? `<a href="${edit}">${escapeHtml(I18N.edit_draft)}</a>` : ''}
        ${view ? `<a href="${view}" target="_blank" rel="noopener">${escapeHtml(I18N.view_draft)}</a>` : ''}
      `;
      box.style.display = '';
    }
  }
  function setBatchProgress(li, pct){
    pct = Math.max(0, Math.min(100, Math.round(pct||0)));
    const _fill = li.querySelector('.urlifywriter-progress-fill');
    const _lab  = li.querySelector('.urlifywriter-progress-label');
    if (_fill) _fill.style.width = pct + '%';
    if (_lab)  _lab.textContent  = pct + '%';
  }
  function setBatchStep(li, text){
    const s = li.querySelector('.step');
    if (s) s.textContent = text || '';
  }

  // Indeterminados por item (WeakMap)
  const _genTimers = new WeakMap();
  const _imgTimers = new WeakMap();
  function startItemGenerateIndeterminate(li, start=50, cap=72, tick=1100){
    stopItemGenerateIndeterminate(li);
    const state = { p: start, cap, id: null };
    state.id = setInterval(() => {
      if (state.p >= state.cap) return;
      const remaining = Math.max(0, state.cap - state.p);
      const base = 0.10 + (remaining/28)*0.28;
      const jitter = (Math.random()*0.12)-0.06;
      const step = Math.max(0.06, base + jitter);
      state.p = Math.min(state.cap, state.p + step);
      setBatchProgress(li, Math.round(state.p));
    }, tick);
    _genTimers.set(li, state);
  }
  function stopItemGenerateIndeterminate(li){
    const st = _genTimers.get(li);
    if (st && st.id) clearInterval(st.id);
    _genTimers.delete(li);
  }
  function startItemImagesIndeterminate(li, start=75, cap=95, tick=900){
    stopItemImagesIndeterminate(li);
    const state = { p: start, cap, id: null };
    state.id = setInterval(() => {
      if (state.p >= state.cap) return;
      const remaining = Math.max(0, state.cap - state.p);
      const base = 0.18 + (remaining/20)*0.22;
      const jitter = (Math.random()*0.12)-0.06;
      const step = Math.max(0.12, base + jitter);
      state.p = Math.min(state.cap, state.p + step);
      setBatchProgress(li, Math.round(state.p));
    }, tick);
    _imgTimers.set(li, state);
  }
  function stopItemImagesIndeterminate(li){
    const st = _imgTimers.get(li);
    if (st && st.id) clearInterval(st.id);
    _imgTimers.delete(li);
  }

  /* ================= Submit ================= */
  if (!form) return;

  // Toggle images advanced
  if (advToggleBtn){
    advToggleBtn.addEventListener('click', () => {
      const show = (advPanel.style.display==='none' || !advPanel.style.display);
      advPanel.style.display = show ? 'block' : 'none';
      advToggleBtn.setAttribute('aria-expanded', show ? 'true' : 'false');
    });
  }
  srcRadios.forEach(r=>r.addEventListener('change', () => {
    const checked = document.querySelector('input[name="urlifywriterimage_source_run"]:checked');
    const val = checked ? checked.value : 'pixabay';
    document.querySelectorAll('#urlifywriter-images-advanced .urlifywriter-subrow').forEach(sr=>{
      const target = sr.getAttribute('data-for');
      if (!target) return;
      sr.style.display = (target === val) ? '' : 'none';
    });
  }));

  form.addEventListener('submit', function(ev){
    ev.preventDefault(); // AJAX flow

    // Validación extra: imágenes originales sin ACK
    const srcChecked = document.querySelector('input[name="urlifywriterimage_source_run"]:checked');
    const image_source = srcChecked ? srcChecked.value : '';
    if (image_source === 'original') {
      if (!elOrigAck || !elOrigAck.checked) {
        alert(I18N.origin_ack_needed || '⚠️ You must acknowledge the copyright disclaimer before using the original image.');
        return;
      }
    }

    const mode = currentMode(); // url|keyword
    let  type = currentType();  // single|batch

    const items = parseLines(itemsArea.value);
    const instrLines = parseLines(instrArea.value);
    const applyAll = !!(instrAllCk && instrAllCk.checked);

    // Si hay >1 ítem y estaba en single, cambiamos a batch automáticamente
    if (type === 'single' && items.length > 1) {
      const batchRadio = document.querySelector('input[name="urlifywriterbatch_type"][value="batch"]');
      if (batchRadio) batchRadio.checked = true;

      setItemsPlaceholder();
      updateInstructionLabel();
      updateInstructionHelp();
      toggleApplyAllVisibility();
      resizeTextareas();

      injectNotice('<div class="notice notice-info"><p>'+ (I18N.multi_detected || 'Multiple items detected. Switched to Batch automatically.') +'</p></div>');
      type = currentType();
    }

    const minw = sMin ? parseInt(sMin.value,10) : 800;
    const maxw = sMax ? parseInt(sMax.value,10) : 1200;

    // Validaciones
    if (!items.length) {
      alert(mode==='url' ? I18N.provide_url : I18N.provide_kw);
      return;
    }

    if (mode === 'url') {
      for (const u of items) {
        if (!/^https?:\/\/.+/i.test(u)) {
          alert((I18N.invalid_url || 'Invalid URL format:') + ' ' + u);
          return;
        }
      }
      if (type === 'batch' && instrLines.length > 0 && !applyAll) {
        if (instrLines.length !== items.length) {
          alert(I18N.instr_1to1);
          return;
        }
      }
    } else {
      if (!instrLines.length) {
        alert(I18N.instr_required_kw);
        return;
      }
      if (type === 'batch' && !applyAll) {
        if (instrLines.length !== items.length) {
          alert(I18N.instr_1to1);
          return;
        }
      }
    }
    // Reset UI
    if (submitLbl) submitLbl.textContent = I18N.work || 'Working…';
    if (bar) { bar.classList.remove('error'); bar.classList.remove('success'); }
    if (resultBox) resultBox.style.display = 'none';
    setProgress(0);

    // Construir scheduler (si procede)
    const schedCfg = getScheduleConfig();
    const scheduler = makeScheduler(schedCfg);

    // === SINGLE ===
    if (type === 'single') {
      if (progressCard) progressCard.style.display = 'block';
      const liEx = stepEl('extract');
      if (liEx) liEx.style.display = (mode==='url') ? 'list-item' : 'none';

      const liValidate = stepEl('validate'); markRun(liValidate);
      markDone(liValidate, I18N.inputs_ok);
      setProgress(20);

      // calcular slot para este único artículo (si hay scheduler)
      let schedMeta = null;
      if (scheduler){
        const slot = scheduler.nextSlot();
        schedMeta = scheduler.metaFor(slot);
      }

      if (mode==='url') {
        const url = items[0];
        extractStep(url)
          .then(data => generateStep({
            mode: 'url',
            url,
            instructions: instrLines.join("\n"),
            min_words: minw,
            max_words: maxw,
            title_hint: data.title_detected || '',
            extracted_text: data.extracted_text || ''
          }))
          .then(article => imagesStep(article).then(imgData => ({ article, imgData })))
          .then(({ article, imgData }) => draftStep(url, article, imgData, schedMeta))
          .catch(()=>{})
          .finally(()=>{ if (submitLbl) submitLbl.textContent = I18N.create || 'Create'; });
      } else {
        const kw = items[0];
        generateStep({
          mode: 'keyword',
          keyword: kw,
          url: '',
          instructions: instrLines, // array de líneas
          min_words: minw,
          max_words: maxw,
          title_hint: kw,
          extracted_text: ''
        })
        .then(article => imagesStep(article).then(imgData => ({ article, imgData })))
        .then(({ article, imgData }) => draftStep('', article, imgData, schedMeta))
        .catch(()=>{})
        .finally(()=>{ if (submitLbl) submitLbl.textContent = I18N.create || 'Create'; });
      }

      return;
    }

    // === BATCH ===
    if (progressCard) progressCard.style.display = 'none';
    if (batchCard) {

      batchCard.style.display = 'block';
      batchList.innerHTML = '';
      batchSummary.textContent = sprintf(I18N.total_items || 'Total items: %d', items.length);
    }

    const queue = items.map((it, idx) => {
      const li = addBatchRow(idx+1, it, mode==='url' ? 'URL' : 'Keyword');
      return { idx, item: it, li };
    });

    (async function runBatch(){

      for (const q of queue) {
        try {
          setBatchState(q.li, I18N.processing, null);
          setBatchStep(q.li,  I18N.inputs_ok);
          setBatchProgress(q.li, 10);

          let article;
          let usedUrl   = '';
          let originImg = '';

          if (mode === 'url') {
            const url = q.item;
            usedUrl = url;
            // EXTRAER
const downloadedParsed = String(I18N.downloaded_parsed || 'Downloaded and parsed.');
setBatchStep(q.li, downloadedParsed.replace('Downloaded and parsed.', 'Extracting article…'));

            setBatchProgress(q.li, 25);
            const ex = await ajax('urlifywriterextract', { url });
            if (!ex || ex.success !== true) throw ex;
            originImg = ex.data?.origin_image_url || '';
            setBatchProgress(q.li, 40);
            // GENERAR

			setBatchStep(q.li, String(I18N.gen_created || 'Created.').replace('Created.','Creating AI article…'));

            setBatchProgress(q.li, 50);
            startItemGenerateIndeterminate(q.li);
            const gen = await ajax('urlifywritergenerate', {
              mode: 'url',
              url,
              instructions: instrLines.length ? instrLines.join("\n") : '',
              min_words: minw, max_words: maxw,
              title_hint: ex.data?.title_detected || '',
              extracted_text: ex.data?.extracted_text || ''
            });

            stopItemGenerateIndeterminate(q.li);
            if (!gen || gen.success !== true) throw gen;
            article = gen.data || {};
            setBatchProgress(q.li, 75);

          } else {
            // KEYWORD
            const kw = q.item;
            const instrForThis = (instrAllCk && instrAllCk.checked) ? instrLines : [ instrLines[q.idx] ];

            setBatchStep(q.li, String(I18N.gen_created || 'Created.').replace('Created.','Creating AI article…'));

            setBatchProgress(q.li, 50);
            startItemGenerateIndeterminate(q.li);
            const gen = await ajax('urlifywritergenerate', {
              mode: 'keyword',
              keyword: kw,
              url: '',
              instructions: instrForThis, // array
              min_words: minw, max_words: maxw,
              title_hint: kw,
              extracted_text: ''
            });
            stopItemGenerateIndeterminate(q.li);
            if (!gen || gen.success !== true) throw gen;
            article = gen.data || {};
            setBatchProgress(q.li, 75);
          }

          // IMÁGENES
          let imgData = { urls: [], used: 'none' };
          if (imgEnabled && imgEnabled.checked) {
            setBatchStep(q.li,  I18N.img_generating);

            const currPct = parseInt(q.li.querySelector('.urlifywriter-progress-label')?.textContent || '75', 10) || 75;
            setBatchProgress(q.li, Math.max(75, currPct));
            startItemImagesIndeterminate(q.li);

            const srcCheckedB = document.querySelector('input[name="urlifywriterimage_source_run"]:checked');
            const image_source_b = srcCheckedB ? srcCheckedB.value : 'pixabay';
            const image_insert_b = elInsert ? (elInsert.value || 'featured') : 'featured';

            if (image_source_b === 'original') {
              stopItemImagesIndeterminate(q.li);
              if (elOrigAck && elOrigAck.checked && mode==='url') {
                imgData = { urls: [], used: 'original', insert: image_insert_b };
              } else {
                imgData = { urls: [], used: 'skip' };
              }
              setBatchProgress(q.li, 88);
            } else {
              const n = (image_source_b === 'pixabay')
                ? Math.max(1, Math.min(5, parseInt(elPixCount?.value || String(DEF.pixabay_n || 1), 10)))
                : Math.max(1, Math.min(4, parseInt(elAiCount?.value  || String(DEF.ai_n || 1), 10)));

              let qimg = (article.seo_title || article.title || '').trim();
              if (Array.isArray(article.tags) && article.tags.length) {
                qimg += ' ' + String(article.tags[0] || '');
              }
              qimg = qimg.trim();

              let style = (image_source_b === 'ai') ? (elAiStyle?.value || '').trim() : '';
              if (image_source_b === 'ai' && !style) {
                const def = String(DEF.ai_style_hint || '');
                if (def) style = def.trim();
              }

              const sectionsToSend = [ (article.title || article.seo_title || '').trim() ].filter(Boolean);

              const imgRes = await ajax('urlifywritergenerate_images', {
                mode: image_source_b === 'ai' ? 'ai' : 'pixabay',
                query: qimg, n,
                prefer_user_api: preferUserApi ? 1 : 0,
                style,
                lang: (document.documentElement.lang || '').slice(0,2),
                sections_json: JSON.stringify(sectionsToSend)
              });
              stopItemImagesIndeterminate(q.li);

              if (imgRes && imgRes.success === true) {
                const urls = Array.isArray(imgRes.data?.urls) ? imgRes.data.urls : [];
                imgData = { urls, used: (imgRes.data?.provider||'none'), insert: image_insert_b };
              } else {
                imgData = { urls: [], used: 'none' };
              }
              setBatchProgress(q.li, 95);
            }
          } else {
            setBatchProgress(q.li, Math.max(90, parseInt(q.li.querySelector('.urlifywriter-progress-label')?.textContent || '90', 10) ));
          }

          // DRAFT
          setBatchStep(q.li, String(I18N.draft_saved || 'Draft created.').replace('Draft created.','Saving…'));

          setBatchProgress(q.li, 96);

          // calcular slot para ESTE artículo si hay scheduler
          let schedMeta = null;
          if (scheduler){
            const slot = scheduler.nextSlot();
            schedMeta = scheduler.metaFor(slot);
          }

          const pixabay_images_per_article = elPixCount ? parseInt(elPixCount.value||'0',10) : 0;
          const ai_images_per_article      = elAiCount  ? parseInt(elAiCount.value ||'0',10) : 0;

          const payloadDraft = {
            url: usedUrl || '',
            title: article.title || '',
            content: article.content || '',
            excerpt: article.excerpt || '',
            category_id: article.category_id || 0,
            seo_title: article.seo_title || '',
            seo_description: article.seo_description || '',
			post_type: currentPostType(),
            pixabay_images_per_article,
            ai_images_per_article,
            images_enabled: (imgEnabled && imgEnabled.checked) ? 1 : 0,
            image_source: (document.querySelector('input[name="urlifywriterimage_source_run"]:checked')?.value || ''),
            image_original_ack: (elOrigAck && elOrigAck.checked) ? 1 : 0,
            origin_image_url: originImg || originImageUrl || '',
            image_insert: (elInsert ? (elInsert.value || 'featured') : 'featured'),
            // Programación
            ...(schedMeta ? schedMeta : {
              publish_mode: publishModeSel ? publishModeSel.value : 'draft',
              schedule_freq: document.querySelector(`input[name="${freqRadiosName}"]:checked`)?.value || '',
              schedule_start_date: dateInput?.value || '',
              schedule_start_time: timeInput?.value || '',
              schedule_count: parseInt(countInput?.value || '1', 10),
              schedule_hours_between: parseInt(hoursBetweenInput?.value || '0', 10)
            })
          };
          if (Array.isArray(article.tags)) {
            article.tags.forEach((t,i)=>{ payloadDraft['tags['+i+']'] = t; });
          }
          if (Array.isArray(imgData.urls) && imgData.urls.length) {
            imgData.urls.forEach((u,i)=>{ payloadDraft['image_urls['+i+']'] = u; });
          }

          const saved = await ajax('urlifywritercreate_draft', payloadDraft);
          if (!saved || saved.success !== true) throw saved;

          setBatchState(q.li, I18N.done, 'success');
          setBatchStep(q.li,  I18N.completed);
          setBatchProgress(q.li, 100);
          setBatchLinks(q.li, saved.data?.edit || '', saved.data?.view || '');

        } catch(e){
          // parar indeterminados si algo falla
          stopItemGenerateIndeterminate(q.li);
          stopItemImagesIndeterminate(q.li);

          const msg = e?.data?.message || 'Failed';
          setBatchState(q.li, msg, 'error');
          setBatchStep(q.li,  I18N.error);
          setBatchProgress(q.li, 100);
        }
      }
      if (submitLbl) submitLbl.textContent = I18N.create || 'Create';
    })();
  });

})();
