(function () {
  const $$ = (s) => document.querySelectorAll(s);

  const LOG_PREFIX = "[URLIFYWRITER][TEST_API]";

  function safeJsonParse(text) {
    try {
      return { ok: true, json: JSON.parse(text) };
    } catch (e) {
      return { ok: false, error: e, text };
    }
  }

  // Show / Hide
  $$(".urlifywriter-toggle-visibility").forEach((btn) => {
    btn.addEventListener("click", () => {
      const row = btn.closest(".urlifywriter-key-row");
      const input = row ? row.querySelector(".urlifywriter-api-input") : null;
      if (!input) return;

      const isPass = input.type === "password";
      input.type = isPass ? "text" : "password";

      const l10n = window.urlifywriterConnectionsL10n || {};
      btn.textContent = isPass ? (l10n.hide || "Hide") : (l10n.show || "Show");
    });
  });

  // Test API
  $$(".urlifywriter-test-api").forEach((btn) => {
    btn.addEventListener("click", async () => {
      const row = btn.closest(".urlifywriter-key-row");
      if (!row) return;

      const input  = row.querySelector(".urlifywriter-api-input");
      const status = row.querySelector(".urlifywriter-api-status");
      const spin   = row.querySelector(".spinner");
      if (!input || !status) return;

      const opt   = btn.dataset.opt || "";
      const nonce = btn.dataset.nonce || "";
      const l10n  = window.urlifywriterConnectionsL10n || {};

      status.classList.remove("ok", "bad");
      status.innerHTML = '<span class="dot"></span>' + (l10n.testing || "Testing…");
      if (spin) spin.classList.add("is-active");

      const bodyParams = new URLSearchParams({
        action: "urlifywritertest_api",
        opt,
        key: input.value || "",
        _wpnonce: nonce,
      });

      // ✅ LOG 1: request
      console.log(`${LOG_PREFIX} Request`, {
        ajaxurl: window.ajaxurl,
        opt,
        nonce_present: !!nonce,
        key_len: (input.value || "").length,
        body: bodyParams.toString().replace(/key=[^&]*/i, "key=***redacted***"),
      });

      try {
        const res = await fetch(window.ajaxurl, {
          method: "POST",
          headers: { "Content-Type": "application/x-www-form-urlencoded; charset=UTF-8" },
          body: bodyParams,
          credentials: "same-origin",
        });

        // ✅ LOG 2: status/headers
        console.log(`${LOG_PREFIX} HTTP`, {
          status: res.status,
          ok: res.ok,
          contentType: res.headers.get("content-type"),
        });

        const rawText = await res.text();

        // ✅ LOG 3: raw response preview
        console.log(`${LOG_PREFIX} Raw response (first 400)`, rawText.slice(0, 400));

        const parsed = safeJsonParse(rawText);

        if (!parsed.ok) {
          console.error(`${LOG_PREFIX} JSON parse FAILED`, parsed.error);
          if (spin) spin.classList.remove("is-active");
          status.classList.add("bad");
          status.innerHTML =
            '<span class="dot"></span>❌ ' +
            "Response is not valid JSON (see console)";
          return;
        }

        const json = parsed.json;

        // ✅ LOG 4: parsed JSON
        console.log(`${LOG_PREFIX} Parsed JSON`, json);

        if (spin) spin.classList.remove("is-active");

        // WordPress: { success: true/false, data: {...} }
        const payload = (json && json.data) ? json.data : {};
        const ok = !!(json && json.success && payload.ok);

        const errorMsg =
          payload.error ||
          payload.message ||
          json.error ||
          json.message ||
          "Error";

        // ✅ LOG 5: interpreted result
        console.log(`${LOG_PREFIX} Interpreted`, { ok, payload, errorMsg });

        if (ok) {
          status.classList.add("ok");
          status.innerHTML =
            '<span class="dot"></span>✅ ' + (l10n.ok || "OK");
        } else {
          status.classList.add("bad");
          status.innerHTML =
            '<span class="dot"></span>❌ ' + errorMsg;
        }
      } catch (e) {
        console.error(`${LOG_PREFIX} Fetch FAILED`, e);
        if (spin) spin.classList.remove("is-active");
        status.classList.add("bad");
        status.innerHTML =
          '<span class="dot"></span>❌ ' +
          (e && e.message ? e.message : "Network error");
      }
    });
  });
})();
