(function(){
  'use strict';

  const CFG = (typeof URLIFYWRITER_AUTOSCAN !== 'undefined') ? URLIFYWRITER_AUTOSCAN : {};
  const AJAX  = CFG.ajaxUrl || (typeof ajaxurl !== 'undefined' ? ajaxurl : '');
  const NONCE = CFG.nonce || '';
  const I18N  = CFG.i18n || {};

  function t(k, fallback){ return (I18N && I18N[k]) ? I18N[k] : (fallback || k); }

  function qs(sel, ctx){ return (ctx||document).querySelector(sel); }
  function qsa(sel, ctx){ return Array.from((ctx||document).querySelectorAll(sel)); }
  function esc(s){ return String(s||'').replace(/[&<>"']/g, c=>({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c])); }
  function notice(html){
    const box = qs('#urlifywriter-autoscan-notices');
    if (box) box.insertAdjacentHTML('beforeend', html);
  }
  function ajax(action, data){
    data = data || {};
    data.action = action;
    data.nonce  = NONCE;
    return fetch(AJAX, {
      method:'POST',
      headers:{'Content-Type':'application/x-www-form-urlencoded; charset=UTF-8'},
      body: new URLSearchParams(data)
    }).then(r=>r.json());
  }

  function fillSourcesSelect(selId, rows){
    const sel = qs(selId);
    if (!sel) return;
    const val = sel.value || '';
    sel.innerHTML = rows.map(r => `<option value="${r.id}">${esc(r.domain)}</option>`).join('');
    if (val) sel.value = val;
  }

  /* ------------------------ Cron status: AutoScan ------------------------ */
  function loadCronScan(){
    const s = {
      scheduled: qs('#urlifywriter-cron-scheduled'),
      next:      qs('#urlifywriter-cron-next'),
      last:      qs('#urlifywriter-cron-last'),
      intSel:    qs('#urlifywriter-cron-interval'),
      msg:       qs('#urlifywriter-cron-msg')
    };
    if (!s.scheduled) return;
    if (s.msg) s.msg.textContent = t('loading','Loading…');

    ajax('urlifywriterautoscan_cron_status', {}).then(res=>{
      if (!res || res.success!==true) throw res;
      const st = res.data.status || {};
      s.scheduled.textContent = st.scheduled ? t('yes','Yes') : t('no','No');
      s.next.textContent = st.next_at || '—';
      s.last.textContent = st.last_run || '—';
      if (s.intSel && st.interval) s.intSel.value = st.interval;
      if (s.msg) s.msg.textContent = '';
    }).catch(err=>{
      if (s.msg) s.msg.textContent = (err && err.data && err.data.message) ? err.data.message : 'Error';
    });
  }

  function startCronScan(){
    const intSel = qs('#urlifywriter-cron-interval');
    const key = intSel ? (intSel.value||'hourly') : 'hourly';
    const msg = qs('#urlifywriter-cron-msg');
    if (msg) msg.textContent = t('starting','Starting…');

    ajax('urlifywriterautoscan_cron_start', {interval:key}).then(res=>{
      if (!res || res.success!==true) throw res;
      if (msg) msg.textContent = t('cronScheduled','Cron scheduled.');
      loadCronScan();
    }).catch(err=>{
      if (msg) msg.textContent = (err && err.data && err.data.message) ? err.data.message : 'Error';
    });
  }

  function stopCronScan(){
    const msg = qs('#urlifywriter-cron-msg');
    if (!confirm(t('confirmStopScan','Stop the AutoScan cron?'))) return;
    if (msg) msg.textContent = t('stopping','Stopping…');

    ajax('urlifywriterautoscan_cron_stop', {}).then(res=>{
      if (!res || res.success!==true) throw res;
      if (msg) msg.textContent = t('cronStopped','Cron stopped.');
      loadCronScan();
    }).catch(err=>{
      if (msg) msg.textContent = (err && err.data && err.data.message) ? err.data.message : 'Error';
    });
  }

  function runCronScan(){
    const msg = qs('#urlifywriter-cron-msg');
    if (msg) msg.textContent = t('running','Running…');

    ajax('urlifywriterautoscan_cron_run_now', {}).then(res=>{
      if (!res || res.success!==true) throw res;
      const d = res.data||{};
      if (msg) msg.textContent = 'OK. scanned='+(d.scanned||0)+' generated='+(d.generated||0);
      loadCronScan();
      loadSources();
    }).catch(err=>{
      if (msg) msg.textContent = (err && err.data && err.data.message) ? err.data.message : 'Error';
    });
  }

  qs('#urlifywriter-cron-refresh')?.addEventListener('click', loadCronScan);
  qs('#urlifywriter-cron-start')?.addEventListener('click', startCronScan);
  qs('#urlifywriter-cron-stop')?.addEventListener('click', stopCronScan);
  qs('#urlifywriter-cron-run')?.addEventListener('click', runCronScan);

  /* ------------------------ Cron status: AutoGen ------------------------ */
  function loadCronGen(){
    const s = {
      scheduled: qs('#urlifywriter-gen-scheduled'),
      next:      qs('#urlifywriter-gen-next'),
      last:      qs('#urlifywriter-gen-last'),
      intSel:    qs('#urlifywriter-gen-interval'),
      msg:       qs('#urlifywriter-gen-msg')
    };
    if (!s.scheduled) return;
    if (s.msg) s.msg.textContent = t('loading','Loading…');

    ajax('urlifywriterautogen_cron_status', {}).then(res=>{
      if (!res || res.success!==true) throw res;
      const st = res.data.status || {};
      s.scheduled.textContent = st.scheduled ? t('yes','Yes') : t('no','No');
      s.next.textContent = st.next_at || '—';
      s.last.textContent = st.last_run || '—';
      if (s.intSel && st.interval) s.intSel.value = st.interval;
      if (s.msg) s.msg.textContent = '';
    }).catch(err=>{
      if (s.msg) s.msg.textContent = (err && err.data && err.data.message) ? err.data.message : 'Error';
    });
  }

  function startCronGen(){
    const intSel = qs('#urlifywriter-gen-interval');
    const key = intSel ? (intSel.value||'urlifywriterevery_15_minutes') : 'urlifywriterevery_15_minutes';
    const msg = qs('#urlifywriter-gen-msg');
    if (msg) msg.textContent = t('starting','Starting…');

    ajax('urlifywriterautogen_cron_start', {interval:key}).then(res=>{
      if (!res || res.success!==true) throw res;
      if (msg) msg.textContent = t('cronScheduled','Cron scheduled.');
      loadCronGen();
    }).catch(err=>{
      if (msg) msg.textContent = (err && err.data && err.data.message) ? err.data.message : 'Error';
    });
  }

  function stopCronGen(){
    const msg = qs('#urlifywriter-gen-msg');
    if (!confirm(t('confirmStopGen','Stop the AutoGen cron?'))) return;
    if (msg) msg.textContent = t('stopping','Stopping…');

    ajax('urlifywriterautogen_cron_stop', {}).then(res=>{
      if (!res || res.success!==true) throw res;
      if (msg) msg.textContent = t('cronStopped','Cron stopped.');
      loadCronGen();
    }).catch(err=>{
      if (msg) msg.textContent = (err && err.data && err.data.message) ? err.data.message : 'Error';
    });
  }

  function runCronGen(){
    const msg = qs('#urlifywriter-gen-msg');
    if (msg) msg.textContent = t('running','Running…');

    ajax('urlifywriterautogen_cron_run_now', {}).then(res=>{
      if (!res || res.success!==true) throw res;
      const d = res.data||{};
      if (msg) msg.textContent = 'OK. processed='+(d.processed||d.generated||0);
      loadCronGen();
      loadItems();
    }).catch(err=>{
      if (msg) msg.textContent = (err && err.data && err.data.message) ? err.data.message : 'Error';
    });
  }

  qs('#urlifywriter-gen-refresh')?.addEventListener('click', loadCronGen);
  qs('#urlifywriter-gen-start')?.addEventListener('click', startCronGen);
  qs('#urlifywriter-gen-stop')?.addEventListener('click', stopCronGen);
  qs('#urlifywriter-gen-run')?.addEventListener('click', runCronGen);

  /* ------------------------ Sources list ------------------------ */
  const srcTBody = qs('#urlifywriter-src-tbody');

  function loadSources(){
    if (!srcTBody) return;
    srcTBody.innerHTML = `<tr><td colspan="9">${esc(t('loading','Loading…'))}</td></tr>`;

    ajax('urlifywriterautoscan_sources_list', {}).then(res=>{
      if (!res || res.success!==true) throw res;

      const rows = res.data.sources || [];
      if (!rows.length){
        srcTBody.innerHTML = `<tr><td colspan="9">No sources yet.</td></tr>`;
      } else {
        srcTBody.innerHTML = rows.map(r => {
          const on   = r.enabled ? t('yes','Yes') : t('no','No');
          const last = r.last_scan_at || '—';
          const next = r.next_scan_at || '—';

          const actToggle = r.enabled ? t('disable','Disable') : t('enable','Enable');

          return `<tr data-id="${r.id}">
            <td>${r.id}</td>
            <td><strong>${esc(r.domain)}</strong></td>
            <td><a href="${esc(r.start_url)}" target="_blank" rel="noopener">${esc(r.start_url)}</a></td>
            <td>${esc(r.scan_interval)}</td>
            <td>${r.max_per_day}</td>
            <td>${on}</td>
            <td>${esc(last)}</td>
            <td>${esc(next)}</td>
            <td>
              <a href="#" data-act="edit">${esc(t('edit','Edit'))}</a> ·
              <a href="#" data-act="toggle">${esc(actToggle)}</a> ·
              <a href="#" data-act="scan">${esc(t('scanNow','Scan now'))}</a> ·
              <a href="#" data-act="delete" class="urlifywriter-danger">${esc(t('delete','Delete'))}</a>
            </td>
          </tr>`;
        }).join('');
      }

      fillSourcesSelect('#urlifywriter-items-source', rows);
      hookRowActions();

      const itemsSel = qs('#urlifywriter-items-source');
      if (rows.length && itemsSel && !itemsSel.value) {
        itemsSel.value = String(rows[0].id);
      }
      loadItems();
    }).catch(()=>{
      srcTBody.innerHTML = `<tr><td colspan="9">Error.</td></tr>`;
    });
  }

  function hookRowActions(){
    qsa('#urlifywriter-src-tbody tr').forEach(tr=>{
      tr.addEventListener('click', (ev)=>{
        const a = ev.target.closest('a[data-act]');
        if (!a) return;
        ev.preventDefault();

        const id  = parseInt(tr.getAttribute('data-id')||'0',10);
        const act = a.getAttribute('data-act');

        if (act==='edit'){
          editRow(id);
        } else if (act==='toggle') {
          const en = /Disable/i.test(a.textContent) ? 0 : 1;
          ajax('urlifywriterautoscan_source_toggle', {id, enabled: en}).then(()=>loadSources());
        } else if (act==='scan') {
          runScan(id);
        } else if (act==='delete') {
          if (!confirm(t('confirmDelete','Delete this source?'))) return;
          ajax('urlifywriterautoscan_source_delete', {id}).then(()=>loadSources());
        }
      });
    });
  }

  function editRow(id){
    const tr = qs(`#urlifywriter-src-tbody tr[data-id="${id}"]`);
    if (!tr) return;

    qs('#urlifywriter-src-id').value         = id;
    qs('#urlifywriter-src-domain').value     = tr.children[1].textContent.trim();
    qs('#urlifywriter-src-start').value      = tr.children[2].querySelector('a')?.getAttribute('href') || '';
    qs('#urlifywriter-src-interval').value   = tr.children[3].textContent.trim();
    qs('#urlifywriter-src-maxday').value     = parseInt(tr.children[4].textContent.trim()||'5',10);
    qs('#urlifywriter-src-enabled').checked  = (tr.children[5].textContent.trim().toLowerCase() === 'yes');

    // Instructions are not in table -> keep empty to avoid overwriting
    window.scrollTo({top:0, behavior:'smooth'});
  }

  qs('#urlifywriter-src-reset')?.addEventListener('click', ()=>{
    qs('#urlifywriter-src-id').value = '0';
    qs('#urlifywriter-src-domain').value = '';
    qs('#urlifywriter-src-start').value  = '';
    qs('#urlifywriter-src-interval').value = 'hourly';
    qs('#urlifywriter-src-maxday').value = '5';
    qs('#urlifywriter-src-instructions').value = '';
    qs('#urlifywriter-src-enabled').checked = true;
  });

  qs('#urlifywriter-src-form')?.addEventListener('submit', (ev)=>{
    ev.preventDefault();

    const payload = {
      id: qs('#urlifywriter-src-id').value||'0',
      domain: qs('#urlifywriter-src-domain').value||'',
      start_url: qs('#urlifywriter-src-start').value||'',
      scan_interval: qs('#urlifywriter-src-interval').value||'hourly',
      max_per_day: qs('#urlifywriter-src-maxday').value||'5',
      enabled: qs('#urlifywriter-src-enabled').checked ? 1 : 0,
      gen_instructions: qs('#urlifywriter-src-instructions').value||'',
      reschedule_cron: 0
    };

    ajax('urlifywriterautoscan_source_save', payload).then(res=>{
      if (!res || res.success!==true) throw res;
      notice('<div class="notice notice-success"><p>'+esc(t('sourceSaved','Source saved.'))+'</p></div>');
      loadSources();
      qs('#urlifywriter-src-id').value = '0';
    }).catch(err=>{
      const msg = err?.data?.message || 'Error';
      notice('<div class="notice notice-error"><p>'+esc(msg)+'</p></div>');
    });
  });

  /* ------------------------ Manual scan source ------------------------ */
  function runScan(id){
    const sid = parseInt(id || (qs('#urlifywriter-items-source')?.value||'0'),10);
    if (!sid) return;

    const msg = qs('#urlifywriter-cron-msg');
    if (msg) msg.textContent = t('scanning','Scanning…');

    ajax('urlifywriterautoscan_scan_now', {id:sid}).then(res=>{
      if (!res || res.success!==true) throw res;
      const ins = res.data.inserted || 0;
      const cand= res.data.candidates || 0;
      const m = res.data.message || '';
      if (msg) msg.textContent = m ? m : `Inserted ${ins} of ${cand} candidates.`;
      loadItems();
      loadSources();
    }).catch(err=>{
      if (msg) msg.textContent = err?.data?.message || 'Error';
    });
  }

  /* ------------------------ Items list / queue ------------------------ */
  let page = 1;

  function loadItems(){
    const srcSel = qs('#urlifywriter-items-source');
    if (srcSel && !srcSel.value && srcSel.options.length) {
      srcSel.value = srcSel.options[0].value;
    }

    const src = parseInt(qs('#urlifywriter-items-source')?.value||'0',10);
    const st  = qs('#urlifywriter-items-status')?.value || '';
    const pp  = parseInt(qs('#urlifywriter-items-perpage')?.value||'20',10);
    const tbody = qs('#urlifywriter-items-tbody');
    if (!tbody) return;

    if (!src){
      tbody.innerHTML = `<tr><td colspan="8">Select a source…</td></tr>`;
      return;
    }

    const params = new URLSearchParams();
    params.set('action','urlifywriterautoscan_items_list');
    params.set('nonce', NONCE);
    params.set('source_id', String(src));
    if (st) params.set('status', st);
    params.set('page', String(page));
    params.set('per_page', String(pp));

    tbody.innerHTML = `<tr><td colspan="8">${esc(t('loading','Loading…'))}</td></tr>`;

    fetch(AJAX, {
      method:'POST',
      headers:{'Content-Type':'application/x-www-form-urlencoded; charset=UTF-8'},
      body: params
    })
    .then(r=>r.json())
    .then(res=>{
      if (!res || res.success!==true) throw res;

      const rows = res.data.items || [];
      const total = res.data.total || 0;

      if (!rows.length){
        tbody.innerHTML = `<tr><td colspan="8">No items.</td></tr>`;
      } else {
        tbody.innerHTML = rows.map(r=>(
          `<tr data-id="${r.id}">
            <td><input type="checkbox" class="urlifywriter-chk"></td>
            <td>${r.id}</td>
            <td>${Math.round(r.priority||0)}</td>
            <td>${esc(r.title||'')}</td>
            <td><a href="${esc(r.url)}" target="_blank" rel="noopener">${esc(r.url)}</a></td>
            <td>
              ${esc(r.status||'')}
              ${ (r.status === 'failed' && r.last_error)
                  ? `<div class="urlifywriter-meta-mini urlifywriter-danger-mini">${esc(r.last_error)}</div>`
                  : '' }
            </td>
            <td>${esc(r.detected_at||'')}</td>
            <td class="actions">
              ${ r.post_id
                  ? `<a href="${esc(r.post_admin)}">Edit</a> · <a href="${esc(r.post_url)}" target="_blank" rel="noopener">View</a> · `
                  : '' }
              ${ (r.status === 'pending' || r.status === 'failed')
                  ? `<a href="#" data-act="ignore" class="urlifywriter-danger">Ignore</a>` : '' }
              ${ r.status === 'failed' ? ` · <a href="#" data-act="retry">Retry</a>` : '' }
            </td>
          </tr>`
        )).join('');
        hookItemActions();
      }

      const pages = Math.max(1, Math.ceil(total / pp));
      const pager = qs('#urlifywriter-items-pager');
      if (pager) {
        pager.innerHTML = `Total: ${total} — Page ${page}/${pages}
          ${page>1?'<a href="#" data-p="-1">« Prev</a>':''}
          ${page<pages?'<a href="#" data-p="+1" style="margin-left:6px;">Next »</a>':''}`;

        pager.querySelectorAll('a[data-p]').forEach(a=>{
          a.addEventListener('click', (ev)=>{
            ev.preventDefault();
            const dir = a.getAttribute('data-p');
            page = (dir==='+1') ? Math.min(pages, page+1) : Math.max(1, page-1);
            loadItems();
          });
        });
      }
    })
    .catch(()=>{
      tbody.innerHTML = `<tr><td colspan="8">Error.</td></tr>`;
    });
  }

  function hookItemActions(){
    qsa('#urlifywriter-items-tbody tr').forEach(tr=>{
      tr.addEventListener('click', (ev)=>{
        const a = ev.target.closest('a[data-act]');
        if (!a) return;
        ev.preventDefault();

        const id = parseInt(tr.getAttribute('data-id')||'0',10);
        const act= a.getAttribute('data-act');

        if (act==='retry'){
          const params = new URLSearchParams();
          params.set('action','urlifywriterautoscan_items_retry');
          params.set('nonce', NONCE);
          params.append('ids[0]', String(id));
          fetch(AJAX, {method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded; charset=UTF-8'}, body: params})
            .then(r=>r.json()).then(()=>loadItems());
        } else if (act==='ignore') {
          ajax('urlifywriterautoscan_item_ignore', {id}).then(()=>loadItems());
        }
      });
    });
  }

  qs('#urlifywriter-items-refresh')?.addEventListener('click', ()=>{ page=1; loadItems(); });
  qs('#urlifywriter-items-source')?.addEventListener('change', ()=>{ page=1; loadItems(); });
  qs('#urlifywriter-items-status')?.addEventListener('change', ()=>{ page=1; loadItems(); });
  qs('#urlifywriter-items-perpage')?.addEventListener('change', ()=>{ page=1; loadItems(); });

  qs('#urlifywriter-items-checkall')?.addEventListener('change', (ev)=>{
    const on = ev.target.checked;
    qsa('#urlifywriter-items-tbody .urlifywriter-chk').forEach(ch => ch.checked = on);
  });

  qs('#urlifywriter-items-retry-failed')?.addEventListener('click', ()=>{
    if (!confirm(t('confirmRetryAll','Retry all failed items?'))) return;
    ajax('urlifywriterautoscan_items_retry', {}).then(()=>{ page=1; loadItems(); });
  });

  /* init */
  loadCronScan();
  loadCronGen();
  loadSources();
})();
