<?php
// includes/post-writer.php
if ( ! defined('ABSPATH') ) { exit; }

/**
 * Crea un borrador en WordPress con el contenido generado y aplica
 * categoría, etiquetas y metadatos SEO si vienen en $generated.
 *
 * @param array  $generated [
 *   'title','content','excerpt',
 *   'category_id' (int), 'tags' (array de strings),
 *   'seo_title','seo_description',
 *   'author_id' (int opcional), 'lang' (opcional)
 * ]
 * @param string $origin_url
 * @return int|WP_Error post_id o error
 */
function urlifywritercreate_draft_from_generated( array $generated, string $origin_url ) {
	$title   = wp_strip_all_tags( $generated['title'] ?? '' );
	$content = (string) ( $generated['content'] ?? '' );

	// Excerpt: usa el proporcionado; si viene vacío, deriva del contenido
	$excerpt = isset($generated['excerpt'])
		? wp_strip_all_tags( $generated['excerpt'] )
		: '';
	if ( $excerpt === '' ) {
		$plain   = trim( preg_replace('/\s+/', ' ', wp_strip_all_tags($content) ) );
		$excerpt = wp_trim_words( $plain, 40, '…' );
	}

	// Autor: prioriza el recibido; si no, opción del plugin; si no, usuario actual
	$author_id = isset($generated['author_id']) ? absint($generated['author_id']) : 0;
	if ( ! $author_id ) {
		$default_author = (int) get_option('urlifywriterarticle_author', 0);
		$author_id = $default_author ?: get_current_user_id();
	}

	$postarr = [
		'post_title'   => $title ?: __('Untitled Draft', 'urlifywriter'),
		'post_content' => $content,
		'post_excerpt' => $excerpt,
		'post_status'  => 'draft',
		'post_type'    => 'post',
		'post_author'  => $author_id,
	];

	$post_id = wp_insert_post( $postarr, true );
	if ( is_wp_error($post_id) ) {
		return $post_id;
	}

	// ====== Términos: categoría y etiquetas ======
	$category_id = isset($generated['category_id']) ? absint($generated['category_id']) : 0;
	if ( $category_id && term_exists( (int) $category_id, 'category' ) ) {
		wp_set_post_categories( $post_id, [ (int) $category_id ], false );
	}

	$tags_sanitized = [];
	if ( ! empty($generated['tags']) && is_array($generated['tags']) ) {
		foreach ( $generated['tags'] as $t ) {
			$t = sanitize_text_field( (string) $t );
			if ( $t !== '' ) $tags_sanitized[] = $t;
		}
	}
	if ( ! empty($tags_sanitized) ) {
		wp_set_post_tags( $post_id, $tags_sanitized, false );
	}

	// ====== Metadatos de origen e idioma ======
	update_post_meta( $post_id, '_urlifywriterorigin_url', esc_url_raw( $origin_url ) );

	// Idioma: usa el recibido o el de ajustes (con fallback al locale WP)
	$lang = isset($generated['lang']) ? (string) $generated['lang'] : '';
	if ( $lang === '' ) {
		if ( function_exists('urlifywriterget_default_lang_from_wp') ) {
			$lang = get_option('urlifywriterarticle_language', urlifywriterget_default_lang_from_wp());
		} else {
			$locale = get_locale();
			$key = strtolower(str_replace('-', '_', $locale));
			if (in_array($key, ['zh_tw','zh_hk'], true))      $lang = 'zh-Hant';
			elseif (in_array($key, ['zh_cn','zh_sg'], true))  $lang = 'zh';
			elseif ($key === 'iw_il' || $key === 'he_il')     $lang = 'he';
			elseif ($key === 'nb_no')                         $lang = 'no';
			else                                              $lang = substr($locale, 0, 2) ?: 'en';
		}
	}
	update_post_meta( $post_id, '_urlifywriterlang', $lang );

	// ====== SEO: Yoast (y compat opcional con Rank Math / AIOSEO) ======
	$seo_title       = isset($generated['seo_title']) ? wp_strip_all_tags($generated['seo_title']) : '';
	$seo_description = isset($generated['seo_description']) ? wp_strip_all_tags($generated['seo_description']) : '';

	if ( $seo_title !== '' ) {
		update_post_meta( $post_id, '_yoast_wpseo_title', $seo_title );
		// Compat opcional:
		update_post_meta( $post_id, 'rank_math_title', $seo_title );
		update_post_meta( $post_id, '_aioseo_title', $seo_title );
	}
	if ( $seo_description !== '' ) {
		update_post_meta( $post_id, '_yoast_wpseo_metadesc', $seo_description );
		// Compat opcional:
		update_post_meta( $post_id, 'rank_math_description', $seo_description );
		update_post_meta( $post_id, '_aioseo_description', $seo_description );
	}

	// Flag interno
	update_post_meta( $post_id, '_urlifywritergenerated_simulation', 1 );

	// Debug (opcional)
	/*if ( defined('WP_DEBUG') && WP_DEBUG ) {
		error_log( sprintf(
			'[URLIFYWRITER] post-writer saved id=%d | cat=%d | tags=%d | seo_title=%s | seo_desc=%s',
			$post_id,
			$category_id,
			count($tags_sanitized),
			$seo_title !== '' ? 'yes' : 'no',
			$seo_description !== '' ? 'yes' : 'no'
		) );
	}*/

	return $post_id;
}
