<?php
// includes/generator.php
if ( ! defined('ABSPATH') ) { exit; }

/**
 * Llamada HTTP firmada al Worker (HMAC)
 */
if ( ! function_exists('urlifywritercloud_post') ) {
	function urlifywritercloud_post( $endpoint, array $payload, $timeout = null ) {
		if ( empty($endpoint) ) {
			return new WP_Error('no_endpoint', __('Remote endpoint is not configured.', 'urlifywriter'));
		}
		if ( ! defined('URLIFYWRITER_LIC_HMAC') ) {
			return new WP_Error('no_hmac', __('HMAC secret is not configured.', 'urlifywriter'));
		}

		$body = wp_json_encode($payload);
		$sig  = hash_hmac('sha256', $body, URLIFYWRITER_LIC_HMAC);

		$effective_timeout = is_null($timeout)
			? ( defined('URLIFYWRITER_REMOTE_TIMEOUT') ? (int) URLIFYWRITER_REMOTE_TIMEOUT : 45 )
			: (int) $timeout;

		$args = [
			'timeout' => $effective_timeout,
			'headers' => [
				'Content-Type' => 'application/json',
				'X-Signature'  => $sig,
			],
			'body' => $body,
		];

		$res = wp_remote_post($endpoint, $args);

		// Errores de transporte (incluye cURL 28)
		if ( is_wp_error($res) ) {
			$msg = $res->get_error_message();
			// Timeout típicamente: "cURL error 28: Operation timed out ..."
			if ( stripos($msg, 'cURL error 28') !== false || stripos($msg, 'timed out') !== false ) {
				return new WP_Error(
					'remote_timeout',
					sprintf(
						/* translators: %ds is the timeout seconds */
						__('The remote generation exceeded the timeout (%ds). Try again or increase the timeout.', 'urlifywriter'),
						$effective_timeout
					),
					[ 'transport_error' => $msg ]
				);
			}
			return $res; // otro error de transporte
		}

		$code = (int) wp_remote_retrieve_response_code($res);
		$raw  = wp_remote_retrieve_body($res);
		$json = json_decode($raw, true);

		// Respuestas no-200 o JSON inválido → errores claros
		if ( $code !== 200 ) {
			$remoteMsg = is_array($json) && !empty($json['error']) ? (string) $json['error'] : '';
			switch ($code) {
				case 401:
					return new WP_Error(
						'auth_failed',
						__('Signature check failed. Verify the HMAC secret in the plugin and in the Worker.', 'urlifywriter'),
						[ 'code'=>$code, 'body'=>$raw, 'json'=>$json ]
					);

			case 403:
				// Worker puede devolver distintos estados y errores semánticos
				if ( is_array($json) ) {
					$state  = (string) ($json['state']  ?? '');
					$err    = (string) ($json['error']  ?? '');
					$detail = (string) ($json['detail'] ?? '');
					$msg    = (string) ($json['message'] ?? '');

					// 🔍 Log de depuración
					/*if ( defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG ) {
						error_log("[URLIFYWRITER][403] state={$state} err={$err} detail={$detail} msg={$msg} raw=" . substr($raw, 0, 300));
					}*/

					// 👉 contempla 'detail' además de 'error' y un posible state específico
					if ( $err === 'limit_reached'
						 || $detail === 'limit_reached'
						 || $state === 'exhausted' ) {
						return new WP_Error(
							'limit_reached',
							__('You have reached your article limit.', 'urlifywriter'),
							[ 'code'=>$code, 'body'=>$raw, 'json'=>$json ]
						);
					}

					if ( $state === 'invalid' || $err === 'license_invalid' ) {
						return new WP_Error(
							'license_invalid',
							__('License is not valid. Go to UrlifyWriter → License and check your key.', 'urlifywriter'),
							[ 'code'=>$code, 'body'=>$raw, 'json'=>$json ]
						);
					}
				}

				return new WP_Error(
					'forbidden',
					$remoteMsg ?: __('Request was rejected by the remote service (403).', 'urlifywriter'),
					[ 'code'=>$code, 'body'=>$raw, 'json'=>$json ]
				);



				case 400:
					if ( $remoteMsg === 'Stale or missing timestamp' ) {
						return new WP_Error(
							'stale_ts',
							__('Request timestamp is out of window. Please try again.', 'urlifywriter'),
							[ 'code'=>$code, 'body'=>$raw, 'json'=>$json ]
						);
					}
					return new WP_Error(
						'bad_request',
						$remoteMsg ?: __('Invalid request sent to the remote service (400).', 'urlifywriter'),
						[ 'code'=>$code, 'body'=>$raw, 'json'=>$json ]
					);

				case 404:
					return new WP_Error(
						'remote_404',
						__('Remote route not found. Check the endpoint URL.', 'urlifywriter'),
						[ 'code'=>$code, 'body'=>$raw, 'json'=>$json ]
					);

				case 423:
					return new WP_Error(
						'remote_404',
						__('No images found.', 'urlifywriter'),
						[ 'code'=>$code, 'body'=>$raw, 'json'=>$json ]
					);

				default:
					if ( $code >= 500 ) {
						return new WP_Error(
							'remote_unavailable',
							__('Remote service is unavailable. Please try again later.', 'urlifywriter'),
							[ 'code'=>$code, 'body'=>$raw, 'json'=>$json ]
						);
					}
					return new WP_Error(
						'bad_http',
						sprintf(
							/* translators: %d is the HTTP status code returned by the remote service. */
							__('Unexpected response from remote service (HTTP %d).', 'urlifywriter'),
							$code
						),
						[ 'code' => $code, 'body' => $raw, 'json' => $json ]
					);

			}
		}

		// 200 OK pero JSON inválido
		if ( ! is_array($json) ) {
			return new WP_Error(
				'bad_json',
				__('Remote service returned invalid JSON.', 'urlifywriter'),
				[ 'code'=>$code, 'body'=>$raw ]
			);
		}

		// Éxito
		return $json;
	}
}



/**
 * Alta nivel: pide al Worker que valide la licencia y devuelva un borrador.
 * Devuelve un array con: title, content, excerpt, tags (para tu flujo actual).
 *
 * @param string $source_url
 * @param array  $args ['max_words','min_words','title_hint','instructions','extracted']
 * @return array|WP_Error
 */
if ( ! function_exists('urlifywritergenerate_article_from_url') ) {
	function urlifywritergenerate_article_from_url( $source_url, array $args = [] ) {
		$endpoint = defined('URLIFYWRITER_ART_API') ? URLIFYWRITER_ART_API : '';
		$license_opt  = get_option('urlifywriterlicense_key', '');

		if ( empty($source_url) ) {
			return new WP_Error('no_url', __('Source URL is required.', 'urlifywriter'));
		}


		$license_to_send = ($license_opt !== '' ? $license_opt : '');
		
		$max_words    = isset($args['max_words'])    ? (int) $args['max_words']    : 800;
		$min_words    = isset($args['min_words'])    ? (int) $args['min_words']    : 600;
		$title_hint   = isset($args['title_hint'])   ? (string) $args['title_hint'] : '';
		$instructions = isset($args['instructions']) ? (string) $args['instructions'] : '';
		$extracted    = isset($args['extracted'])    ? (string) $args['extracted']    : '';

		// ⬇️ Nuevo: idioma desde ajustes (fallback al idioma de WP mapeado)
		if ( ! function_exists('urlifywriterget_default_lang_from_wp') ) {
			// Si no cargaste la función en este contexto, usa este fallback simple:
			function urlifywriterget_default_lang_from_wp() {
				$locale = get_locale();
				$key = strtolower(str_replace('-', '_', $locale));
				if (in_array($key, ['zh_tw','zh_hk'], true)) return 'zh-Hant';
				if (in_array($key, ['zh_cn','zh_sg'], true)) return 'zh';
				if ($key === 'iw_il' || $key === 'he_il') return 'he';
				if ($key === 'nb_no') return 'no';
				return substr($locale, 0, 2) ?: 'en';
			}
		}
		$lang = get_option('urlifywriterarticle_language', urlifywriterget_default_lang_from_wp());

		$payload = [
			'ts'            => (int) round( microtime(true) * 1000 ),
			'license_key'   => $license_to_send,
			'domain'         => (string) wp_parse_url( home_url(), PHP_URL_HOST ),
			'site_url'      => (string) home_url('/'),
			'plugin'        => 'urlifywriter',
			'plugin_version'=> defined('URLIFYWRITER_VERSION') ? URLIFYWRITER_VERSION : '1.0.0',
			'wp_version'    => get_bloginfo('version'),
			'php_version'   => PHP_VERSION,

			'url'           => (string) $source_url,
			'max_words'     => max(200, min(3000, $max_words)),
			'min_words'     => max(100, min(3000, $min_words)),
			'title_hint'    => $title_hint,
			'instructions'  => $instructions,
			'extracted'     => $extracted,

			// ⬇️ Nuevo: idioma
			'lang'          => $lang,
		];
	

		$res = urlifywritercloud_post($endpoint, $payload);
		if ( is_wp_error($res) ) return $res;

		if ( empty($res['ok']) ) {
			return new WP_Error('remote_not_ok', __('Remote service returned a non-OK response.', 'urlifywriter'));
		}
		if ( isset($res['state']) && $res['state'] !== 'valid' ) {
			return new WP_Error('license_invalid', __('License is not valid.', 'urlifywriter'));
		}

		$title   = isset($res['title'])   ? (string) $res['title']   : 'Untitled Draft';
		$content = isset($res['content']) ? (string) $res['content'] : '<p>Empty content.</p>';
		$tags    = isset($res['tags']) && is_array($res['tags']) ? $res['tags'] : [];

		$plain = wp_strip_all_tags( $content );
		$plain = trim( preg_replace('/\s+/', ' ', $plain) );
		$sent  = preg_split('/(?<=[\.\!\?])\s+/u', $plain);
		$excerpt = implode(' ', array_slice(array_filter((array)$sent), 0, 2));
		if ($excerpt === '') $excerpt = wp_trim_words($plain, 40, '…');

		return [
			'title'   => $title,
			'content' => $content,
			'excerpt' => $excerpt,
			'tags'    => $tags,
		];
	}
}