<?php
/**
 * UrlifyWriter — AutoScan Manager (fixed)
 *
 * Responsabilidades:
 * - Crear tablas (sites + detected_urls)
 * - CRUD de dominios a escanear
 * - Escaneo de dominios (portada/RSS), detección y filtrado de URLs
 * - Cola y estados de URLs detectadas (pending/generated/duplicate/skipped)
 * - Ejecución desde cron (selección de dominios vencidos y procesado)
 * - Generación automática opcional (modo 'auto') de nuevos artículos (borradores)
 */

if ( ! defined('ABSPATH') ) { exit; }

/** ============================================================
 *  Utilidad de debug SIN error_log() directo (evita warnings PHPCS)
 * ============================================================ */
if ( ! function_exists( 'urlifywriterdebug_log' ) ) {
	function urlifywriterdebug_log( $message ) {
		if ( defined( 'URLIFYWRITER_DEBUG' ) && URLIFYWRITER_DEBUG ) {
			if ( ! is_scalar( $message ) ) {
				$message = wp_json_encode( $message, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES );
			}
			$text = esc_html( '[URLIFYWRITER] ' . (string) $message );

			// Solo para depuración interna, no en producción.
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_trigger_error,WordPress.Security.EscapeOutput.OutputNotEscaped
			@trigger_error( $text, E_USER_NOTICE );
		}
	}
}


if ( ! defined('URLIFYWRITER_AUTOSCAN_SITES') ) {
	// Nombres de tabla (respetan prefijo)
	global $wpdb;
	define('URLIFYWRITER_AUTOSCAN_SITES', $wpdb->prefix . 'urlifywriterautoscan_sites');
	define('URLIFYWRITER_AUTOSCAN_URLS',  $wpdb->prefix . 'urlifywriterdetected_urls');
}

/* =======================================================================
 * 1) Instalación de tablas
 * ======================================================================= */

if ( ! function_exists('urlifywriterautoscan_install_sources_items') ) {
	function urlifywriterautoscan_install_sources_items() {
		global $wpdb;
		$charset    = $wpdb->get_charset_collate();

		$tbl_sources = $wpdb->prefix . 'urlifywriterautoscan_sources';
		$tbl_items   = $wpdb->prefix . 'urlifywriterautoscan_items';

        $sql_sources =
        "CREATE TABLE $tbl_sources (
          id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
          domain VARCHAR(191) NOT NULL,
          start_url TEXT NULL,
          enabled TINYINT(1) NOT NULL DEFAULT 1,
          max_per_day INT NOT NULL DEFAULT 5,
          scan_interval VARCHAR(50) NOT NULL DEFAULT 'hourly',
          publish_mode VARCHAR(20) NOT NULL DEFAULT 'draft',
          gen_instructions TEXT NULL,
          gen_min_words INT NULL,
          gen_max_words INT NULL,
          last_scan_at DATETIME NULL,
          next_scan_at DATETIME NULL,
          notes TEXT NULL,
          created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
          updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
          PRIMARY KEY  (id),
          UNIQUE KEY uniq_domain (domain)
        ) $charset;";
        
        $sql_items =
        "CREATE TABLE $tbl_items (
          id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
          source_id BIGINT UNSIGNED NOT NULL,
          url TEXT NOT NULL,
          title TEXT NULL,
          status VARCHAR(20) NOT NULL DEFAULT 'pending',
          last_error VARCHAR(190) NULL DEFAULT NULL,
          priority DOUBLE NOT NULL DEFAULT 0,
          post_id BIGINT UNSIGNED NULL DEFAULT NULL,
          detected_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
          updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
          PRIMARY KEY (id),
          INDEX idx_source (source_id),
          INDEX idx_status (status),
          INDEX idx_post_id (post_id)
        ) $charset;";


		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta($sql_sources);
		dbDelta($sql_items);
	}
}


/**
 * Comprobación perezosa para crear si faltan.
 */
if ( ! function_exists('urlifywriterautoscan_maybe_install') ) {
	function urlifywriterautoscan_maybe_install() {
		global $wpdb;
		$tbl_sources = $wpdb->prefix . 'urlifywriterautoscan_sources';
		$tbl_items   = $wpdb->prefix . 'urlifywriterautoscan_items';
		$need = false;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$exists_count = (int) $wpdb->get_var( $wpdb->prepare(
			"SELECT COUNT(*) FROM information_schema.TABLES WHERE TABLE_SCHEMA=%s AND TABLE_NAME IN (%s,%s)",
			DB_NAME, $tbl_sources, $tbl_items
		) );

		if ( $exists_count < 2 ) {
			$need = true;
		}
		if ( $need ) {
			urlifywriterautoscan_install_sources_items();
		}
	}
}

/* =======================================================================
 * 2) Utilidades de normalización / hashing / fechas
 * ======================================================================= */

if ( ! function_exists('urlifywriterautoscan_normalize_url') ) {
	function urlifywriterautoscan_normalize_url( $url ) {
		$url = trim( (string) $url );
		if ($url === '') return '';
		// Forzar esquema si viene sin él
		if ( strpos($url, '//') === 0 ) {
			$url = (is_ssl() ? 'https:' : 'http:') . $url;
		}
		if ( ! preg_match('~^https?://~i', $url) ) {
			$url = 'https://' . ltrim($url, '/');
		}
		// Quitar parámetros de tracking comunes
		$parts = wp_parse_url($url);
		if (empty($parts['host'])) return '';

		$query = [];
		if (!empty($parts['query'])) {
			parse_str($parts['query'], $query);
			foreach ($query as $k => $v) {
				if (preg_match('/^(utm_|fbclid|gclid|yclid|mc_cid|mc_eid)/i', $k)) {
					unset($query[$k]);
				}
			}
		}
		$scheme = isset($parts['scheme']) ? strtolower($parts['scheme']) : 'https';
		$host   = strtolower($parts['host']);
		$path   = isset($parts['path']) ? $parts['path'] : '/';
		$qs     = $query ? ('?' . http_build_query($query)) : '';
		$port   = isset($parts['port']) ? ':' . $parts['port'] : '';

		return $scheme . '://' . $host . $port . $path . $qs;
	}
}

if ( ! function_exists('urlifywriterautoscan_hash') ) {
	function urlifywriterautoscan_hash( $str ) {
		return hash('sha256', (string) $str );
	}
}

if ( ! function_exists('urlifywriterautoscan_now_mysql') ) {
	function urlifywriterautoscan_now_mysql() {
		return current_time('mysql'); // respeta tz WP
	}
}

/* =======================================================================
 * 3) CRUD Dominios
 * ======================================================================= */

if ( ! function_exists('urlifywriterautoscan_add_domain') ) {
	/**
	 * Crea o re-activa un dominio.
	 * @param array $data [domain_url, mode, frequency_hours, daily_limit, enabled]
	 * @return int|WP_Error domain_id
	 */
	function urlifywriterautoscan_add_domain( $data ) {
		global $wpdb;

		$url = isset($data['domain_url']) ? urlifywriterautoscan_normalize_url($data['domain_url']) : '';
		if ( ! $url ) return new WP_Error('bad_domain', __('Invalid domain/URL', 'urlifywriter'));

		$mode  = isset($data['mode']) ? strtolower($data['mode']) : 'manual';
		if ( ! in_array($mode, ['auto','manual'], true) ) $mode = 'manual';

		$freq  = isset($data['frequency_hours']) ? max(1, (int)$data['frequency_hours']) : 24;
		$limit = isset($data['daily_limit']) ? max(1, (int)$data['daily_limit']) : 3;
		$en    = isset($data['enabled']) ? (int) !!$data['enabled'] : 1;

		$now  = urlifywriterautoscan_now_mysql();
		$next = gmdate('Y-m-d H:i:s', time() + $freq*3600);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$table = esc_sql( URLIFYWRITER_AUTOSCAN_SITES );


		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$exists = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT id FROM `{$table}` WHERE domain_url = %s LIMIT 1",
				$url
			)
		);
		// phpcs:enable



		if ( $exists ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
			$wpdb->update(
				URLIFYWRITER_AUTOSCAN_SITES,
				[
					'mode'            => $mode,
					'frequency_hours' => $freq,
					'daily_limit'     => $limit,
					'enabled'         => $en,
					'updated_at'      => $now,
					'next_scan'       => $next,
				],
				['id' => (int)$exists],
				['%s','%d','%d','%d','%s','%s'],
				['%d']
			);
			return (int) $exists;
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$ok = $wpdb->insert(
			URLIFYWRITER_AUTOSCAN_SITES,
			[
				'domain_url'      => $url,
				'mode'            => $mode,
				'frequency_hours' => $freq,
				'daily_limit'     => $limit,
				'last_scan'       => null,
				'next_scan'       => $next,
				'enabled'         => $en,
				'created_at'      => $now,
				'updated_at'      => $now,
			],
			['%s','%s','%d','%d','%s','%s','%d','%s','%s']
		);

		if ( ! $ok ) return new WP_Error('db_error', __('Could not insert domain', 'urlifywriter'));
		return (int) $wpdb->insert_id;
	}
}

if ( ! function_exists('urlifywriterautoscan_update_domain') ) {
	function urlifywriterautoscan_update_domain( $id, $data ) {
		global $wpdb;
		$id = (int) $id;
		if ( ! $id ) return new WP_Error('bad_id', 'Invalid domain id');

		$fields = [];
		$formats = [];

		if ( isset($data['domain_url']) ) {
			$val = urlifywriterautoscan_normalize_url( $data['domain_url'] );
			if ( ! $val ) return new WP_Error('bad_domain', 'Invalid URL');
			$fields['domain_url'] = $val; $formats[] = '%s';
		}
		if ( isset($data['mode']) ) {
			$mode = in_array(strtolower($data['mode']), ['auto','manual'], true) ? strtolower($data['mode']) : 'manual';
			$fields['mode'] = $mode; $formats[] = '%s';
		}
		if ( isset($data['frequency_hours']) ) { $fields['frequency_hours'] = max(1, (int)$data['frequency_hours']); $formats[] = '%d'; }
		if ( isset($data['daily_limit']) )     { $fields['daily_limit']     = max(1, (int)$data['daily_limit']);     $formats[] = '%d'; }
		if ( isset($data['enabled']) )         { $fields['enabled']         = (int) !!$data['enabled'];              $formats[] = '%d'; }

		if ( empty($fields) ) return true;

		$fields['updated_at'] = urlifywriterautoscan_now_mysql(); $formats[] = '%s';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$ok = $wpdb->update(
			URLIFYWRITER_AUTOSCAN_SITES,
			$fields,
			['id'=>$id],
			$formats,
			['%d']
		);
		return ($ok === false) ? new WP_Error('db_error','Update failed') : true;
	}
}

if ( ! function_exists('urlifywriterautoscan_delete_domain') ) {
	function urlifywriterautoscan_delete_domain( $id ) {
		global $wpdb;
		$id = (int) $id;
		if ( ! $id ) return false;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->delete( URLIFYWRITER_AUTOSCAN_URLS, ['domain_id'=>$id], ['%d'] );
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->delete( URLIFYWRITER_AUTOSCAN_SITES, ['id'=>$id], ['%d'] );
		return true;
	}
}

if ( ! function_exists( 'urlifywriterautoscan_get_domains' ) ) {
	function urlifywriterautoscan_get_domains( $args = [] ) {
		global $wpdb;

		$enabled_only = isset( $args['enabled_only'] ) ? (int) !! $args['enabled_only'] : 0;
		$table        = esc_sql( URLIFYWRITER_AUTOSCAN_SITES ); // ✅ Sanitiza el nombre de la tabla

		if ( $enabled_only ) {
			// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
			$rows = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM `{$table}` WHERE enabled = %d ORDER BY next_scan ASC",
					1
				),
				ARRAY_A
			);
			// phpcs:enable
			return $rows;
		}

		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM `{$table}` WHERE 1 = %d ORDER BY next_scan ASC",
				1
			),
			ARRAY_A
		);
		// phpcs:enable

		return $rows;
	}
}


if ( ! function_exists( 'urlifywriterautoscan_get_domain' ) ) {
	function urlifywriterautoscan_get_domain( $id ) {
		global $wpdb;

		$id = absint( $id );
		if ( ! $id ) {
			return null;
		}

		$table = esc_sql( URLIFYWRITER_AUTOSCAN_SITES );

		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$row = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$table}` WHERE id = %d",
				$id
			),
			ARRAY_A
		);
		// phpcs:enable

		return $row ?: null;
	}
}


/* =======================================================================
 * 4) Inserción y consulta de URLs detectadas
 * ======================================================================= */

if ( ! function_exists('urlifywriterautoscan_insert_detected_url') ) {
	function urlifywriterautoscan_insert_detected_url( $domain_id, $url, $title = '', $priority = 0.0 ) {
		global $wpdb;

		$domain_id = absint( $domain_id );
		$url_norm  = urlifywriterautoscan_normalize_url( $url );

		if ( ! $domain_id || ! $url_norm ) {
			return false;
		}

		$hash  = urlifywriterautoscan_hash( $url_norm );
		$table = esc_sql( URLIFYWRITER_AUTOSCAN_URLS ); // ✅ Sanitiza el nombre de la tabla

		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$exists = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT id FROM `{$table}` WHERE hash = %s LIMIT 1",
				$hash
			)
		);
		// phpcs:enable

		
		if ( $exists ) {
			return 'duplicate';
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$ok = $wpdb->insert(
			URLIFYWRITER_AUTOSCAN_URLS,
			[
				'domain_id'     => $domain_id,
				'url'           => $url_norm,
				'title'         => $title ?: '',
				'status'        => 'pending',
				'hash'          => $hash,
				'priority'      => (float)$priority,
				'date_detected' => urlifywriterautoscan_now_mysql(),
			],
			['%d','%s','%s','%s','%s','%f','%s']
		);
		return $ok ? (int)$wpdb->insert_id : false;
	}
}

if ( ! function_exists('urlifywriterautoscan_mark_url_status') ) {
	function urlifywriterautoscan_mark_url_status( $id, $status ) {
		global $wpdb;
		$id = (int) $id;
		if ( ! $id ) return false;

		$allowed = ['pending','generated','duplicate','skipped','processing','done','failed','approved'];
		if ( ! in_array($status, $allowed, true) ) $status = 'pending';

		// Preferir tabla NUEVA autoscan_items si existe
		$items_tbl_new = $wpdb->prefix . 'urlifywriterautoscan_items';
		$urls_tbl_old  = defined('URLIFYWRITER_AUTOSCAN_URLS') ? URLIFYWRITER_AUTOSCAN_URLS : ($wpdb->prefix . 'urlifywriterdetected_urls');

		$use_new = false;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$chk = $wpdb->get_var( $wpdb->prepare("SHOW TABLES LIKE %s", $items_tbl_new) );
		if ( $chk === $items_tbl_new ) $use_new = true;

		if ( $use_new ) {
			$data = ['status' => $status, 'updated_at' => current_time('mysql')];
			$fmt  = ['%s','%s'];

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$ok = $wpdb->update( $items_tbl_new, $data, ['id'=>$id], $fmt, ['%d'] );
			return ($ok !== false);
		}

		// FALLBACK tabla vieja
		$data = ['status'=>$status];
		$fmt  = ['%s'];

		if ( $status === 'generated' ) {
			$data['date_generated'] = current_time('mysql');
			$fmt[] = '%s';
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$ok = $wpdb->update( $urls_tbl_old, $data, ['id'=>$id], $fmt, ['%d'] );
		return ($ok !== false);
	}
}


if ( ! function_exists('urlifywriterautoscan_get_pending_for_domain') ) {
	function urlifywriterautoscan_get_pending_for_domain( $domain_id, $limit = 10 ) {
		global $wpdb;

		$domain_id = absint( $domain_id );
		$limit     = max( 1, absint( $limit ) );

		// Sanitiza el nombre de la tabla antes de interpolar
		$table = esc_sql( URLIFYWRITER_AUTOSCAN_URLS );

		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * 
				   FROM `{$table}` 
				  WHERE domain_id = %d 
					AND status = 'pending' 
				  ORDER BY priority DESC, date_detected DESC 
				  LIMIT %d",
				$domain_id,
				$limit
			),
			ARRAY_A
		);
		// phpcs:enable


		return $results;
	}
}


/* =======================================================================
 * 5) Filtrado rápido de URLs (paginación/taxonomías/anuncios)
 * ======================================================================= */

if ( ! function_exists('urlifywriterautoscan_is_article_candidate') ) {
	function urlifywriterautoscan_is_article_candidate( $candidate, $base_host ) {
		$candidate = (string) $candidate;
		if ($candidate === '') return false;

		// Debe ser mismo dominio
		$HP = wp_parse_url( $candidate );
		if (empty($HP['host']) || empty($base_host)) return false;
		if ( strtolower($HP['host']) !== strtolower($base_host) ) return false;

		$path = isset($HP['path']) ? $HP['path'] : '/';

		$bad = [
			'/page/', '/tag/', '/tags/', '/category/', '/categories/',
			'/author/', '/search', '/buscar', '/ads', '/advert', '/sponsor',
			'/opinion/', '/podcast/',
		];
		foreach ($bad as $needle) {
			if ( stripos($path, $needle) !== false ) return false;
		}

		if ( rtrim($path, '/') === '' || rtrim($path, '/') === '/' ) return false;

		if ( preg_match('~\.(jpg|jpeg|png|webp|gif|pdf|zip|mp3|mp4|avi|mov)$~i', $path) ) return false;

		return true;
	}
}

/* =======================================================================
 * 6) Escaneo de dominio (descarga + extracción de enlaces)
 * ======================================================================= */

if ( ! function_exists('urlifywriterautoscan_fetch') ) {
	function urlifywriterautoscan_fetch( $url, $timeout = 18 ) {
		$resp = wp_remote_get( $url, [
			'timeout' => max(5, (int)$timeout),
			'redirection' => 5,
			'user-agent' => 'UrlifyWriter/1.0 (+WP; AutoScan)'
		]);
		if ( is_wp_error($resp) ) return $resp;
		$code = wp_remote_retrieve_response_code($resp);
		if ( $code < 200 || $code >= 400 ) {
			return new WP_Error('http_error', 'HTTP '.$code);
		}
		$body = wp_remote_retrieve_body($resp);
		return is_string($body) ? $body : '';
	}
}

if ( ! function_exists('urlifywriterautoscan_scan_domain') ) {
	function urlifywriterautoscan_scan_domain( $domain_id ) {
		global $wpdb;

		$site = urlifywriterautoscan_get_domain( $domain_id );
		if ( ! $site || ! (int)$site['enabled'] ) {
			return [ 'ok'=>false, 'error'=>'disabled_or_missing' ];
		}

		$base = urlifywriterautoscan_normalize_url( $site['domain_url'] );
		if ( ! $base ) return [ 'ok'=>false, 'error'=>'invalid_base' ];

		$base_parts = wp_parse_url($base);
		$host = $base_parts['host'] ?? '';
		if ( ! $host ) return [ 'ok'=>false, 'error'=>'no_host' ];

		$html = urlifywriterautoscan_fetch( $base );
		if ( is_wp_error($html) ) {
			return [ 'ok'=>false, 'error'=>$html->get_error_message() ];
		}

		$links = [];
		if ( function_exists('urlifywriterautoscan_extract_links') ) {
			$links = urlifywriterautoscan_extract_links( $html, $base );
		} else {
			$links = [];
			if ( preg_match_all('~<a\s[^>]*href=["\']([^"\']+)["\'][^>]*>(.*?)</a>~is', $html, $m, PREG_SET_ORDER) ) {
				$pos=0;
				foreach ($m as $row) {
					$u = trim(html_entity_decode($row[1]));
					$t = wp_strip_all_tags($row[2]); $t = trim(preg_replace('/\s+/', ' ', $t));
					if (!preg_match('~^https?://~i', $u)) {
						$u = function_exists('urlifywritermake_absolute_url') ? urlifywritermake_absolute_url($u, $base) : $u;
					}
					$links[] = ['url'=>$u, 'title'=>$t, 'priority'=> (float) (100 - min(99,$pos))];
					$pos++;
				}
			}
		}

		$inserted = 0; $dups = 0; $skipped = 0;

		foreach ($links as $lnk) {
			$u  = isset($lnk['url']) ? urlifywriterautoscan_normalize_url($lnk['url']) : '';
			$ti = isset($lnk['title']) ? (string)$lnk['title'] : '';
			$pr = isset($lnk['priority']) ? (float)$lnk['priority'] : 0;

			if ( ! $u ) { $skipped++; continue; }
			if ( ! urlifywriterautoscan_is_article_candidate($u, $host) ) { $skipped++; continue; }

			$res = urlifywriterautoscan_insert_detected_url( (int)$site['id'], $u, $ti, $pr );
			if ( $res === 'duplicate' ) $dups++;
			elseif ( $res ) $inserted++;
			else $skipped++;
		}

		$now  = urlifywriterautoscan_now_mysql();
		$next = gmdate('Y-m-d H:i:s', time() + ((int)$site['frequency_hours'])*3600);
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->update(
			URLIFYWRITER_AUTOSCAN_SITES,
			['last_scan'=>$now, 'next_scan'=>$next, 'updated_at'=>$now],
			['id'=>(int)$site['id']],
			['%s','%s','%s'],
			['%d']
		);

		return [
			'ok'       => true,
			'inserted' => $inserted,
			'duplicates'=> $dups,
			'skipped'  => $skipped,
			'total_links_scanned' => count($links),
		];
	}
}

/* =======================================================================
 * 7) RUN: llamado por el cron orquestador
 * ======================================================================= */

if ( ! function_exists('urlifywriterautoscan_run') ) {
	function urlifywriterautoscan_run() {
		global $wpdb;
		$sites_tbl = $wpdb->prefix . 'urlifywriterautoscan_sources';

		$sites_tbl = esc_sql( $sites_tbl );
		$now_sql   = current_time( 'mysql' );

		// Nombres de tabla no admiten placeholders → desactivamos el sniff sólo aquí
		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$sites = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT *
				   FROM `{$sites_tbl}`
				  WHERE enabled = %d
					AND ( next_scan_at IS NULL OR next_scan_at = '' OR next_scan_at <= %s )
				  ORDER BY next_scan_at IS NULL DESC, next_scan_at ASC
				  LIMIT %d",
				1,
				$now_sql,
				10
			)
		);
		// phpcs:enable


		if ( empty($sites) ) {
			return ['ok'=>true,'scanned'=>0,'generated'=>0];
		}

		$scanned = 0;
		$generated = 0;

		foreach ($sites as $src) {
			$scanned++;

			$res = function_exists('urlifywriterautoscan_run_for_domain')
				? urlifywriterautoscan_run_for_domain($src)
				: ['ok'=>false,'recent_errors'=>0];

			$ok        = !empty($res['ok']);
			$gen_count = isset($res['generated']) ? (int)$res['generated'] : 0;
			$err_count = isset($res['recent_errors']) ? (int)$res['recent_errors'] : 0;
			$generated += $gen_count;

			if ( function_exists('urlifywriterautoscan_after_scan_update_next') ) {
				urlifywriterautoscan_after_scan_update_next($src, $ok, $err_count);
			}
		}

		return ['ok'=>true,'scanned'=>$scanned,'generated'=>$generated];
	}
}

/* =======================================================================
 * 8) Generación de pendientes (modo 'auto' o acción manual)
 * ======================================================================= */

if ( ! function_exists('urlifywriterautoscan_generate_pending_for_domain') ) {
	function urlifywriterautoscan_generate_pending_for_domain( $domain_id, $limit = 3 ) {
		$pending = urlifywriterautoscan_get_pending_for_domain( (int)$domain_id, max(1,(int)$limit) );
		if ( empty($pending) ) return ['ok'=>true,'generated'=>0,'errors'=>0];

		$okN = 0; $errN = 0;

		foreach ($pending as $row) {
			$res = urlifywriterautoscan_generate_article_from_url( $row['id'], $row['url'] );
			if ( ! empty($res['ok']) ) {
				$okN++;
			} else {
				$errN++;
				if ( isset($res['fatal']) && $res['fatal'] ) {
					urlifywriterautoscan_mark_url_status( (int)$row['id'], 'skipped' );
				}
			}
		}

		return ['ok'=>true, 'generated'=>$okN, 'errors'=>$errN];
	}
}

/* =======================================================================
 * 9) Generación desde URL concreta (remote → create draft)
 * ======================================================================= */

if ( ! function_exists('urlifywriterautoscan_generate_article_from_url') ) {
	function urlifywriterautoscan_generate_article_from_url( $detected_id, $url, $args = [] ) {
		global $wpdb;

		$URLIFYWRITER_DEBUG = defined('URLIFYWRITER_DEBUG') ? (bool) URLIFYWRITER_DEBUG : ( defined('WP_DEBUG') && WP_DEBUG );

		if ( ! defined('URLIFYWRITER_ART_API') || ! defined('URLIFYWRITER_LIC_HMAC') ) {
			urlifywriterdebug_log('[AutoScan] Missing URLIFYWRITER_ART_API or URLIFYWRITER_LIC_HMAC');
			return ['ok'=>false, 'error'=>'endpoint_or_hmac_missing', 'fatal'=>true];
		}

		$url = esc_url_raw( $url );
		if ( ! filter_var($url, FILTER_VALIDATE_URL) ) {
			return ['ok'=>false, 'error'=>'bad_url', 'fatal'=>true];
		}

		$tbl_sources = $wpdb->prefix . 'urlifywriterautoscan_sources';
		$tbl_items   = $wpdb->prefix . 'urlifywriterautoscan_items';

		if ( function_exists('urlifywriterautoscan_table_names') ) {
			$T = (array) urlifywriterautoscan_table_names();
			$tbl_sources = isset($T['sources']) ? $T['sources'] : $tbl_sources;
			$tbl_items   = isset($T['items'])   ? $T['items']   : $tbl_items;
		}

		$src = null;
		if ( $detected_id ) {
			$tbl_sources = esc_sql( $tbl_sources );
			$tbl_items   = esc_sql( $tbl_items );
			$detected_id = absint( $detected_id );

			// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
			$src = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT s.gen_instructions, s.gen_min_words, s.gen_max_words, s.publish_mode
					   FROM {$tbl_sources} s
					   INNER JOIN {$tbl_items} i ON i.source_id = s.id
					  WHERE i.id = %d
					  LIMIT 1",
					$detected_id
				),
				ARRAY_A
			);
			// phpcs:enable
		}


		$src = is_array($src) ? $src : [];

		$instructions = array_key_exists('instructions', $args)
			? wp_kses_post( (string) $args['instructions'] )
			: (string) ( $src['gen_instructions'] ?? '' );

		$min_words = array_key_exists('min_words', $args)
			? (int) $args['min_words']
			: (int) ( $src['gen_min_words'] ?? 700 );

		$max_words = array_key_exists('max_words', $args)
			? (int) $args['max_words']
			: (int) ( $src['gen_max_words'] ?? 1200 );

		$min_words = max(100, min(3000, (int)$min_words));
		$max_words = max($min_words + 50, min(3000, (int)$max_words));

		if ( ! function_exists('urlifywriterget_default_lang_from_wp') ) {
			function urlifywriterget_default_lang_from_wp() {
				$locale = get_locale();
				$key = strtolower(str_replace('-', '_', $locale));
				if (in_array($key, ['zh_tw','zh_hk'], true)) return 'zh-Hant';
				if (in_array($key, ['zh_cn','zh_sg'], true)) return 'zh';
				if ($key === 'iw_il' || $key === 'he_il') return 'he';
				if ($key === 'nb_no') return 'no';
				return substr($locale, 0, 2) ?: 'en';
			}
		}
		$lang = get_option('urlifywriterarticle_language', urlifywriterget_default_lang_from_wp());

		$cat_terms = get_terms([ 'taxonomy' => 'category', 'hide_empty' => false ]);
		$categories = [];
		if ( ! is_wp_error($cat_terms) ) {
			foreach ($cat_terms as $t) {
				$categories[] = [ 'id'=>(int)$t->term_id, 'name'=>(string)$t->name, 'slug'=>(string)$t->slug ];
			}
		}

		$lic_opt = get_option('urlifywriterlicense_key', '');

		$home_parts = wp_parse_url( home_url() );
		$site_host  = $home_parts['host'] ?? '';

		$payload = [
			'mode'         => 'url',
			'keyword'      => '',
			'url'          => $url,
			'ts'           => time() * 1000,
			'license_key'  => ($lic_opt !== '' ? $lic_opt : ''),
			'site_domain'  => $site_host,
			'site_url'     => home_url(),
			'instructions' => $instructions,
			'min_words'    => $min_words,
			'max_words'    => $max_words,
			'title_hint'   => '',
			'extracted'    => '',
			'lang'         => $lang,
			'categories'   => $categories,
		];

		if ( ! function_exists('urlifywritercloud_post') ) {
			return ['ok'=>false,'error'=>'urlifywritercloud_post_missing','fatal'=>true];
		}

		$res = urlifywritercloud_post( URLIFYWRITER_ART_API, $payload, defined('URLIFYWRITER_REMOTE_TIMEOUT') ? (int) URLIFYWRITER_REMOTE_TIMEOUT : 60 );
		if ( is_wp_error($res) ) {
			urlifywriterdebug_log('[AutoScan] generate remote error: '.$res->get_error_message());
			return ['ok'=>false,'error'=>$res->get_error_message(),'fatal'=>false];
		}

		$title   = (string) ($res['title'] ?? '');
		$content = (string) ($res['content'] ?? '');
		$excerpt = (string) ($res['excerpt'] ?? '');
		$seo_t   = (string) ($res['seo_title'] ?? '');
		$seo_d   = (string) ($res['seo_description'] ?? '');
		$cat_id  = isset($res['category_id']) ? (int)$res['category_id'] : 0;
		$tags    = is_array($res['tags'] ?? null) ? $res['tags'] : [];
		$keyword = isset($res['keyword']) ? (string) $res['keyword'] : '';

		$post_type = isset($args['post_type']) ? sanitize_key($args['post_type']) : 'post';

		$publish_mode = isset($args['publish_mode'])
			? $args['publish_mode']
			: ( isset($src['publish_mode']) ? $src['publish_mode'] : 'draft' );

		$publish_mode = in_array($publish_mode, ['draft','publish','schedule'], true) ? $publish_mode : 'draft';
		$post_status  = ($publish_mode === 'publish') ? 'publish' : 'draft';

		$default_author = (int) get_option('urlifywriterarticle_author', 0);
		$post_author_id = $default_author ?: get_current_user_id();

		$postarr = [
			'post_title'   => $title ?: __('Untitled Draft', 'urlifywriter'),
			'post_content' => $content ?: '',
			'post_excerpt' => $excerpt ?: '',
			'post_status'  => $post_status,
			'post_author'  => $post_author_id,
			'post_date'    => current_time('mysql'),
			'post_type'    => in_array($post_type, ['post','page'], true) ? $post_type : 'post',
		];

		$post_id = wp_insert_post( $postarr, true );
		if ( is_wp_error($post_id) ) {
			return ['ok'=>false, 'error'=>$post_id->get_error_message(), 'fatal'=>false];
		}

		if ( $detected_id ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$wpdb->update(
				$tbl_items,
				[
					'post_id'     => (int) $post_id,
					'updated_at'  => current_time('mysql'),
				],
				['id' => (int) $detected_id],
				['%d','%s'],
				['%d']
			);
		}

		update_post_meta($post_id, '_urlifywriterlang', $lang);
		update_post_meta($post_id, '_urlifywritersource_url', esc_url_raw($url));

		$set_cat_id = 0;
		if ( $post_type === 'post' ) {
			if ( $cat_id ) {
				$exists = term_exists($cat_id, 'category');
				if ( $exists ) {
					wp_set_post_categories($post_id, [ $cat_id ], false);
					$set_cat_id = $cat_id;
				}
			} else {
				$default_cat = (int) get_option('default_category');
				if ( $default_cat ) {
					wp_set_post_categories($post_id, [ $default_cat ], false);
					$set_cat_id = $default_cat;
				}
			}
			if ( ! empty($tags) ) {
				wp_set_post_tags($post_id, array_map('sanitize_text_field', $tags), false);
			}
		}

		$final_seo_title = $seo_t ?: $title;
		if ( $final_seo_title ) update_post_meta($post_id, '_yoast_wpseo_title', $final_seo_title);

		$final_seo_desc = $seo_d;
		if ( ! $final_seo_desc ) {
			$plain = $excerpt ? $excerpt : wp_strip_all_tags($content);
			$plain = trim( preg_replace('/\s+/', ' ', $plain) );
			$final_seo_desc = mb_substr($plain, 0, 155);
			if ( mb_strlen($plain) > 155 ) {
				$final_seo_desc = preg_replace('/\s+\S*$/u', '', mb_substr($plain, 0, 155)) . '…';
			}
		}
		update_post_meta($post_id, '_yoast_wpseo_metadesc', $final_seo_desc);

		if ( $post_type === 'post' && $set_cat_id ) {
			update_post_meta($post_id, '_yoast_wpseo_primary_category', (int) $set_cat_id);
		}

		$focuskw = '';
		if ( $keyword !== '' ) {
			$focuskw = $keyword;
		} elseif ( ! empty($tags) ) {
			$first = reset($tags);
			if ( is_string($first) && $first !== '' ) $focuskw = $first;
		} elseif ( ! empty($title) ) {
			$focuskw = $title;
		}
		if ( $focuskw ) {
			update_post_meta($post_id, '_yoast_wpseo_focuskw', sanitize_text_field($focuskw));
		}

		if ( $detected_id ) {
			urlifywriterautoscan_mark_url_status( (int)$detected_id, 'generated' );
		}

		if ( function_exists('urlifywriterreport_license_use') ) {
			try { urlifywriterreport_license_use(1, 0); } catch (\Throwable $e) {}
		}

		return ['ok'=>true, 'post_id'=>(int)$post_id];
	}
}


/* =======================================================================
 * 10) Helpers varios
 * ======================================================================= */

if ( ! function_exists('urlifywriterautoscan_bump_next_scan') ) {
	function urlifywriterautoscan_bump_next_scan( $domain_id ) {
		global $wpdb;
		$site = urlifywriterautoscan_get_domain( (int)$domain_id );
		if ( ! $site ) return;

		$freq = max(1, (int)$site['frequency_hours']);
		$now  = urlifywriterautoscan_now_mysql();
		$next = gmdate('Y-m-d H:i:s', time() + $freq*3600);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->update(
			URLIFYWRITER_AUTOSCAN_SITES,
			['last_scan'=>$now, 'next_scan'=>$next, 'updated_at'=>$now],
			['id'=>(int)$site['id']],
			['%s','%s','%s'],
			['%d']
		);
	}
}

/* =======================================================================
 * NUEVO: Runner por dominio para el esquema "sources/items"
 * ======================================================================= */
if ( ! function_exists('urlifywriterautoscan_run_for_domain') ) {
	function urlifywriterautoscan_run_for_domain( $src ) {
		global $wpdb;

		$sources_tbl = $wpdb->prefix . 'urlifywriterautoscan_sources';
		$items_tbl   = $wpdb->prefix . 'urlifywriterautoscan_items';

		$get = function($k) use ($src) {
			if (is_array($src))  return $src[$k] ?? null;
			if (is_object($src)) return $src->$k ?? null;
			return null;
		};

		$src_id      = (int) ($get('id') ?? 0);
		$enabled     = (int) ($get('enabled') ?? 0);
		$start_url   = (string) ($get('start_url') ?? '');
		$domain      = (string) ($get('domain') ?? '');
		$max_per_day = (int) ($get('max_per_day') ?? 5);
		$publish_mode= (string) ($get('publish_mode') ?? 'draft');

		if ( !$src_id || !$enabled )      return ['ok'=>false, 'recent_errors'=>1, 'reason'=>'disabled_or_missing'];
		if ( $start_url === '' && $domain !== '' ) {
			$start_url = 'https://' . $domain . '/';
		}
		if ( ! $start_url )               return ['ok'=>false, 'recent_errors'=>1, 'reason'=>'no_start_url'];

		$resp = wp_remote_get( $start_url, [
			'timeout'     => 18,
			'redirection' => 5,
			'user-agent'  => 'UrlifyWriter/1.0 (+WP; AutoScan Cron)',
		]);
		if ( is_wp_error($resp) ) {
			urlifywriterdebug_log('[AutoScan] fetch error: '.$resp->get_error_message().' (src '.$src_id.')');
			return ['ok'=>false, 'recent_errors'=>1, 'reason'=>'http_error'];
		}
		$code = (int) wp_remote_retrieve_response_code($resp);
		if ( $code < 200 || $code >= 400 ) {
			urlifywriterdebug_log('[AutoScan] HTTP '.$code.' fetching '.$start_url.' (src '.$src_id.')');
			return ['ok'=>false, 'recent_errors'=>1, 'reason'=>'http_'.$code];
		}
		$html = (string) wp_remote_retrieve_body($resp);

		if ( ! function_exists('urlifywriterautoscan_extract_links') ) {
			urlifywriterdebug_log('[AutoScan] crawler core not loaded (urlifywriterautoscan_extract_links missing)');
			return ['ok'=>false, 'recent_errors'=>1, 'reason'=>'crawler_missing'];
		}

		$candidates = urlifywriterautoscan_extract_links( $html, $start_url );
		$total_cand = is_array($candidates) ? count($candidates) : 0;
		if ( ! $total_cand ) {
			return ['ok'=>true, 'generated'=>0, 'inserted'=>0, 'candidates'=>0, 'recent_errors'=>0];
		}

		$midnight_sql = date_i18n('Y-m-d 00:00:00', current_time('timestamp'));
		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$today_count  = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(1) FROM `{$items_tbl}`
				  WHERE source_id = %d
					AND detected_at >= %s
					AND status IN ('pending','approved','processing','done')",
				$src_id,
				$midnight_sql
			)
		);
		// phpcs:enable
		$quota = ($max_per_day > 0) ? max(0, $max_per_day - $today_count) : PHP_INT_MAX;

		$inserted = 0;
		foreach ($candidates as $row) {
			if ( $inserted >= $quota ) break;

			$url = esc_url_raw( (string) ($row['url'] ?? '') );
			if ( ! $url ) continue;

			// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
			$exists = (int) $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(1) FROM `{$items_tbl}` WHERE source_id = %d AND url = %s",
					$src_id,
					$url
				)
			);
			// phpcs:enable
			if ( $exists ) continue;

			$title    = wp_strip_all_tags( (string) ($row['title'] ?? '') );
			$priority = (float) ($row['priority'] ?? 0);

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
			$ok = $wpdb->insert( $items_tbl, [
				'source_id'   => $src_id,
				'url'         => $url,
				'title'       => $title,
				'status'      => 'pending',
				'priority'    => $priority,
				'detected_at' => current_time('mysql'),
				'updated_at'  => current_time('mysql'),
			], ['%d','%s','%s','%s','%f','%s','%s'] );

			if ( $ok ) $inserted++;
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->update( $sources_tbl, [
			'last_scan_at' => current_time('mysql'),
			'updated_at'   => current_time('mysql'),
		], ['id'=>$src_id], ['%s','%s'], ['%d'] );

		return [
			'ok'         => true,
			'generated'  => 0,
			'inserted'   => (int) $inserted,
			'candidates' => (int) $total_cand,
			'recent_errors' => 0,
		];
	}
}

if ( ! function_exists('urlifywriterautoscan_tables') ) {
	function urlifywriterautoscan_tables() {
		global $wpdb;
		return [
			'sources' => $wpdb->prefix . 'urlifywriterautoscan_sources',
			'items'   => $wpdb->prefix . 'urlifywriterautoscan_items',
		];
	}
}

/**
 * Procesa la cola de items detectados
 */
if ( ! function_exists('urlifywriterautoscan_process_queue') ) {
	function urlifywriterautoscan_process_queue( $limit = 5 ) {
		global $wpdb;
		$T = urlifywriterautoscan_tables();

		// Sanea identificadores
		$items_tbl   = esc_sql( $T['items'] );
		$sources_tbl = esc_sql( $T['sources'] );

		static $last_error_checked = false;
		if ( ! $last_error_checked ) {

			// SHOW COLUMNS (identificador saneado + valor preparado)
			// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
			$col = $wpdb->get_var(
				$wpdb->prepare(
					"SHOW COLUMNS FROM `{$items_tbl}` LIKE %s",
					'last_error'
				)
			);
			// phpcs:enable

			if ( ! $col ) {
			// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
			$wpdb->query( "ALTER TABLE `{$items_tbl}` ADD COLUMN `last_error` VARCHAR(190) NULL DEFAULT NULL AFTER `status`" );
			// phpcs:enable
			}
			$last_error_checked = true;
		}

		$limit = max( 1, min( 20, (int) $limit ) );
		$t0    = microtime( true );

		$min_words = (int) get_option( 'urlifywritertarget_min_words', 800 );
		$max_words = (int) get_option( 'urlifywritertarget_max_words', 1200 );
		$min_words = max( 100, min( 3000, $min_words ) );
		$max_words = max( 150, min( 4000, $max_words ) );
		if ( $min_words > $max_words ) { $tmp = $min_words; $min_words = $max_words; $max_words = $tmp; }

		// SELECT con JOIN (identificadores saneados + valores preparados)
		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
		$rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT it.*, src.publish_mode, src.gen_instructions
				   FROM `{$items_tbl}` it
				   JOIN `{$sources_tbl}` src ON src.id = it.source_id
				  WHERE it.status = %s
				  ORDER BY it.priority DESC, it.detected_at ASC
				  LIMIT %d",
				'pending',
				$limit
			),
			ARRAY_A
		);
		// phpcs:enable


		urlifywriterdebug_log(sprintf('[ProcessQueue] start limit=%d selected=%d min=%d max=%d',
			$limit, is_array($rows)?count($rows):0, $min_words, $max_words
		));

		if ( empty($rows) ) {
			urlifywriterdebug_log('[ProcessQueue] no pending items');
			return ['ok'=>true, 'processed'=>0];
		}

		$processed = 0;
		$errors    = 0;

		foreach ( $rows as $row ) {
			$item_id = (int) $row['id'];
			$url_raw = (string) $row['url'];
			$url     = esc_url_raw( $url_raw );
			$u_log   = (strlen($url_raw) > 180) ? substr($url_raw,0,177).'...' : $url_raw;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
			$ok = $wpdb->update(
				$T['items'],
				['status'=>'processing','updated_at'=>current_time('mysql'), 'last_error'=>null],
				['id'=>$item_id],
				['%s','%s','%s'],
				['%d']
			);

			urlifywriterdebug_log(sprintf('[ProcessQueue] #%d lock=%s url=%s',
				$item_id, ($ok===false?'FAIL':'OK'), $u_log
			));
			if ( $ok === false ) { $errors++; continue; }

			$publish_mode     = (string) ($row['publish_mode'] ?: 'publish');
			$gen_instructions = (string) ($row['gen_instructions'] ?? '');

			$args = [
				'publish_mode'  => $publish_mode,
				'min_words'     => $min_words,
				'max_words'     => $max_words,
				'instructions'  => $gen_instructions,
			];

			try {
				if ( function_exists('urlifywriterautoscan_generate_article_from_url') ) {
					$res = urlifywriterautoscan_generate_article_from_url( $item_id, $url, $args );
				} else {
					$res = ['ok'=>false, 'error'=>'generator_missing', 'fatal'=>true];
				}

				if ( empty($res['ok']) ) {
					$err_msg   = isset($res['error']) ? (is_string($res['error']) ? $res['error'] : wp_json_encode($res['error'])) : 'unknown_error';
					$err_store = substr( wp_strip_all_tags($err_msg), 0, 185 );

					// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
					$wpdb->update(
						$T['items'],
						['status'=>'failed','updated_at'=>current_time('mysql'), 'last_error'=>$err_store],
						['id'=>$item_id],
						['%s','%s','%s'],
						['%d']
					);
					$errors++;
					urlifywriterdebug_log(sprintf('[ProcessQueue] #%d FAILED error=%s', $item_id, $err_store));
					continue;
				}

				$post_id = isset($res['post_id']) ? (int)$res['post_id'] : 0;

				if ( $post_id > 0 ) {
					$img_enabled = (bool) get_option('urlifywriterimages_enabled', 0);
					if ( $img_enabled ) {
						$img_source   = (string) get_option('urlifywriterimage_source', 'pixabay');
						$img_insert   = (string) get_option('urlifywriterimage_insert', 'both');
						$orig_ack     = (int) get_option('urlifywriterimage_original_ack', 0);
						$pix_n        = max(0, (int) get_option('urlifywriterpixabay_images_per_article', 1));
						$ai_n         = max(0, (int) get_option('urlifywriterai_images_per_article', 0));
						$ai_style     = (string) get_option('urlifywriterai_style_hint', '');
						$prefer_user  = (bool) ( get_option('urlifywriterapi_pixabay_key') || get_option('urlifywriterapi_openai_key') );

						if ( function_exists('urlifywriterautogen_attach_images_for_post') ) {
							urlifywriterdebug_log(sprintf('[Images] #%d start source=%s insert=%s', $item_id, $img_source, $img_insert));
							urlifywriterautogen_attach_images_for_post( $post_id, [
								'source_url'      => $url,
								'source'          => $img_source,
								'insert'          => $img_insert,
								'original_ack'    => $orig_ack,
								'pixabay_n'       => $pix_n,
								'ai_n'            => $ai_n,
								'prefer_user_api' => $prefer_user ? 1 : 0,
								'style_hint'      => $ai_style,
							] );
						} else {
							urlifywriter_fallback_attach_images_for_post( $post_id, $url, [
								'source'          => $img_source,
								'insert'          => $img_insert,
								'original_ack'    => $orig_ack,
								'pixabay_n'       => $pix_n,
								'ai_n'            => $ai_n,
								'prefer_user_api' => $prefer_user ? 1 : 0,
								'style_hint'      => $ai_style,
							] );
						}
					}
				}

				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->update(
					$T['items'],
					['status'=>'done','updated_at'=>current_time('mysql'), 'last_error'=>null],
					['id'=>$item_id],
					['%s','%s','%s'],
					['%d']
				);
				$processed++;

				urlifywriterdebug_log(sprintf('[ProcessQueue] #%d DONE%s', $item_id, $post_id ? " post_id={$post_id}" : ''));

			} catch (\Throwable $e) {
				$err_store = substr( wp_strip_all_tags($e->getMessage()), 0, 185 );
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
				$wpdb->update(
					$T['items'],
					['status'=>'failed','updated_at'=>current_time('mysql'), 'last_error'=>$err_store],
					['id'=>$item_id],
					['%s','%s','%s'],
					['%d']
				);
				urlifywriterdebug_log('[ProcessQueue] #'.$item_id.' EXCEPTION: '.$err_store);
				$errors++;
			}
		}

		$ms = (int) round( (microtime(true)-$t0) * 1000 );
		urlifywriterdebug_log(sprintf('[ProcessQueue] done processed=%d errors=%d time_ms=%d',
			$processed, $errors, $ms
		));

		return ['ok'=>true, 'processed'=>$processed, 'errors'=>$errors];
	}
}

/* ========================================================================
 *  Helpers básicos para imágenes
 * ======================================================================== */

if ( ! function_exists('urlifywriteruniform_section_indices') ) {
	function urlifywriteruniform_section_indices( $sectionCount, $imageCount ) {
		$out = [];
		$S = max(0, (int)$sectionCount);
		$M = max(0, (int)$imageCount);
		if ($M === 0 || $S === 0) return $out;
		for ($k = 0; $k < $M; $k++) {
			$pos = (($k + 0.5) * $S) / $M;
			$idx = (int) floor($pos);
			if ($idx < 0) $idx = 0;
			if ($idx >= $S) $idx = $S - 1;
			$out[] = $idx;
		}
		return $out;
	}
}

if ( ! function_exists('urlifywriter_fallback_attach_images_for_post') ) {
	function urlifywriter_fallback_attach_images_for_post( $post_id, $source_url, $opts = [] ) {
		$DBG          = ( defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG );
		$source       = isset($opts['source']) ? $opts['source'] : 'pixabay';
		$insert       = isset($opts['insert']) ? $opts['insert'] : 'both';
		$orig_ack     = !empty($opts['original_ack']);
		$pix_n        = max(0, (int) ($opts['pixabay_n'] ?? 1));
		$ai_n         = max(0, (int) ($opts['ai_n'] ?? 0));
		$prefer_user  = !empty($opts['prefer_user_api']);
		$style_hint   = (string) ($opts['style_hint'] ?? '');

		$post = get_post($post_id);
		if ( ! $post ) {
			if ($DBG) urlifywriterdebug_log("[Images] Fallback: post $post_id not found");
			return;
		}

		$want_featured = in_array($insert, ['featured','both'], true);
		$want_inline   = in_array($insert, ['inline','both'], true);

		$title   = get_the_title($post_id) ?: '';
		$focuskw = trim( (string) get_post_meta($post_id, '_yoast_wpseo_focuskw', true) );

		if ($DBG) {
			urlifywriterdebug_log(sprintf(
				'[Images] Fallback start post_id=%d source=%s insert=%s ack=%d pix_n=%d ai_n=%d prefer_user=%d style_len=%d',
				(int)$post_id, $source, $insert, $orig_ack?1:0, $pix_n, $ai_n, $prefer_user?1:0, strlen($style_hint)
			));
		}

		/* ===================== 1) ORIGINAL ====================== */
		if ( $source === 'original' ) {
			if ( ! $orig_ack ) {
				if ($DBG) urlifywriterdebug_log('[Images] original: skipped (no ACK)');
				return;
			}
			if ($DBG) urlifywriterdebug_log('[Images] original: fetching HTML ' . $source_url);

			$resp = wp_remote_get( $source_url, [
				'timeout' => 18,
				'redirection' => 5,
				'user-agent' => 'UrlifyWriter/1.0 (+WP; Img-Scan)'
			] );
			if ( is_wp_error($resp) ) {
				if ($DBG) urlifywriterdebug_log('[Images] original: fetch error ' . $resp->get_error_message());
				return;
			}
			$code = (int) wp_remote_retrieve_response_code($resp);
			if ( $code >= 400 ) {
				$log('original blocked http=' . $code . ' -> fallback to provider if available');

				// Si original falla, probamos pixabay/ai (si hay configuración)
				// Preferencia: si el usuario eligió original, pero hay pixabay_n o ai_n configurados, usamos pixabay por defecto.
				$fallback_source = 'pixabay';
				if ( !empty($opts['fallback_source']) ) {
					$fallback_source = (string) $opts['fallback_source'];
				}

				// Heurística: si ai_n > 0 y pix_n == 0, usa ai
				if ( $ai_n > 0 && $pix_n <= 0 ) {
					$fallback_source = 'ai';
				}

				// Reescribe "source" y continúa al bloque 2 sin salir de la función
				$source = in_array($fallback_source, ['pixabay','ai'], true) ? $fallback_source : 'pixabay';
				$log('fallback_source=' . $source);

				// Saltamos a la parte 2
			}
			else{
				$html = (string) wp_remote_retrieve_body($resp);

				$img = '';
				if ( function_exists('urlifywriterextract_featured_image') ) {
					$img = urlifywriterextract_featured_image($html);
				} else {
					if ( function_exists('urlifywriterautogen_extract_og_image') ) $img = urlifywriterautogen_extract_og_image($html);
					if ( ! $img && function_exists('urlifywriterautogen_extract_first_img') ) $img = urlifywriterautogen_extract_first_img($html);
				}
				if ( ! $img ) return;

				$att_id = function_exists('urlifywriterautogen_attach_external_image')
					? urlifywriterautogen_attach_external_image($img, $post_id, $title)
					: 0;

				if ($DBG) urlifywriterdebug_log('[Images] original: attach id=' . (int)$att_id);

				if ( $att_id ) {
					wp_update_post(['ID' => $att_id, 'post_title' => sanitize_text_field($title)]);
					update_post_meta($att_id, '_wp_attachment_image_alt', sanitize_text_field($title));
					if ( function_exists('urlifywriterautogen_rename_attachment_file') && function_exists('urlifywriterautogen_slugify') ) {
						urlifywriterautogen_rename_attachment_file($att_id, urlifywriterautogen_slugify($title));
					}

					if ( $want_featured ) {
						set_post_thumbnail($post_id, $att_id);
						if ($DBG) urlifywriterdebug_log('[Images] original: set featured='.(int)$att_id);
					}
					if ( $want_inline ) {
						$old = $post->post_content;
						$new = function_exists('urlifywriterautogen_inject_image_html')
							? urlifywriterautogen_inject_image_html($old, $att_id, 'after_first')
							: ($old . "\n\n" . wp_get_attachment_image($att_id, 'large', false, ['loading'=>'eager']));
						if ( $new && $new !== $old ) {
							wp_update_post(['ID'=>$post_id, 'post_content'=>$new]);
							if ($DBG) urlifywriterdebug_log('[Images] original: inline injected');
						}
					}
				}
				return;
			}
		}

		/* ============== 2) PIXABAY / AI con secciones ============== */
		$n = ($source === 'pixabay') ? max(1, $pix_n) : max(1, min(4, $ai_n ?: 1));
		$tagsArr = wp_get_post_terms($post_id, 'post_tag', ['fields'=>'names']);
		$q = trim( $title . ' ' . (is_array($tagsArr)&&$tagsArr ? $tagsArr[0] : '') );

		$sections = [];
		if ( preg_match_all('~<h2[^>]*>(.*?)</h2>~i', (string) $post->post_content, $m) ) {
			foreach ($m[1] as $sec) {
				$sec = trim(wp_strip_all_tags($sec));
				if (strlen($sec) > 2) $sections[] = $sec;
			}
		}
		$sectionCount = count($sections);

		if ($DBG) {
			urlifywriterdebug_log(sprintf('[Images] fetch provider=%s n=%d query="%s" prefer_user=%d style="%s"',
				($source==='ai'?'ai':'pixabay'), $n, $q, $prefer_user?1:0, $style_hint
			));
		}

		$urls = apply_filters('urlifywriterautogen_fetch_images', [], $q, ($source==='ai'?'ai':'pixabay'), $n, [
			'prefer_user_api' => $prefer_user ? 1 : 0,
			'style_hint'      => $style_hint,
			'post_id'         => $post_id,
			'sections'        => $sections,
		]);

		if ($DBG) {
			$sample = is_array($urls) ? array_slice($urls, 0, 3) : [];
			urlifywriterdebug_log('[Images] provider returned count='.(is_array($urls)?count($urls):0).' sample='.wp_json_encode($sample));
		}

		if ( ! is_array($urls) || empty($urls) ) {
			if ($DBG) urlifywriterdebug_log('[Images] provider: no URLs');
			return;
		}

		$force_inline_if_multi = (count($urls) > 1 && $insert === 'featured');
		$want_inline = $want_inline || $force_inline_if_multi;

		$nonFeaturedCount = max(0, count($urls) - ($want_featured ? 1 : 0));
		$uniformTargets   = ($sectionCount > 0 && $nonFeaturedCount > 0)
			? urlifywriteruniform_section_indices($sectionCount, $nonFeaturedCount)
			: [];

		$featured_att_id = 0;
		$attached_nf     = [];

		foreach ( $urls as $i => $u ) {
			if ( ! $u ) continue;

			$is_featured_slot = ($i === 0 && $want_featured);

			$sec_txt = '';
			$sec_idx_for_insert = -1;
			if ( !$is_featured_slot && $sectionCount > 0 && !empty($uniformTargets) ) {
				$rel = $i - ($want_featured ? 1 : 0);
				if ( isset($uniformTargets[$rel]) ) {
					$sec_idx_for_insert = (int) $uniformTargets[$rel];
					$sec_txt = $sections[$sec_idx_for_insert] ?? '';
				}
			}

			if ( $is_featured_slot ) {
				$display_title = $title;
			} else {
				if ( $focuskw !== '' && $sec_txt !== '' ) {
					$display_title = $focuskw . ': ' . $sec_txt;
				} elseif ( $focuskw !== '' ) {
					$display_title = $focuskw . ': ' . ($sec_txt ?: $title);
				} else {
					$display_title = $sec_txt ?: $title;
				}
			}

			$alt_text = $display_title;

			$att_id = function_exists('urlifywriterautogen_attach_external_image')
				? urlifywriterautogen_attach_external_image($u, $post_id, $alt_text)
				: 0;

			if ($DBG) urlifywriterdebug_log(sprintf('[Images] attach [%d] url=%s id=%d', $i, $u, (int)$att_id));

			if ( $att_id ) {
				wp_update_post([
					'ID'         => $att_id,
					'post_title' => sanitize_text_field($display_title),
				]);
				update_post_meta($att_id, '_wp_attachment_image_alt', sanitize_text_field($alt_text));

				if ( function_exists('urlifywriterautogen_rename_attachment_file') && function_exists('urlifywriterautogen_slugify') ) {
					$slug = urlifywriterautogen_slugify($display_title);
					urlifywriterautogen_rename_attachment_file($att_id, $slug);
				}

				if ( $is_featured_slot ) {
					$featured_att_id = $att_id;
				} else {
					$attached_nf[] = ['id' => $att_id, 'sec_idx' => $sec_idx_for_insert];
				}
			}
		}

		if ( $want_featured && $featured_att_id ) {
			set_post_thumbnail($post_id, $featured_att_id);
			if ($DBG) urlifywriterdebug_log('[Images] featured set id='.(int)$featured_att_id);
		}

		if ( $want_inline && !empty($attached_nf) ) {
			$content  = $post->post_content;
			$injected = 0;

			foreach ( $attached_nf as $row ) {
				$att_id = (int) $row['id'];
				$target = (int) $row['sec_idx'];

				if ( $target >= 0 && isset($sections[$target]) ) {
					$sec_pat = preg_quote($sections[$target], '~');
					$img_html = wp_get_attachment_image($att_id, 'large', false, [
						'alt' => get_post_meta($att_id, '_wp_attachment_image_alt', true) ?: ''
					]);
					$figure = '<figure class="aligncenter">'.$img_html.'</figure>';
					$new = preg_replace("~(<h2[^>]*>\s*{$sec_pat}\s*</h2>)~i", '$1'."\n\n".$figure."\n\n", $content, 1);
					if ( $new && $new !== $content ) {
						$content = $new;
						$injected++;
						continue;
					}
				}

				$content = function_exists('urlifywriterautogen_inject_image_html')
					? urlifywriterautogen_inject_image_html($content, $att_id, 'after_first')
					: ($content . "\n\n" . wp_get_attachment_image($att_id, 'large', false, ['loading'=>'eager']));
				$injected++;
			}

			if ( $content !== $post->post_content ) {
				wp_update_post(['ID'=>$post_id, 'post_content'=>$content]);
				if ($DBG) urlifywriterdebug_log('[Images] inline injected count='.$injected);
			} else if ($DBG) {
				urlifywriterdebug_log('[Images] inline: content unchanged');
			}
		}

		if ($DBG) urlifywriterdebug_log('[Images] done attached_total='.(int)($featured_att_id ? 1 : 0 + count($attached_nf)));
	}
}

if ( ! function_exists('urlifywriterautogen_attach_external_image') ) {
	function urlifywriterautogen_attach_external_image($image_url, $post_id, $alt='') {
		$DBG = ( defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG );
		if ($DBG) urlifywriterdebug_log(sprintf('[Images] sideload start url=%s post_id=%d', $image_url, (int)$post_id));

		if ( ! $image_url || ! $post_id ) return 0;
		require_once ABSPATH.'wp-admin/includes/file.php';
		require_once ABSPATH.'wp-admin/includes/media.php';
		require_once ABSPATH.'wp-admin/includes/image.php';

		if (strpos($image_url, '//') === 0) {
			$image_url = (is_ssl() ? 'https:' : 'http:') . $image_url;
		}
		if (!preg_match('~^https?://~i', $image_url)) {
			if ($DBG) urlifywriterdebug_log('[Images] sideload: bad scheme');
			return 0;
		}

		$tmp = download_url($image_url, 30);
		if (is_wp_error($tmp)) {
			if ($DBG) urlifywriterdebug_log('[Images] sideload: download error ' . $tmp->get_error_message());
			return 0;
		}
		if ($DBG) urlifywriterdebug_log('[Images] sideload: tmp=' . $tmp);

		$img_parts = wp_parse_url($image_url);
		$img_path  = $img_parts['path'] ?? '';
		$name_base = $img_path ? basename($img_path) : 'image.jpg';

		$file_array = [
			'name'     => $name_base,
			'tmp_name' => $tmp,
		];
		if (!preg_match('~\.(jpe?g|png|webp)$~i', $file_array['name'])) {
			$file_array['name'] .= '.jpg';
		}

		$att_id = media_handle_sideload($file_array, $post_id);
		if (is_wp_error($att_id)) {
			// Sustituye unlink() por wp_delete_file()
			wp_delete_file($tmp);
			if ($DBG) urlifywriterdebug_log('[Images] sideload: media_handle error ' . $att_id->get_error_message());
			return 0;
		}
		if ($alt) {
			update_post_meta($att_id, '_wp_attachment_image_alt', wp_strip_all_tags($alt));
		}

		if ($DBG) urlifywriterdebug_log('[Images] sideload: done id='.(int)$att_id);
		return (int) $att_id;
	}
}

if ( ! function_exists('urlifywriterautogen_inject_image_html') ) {
	function urlifywriterautogen_inject_image_html($content, $attachment_id, $position='after_first') {
		$DBG = ( defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG );
		if ($DBG) urlifywriterdebug_log(sprintf('[Images] inject pos=%s att_id=%d', $position, (int)$attachment_id));

		$img_html = wp_get_attachment_image($attachment_id, 'large', false, ['loading'=>'eager']);
		$figure   = '<figure class="aligncenter">'.$img_html.'</figure>';

		if ($position === 'before') {
			return $figure . "\n\n" . $content;
		}
		if ($position === 'after_first') {
			$parts = preg_split('~(\r?\n){2,}~', (string)$content, 2);
			if (count($parts) === 2) {
				return $parts[0]."\n\n".$figure."\n\n".$parts[1];
			}
		}
		return (string)$content . "\n\n" . $figure;
	}
}

if ( ! function_exists('urlifywriterautogen_extract_og_image') ) {
	function urlifywriterautogen_extract_og_image($html) {
		$DBG = ( defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG );
		if (!$html) { if ($DBG) urlifywriterdebug_log('[Images] og: empty html'); return ''; }
		if (!preg_match_all('~<meta[^>]+(property|name)=["\'](?:og:image|twitter:image)["\'][^>]*>~i', $html, $m)) {
			if ($DBG) urlifywriterdebug_log('[Images] og: none');
			return '';
		}
		foreach ($m[0] as $tag) {
			if (preg_match('~content=["\']([^"\']+)~i', $tag, $u)) {
				$url = trim($u[1]);
				if ($url) {
					if ($DBG) urlifywriterdebug_log('[Images] og: pick ' . $url);
					return $url;
				}
			}
		}
		if ($DBG) urlifywriterdebug_log('[Images] og: no content');
		return '';
	}
}

if ( ! function_exists('urlifywriterautogen_extract_first_img') ) {
	function urlifywriterautogen_extract_first_img($html) {
		$DBG = ( defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG );
		if (!$html) { if ($DBG) urlifywriterdebug_log('[Images] first-img: empty html'); return ''; }
		if (!preg_match_all('~<img\s+[^>]*src=["\']([^"\']+)["\'][^>]*>~i', $html, $m, PREG_SET_ORDER)) {
			if ($DBG) urlifywriterdebug_log('[Images] first-img: none');
			return '';
		}
		foreach ($m as $img) {
			$src = trim($img[1]);
			if (!$src) continue;
			$parts = wp_parse_url($src);
			$path  = $parts['path'] ?? '';
			if (preg_match('~\.(svg|gif)$~i', $path)) continue;
			if ($DBG) urlifywriterdebug_log('[Images] first-img: pick ' . $src);
			return $src;
		}
		if ($DBG) urlifywriterdebug_log('[Images] first-img: no usable img');
		return '';
	}
}

if ( ! function_exists('urlifywriterautogen_slugify') ) {
	function urlifywriterautogen_slugify($text) {
		$text = remove_accents( strtolower( sanitize_text_field($text) ) );
		$text = preg_replace('~[^a-z0-9]+~', '-', $text);
		$text = trim($text, '-');
		if ($text === '') $text = 'image';
		return $text;
	}
}

/**
 * Renombra el fichero físico de un attachment a <slug>.<ext> y actualiza metadatos.
 * Usa WP_Filesystem::move() en lugar de rename()
 */
if ( ! function_exists('urlifywriterautogen_rename_attachment_file') ) {
	function urlifywriterautogen_rename_attachment_file($att_id, $new_slug) {
		$file = get_attached_file($att_id);
		if ( ! $file || ! file_exists($file) ) return false;

		$dir  = dirname($file);
		$ext  = pathinfo($file, PATHINFO_EXTENSION);
		if (!$ext) $ext = 'jpg';

		$new_basename = $new_slug . '.' . $ext;
		$new_path     = trailingslashit($dir) . $new_basename;

		$i = 2;
		while (file_exists($new_path)) {
			$new_path = trailingslashit($dir) . $new_slug . '-' . $i . '.' . $ext;
			$i++;
		}

		// WP_Filesystem
		if ( ! function_exists('WP_Filesystem') ) {
			require_once ABSPATH.'wp-admin/includes/file.php';
		}
		global $wp_filesystem;
		if ( ! $wp_filesystem ) {
			WP_Filesystem(); // intenta inicializar
		}

		$moved = false;
		if ( $wp_filesystem && is_object($wp_filesystem) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename
			$moved = $wp_filesystem->move( $file, $new_path, true );
		} else {
			// Último recurso (algunos entornos sin FS creds): usar rename con phpcs ignore
			// phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename
			$moved = @rename($file, $new_path);
		}

		if ( $moved ) {
			update_attached_file($att_id, $new_path);

			require_once ABSPATH.'wp-admin/includes/image.php';
			$metadata = wp_generate_attachment_metadata($att_id, $new_path);
			if ( $metadata && ! is_wp_error($metadata) ) {
				wp_update_attachment_metadata($att_id, $metadata);
			}

			$upload_dir = wp_get_upload_dir();
			if(!empty($upload_dir['basedir']) && str_starts_with($new_path, $upload_dir['basedir'])) {
				$relative = ltrim( str_replace($upload_dir['basedir'], '', $new_path), '/\\' );
				update_post_meta($att_id, '_wp_attached_file', $relative);
			}

			return true;
		}

		return false;
	}
}
