<?php
/**
 * UrlifyWriter — AutoScan + AutoGen Cron
 *
 * AutoScan:
 * - Registra intervalos de cron personalizados
 * - Programa/reescribe el evento 'urlifywriterautoscan_cron'
 * - Callback del cron que ejecuta urlifywriterautoscan_run()
 * - Endpoint REST y WP-CLI opcionales
 *
 * AutoGen (nuevo):
 * - Segundo cron 'urlifywriterautogen_cron' para procesar la cola (pending/approved → generar)
 * - Programa/reescribe el evento 'urlifywriterautogen_cron'
 * - Callback del cron que ejecuta urlifywriterautogen_run() → usa urlifywriterautoscan_process_queue($limit)
 * - Endpoint REST y WP-CLI opcionales
 *
 * Compat:
 * - Esquema nuevo: {prefix}urlifywriterautoscan_sources (scan_interval, next_scan_at, last_scan_at)
 * - Esquema clásico: {prefix}urlifywriterautoscan_sites (frequency_hours, next_scan, last_scan)
 */

if ( ! defined('ABSPATH') ) { exit; }

/* ============================================================
 * 0) Utilidades comunes
 * ============================================================ */
if ( ! function_exists('urlifywriterarr_get') ) {
	function urlifywriterarr_get($arr,$k,$d=null){ return isset($arr[$k]) ? $arr[$k] : $d; }
}
if ( ! function_exists('urlifywriteris_assoc') ) {
	function urlifywriteris_assoc(array $a){ return array_keys($a)!==range(0,count($a)-1); }
}

/**
 * Detecta qué tabla/columnas usar (sources vs sites).
 */
if ( ! function_exists('urlifywriterautoscan_detect_sources_schema') ) {
	function urlifywriterautoscan_detect_sources_schema() {
		global $wpdb;
		$src_tbl   = $wpdb->prefix . 'urlifywriterautoscan_sources';
		$sites_tbl = $wpdb->prefix . 'urlifywriterautoscan_sites';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table detection; no WP core API and caching intentionally disabled.
		$table_match = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $src_tbl ) );

		if ( $table_match === $src_tbl ) {
			return [
				'table'        => $src_tbl,
				'col_id'       => 'id',
				'col_enabled'  => 'enabled',
				'col_next'     => 'next_scan_at',
				'col_last'     => 'last_scan_at',
				'col_interval' => 'scan_interval',
				'is_sources'   => true,
			];
		}
		return [
			'table'        => $sites_tbl,
			'col_id'       => 'id',
			'col_enabled'  => 'enabled',
			'col_next'     => 'next_scan',
			'col_last'     => 'last_scan',
			'col_interval' => 'frequency_hours',
			'is_sources'   => false,
		];
	}
}


/**
 * Convierte schedule key → minutos.
 */
if ( ! function_exists('urlifywriterschedule_key_to_minutes') ) {
	function urlifywriterschedule_key_to_minutes( $key ) {
		switch ($key) {
			case 'urlifywriterevery_5_minutes':   return 5;
			case 'urlifywriterevery_15_minutes':  return 15;
			case 'urlifywriterevery_30_minutes':  return 30;
			case 'hourly':                return 60;
			case 'twicedaily':            return 12*60; // compat
			case 'daily':                 return 24*60;
			case 'urlifywriterevery_3_hours':     return 3*60;
			case 'urlifywriterevery_6_hours':     return 6*60;
			case 'urlifywriterevery_12_hours':    return 12*60;
			default:                      return 60;
		}
	}
}

/* ============================================================
 * 1) Intervalos personalizados de WP-Cron
 * ============================================================ */
if ( ! function_exists('urlifywriterautoscan_cron_schedules') ) {
	function urlifywriterautoscan_cron_schedules( $schedules ) {
		$schedules['urlifywriterevery_5_minutes'] = [
			'interval' => 5 * 60,
			'display'  => __('Every 5 minutes (UrlifyWriter)', 'urlifywriter'),
		];
		$schedules['urlifywriterevery_15_minutes'] = [
			'interval' => 15 * 60,
			'display'  => __('Every 15 minutes (UrlifyWriter)', 'urlifywriter'),
		];
		$schedules['urlifywriterevery_30_minutes'] = [
			'interval' => 30 * 60,
			'display'  => __('Every 30 minutes (UrlifyWriter)', 'urlifywriter'),
		];
		$schedules['urlifywriterevery_3_hours'] = [
			'interval' => 3 * 3600,
			'display'  => __('Every 3 hours (UrlifyWriter)', 'urlifywriter'),
		];
		$schedules['urlifywriterevery_6_hours'] = [
			'interval' => 6 * 3600,
			'display'  => __('Every 6 hours (UrlifyWriter)', 'urlifywriter'),
		];
		$schedules['urlifywriterevery_12_hours'] = [
			'interval' => 12 * 3600,
			'display'  => __('Every 12 hours (UrlifyWriter)', 'urlifywriter'),
		];
		return $schedules;
	}
	add_filter('cron_schedules', 'urlifywriterautoscan_cron_schedules');
}

/* ============================================================
 * 2) Programación del evento (AutoScan)
 * ============================================================ */
if ( ! function_exists('urlifywriterautoscan_get_interval_key') ) {
	function urlifywriterautoscan_get_interval_key() {
		$key = get_option('urlifywriterautoscan_interval', '');
		$allowed = [
			'urlifywriterevery_5_minutes','urlifywriterevery_15_minutes','urlifywriterevery_30_minutes',
			'hourly','twicedaily','daily','urlifywriterevery_3_hours','urlifywriterevery_6_hours','urlifywriterevery_12_hours',
		];
		if ( ! in_array($key, $allowed, true) ) $key = 'hourly';
		return $key;
	}
}
if ( ! function_exists('urlifywriterautoscan_schedule_event') ) {
	function urlifywriterautoscan_schedule_event( $interval_key = null ) {
		$interval_key = $interval_key ?: urlifywriterautoscan_get_interval_key();
		if ( ! wp_next_scheduled('urlifywriterautoscan_cron') ) {
			wp_schedule_event( time() + 60, $interval_key, 'urlifywriterautoscan_cron' );
		}
	}
}
if ( ! function_exists('urlifywriterautoscan_clear_event') ) {
	function urlifywriterautoscan_clear_event() {
		if ( function_exists('wp_clear_scheduled_hook') ) {
			wp_clear_scheduled_hook('urlifywriterautoscan_cron'); // borra TODAS las instancias
		} else {
			// Compatibilidad con WP antiguos
			wp_unschedule_hook('urlifywriterautoscan_cron');
		}
	}
}

if ( ! function_exists('urlifywriterautoscan_reschedule_event') ) {
	function urlifywriterautoscan_reschedule_event( $new_interval_key ) {
		update_option('urlifywriterautoscan_interval', $new_interval_key);
		urlifywriterautoscan_clear_event();
		urlifywriterautoscan_schedule_event( $new_interval_key );
	}
}

/* ============================================================
 * 3) Callback del evento (AutoScan)
 * ============================================================ */
if ( ! function_exists('urlifywriterautoscan_cron_callback') ) {
	function urlifywriterautoscan_cron_callback() {
		if ( ! function_exists('urlifywriterautoscan_run') ) return;
		$res = urlifywriterautoscan_run();
		update_option('urlifywriterautoscan_last_run', current_time('mysql'));
		if ( defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug-only logging, guarded by URLIFYWRITER_DEBUG
			error_log(sprintf('[URLIFYWRITER][AutoScan Cron] scanned=%d generated=%d',
				isset($res['scanned']) ? (int)$res['scanned'] : 0,
				isset($res['generated']) ? (int)$res['generated'] : 0
			));
		}
	}
	add_action('urlifywriterautoscan_cron', 'urlifywriterautoscan_cron_callback');
}

/* ============================================================
 * 4) Endpoint REST (AutoScan)
 *    /wp-json/autopress/v1/autoscan/run
 * ============================================================ */
add_action( 'rest_api_init', function () {

	register_rest_route( 'autopress/v1', '/autoscan/run', array(
		'methods'  => array( 'POST', 'GET' ),
		'callback' => function( WP_REST_Request $request ) {

			$res = function_exists( 'urlifywriterautoscan_run' )
				? urlifywriterautoscan_run()
				: array( 'ok' => false, 'error' => 'missing_core' );

			return new WP_REST_Response( $res, 200 );
		},
		'permission_callback' => function( WP_REST_Request $request ) {

			// ✅ Admin del sitio
			if ( current_user_can( 'manage_options' ) ) {
				return true;
			}

			// ✅ Webhook key
			$key_opt = (string) get_option( 'urlifywriterautoscan_webhook_key', '' );
			if ( $key_opt === '' ) {
				return new WP_Error( 'forbidden', 'Forbidden', array( 'status' => 403 ) );
			}

			$provided = (string) $request->get_param( 'key' );

			// Opcional: permitir header
			if ( $provided === '' ) {
				$provided = (string) $request->get_header( 'x-webhook-key' );
			}

			if ( $provided !== '' && hash_equals( $key_opt, $provided ) ) {
				return true;
			}

			return new WP_Error( 'forbidden', 'Forbidden', array( 'status' => 403 ) );
		},
	) );

} );


/* ============================================================
 * 5) Comando WP-CLI (AutoScan)
 * ============================================================ */
if ( defined('WP_CLI') && WP_CLI ) {
	class URLIFYWRITER_Autoscan_CLI {
		public function __invoke( $args, $assoc_args ) {
			if ( isset($assoc_args['reschedule']) ) {
				$int = sanitize_text_field( $assoc_args['reschedule'] );
				urlifywriterautoscan_reschedule_event( $int );
				\WP_CLI::success( 'AutoScan rescheduled to: ' . $int );
			}
			if ( function_exists('urlifywriterautoscan_run') ) {
				$res = urlifywriterautoscan_run();
				\WP_CLI::line( 'Result: ' . wp_json_encode($res) );
				if ( ! empty($res['ok']) ) \WP_CLI::success('Autoscan finished.');
				else \WP_CLI::error('Autoscan failed.');
			} else {
				\WP_CLI::error('Core function urlifywriterautoscan_run() not available.');
			}
		}
	}
	\WP_CLI::add_command( 'urlifywriter autoscan', 'URLIFYWRITER_Autoscan_CLI' );
}

/* ============================================================
 * 6) Activación / desactivación (cubre ambos crons)
 * ============================================================ */
if ( ! function_exists('urlifywriterautoscan_activate') ) {
	function urlifywriterautoscan_activate() {
		if ( ! get_option('urlifywriterautoscan_interval') ) update_option('urlifywriterautoscan_interval', 'hourly');
		if ( ! get_option('urlifywriterautogen_interval') )  update_option('urlifywriterautogen_interval',  'urlifywriterevery_15_minutes');
		if ( ! get_option('urlifywriterautogen_batch') )     update_option('urlifywriterautogen_batch', 5);
		urlifywriterautoscan_schedule_event();
		urlifywriterautogen_schedule_event( get_option('urlifywriterautogen_interval','urlifywriterevery_15_minutes') );
	}
}
if ( ! function_exists('urlifywriterautoscan_deactivate') ) {
	function urlifywriterautoscan_deactivate() {
		urlifywriterautoscan_clear_event();
		urlifywriterautogen_clear_event();
	}
}

/* ============================================================
 * 7) Config / Jitter / Stagger / Backoff (AutoScan)
 * ============================================================ */
if ( ! defined('URLIFYWRITER_AUTOSCAN_MIN_INTERVAL_MIN') ) define('URLIFYWRITER_AUTOSCAN_MIN_INTERVAL_MIN', 30);
if ( ! defined('URLIFYWRITER_AUTOSCAN_JITTER_PCT') )      define('URLIFYWRITER_AUTOSCAN_JITTER_PCT', 0.25);
if ( ! defined('URLIFYWRITER_AUTOSCAN_STAGGER_MIN_MIN') ) define('URLIFYWRITER_AUTOSCAN_STAGGER_MIN_MIN', 2);
if ( ! defined('URLIFYWRITER_AUTOSCAN_STAGGER_MAX_MIN') ) define('URLIFYWRITER_AUTOSCAN_STAGGER_MAX_MIN', 7);
if ( ! defined('URLIFYWRITER_AUTOSCAN_BACKOFF_MIN') )     define('URLIFYWRITER_AUTOSCAN_BACKOFF_MIN', 720);

if ( ! function_exists('urlifywriterautoscan_clamp') ) {
	function urlifywriterautoscan_clamp($v, $min, $max){ return max($min, min($max, $v)); }
}
if ( ! function_exists('urlifywriterautoscan_compute_next_scan') ) {
	function urlifywriterautoscan_compute_next_scan( $base_minutes, $source_id = 0 ) {
		$base  = max( (int) URLIFYWRITER_AUTOSCAN_MIN_INTERVAL_MIN, (int) $base_minutes );
		$pct   = urlifywriterautoscan_clamp( (float) URLIFYWRITER_AUTOSCAN_JITTER_PCT, 0.0, 0.95 );
		$delta = (int) round( $base * $pct );

		// Reemplazo de rand()/mt_* por wp_rand() y cálculo determinista del stagger.
		$jitter = $delta ? wp_rand( -$delta, $delta ) : 0;

		$st_min = (int) URLIFYWRITER_AUTOSCAN_STAGGER_MIN_MIN;
		$st_max = (int) URLIFYWRITER_AUTOSCAN_STAGGER_MAX_MIN;
		if ( $st_max < $st_min ) { $st_max = $st_min; }

		// Stagger determinista sin seeding global del RNG.
		$range   = $st_max - $st_min;
		$hash    = crc32( 'urlifywriter' . (int) $source_id );
		$stagger = $range > 0 ? ( $st_min + ( $hash % ( $range + 1 ) ) ) : $st_min;

		$next_minutes = max( URLIFYWRITER_AUTOSCAN_MIN_INTERVAL_MIN, $base + $jitter + $stagger );
		return current_time( 'timestamp' ) + ( $next_minutes * 60 );
	}
}
if ( ! function_exists( 'urlifywriterautoscan_after_scan_update_next' ) ) {
	function urlifywriterautoscan_after_scan_update_next( $source_row, $scan_ok, $error_count = 0 ) {
		global $wpdb;

		$meta = urlifywriterautoscan_detect_sources_schema(); // columnas/tabla controladas por el plugin (whitelist)
		$tbl  = $meta['table'];

		$get = function( $key ) use ( $source_row ) {
			if ( is_object( $source_row ) && isset( $source_row->$key ) ) { return $source_row->$key; }
			if ( is_array( $source_row )  && isset( $source_row[ $key ] ) ) { return $source_row[ $key ]; }
			return null;
		};

		$id = (int) $get( $meta['col_id'] );
		if ( ! $id ) {
			return;
		}

		$now_ts = current_time( 'timestamp' );

		// Determinar minutos base según esquema.
		if ( ! empty( $meta['is_sources'] ) ) {
			$skey         = (string) $get( $meta['col_interval'] );
			$base_minutes = urlifywriterschedule_key_to_minutes( $skey ?: 'hourly' ); // clave whitelisteada en esa función
		} else {
			$hours        = (int) $get( $meta['col_interval'] );
			$base_minutes = max( 60, $hours * 60 );
		}

		// Backoff si hay 3+ errores consecutivos.
		if ( ! $scan_ok && (int) $error_count >= 3 ) {
			$next_ts = $now_ts + ( (int) URLIFYWRITER_AUTOSCAN_BACKOFF_MIN * 60 );
		} else {
			$next_ts = urlifywriterautoscan_compute_next_scan( $base_minutes, $id );
		}

		// Preparar datos (formateo consistente para DATETIME).
		$data = array(
			$meta['col_last'] => date_i18n( 'Y-m-d H:i:s', $now_ts ),
			$meta['col_next'] => date_i18n( 'Y-m-d H:i:s', $next_ts ),
		);
		$fmt  = array( '%s', '%s' );

		// UPDATE seguro en tabla propia del plugin. No hay API core; sin caché a propósito.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom plugin table update; $wpdb->update() prepara internamente. Caché intencionalmente desactivada.
		$wpdb->update(
			$tbl,
			$data,
			array( $meta['col_id'] => (int) $id ),
			$fmt,
			array( '%d' )
		);
	}
}


/* ============================================================
 * 8) Selección de fuentes vencidas (helper opcional)
 * ============================================================ */
if ( ! function_exists('urlifywriterautoscan_select_due_sources') ) {
	function urlifywriterautoscan_select_due_sources( $limit = 10 ) {
		global $wpdb;

		$meta    = urlifywriterautoscan_detect_sources_schema();
		$is_src  = ! empty( $meta['is_sources'] );
		$limit   = max( 1, (int) $limit );
		$now_sql = current_time( 'mysql' );

		if ( $is_src ) {
			// Esquema NUEVO: {prefix}urlifywriterautoscan_sources
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			return $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM {$wpdb->prefix}urlifywriterautoscan_sources
					  WHERE enabled = 1
					    AND (next_scan_at IS NULL OR next_scan_at = '' OR next_scan_at <= %s)
					  ORDER BY next_scan_at IS NULL DESC, next_scan_at ASC
					  LIMIT %d",
					$now_sql,
					$limit
				)
			);
		}

		// Esquema CLÁSICO: {prefix}urlifywriterautoscan_sites
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		return $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}urlifywriterautoscan_sites
				  WHERE enabled = 1
				    AND (next_scan IS NULL OR next_scan = '' OR next_scan <= %s)
				  ORDER BY next_scan IS NULL DESC, next_scan ASC
				  LIMIT %d",
				$now_sql,
				$limit
			)
		);
	}
}



/* ============================================================
 * 9) === AUTO-GEN (segundo cron) =============================
 * ============================================================ */

/** Intervalo guardado para AutoGen */
if ( ! function_exists('urlifywriterautogen_get_interval_key') ) {
	function urlifywriterautogen_get_interval_key() {
		$key = get_option('urlifywriterautogen_interval', 'urlifywriterevery_15_minutes');
		$allowed = ['urlifywriterevery_5_minutes','urlifywriterevery_15_minutes','urlifywriterevery_30_minutes','hourly','urlifywriterevery_3_hours'];
		if ( ! in_array($key, $allowed, true) ) $key = 'urlifywriterevery_15_minutes';
		return $key;
	}
}

/** Programar / reprogramar / limpiar AutoGen */
if ( ! function_exists('urlifywriterautogen_schedule_event') ) {
	function urlifywriterautogen_schedule_event( $interval_key = null ) {
		$interval_key = $interval_key ?: urlifywriterautogen_get_interval_key();
		if ( ! wp_next_scheduled('urlifywriterautogen_cron') ) {
			wp_schedule_event( time() + 20, $interval_key, 'urlifywriterautogen_cron' );
		}
	}
}
if ( ! function_exists('urlifywriterautogen_reschedule_event') ) {
	function urlifywriterautogen_reschedule_event( $interval_key ) {
		update_option('urlifywriterautogen_interval', $interval_key);
		$ts = wp_next_scheduled('urlifywriterautogen_cron');
		if ($ts) wp_unschedule_event($ts, 'urlifywriterautogen_cron');
		wp_schedule_event( time() + 20, $interval_key, 'urlifywriterautogen_cron' );
	}
}
if ( ! function_exists('urlifywriterautogen_clear_event') ) {
	function urlifywriterautogen_clear_event() {
		$ts = wp_next_scheduled('urlifywriterautogen_cron');
		if ($ts) wp_unschedule_event($ts, 'urlifywriterautogen_cron');
	}
}

/** Callback del cron de AutoGen */
if ( ! function_exists('urlifywriterautogen_run') ) {
	function urlifywriterautogen_run(){
		$t0         = microtime(true);
		$limit_opt  = (int) get_option('urlifywriterautogen_batch', 5);
		$limit      = max(1, min(20, $limit_opt));
		$interval_k = get_option('urlifywriterautogen_interval', 'urlifywriterevery_15_minutes');

		if ( defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug-only logging, guarded by URLIFYWRITER_DEBUG
			error_log(sprintf('[URLIFYWRITER][AutoGen Cron] start limit=%d (opt=%d) interval=%s',
				$limit, $limit_opt, $interval_k
			));
		}

		$res = ['ok'=>false, 'processed'=>0, 'errors'=>0];

		if ( function_exists('urlifywriterautoscan_process_queue') ) {
			try {
				$r = urlifywriterautoscan_process_queue($limit);
				if ( is_array($r) ) {
					$res['ok']        = ! empty($r['ok']);
					$res['processed'] = isset($r['processed']) ? (int)$r['processed'] : 0;
					$res['errors']    = isset($r['errors']) ? (int)$r['errors'] : 0;
				} else {
					$res = ['ok'=>true, 'processed'=>0, 'errors'=>0];
				}
			} catch (\Throwable $e) {
				$res = ['ok'=>false, 'processed'=>0, 'errors'=>1];
				if ( defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG ) {
					// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug-only logging, guarded by URLIFYWRITER_DEBUG
					error_log('[URLIFYWRITER][AutoGen Cron] exception: '.$e->getMessage());
				}
			}
		} else {
			if ( defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug-only logging, guarded by URLIFYWRITER_DEBUG
				error_log('[URLIFYWRITER][AutoGen Cron] missing function: urlifywriterautoscan_process_queue');
			}
		}

		update_option('urlifywriterautogen_last_run', current_time('mysql'));

		if ( defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG ) {
			$ms      = (int) round((microtime(true) - $t0) * 1000);
			$ts_next = wp_next_scheduled('urlifywriterautogen_cron');
			$next_at = $ts_next ? date_i18n('Y-m-d H:i:s', $ts_next) : '';
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug-only logging, guarded by URLIFYWRITER_DEBUG
			error_log(sprintf('[URLIFYWRITER][AutoGen Cron] done ok=%s processed=%d errors=%d time_ms=%d next=%s',
				$res['ok'] ? '1' : '0',
				(int)$res['processed'],
				(int)$res['errors'],
				$ms,
				$next_at
			));
		}

		return $res;
	}
	add_action('urlifywriterautogen_cron', 'urlifywriterautogen_run');
}


/* REST para AutoGen: /wp-json/autopress/v1/autogen/run */
add_action( 'rest_api_init', function () {

	register_rest_route( 'autopress/v1', '/autogen/run', array(
		'methods'  => array( 'POST', 'GET' ),
		'callback' => function( WP_REST_Request $request ) {

			$res = urlifywriterautogen_run();
			return new WP_REST_Response( $res, 200 );
		},
		'permission_callback' => function( WP_REST_Request $request ) {

			// ✅ Admin del sitio
			if ( current_user_can( 'manage_options' ) ) {
				return true;
			}

			// ✅ Webhook key
			$key_opt = (string) get_option( 'urlifywriterautogen_webhook_key', '' );
			if ( $key_opt === '' ) {
				return new WP_Error( 'forbidden', 'Forbidden', array( 'status' => 403 ) );
			}

			$provided = (string) $request->get_param( 'key' );

			// Opcional: permitir header
			if ( $provided === '' ) {
				$provided = (string) $request->get_header( 'x-webhook-key' );
			}

			if ( $provided !== '' && hash_equals( $key_opt, $provided ) ) {
				return true;
			}

			return new WP_Error( 'forbidden', 'Forbidden', array( 'status' => 403 ) );
		},
	) );

} );


/* WP-CLI para AutoGen */
if ( defined('WP_CLI') && WP_CLI ) {
	class URLIFYWRITER_Autogen_CLI {
		public function __invoke( $args, $assoc_args ) {
			if ( isset($assoc_args['reschedule']) ) {
				$int = sanitize_text_field( $assoc_args['reschedule'] );
				urlifywriterautogen_reschedule_event( $int );
				\WP_CLI::success( 'AutoGen rescheduled to: ' . $int );
			}
			if ( isset($assoc_args['batch']) ) {
				$batch = max(1, (int)$assoc_args['batch']);
				update_option('urlifywriterautogen_batch', $batch);
				\WP_CLI::success( 'AutoGen batch set to: ' . $batch );
			}
			$res = urlifywriterautogen_run();
			\WP_CLI::line( 'Result: ' . wp_json_encode($res) );
			if ( ! empty($res['ok']) ) \WP_CLI::success('Autogen finished.');
			else \WP_CLI::error('Autogen failed.');
		}
	}
	\WP_CLI::add_command( 'urlifywriter autogen', 'URLIFYWRITER_Autogen_CLI' );
}
