<?php
/**
 * UrlifyWriter — AutoScan AJAX
 *
 * Admin AJAX endpoints to manage:
 *  - Sources (domains) for autoscan
 *  - Test crawl (preview of detected links)
 *  - Trigger scans / manual queue processing
 *  - List / ignore detected items
 *
 * This file outputs JSON responses only (no HTML/scripts).
 */

if ( ! defined('ABSPATH') ) { exit; }

/* -----------------------------------------------------------
 * Helpers
 * ----------------------------------------------------------- */

if ( ! function_exists('urlifywriterautoscan_tables') ) {
	function urlifywriterautoscan_tables() {
		global $wpdb;
		return [
			'sources' => $wpdb->prefix . 'urlifywriterautoscan_sources',
			'items'   => $wpdb->prefix . 'urlifywriterautoscan_items',
		];
	}
}

if ( ! function_exists('urlifywriterautoscan_cap_manage') ) {
	function urlifywriterautoscan_cap_manage() {
		return current_user_can('manage_options');
	}
}
if ( ! function_exists('urlifywriterautoscan_cap_operate') ) {
	function urlifywriterautoscan_cap_operate() {
		return ( current_user_can('edit_posts') || current_user_can('edit_pages') );
	}
}

if ( ! function_exists('urlifywriterautoscan_sanitize_url') ) {
	function urlifywriterautoscan_sanitize_url( $url ) {
		$url = trim( (string) $url );
		if ( $url === '' ) return '';
		if ( strpos($url, '//') === 0 ) $url = (is_ssl() ? 'https:' : 'http:') . $url;
		if ( ! preg_match('~^https?://~i', $url) ) $url = 'https://' . ltrim($url, '/');
		$clean = esc_url_raw( $url );
		return $clean ?: '';
	}
}

/* -----------------------------------------------------------
 * Nonces
 * ----------------------------------------------------------- */

define('URLIFYWRITER_AUTOSCAN_NONCE_ACTION', 'urlifywriterautoscan');
define('URLIFYWRITER_AUTOSCAN_NONCE_FIELD',  'nonce');

/* -----------------------------------------------------------
 * 1) List sources
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_sources_list', function () {
	if ( ! urlifywriterautoscan_cap_operate() ) {
		wp_send_json_error( [ 'message' => __( 'You do not have permission.', 'urlifywriter' ) ], 403 );
	}
	check_ajax_referer( URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD );

	global $wpdb;
	$T = urlifywriterautoscan_tables();

	// Sanea el nombre de la tabla
	$sources_tbl = esc_sql( $T['sources'] );

	// Nombres de tabla no admiten placeholders → documentado, sin caché
	// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
	$rows = $wpdb->get_results(
		"
		SELECT id, domain, start_url, enabled, max_per_day, scan_interval,
			   publish_mode, last_scan_at, next_scan_at, created_at, updated_at,
			   gen_instructions
		FROM `{$sources_tbl}`
		ORDER BY id DESC
		",
		ARRAY_A
	);
	// phpcs:enable


	wp_send_json_success( [
		'sources' => array_map(
			function ( $r ) {
				$r['id']               = (int) $r['id'];
				$r['enabled']          = (int) $r['enabled'];
				$r['max_per_day']      = (int) $r['max_per_day'];
				$r['scan_interval']    = (string) ( $r['scan_interval'] ?? 'hourly' );
				$r['publish_mode']     = 'publish'; // UI fija
				$r['last_scan_at']     = (string) ( $r['last_scan_at'] ?? '' );
				$r['next_scan_at']     = (string) ( $r['next_scan_at'] ?? '' );
				$r['gen_instructions'] = (string) ( $r['gen_instructions'] ?? '' );
				return $r;
			},
			$rows ?: []
		),
	] );
} );

/* -----------------------------------------------------------
 * 2) Create / update source
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_source_save', function () {
	if ( ! urlifywriterautoscan_cap_manage() ) {
		wp_send_json_error( [ 'message' => __( 'You do not have permission.', 'urlifywriter' ) ], 403 );
	}
	check_ajax_referer( URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD );

	global $wpdb;
	$T = urlifywriterautoscan_tables();

	$id           = isset( $_POST['id'] ) ? absint( wp_unslash( $_POST['id'] ) ) : 0;
	$domain_raw   = isset( $_POST['domain'] ) ? wp_strip_all_tags( wp_unslash( $_POST['domain'] ) ) : '';
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	$start_url    = isset( $_POST['start_url'] ) ? urlifywriterautoscan_sanitize_url( wp_unslash( $_POST['start_url'] ) ) : '';
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	$enabled      = isset( $_POST['enabled'] ) ? (int) !! wp_unslash( $_POST['enabled'] ) : 0;
	$max_per_day  = isset( $_POST['max_per_day'] ) ? max( 0, absint( wp_unslash( $_POST['max_per_day'] ) ) ) : 5;
	$scan_int     = isset( $_POST['scan_interval'] ) ? sanitize_text_field( wp_unslash( $_POST['scan_interval'] ) ) : 'hourly';
	$publish_mode = 'publish';
	$notes        = isset( $_POST['notes'] ) ? wp_kses_post( wp_unslash( $_POST['notes'] ) ) : '';
	$gen_instructions = isset( $_POST['gen_instructions'] ) ? wp_kses_post( wp_unslash( $_POST['gen_instructions'] ) ) : '';

	$allowed_intervals = [
		'urlifywriterevery_30_minutes','hourly','urlifywriterevery_3_hours','urlifywriterevery_6_hours','urlifywriterevery_12_hours','daily',
	];
	if ( ! in_array( $scan_int, $allowed_intervals, true ) ) {
		$scan_int = 'hourly';
	}

	$domain = '';
	if ( $domain_raw !== '' ) {
		$domain = preg_replace( '~^https?://~i', '', trim( $domain_raw ) );
		$domain = preg_replace( '~/.+$~', '', $domain );
	}
	if ( $domain === '' && $start_url ) {
		$domain = wp_parse_url( $start_url, PHP_URL_HOST ) ?: '';
	}
	if ( $domain === '' ) {
		wp_send_json_error( [ 'message' => __( 'Domain is required.', 'urlifywriter' ) ], 400 );
	}
	if ( $start_url === '' ) {
		$start_url = 'https://' . $domain . '/';
	}

	$data = [
		'domain'           => strtolower( $domain ),
		'start_url'        => $start_url,
		'enabled'          => $enabled,
		'max_per_day'      => $max_per_day,
		'scan_interval'    => $scan_int,
		'publish_mode'     => $publish_mode,
		'notes'            => $notes,
		'gen_instructions' => ( $gen_instructions !== '' ? $gen_instructions : null ),
		'updated_at'       => current_time( 'mysql' ),
	];

	if ( $id > 0 ) {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$ok = $wpdb->update( $T['sources'], $data, [ 'id' => $id ], null, [ '%d' ] );
		if ( $ok === false ) {
			wp_send_json_error( [ 'message' => __( 'Could not update source.', 'urlifywriter' ) ], 500 );
		}
	} else {
		$data['created_at'] = current_time( 'mysql' );
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$ok = $wpdb->insert( $T['sources'], $data );
		if ( ! $ok ) {
			wp_send_json_error( [ 'message' => __( 'Could not create source.', 'urlifywriter' ) ], 500 );
		}
		$id = (int) $wpdb->insert_id;
	}

	// Programar next_scan_at
	if ( function_exists( 'urlifywriterschedule_key_to_minutes' ) && function_exists( 'urlifywriterautoscan_compute_next_scan' ) ) {
		$mins    = urlifywriterschedule_key_to_minutes( $scan_int );
		$next_ts = urlifywriterautoscan_compute_next_scan( $mins, (int) $id );
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->update(
			$T['sources'],
			[ 'next_scan_at' => date_i18n( 'Y-m-d H:i:s', $next_ts ) ],
			[ 'id' => (int) $id ],
			[ '%s' ],
			[ '%d' ]
		);
	}

	// Optional: reprogramar cron global
	if ( function_exists( 'urlifywriterautoscan_reschedule_event' ) && ! empty( $_POST['reschedule_cron'] ) ) {
		urlifywriterautoscan_reschedule_event( $scan_int );
	}

	wp_send_json_success( [ 'id' => (int) $id ] );
} );

/* -----------------------------------------------------------
 * 3) Delete source (and its items)
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_source_delete', function () {
	if ( ! urlifywriterautoscan_cap_manage() ) {
		wp_send_json_error( [ 'message' => __( 'You do not have permission.', 'urlifywriter' ) ], 403 );
	}
	check_ajax_referer( URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD );

	$id = isset( $_POST['id'] ) ? absint( wp_unslash( $_POST['id'] ) ) : 0;
	if ( $id <= 0 ) {
		wp_send_json_error( [ 'message' => __( 'Invalid source id.', 'urlifywriter' ) ], 400 );
	}
	global $wpdb;
	$T = urlifywriterautoscan_tables();

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->delete( $T['items'], [ 'source_id' => $id ], [ '%d' ] );
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->delete( $T['sources'], [ 'id' => $id ], [ '%d' ] );

	wp_send_json_success( [ 'deleted' => true ] );
} );

/* -----------------------------------------------------------
 * 4) Toggle enable/disable source
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_source_toggle', function () {
	if ( ! urlifywriterautoscan_cap_manage() ) {
		wp_send_json_error( [ 'message' => __( 'You do not have permission.', 'urlifywriter' ) ], 403 );
	}
	check_ajax_referer( URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD );

	global $wpdb;
	$T = urlifywriterautoscan_tables();
	$id = isset( $_POST['id'] ) ? absint( wp_unslash( $_POST['id'] ) ) : 0;
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	$en = isset( $_POST['enabled'] ) ? (int) !! wp_unslash( $_POST['enabled'] ) : 0;
	if ( $id <= 0 ) {
		wp_send_json_error( [ 'message' => __( 'Invalid source id.', 'urlifywriter' ) ], 400 );
	}

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$ok = $wpdb->update( $T['sources'], [ 'enabled' => $en, 'updated_at' => current_time( 'mysql' ) ], [ 'id' => $id ], null, [ '%d' ] );
	if ( $ok === false ) {
		wp_send_json_error( [ 'message' => __( 'Failed to update.', 'urlifywriter' ) ], 500 );
	}

	// Programar/limpiar next_scan_at al (des)activar
	if ( $en ) {
		if ( function_exists( 'urlifywriterschedule_key_to_minutes' ) && function_exists( 'urlifywriterautoscan_compute_next_scan' ) ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$row     = $wpdb->get_row( $wpdb->prepare( "SELECT scan_interval FROM {$T['sources']} WHERE id=%d", $id ), ARRAY_A );
			$mins    = urlifywriterschedule_key_to_minutes( (string) ( $row['scan_interval'] ?? 'hourly' ) );
			$next_ts = urlifywriterautoscan_compute_next_scan( $mins, (int) $id );
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->update( $T['sources'], [ 'next_scan_at' => date_i18n( 'Y-m-d H:i:s', $next_ts ) ], [ 'id' => $id ], [ '%s' ], [ '%d' ] );
		}
	} else {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->update( $T['sources'], [ 'next_scan_at' => null ], [ 'id' => $id ], [ '%s' ], [ '%d' ] );
	}

	wp_send_json_success( [ 'id' => $id, 'enabled' => $en ] );
} );

/* -----------------------------------------------------------
 * 5) Test crawl (preview candidates)
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_test_crawl', function () {
	if ( ! urlifywriterautoscan_cap_operate() ) {
		wp_send_json_error( [ 'message' => __( 'You do not have permission.', 'urlifywriter' ) ], 403 );
	}
	check_ajax_referer( URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD );

	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	$start_url = isset( $_POST['start_url'] ) ? urlifywriterautoscan_sanitize_url( wp_unslash( $_POST['start_url'] ) ) : '';
	if ( ! $start_url ) {
		wp_send_json_error( [ 'message' => __( 'Invalid URL.', 'urlifywriter' ) ], 400 );
	}

	$resp = wp_remote_get(
		$start_url,
		[
			'timeout'     => 18,
			'redirection' => 5,
			'user-agent'  => 'UrlifyWriter/1.0 (+WP; AutoScan Test)',
		]
	);

	if ( is_wp_error( $resp ) ) {
		wp_send_json_error( [ 'message' => $resp->get_error_message() ], 500 );
	}
	$code = wp_remote_retrieve_response_code( $resp );
	if ( $code < 200 || $code >= 400 ) {
		wp_send_json_error( [ 'message' => 'HTTP ' . $code ], 500 );
	}
	$html = (string) wp_remote_retrieve_body( $resp );

	if ( ! function_exists( 'urlifywriterautoscan_extract_links' ) ) {
		wp_send_json_error( [ 'message' => __( 'Crawler core not loaded.', 'urlifywriter' ) ], 500 );
	}

	$cand = urlifywriterautoscan_extract_links( $html, $start_url );
	wp_send_json_success(
		[
			'count'      => count( $cand ),
			'candidates' => array_slice( $cand, 0, 25 ),
		]
	);
} );

/* -----------------------------------------------------------
 * 6) Scan a source now (insert candidates in queue) con cap diario
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_scan_now', function () {
	if ( ! urlifywriterautoscan_cap_operate() ) {
		wp_send_json_error( [ 'message' => __( 'You do not have permission.', 'urlifywriter' ) ], 403 );
	}
	check_ajax_referer( URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD );

	$id = isset( $_POST['id'] ) ? absint( wp_unslash( $_POST['id'] ) ) : 0;
	if ( $id <= 0 ) {
		wp_send_json_error( [ 'message' => __( 'Invalid source id.', 'urlifywriter' ) ], 400 );
	}

	global $wpdb;
	$T = urlifywriterautoscan_tables();

	// Cargar fuente
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	$src = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$T['sources']} WHERE id=%d", $id ), ARRAY_A );
	if ( ! $src ) {
		wp_send_json_error( [ 'message' => __( 'Source not found.', 'urlifywriter' ) ], 404 );
	}

	// Cap diario: contar detectados hoy NO ignorados
	$today = date_i18n( 'Y-m-d' );
// Sanitiza el nombre de la tabla antes de interpolarla
$items_tbl = esc_sql( $T['items'] );

	// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
	$detected_today = (int) $wpdb->get_var(
		$wpdb->prepare(
			"SELECT COUNT(1)
			   FROM `{$items_tbl}`
			  WHERE source_id = %d
				AND DATE(detected_at) = %s
				AND status <> 'ignored'",
			$id,
			$today
		)
	);
	// phpcs:enable

	$max_per_day_src = max( 0, (int) ( $src['max_per_day'] ?? 0 ) );

	// Si alcanzó cap: actualizar last/next y salir
	if ( $max_per_day_src > 0 && $detected_today >= $max_per_day_src ) {
		$last     = current_time( 'mysql' );
		$next_sql = null;
		if ( function_exists( 'urlifywriterschedule_key_to_minutes' ) && function_exists( 'urlifywriterautoscan_compute_next_scan' ) ) {
			$mins    = urlifywriterschedule_key_to_minutes( (string) $src['scan_interval'] );
			$next_ts = urlifywriterautoscan_compute_next_scan( $mins, (int) $id );
			$next_sql = date_i18n( 'Y-m-d H:i:s', $next_ts );
		}
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->update(
			$T['sources'],
			[ 'last_scan_at' => $last, 'next_scan_at' => $next_sql ],
			[ 'id' => $id ],
			[ '%s', '%s' ],
			[ '%d' ]
		);

		wp_send_json_success(
			[
				'ok'         => true,
				'inserted'   => 0,
				'candidates' => 0,
				'message'    => __( 'Daily discovery cap reached for this source. Scan skipped.', 'urlifywriter' ),
			]
		);
	}

	// Delegar si hay manager
	if ( function_exists( 'urlifywriterautoscan_scan_source' ) ) {
		$res = urlifywriterautoscan_scan_source( $id );
		wp_send_json_success( $res ?: [ 'ok' => true ] );
	}

	// Fallback simple
	$start_url = $src['start_url'];
	$resp      = wp_remote_get(
		$start_url,
		[
			'timeout'     => 18,
			'redirection' => 5,
			'user-agent'  => 'UrlifyWriter/1.0 (+WP; AutoScan Fallback)',
		]
	);
	if ( is_wp_error( $resp ) ) {
		wp_send_json_error( [ 'message' => $resp->get_error_message() ], 500 );
	}
	if ( (int) wp_remote_retrieve_response_code( $resp ) >= 400 ) {
		wp_send_json_error( [ 'message' => 'HTTP ' . wp_remote_retrieve_response_code( $resp ) ], 500 );
	}

	if ( ! function_exists( 'urlifywriterautoscan_extract_links' ) ) {
		wp_send_json_error( [ 'message' => __( 'Crawler core not loaded.', 'urlifywriter' ) ], 500 );
	}
	$html = (string) wp_remote_retrieve_body( $resp );
	$cand = urlifywriterautoscan_extract_links( $html, $start_url );

	// Limitar inserciones al remanente del día
	$remaining = ( $max_per_day_src > 0 ) ? max( 0, $max_per_day_src - $detected_today ) : PHP_INT_MAX;

	$inserted = 0;
	foreach ( $cand as $row ) {
		if ( $inserted >= $remaining ) {
			break;
		}

		$url = esc_url_raw( $row['url'] );
		if ( ! $url ) {
			continue;
		}

		// Dedupe exacto por URL
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$exists = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(1) FROM {$T['items']} WHERE source_id=%d AND url=%s", $id, $url ) );
		if ( $exists ) {
			continue;
		}

		// Insertar
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$ins = $wpdb->insert(
			$T['items'],
			[
				'source_id'   => $id,
				'url'         => $url,
				'title'       => wp_strip_all_tags( $row['title'] ?? '' ),
				'status'      => 'pending', // pending|approved|ignored|processing|done|failed
				'priority'    => floatval( $row['priority'] ?? 0 ),
				'detected_at' => current_time( 'mysql' ),
				'updated_at'  => current_time( 'mysql' ),
			]
		);
		if ( $ins ) {
			$inserted++;
			$detected_today++;
			if ( $max_per_day_src > 0 && $detected_today >= $max_per_day_src ) {
				break;
			}
		}
	}

	// Actualizar last y programar next
	$last     = current_time( 'mysql' );
	$next_sql = null;
	if ( function_exists( 'urlifywriterschedule_key_to_minutes' ) && function_exists( 'urlifywriterautoscan_compute_next_scan' ) ) {
		$mins    = urlifywriterschedule_key_to_minutes( (string) $src['scan_interval'] );
		$next_ts = urlifywriterautoscan_compute_next_scan( $mins, (int) $id );
		$next_sql = date_i18n( 'Y-m-d H:i:s', $next_ts );
	}
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->update(
		$T['sources'],
		[ 'last_scan_at' => $last, 'next_scan_at' => $next_sql ],
		[ 'id' => $id ],
		[ '%s', '%s' ],
		[ '%d' ]
	);

	wp_send_json_success( [ 'ok' => true, 'inserted' => $inserted, 'candidates' => count( $cand ) ] );
} );

/* -----------------------------------------------------------
 * 7) List items (simple pagination)
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_items_list', function () {
	if ( ! urlifywriterautoscan_cap_operate() ) {
		wp_send_json_error( [ 'message' => __( 'You do not have permission.', 'urlifywriter' ) ], 403 );
	}
	check_ajax_referer( URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD );

	global $wpdb;
	$T = urlifywriterautoscan_tables();

	$source_id = isset( $_REQUEST['source_id'] ) ? absint( wp_unslash( $_REQUEST['source_id'] ) ) : 0;
	$status    = isset( $_REQUEST['status'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['status'] ) ) : '';
	$page      = max( 1, isset( $_REQUEST['page'] ) ? absint( wp_unslash( $_REQUEST['page'] ) ) : 1 );
	$per_page  = max( 1, min( 100, isset( $_REQUEST['per_page'] ) ? absint( wp_unslash( $_REQUEST['per_page'] ) ) : 20 ) );
	$offset    = ( $page - 1 ) * $per_page;

	if ( ! $source_id ) {
		wp_send_json_success( [ 'total' => 0, 'page' => $page, 'per_page' => $per_page, 'items' => [] ] );
	}

	if ( $status !== '' ) {
		// Construimos las consultas y suprimimos el aviso por interpolar el nombre de la tabla
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$count_sql = "SELECT COUNT(1) FROM {$T['items']} WHERE source_id=%d AND status=%s";
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		$total = (int) $wpdb->get_var( $wpdb->prepare( $count_sql, $source_id, $status ) );

		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$list_sql = "SELECT id, source_id, priority, title, url, status, last_error, post_id, detected_at
		             FROM {$T['items']} WHERE source_id=%d AND status=%s
		             ORDER BY (status='pending') DESC, detected_at DESC, priority DESC
		             LIMIT %d OFFSET %d";
		// Sanea el nombre de la tabla antes de interpolarlo
		$items_tbl = esc_sql( $T['items'] );

		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
		$rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT id, source_id, priority, title, url, status, last_error, post_id, detected_at
				 FROM `{$items_tbl}`
				 WHERE source_id = %d AND status = %s
				 ORDER BY (status='pending') DESC, detected_at DESC, priority DESC
				 LIMIT %d OFFSET %d",
				$source_id,
				$status,
				$per_page,
				$offset
			),
			ARRAY_A
		);
		// phpcs:enable

	} else {
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$count_sql = "SELECT COUNT(1) FROM {$T['items']} WHERE source_id=%d";
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		$total = (int) $wpdb->get_var( $wpdb->prepare( $count_sql, $source_id ) );

		// Sanitiza el nombre de la tabla
		$items_tbl = esc_sql( $T['items'] );

		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
		$rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT id, source_id, priority, title, url, status, last_error, post_id, detected_at
				 FROM `{$items_tbl}`
				 WHERE source_id = %d
				 ORDER BY (status = 'pending') DESC, detected_at DESC, priority DESC
				 LIMIT %d OFFSET %d",
				$source_id,
				$per_page,
				$offset
			),
			ARRAY_A
		);
		// phpcs:enable




	}

	wp_send_json_success( [
		'total'    => $total,
		'page'     => $page,
		'per_page' => $per_page,
		'items'    => array_map(
			function ( $r ) {
				$r['id']         = (int) $r['id'];
				$r['source_id']  = (int) $r['source_id'];
				$r['priority']   = (float) $r['priority'];
				$r['last_error'] = (string) ( $r['last_error'] ?? '' );
				$r['post_id']    = isset( $r['post_id'] ) ? (int) $r['post_id'] : 0;

				if ( $r['post_id'] > 0 ) {
					$edit            = get_edit_post_link( $r['post_id'], '' );
					$view            = get_permalink( $r['post_id'] );
					$r['post_admin'] = $edit ? (string) $edit : admin_url( 'post.php?post=' . $r['post_id'] . '&action=edit' );
					$r['post_url']   = $view ? (string) $view : home_url( '/?p=' . $r['post_id'] );
				} else {
					$r['post_admin'] = '';
					$r['post_url']   = '';
				}
				return $r;
			},
			$rows ?: []
		),
	] );
} );


/* -----------------------------------------------------------
 * 8) Approve item (compat)
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_item_approve', function () {
	if ( ! urlifywriterautoscan_cap_operate() ) {
		wp_send_json_error( [ 'message' => __( 'You do not have permission.', 'urlifywriter' ) ], 403 );
	}
	check_ajax_referer( URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD );

	global $wpdb;
	$T = urlifywriterautoscan_tables();

	$item_id  = isset( $_POST['id'] ) ? absint( wp_unslash( $_POST['id'] ) ) : 0;
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	$generate = isset( $_POST['generate'] ) ? (int) !! wp_unslash( $_POST['generate'] ) : 0;

	if ( $item_id <= 0 ) {
		wp_send_json_error( [ 'message' => __( 'Invalid item id.', 'urlifywriter' ) ], 400 );
	}

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	$item = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$T['items']} WHERE id=%d", $item_id ), ARRAY_A );
	if ( ! $item ) {
		wp_send_json_error( [ 'message' => __( 'Item not found.', 'urlifywriter' ) ], 404 );
	}

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->update(
		$T['items'],
		[
			'status'     => $generate ? 'processing' : 'approved',
			'updated_at' => current_time( 'mysql' ),
		],
		[ 'id' => $item_id ],
		null,
		[ '%d' ]
	);

	$result = [ 'ok' => true, 'id' => $item_id, 'status' => ( $generate ? 'processing' : 'approved' ) ];

	if ( $generate && function_exists( 'urlifywriterautoscan_generate_item' ) ) {
		try {
			$gen                   = urlifywriterautoscan_generate_item( $item_id );
			$result['generation'] = $gen ?: [ 'queued' => true ];
		} catch ( \Throwable $e ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->update( $T['items'], [ 'status' => 'failed', 'updated_at' => current_time( 'mysql' ) ], [ 'id' => $item_id ] );
			wp_send_json_error( [ 'message' => $e->getMessage() ], 500 );
		}
	}

	wp_send_json_success( $result );
} );

/* -----------------------------------------------------------
 * 9) Ignore item
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_item_ignore', function () {
	if ( ! urlifywriterautoscan_cap_operate() ) {
		wp_send_json_error( [ 'message' => __( 'You do not have permission.', 'urlifywriter' ) ], 403 );
	}
	check_ajax_referer( URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD );

	global $wpdb;
	$T = urlifywriterautoscan_tables();

	$item_id = isset( $_POST['id'] ) ? absint( wp_unslash( $_POST['id'] ) ) : 0;
	if ( $item_id <= 0 ) {
		wp_send_json_error( [ 'message' => __( 'Invalid item id.', 'urlifywriter' ) ], 400 );
	}

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$ok = $wpdb->update(
		$T['items'],
		[
			'status'     => 'ignored',
			'updated_at' => current_time( 'mysql' ),
		],
		[ 'id' => $item_id ],
		null,
		[ '%d' ]
	);

	if ( $ok === false ) {
		wp_send_json_error( [ 'message' => __( 'Update failed.', 'urlifywriter' ) ], 500 );
	}

	wp_send_json_success( [ 'ok' => true, 'id' => $item_id ] );
} );

/* -----------------------------------------------------------
 * 10) Bulk approve (iterativo, evita IN (...))
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_items_approve_bulk', function () {
	if ( ! urlifywriterautoscan_cap_operate() ) {
		wp_send_json_error( [ 'message' => __( 'You do not have permission.', 'urlifywriter' ) ], 403 );
	}
	check_ajax_referer( URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD );

	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	$ids = isset( $_POST['ids'] ) ? (array) wp_unslash( $_POST['ids'] ) : [];
	$ids = array_values( array_unique( array_map( 'absint', $ids ) ) );
	if ( empty( $ids ) ) {
		wp_send_json_error( [ 'message' => __( 'No items provided.', 'urlifywriter' ) ], 400 );
	}

	global $wpdb;
	$T = urlifywriterautoscan_tables();

	$updated = 0;
	foreach ( $ids as $iid ) {
		// Solo pasar de pending/ignored a approved
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$st = $wpdb->get_var( $wpdb->prepare( "SELECT status FROM {$T['items']} WHERE id=%d", $iid ) );
		if ( in_array( $st, [ 'pending', 'ignored' ], true ) ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$ok = $wpdb->update(
				$T['items'],
				[ 'status' => 'approved', 'updated_at' => current_time( 'mysql' ) ],
				[ 'id' => $iid ],
				null,
				[ '%d' ]
			);
			if ( $ok !== false ) { $updated++; }
		}
	}

	wp_send_json_success( [ 'ok' => true, 'count' => $updated ] );
} );

/* -----------------------------------------------------------
 * 11) Retry failed (ids[] opcional; sin IN)
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_items_retry', function () {
	if ( ! urlifywriterautoscan_cap_operate() ) {
		wp_send_json_error( [ 'message' => __( 'You do not have permission.', 'urlifywriter' ) ], 403 );
	}
	check_ajax_referer( URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD );

	global $wpdb;
	$T = urlifywriterautoscan_tables();

	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	$ids = isset( $_POST['ids'] ) ? (array) wp_unslash( $_POST['ids'] ) : [];
	$ids = array_values( array_filter( array_map( 'absint', $ids ) ) );

	if ( empty( $ids ) ) {
		// Obtener todos los fallidos y actualizarlos individualmente
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$ids = $wpdb->get_col( $wpdb->prepare( "SELECT id FROM {$T['items']} WHERE status=%s", 'failed' ) );
	}

	$changed = 0;
	foreach ( $ids as $iid ) {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$ok = $wpdb->update(
			$T['items'],
			[ 'status' => 'pending', 'updated_at' => current_time( 'mysql' ) ],
			[ 'id' => (int) $iid, 'status' => 'failed' ],
			null,
			[ '%d', '%s' ]
		);
		if ( $ok !== false ) { $changed++; }
	}

	wp_send_json_success( [ 'ok' => true, 'count' => $changed ] );
} );

/* -----------------------------------------------------------
 * 12) Process queue (approved → generate) manually (compat)
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_process_queue', function () {
	if ( ! urlifywriterautoscan_cap_operate() ) {
		wp_send_json_error(['message'=>__('You do not have permission.', 'urlifywriter')], 403);
	}
	check_ajax_referer(URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD);

	$limit = isset($_POST['limit']) ? max(1, min(20, absint(wp_unslash($_POST['limit'])))) : 5;

	if ( function_exists('urlifywriterautoscan_process_queue') ) {
		$res = urlifywriterautoscan_process_queue( $limit );
		wp_send_json_success( $res ?: ['ok'=>true] );
	}

	wp_send_json_error(['message'=>__('Queue processor not available.', 'urlifywriter')], 500);
});

/* -----------------------------------------------------------
 * 13) Cron status (global ticker)
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_cron_status', function () {
	if ( ! current_user_can('manage_options') ) {
		wp_send_json_error(['message'=>__('You do not have permission.', 'urlifywriter')], 403);
	}
	check_ajax_referer(URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD);

	$ts_next = wp_next_scheduled('urlifywriterautoscan_cron');
	$key     = get_option('urlifywriterautoscan_interval', 'hourly');
	$last    = get_option('urlifywriterautoscan_last_run', '');

	wp_send_json_success([
		'status' => [
			'scheduled'        => (bool) $ts_next,
			'next_ts'          => (int) ($ts_next ?: 0),
			'next_at'          => $ts_next ? date_i18n('Y-m-d H:i:s', $ts_next) : '',
			'interval'         => $key,
			'last_run'         => (string) $last,
			'now'              => current_time('mysql'),
			'wp_cron_disabled' => ( defined('DISABLE_WP_CRON') && DISABLE_WP_CRON ) ? 1 : 0,
		],
	]);
});

/* -----------------------------------------------------------
 * 14) Cron start / reschedule
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_cron_start', function () {
	if ( ! current_user_can('manage_options') ) {
		wp_send_json_error(['message'=>__('You do not have permission.', 'urlifywriter')], 403);
	}
	check_ajax_referer(URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD);

	$interval = isset($_POST['interval']) ? sanitize_text_field( wp_unslash($_POST['interval']) ) : 'hourly';
	$allowed  = ['urlifywriterevery_5_minutes','urlifywriterevery_15_minutes','urlifywriterevery_30_minutes','hourly','urlifywriterevery_3_hours','urlifywriterevery_6_hours','urlifywriterevery_12_hours','daily'];
	if ( ! in_array($interval, $allowed, true) ) $interval = 'hourly';

	if ( ! function_exists('urlifywriterautoscan_reschedule_event') || ! function_exists('urlifywriterautoscan_schedule_event') ) {
		wp_send_json_error(['message'=>__('Cron helpers missing.', 'urlifywriter')], 500);
	}

	update_option('urlifywriterautoscan_interval', $interval);

	$next = wp_next_scheduled('urlifywriterautoscan_cron');
	if ( $next ) {
		urlifywriterautoscan_reschedule_event( $interval );
	} else {
		wp_schedule_event( time() + 10, $interval, 'urlifywriterautoscan_cron' );
	}

	$ts_next = wp_next_scheduled('urlifywriterautoscan_cron');
	wp_send_json_success([
		'scheduled' => (bool) $ts_next,
		'next_at'   => $ts_next ? date_i18n('Y-m-d H:i:s', $ts_next) : '',
		'interval'  => $interval,
	]);
});

/* -----------------------------------------------------------
 * 15) Cron stop
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_cron_stop', function () {
	if ( ! current_user_can('manage_options') ) {
		wp_send_json_error(['message'=>__('You do not have permission.', 'urlifywriter')], 403);
	}
	check_ajax_referer(URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD);

	if ( ! function_exists('urlifywriterautoscan_clear_event') ) {
		wp_send_json_error(['message'=>__('Cron helpers missing.', 'urlifywriter')], 500);
	}
	urlifywriterautoscan_clear_event();

	wp_send_json_success(['stopped'=>true]);
});

/* -----------------------------------------------------------
 * 16) Cron run now
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautoscan_cron_run_now', function () {
	if ( ! current_user_can('manage_options') ) {
		wp_send_json_error(['message'=>__('You do not have permission.', 'urlifywriter')], 403);
	}
	check_ajax_referer(URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD);

	if ( ! function_exists('urlifywriterautoscan_run') ) {
		wp_send_json_error(['message'=>__('Runner not available.', 'urlifywriter')], 500);
	}
	$res = urlifywriterautoscan_run();
	update_option('urlifywriterautoscan_last_run', current_time('mysql'));

	wp_send_json_success([
		'ok'        => ! empty($res['ok']),
		'scanned'   => isset($res['scanned']) ? (int)$res['scanned'] : 0,
		'generated' => isset($res['generated']) ? (int)$res['generated'] : 0,
	]);
});

/* -----------------------------------------------------------
 * 17) AutoGen cron status
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautogen_cron_status', function () {
	if ( ! current_user_can('manage_options') ) {
		wp_send_json_error(['message'=>__('You do not have permission.', 'urlifywriter')], 403);
	}
	check_ajax_referer(URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD);

	$ts_next = wp_next_scheduled('urlifywriterautogen_cron');
	$key     = get_option('urlifywriterautogen_interval', 'urlifywriterevery_5_minutes');
	$last    = get_option('urlifywriterautogen_last_run', '');

	wp_send_json_success([
		'status' => [
			'scheduled' => (bool) $ts_next,
			'next_ts'   => (int) ($ts_next ?: 0),
			'next_at'   => $ts_next ? date_i18n('Y-m-d H:i:s', $ts_next) : '',
			'interval'  => $key,
			'last_run'  => (string) $last,
			'now'       => current_time('mysql'),
		],
	]);
});

/* -----------------------------------------------------------
 * 18) AutoGen cron start / reschedule
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautogen_cron_start', function () {
	if ( ! current_user_can('manage_options') ) {
		wp_send_json_error(['message'=>__('You do not have permission.', 'urlifywriter')], 403);
	}
	check_ajax_referer(URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD);

	$interval = isset($_POST['interval']) ? sanitize_text_field( wp_unslash($_POST['interval']) ) : 'urlifywriterevery_15_minutes';
	$allowed  = ['urlifywriterevery_5_minutes','urlifywriterevery_15_minutes','urlifywriterevery_30_minutes','hourly','urlifywriterevery_3_hours'];
	if ( ! in_array($interval, $allowed, true) ) $interval = 'urlifywriterevery_15_minutes';

	if ( ! function_exists('urlifywriterautogen_reschedule_event') || ! function_exists('urlifywriterautogen_schedule_event') ) {
		wp_send_json_error(['message'=>__('Cron helpers missing.', 'urlifywriter')], 500);
	}

	update_option('urlifywriterautogen_interval', $interval);

	$next = wp_next_scheduled('urlifywriterautogen_cron');
	if ( $next ) {
		urlifywriterautogen_reschedule_event( $interval );
	} else {
		wp_schedule_event( time() + 10, $interval, 'urlifywriterautogen_cron' );
	}

	$ts_next = wp_next_scheduled('urlifywriterautogen_cron');
	wp_send_json_success([
		'scheduled' => (bool) $ts_next,
		'next_at'   => $ts_next ? date_i18n('Y-m-d H:i:s', $ts_next) : '',
		'interval'  => $interval,
	]);
});

/* -----------------------------------------------------------
 * 19) AutoGen cron stop
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautogen_cron_stop', function () {
	if ( ! current_user_can('manage_options') ) {
		wp_send_json_error(['message'=>__('You do not have permission.', 'urlifywriter')], 403);
	}
	check_ajax_referer(URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD);

	if ( ! function_exists('urlifywriterautogen_clear_event') ) {
		wp_send_json_error(['message'=>__('Cron helpers missing.', 'urlifywriter')], 500);
	}
	urlifywriterautogen_clear_event();

	wp_send_json_success(['stopped'=>true]);
});

/* -----------------------------------------------------------
 * 20) AutoGen cron run now
 * ----------------------------------------------------------- */
add_action('wp_ajax_urlifywriterautogen_cron_run_now', function () {
	if ( ! current_user_can('manage_options') ) {
		wp_send_json_error(['message'=>__('You do not have permission.', 'urlifywriter')], 403);
	}
	check_ajax_referer(URLIFYWRITER_AUTOSCAN_NONCE_ACTION, URLIFYWRITER_AUTOSCAN_NONCE_FIELD);

	if ( ! function_exists('urlifywriterautogen_run') ) {
		wp_send_json_error(['message'=>__('Runner not available.', 'urlifywriter')], 500);
	}

	$res = urlifywriterautogen_run();
	update_option('urlifywriterautogen_last_run', current_time('mysql'));

	wp_send_json_success( $res ?: ['ok'=>true] );
});

/**
 * Provider por defecto para `urlifywriterautogen_fetch_images`
 */
add_filter('urlifywriterautogen_fetch_images', 'urlifywriterprovider_worker_fetch_images', 10, 5);
function urlifywriterprovider_worker_fetch_images($urls, $query, $mode, $n, $args){
	$DBG = (defined('URLIFYWRITER_DEBUG') && URLIFYWRITER_DEBUG);

	$mode  = in_array($mode, ['pixabay','ai'], true) ? $mode : 'pixabay';
	$n     = max(1, min(10, (int)$n));
	$query = trim(preg_replace('/[#,;:|\/\\\\\\[\\]\\(\\)\\{\\}"\'`<>^~]+/u', ' ', (string)$query));
	if ($query === '') $query = 'travel landscape';

	$prefer_user = !empty($args['prefer_user_api']);
	$style_hint  = isset($args['style_hint']) ? (string)$args['style_hint'] : '';
	$post_id     = isset($args['post_id']) ? (int)$args['post_id'] : 0;

	if ( ! defined('URLIFYWRITER_IMG_API') || empty(URLIFYWRITER_IMG_API) ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		if ($DBG) error_log('[URLIFYWRITER][IMG Provider] URLIFYWRITER_IMG_API not set');
		return [];
	}
	if ( ! defined('URLIFYWRITER_LIC_HMAC') || empty(URLIFYWRITER_LIC_HMAC) ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		if ($DBG) error_log('[URLIFYWRITER][IMG Provider] URLIFYWRITER_LIC_HMAC not set');
		return [];
	}

	$user_px = get_option('urlifywriterapi_pixabay_key', '');
	$user_oa = get_option('urlifywriterapi_openai_key', '');

	$sections = [];
	if ($post_id) {
		$sections = urlifywriter_compute_image_sections_for_post($post_id, min($n, 4));
	}

	if ($mode === 'ai') {
		$style_hint = trim(wp_strip_all_tags($style_hint));
		if ($style_hint === '') {
			$style_hint = (string) get_option('urlifywriterai_style_hint','');
			$style_hint = trim(wp_strip_all_tags($style_hint));
		}
		if ($style_hint !== '') $style_hint = mb_substr($style_hint, 0, 200);
	}

	$lic_opt = get_option('urlifywriterlicense_key','');

	$payload = [
		'ts'          => time() * 1000,
		'license_key' => ($lic_opt !== '' ? $lic_opt : ''),
		'site_domain' => wp_parse_url(home_url(), PHP_URL_HOST),
		'mode'        => $mode,
		'query'       => $query,
		'n'           => $n,
		'user_api'    => [
			'pixabay' => $user_px ?: '',
			'openai'  => $user_oa ?: '',
		],
		'prefer_user_api' => $prefer_user ? true : false,
		'lang'            => get_locale() ? substr(get_locale(),0,2) : 'en',
		'sections'        => $sections,
		'debug'           => $DBG ? 1 : 0,
	];

	if ($mode === 'ai' && $style_hint !== '') {
		$payload['style'] = $style_hint;
	}

	if ($DBG) {
		$log = $payload;
		if (!empty($log['user_api']['pixabay'])) $log['user_api']['pixabay'] = '***' . substr($log['user_api']['pixabay'], -4);
		if (!empty($log['user_api']['openai']))  $log['user_api']['openai']  = '***' . substr($log['user_api']['openai'],  -4);
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log('[URLIFYWRITER][IMG Provider] payload=' . wp_json_encode($log));
	}

	$res = urlifywritercloud_post(URLIFYWRITER_IMG_API, $payload);

	if (is_wp_error($res)) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		if ($DBG) error_log('[URLIFYWRITER][IMG Provider] ERROR ' . $res->get_error_code() . ': ' . $res->get_error_message());
		return [];
	}
	if (isset($res['code']) && $res['code'] === 'content_policy_violation') {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		if ($DBG) error_log('[URLIFYWRITER][IMG Provider] blocked by provider policy');
		return [];
	}

	$urls = (array) ($res['urls'] ?? []);
	if ($DBG) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log('[URLIFYWRITER][IMG Provider] got=' . count($urls) . ' provider=' . (string)($res['provider'] ?? $mode));
	}

	return array_values($urls);
}

/**
 * Extrae hasta $max secciones (H2/H3 o párrafos) del post para orientar al worker.
 */
function urlifywriter_compute_image_sections_for_post($post_id, $max = 3){
	$max = max(1, min(10, (int)$max));
	$html = get_post_field('post_content', $post_id);
	if (!is_string($html) || $html === '') {
		$t = get_the_title($post_id);
		return $t ? [ wp_strip_all_tags($t) ] : [];
	}
	$sections = [];

	if (preg_match_all('~<h(2|3)[^>]*>(.*?)</h\1>~is', $html, $m)) {
		foreach ($m[2] as $h) {
			$txt = trim(wp_strip_all_tags($h));
			if ($txt !== '') $sections[] = $txt;
			if (count($sections) >= $max) break;
		}
	}

	if (count($sections) < $max && preg_match_all('~<p[^>]*>(.*?)</p>~is', $html, $p)) {
		foreach ($p[1] as $ph) {
			$txt = trim(wp_strip_all_tags($ph));
			if (mb_strlen($txt) >= 40) $sections[] = mb_substr($txt, 0, 140);
			if (count($sections) >= $max) break;
		}
	}

	$sections = array_values(array_unique(array_filter($sections)));
	if (empty($sections)) {
		$t = get_the_title($post_id);
		if ($t) $sections[] = $t;
	}
	return array_slice($sections, 0, $max);
}
