<?php
// includes/ajax.php
if ( ! defined( 'ABSPATH' ) ) { exit; }

/* ==========================================================
 * Helper: slugify (shared)
 * ========================================================== */
if ( ! function_exists( 'urlifywriterslug' ) ) {
	function urlifywriterslug( $text, $fallback = 'image' ) {
		$text = wp_strip_all_tags( (string) $text );
		$text = remove_accents( $text );
		$text = preg_replace( '~[^\pL\d]+~u', '-', $text );
		$text = trim( $text, '-' );
		$text = preg_replace( '~[^-\w]+~', '', $text );
		$text = strtolower( $text );
		return $text !== '' ? $text : $fallback;
	}
}

/* ==========================================================
 * Helper: POST remoto con firma HMAC (solo si no existe)
 * ========================================================== */
if ( ! function_exists( 'urlifywritercloud_post' ) ) {
	function urlifywritercloud_post( $endpoint, $payload, $timeout = null ) {
		$timeout = is_null( $timeout )
			? ( defined( 'URLIFYWRITER_REMOTE_TIMEOUT' ) ? (int) URLIFYWRITER_REMOTE_TIMEOUT : 45 )
			: (int) $timeout;

		if ( ! filter_var( $endpoint, FILTER_VALIDATE_URL ) ) {
			return new WP_Error( 'bad_endpoint', 'Invalid remote endpoint URL.' );
		}

		$body = wp_json_encode( $payload );
		if ( false === $body ) {
			return new WP_Error( 'json_encode_error', 'Could not JSON-encode request payload.' );
		}

		$headers = array( 'Content-Type' => 'application/json' );

		// Firma HMAC hex (compatible con el Worker: header X-Signature)
		if ( defined( 'URLIFYWRITER_LIC_HMAC' ) && URLIFYWRITER_LIC_HMAC ) {
			$headers['X-Signature'] = hash_hmac( 'sha256', $body, URLIFYWRITER_LIC_HMAC );
		}

		$args = array(
			'method'  => 'POST',
			'timeout' => $timeout,
			'headers' => $headers,
			'body'    => $body,
		);

		$resp = wp_remote_request( $endpoint, $args );
		if ( is_wp_error( $resp ) ) {
			return $resp;
		}

		$code = (int) wp_remote_retrieve_response_code( $resp );
		$text = (string) wp_remote_retrieve_body( $resp );
		$json = json_decode( $text, true );

		// 2xx → OK si es JSON
		if ( $code >= 200 && $code < 300 ) {
			if ( is_array( $json ) ) {
				return $json;
			}
			return new WP_Error(
				'remote_json',
				'Remote returned non-JSON success.',
				array(
					'http_code'     => $code,
					'body_preview'  => substr( $text, 0, 400 ),
				)
			);
		}

		// 4xx/5xx → error WP_Error
		$msg = '';
		if ( is_array( $json ) && isset( $json['error'] ) ) {
			$msg = (string) $json['error'];
		}
		if ( ! $msg ) {
			$msg = 'HTTP ' . $code;
		}

		return new WP_Error(
			'remote_error',
			$msg,
			array(
				'http_code'     => $code,
				'body_preview'  => substr( $text, 0, 600 ),
				'remote_code'   => isset( $json['code'] ) ? (string) $json['code'] : '',
			)
		);
	}
}

/* ==========================================================
 * EXTRACT: descarga y parsea el artículo
 * ========================================================== */
add_action( 'wp_ajax_urlifywriterextract', function () {
	if ( ! current_user_can( 'edit_posts' ) ) {
		wp_send_json_error(
			array( 'code' => 'forbidden', 'message' => __( 'You do not have permission.', 'urlifywriter' ) ),
			403
		);
	}

	check_ajax_referer( 'urlifywriterurl_flow', 'nonce' );

	$url = isset( $_POST['url'] ) ? esc_url_raw( wp_unslash( $_POST['url'] ) ) : '';
	if ( ! filter_var( $url, FILTER_VALIDATE_URL ) ) {
		wp_send_json_error(
			array( 'code' => 'invalid_url', 'message' => __( 'Invalid URL format.', 'urlifywriter' ) ),
			400
		);
	}

	$extracted = urlifywriterextract_from_url_text( $url );
	if ( is_wp_error( $extracted ) ) {
		wp_send_json_error(
			array(
				'code'    => $extracted->get_error_code() ? $extracted->get_error_code() : 'extract_failed',
				'message' => $extracted->get_error_message(),
			),
			500
		);
	}

	// Compactar texto para la siguiente fase
	$extracted_text = '';
	if ( ! empty( $extracted['blocks'] ) && is_array( $extracted['blocks'] ) ) {
		$col = array();
		foreach ( $extracted['blocks'] as $b ) {
			if ( ! empty( $b['text'] ) && in_array( $b['type'], array( 'p', 'h2', 'h3' ), true ) ) {
				$col[] = $b['text'];
			}
			if ( strlen( implode( "\n", $col ) ) > 6000 ) {
				break;
			}
		}
		$extracted_text = implode( "\n", $col );
	} else {
		$extracted_text = (string) ( $extracted['content'] ?? '' );
	}

	$title_detected = trim( (string) ( $extracted['title'] ?? '' ) );

	// mini preview
	$preview_source = (string) ( $extracted['excerpt'] ?? '' );
	if ( '' === $preview_source && ! empty( $extracted['blocks'] ) ) {
		foreach ( $extracted['blocks'] as $b ) {
			if ( ( $b['type'] ?? '' ) === 'p' && ! empty( $b['text'] ) ) {
				$preview_source = $b['text'];
				break;
			}
		}
	}

	$sentences    = preg_split( '/(?<=[\.\!\?])\s+/u', trim( (string) $preview_source ) );
	$mini_preview = implode( ' ', array_slice( array_filter( (array) $sentences ), 0, 2 ) );

	wp_send_json_success(
		array(
			'title_detected'   => $title_detected,
			'mini_preview'     => $mini_preview,
			'extracted_text'   => $extracted_text,
			'origin_image_url' => isset( $extracted['origin_image_url'] ) ? esc_url_raw( $extracted['origin_image_url'] ) : '',
		)
	);
} );

/* ==========================================================
 * Helper: idioma por defecto a partir del locale WP (fallback)
 * ========================================================== */
if ( ! function_exists( 'urlifywriterget_default_lang_from_wp' ) ) {
	function urlifywriterget_default_lang_from_wp() {
		$locale = get_locale();
		$locale = is_string( $locale ) ? $locale : 'en_US';
		$key    = strtolower( str_replace( '-', '_', $locale ) );

		if ( in_array( $key, array( 'zh_tw', 'zh_hk' ), true ) ) { return 'zh-Hant'; }
		if ( in_array( $key, array( 'zh_cn', 'zh_sg' ), true ) ) { return 'zh'; }
		if ( 'iw_il' === $key || 'he_il' === $key ) { return 'he'; }
		if ( 'nb_no' === $key ) { return 'no'; }

		$short = substr( $locale, 0, 2 );
		return $short ? $short : 'en';
	}
}

/* ==========================================================
 * GENERATE: llama al Worker para generar el artículo
 * ========================================================== */
if ( ! function_exists( 'urlifywriterdebug_log' ) ) {
	function urlifywriterdebug_log( $message ) {
		/**
		 * Permite a los desarrolladores enganchar su propio logger sin introducir error_log().
		 * Ej.: add_action('urlifywriterdebug_log', function($m){ wc_get_logger()->debug($m, ['source'=>'urlifywriter']); });
		 */
		do_action( 'urlifywriterdebug_log', (string) $message );
	}
}

add_action( 'wp_ajax_urlifywritergenerate', function () {
	if ( ! current_user_can( 'edit_posts' ) ) {
		wp_send_json_error(
			array( 'code' => 'forbidden', 'message' => __( 'You do not have permission.', 'urlifywriter' ) ),
			403
		);
	}
	check_ajax_referer( 'urlifywriterurl_flow', 'nonce' );

	// Nuevo: modo y keyword
	$mode = isset( $_POST['mode'] ) ? sanitize_text_field( wp_unslash( $_POST['mode'] ) ) : 'url';
	$mode = in_array( $mode, array( 'url', 'keyword' ), true ) ? $mode : 'url';

	$keyword = isset( $_POST['keyword'] ) ? sanitize_text_field( wp_unslash( $_POST['keyword'] ) ) : '';

	// Campos existentes
	$url          = isset( $_POST['url'] ) ? esc_url_raw( wp_unslash( $_POST['url'] ) ) : '';
	$instructions = isset( $_POST['instructions'] ) ? wp_kses_post( wp_unslash( $_POST['instructions'] ) ) : '';

	$min_words_raw = isset( $_POST['min_words'] ) ? absint( wp_unslash( $_POST['min_words'] ) ) : 800;
	$max_words_raw = isset( $_POST['max_words'] ) ? absint( wp_unslash( $_POST['max_words'] ) ) : 1200;

	$min_words  = max( 100, min( 3000, $min_words_raw ) );
	$max_words  = max( 200, min( 3000, $max_words_raw ) );

	$title_hint   = isset( $_POST['title_hint'] ) ? sanitize_text_field( wp_unslash( $_POST['title_hint'] ) ) : '';
	$extracted_txt = isset( $_POST['extracted_text'] ) ? wp_kses_post( wp_unslash( $_POST['extracted_text'] ) ) : '';

	// Validaciones según modo
	if ( 'url' === $mode ) {
		if ( ! filter_var( $url, FILTER_VALIDATE_URL ) ) {
			wp_send_json_error(
				array( 'code' => 'invalid_url', 'message' => __( 'Invalid URL format.', 'urlifywriter' ) ),
				400
			);
		}
	} else {
		if ( '' === $keyword ) {
			wp_send_json_error(
				array( 'code' => 'invalid_keyword', 'message' => __( 'Keyword is required.', 'urlifywriter' ) ),
				400
			);
		}
		if ( '' === trim( (string) $instructions ) ) {
			wp_send_json_error(
				array( 'code' => 'instructions_required', 'message' => __( 'Instructions are required in keyword mode.', 'urlifywriter' ) ),
				400
			);
		}
		$url          = '';
		$extracted_txt = '';
		if ( '' === $title_hint ) {
			$title_hint = $keyword;
		}
	}

	$lang = (string) get_option( 'urlifywriterarticle_language', urlifywriterget_default_lang_from_wp() );

	// Categorías para sugerir al worker
	$cat_terms  = get_terms( array( 'taxonomy' => 'category', 'hide_empty' => false ) );
	$categories = array();
	if ( ! is_wp_error( $cat_terms ) ) {
		foreach ( $cat_terms as $t ) {
			$categories[] = array(
				'id'   => (int) $t->term_id,
				'name' => (string) $t->name,
				'slug' => (string) $t->slug,
			);
		}
	}

	$lic_opt = (string) get_option( 'urlifywriterlicense_key', '' );

	$home_parts  = wp_parse_url( home_url() );
	$site_domain = ( is_array( $home_parts ) && isset( $home_parts['host'] ) ) ? (string) $home_parts['host'] : '';

	$payload = array(
		'mode'         => $mode,
		'keyword'      => $keyword,
		'url'          => $url,
		'ts'           => time() * 1000,
		'license_key'  => ( '' !== $lic_opt ? $lic_opt : '' ),
		'site_domain'  => $site_domain,
		'site_url'     => home_url( '/' ),
		'instructions' => $instructions,
		'min_words'    => $min_words,
		'max_words'    => $max_words,
		'title_hint'   => $title_hint,
		'extracted'    => $extracted_txt,
		'lang'         => $lang,
		'categories'   => $categories,
	);

	$URLIFYWRITER_DEBUG = defined( 'URLIFYWRITER_DEBUG' )
		? (bool) URLIFYWRITER_DEBUG
		: ( defined( 'WP_DEBUG' ) && WP_DEBUG );

	if ( $URLIFYWRITER_DEBUG ) {
		urlifywriterdebug_log( '[URLIFYWRITER] /urlifywritergenerate start mode=' . $mode . ' | url=' . $url . ' | kw=' . $keyword . ' | lang=' . $lang );
		if ( defined( 'URLIFYWRITER_ART_API' ) ) {
			urlifywriterdebug_log( '[URLIFYWRITER] /urlifywritergenerate endpoint=' . URLIFYWRITER_ART_API );
		}
	}

	if ( ! defined( 'URLIFYWRITER_ART_API' ) || empty( URLIFYWRITER_ART_API ) ) {
		wp_send_json_error(
			array( 'code' => 'no_endpoint', 'message' => __( 'Remote endpoint is not configured.', 'urlifywriter' ) ),
			500
		);
	}
	if ( ! defined( 'URLIFYWRITER_LIC_HMAC' ) || empty( URLIFYWRITER_LIC_HMAC ) ) {
		wp_send_json_error(
			array( 'code' => 'no_hmac', 'message' => __( 'HMAC secret is not configured.', 'urlifywriter' ) ),
			500
		);
	}

	$res = urlifywritercloud_post( URLIFYWRITER_ART_API, $payload );

	if ( is_wp_error( $res ) ) {
		if ( $URLIFYWRITER_DEBUG ) {
			urlifywriterdebug_log( '[URLIFYWRITER] /urlifywritergenerate ERROR code=' . $res->get_error_code() . ' | msg=' . $res->get_error_message() );
		}
		wp_send_json_error(
			array(
				'code'    => $res->get_error_code(),
				'message' => $res->get_error_message(),
			),
			500
		);
	}

	wp_send_json_success(
		array(
			'title'           => (string) ( $res['title'] ?? '' ),
			'content'         => (string) ( $res['content'] ?? '' ),
			'tags'            => is_array( $res['tags'] ?? null ) ? $res['tags'] : array(),
			'excerpt'         => (string) ( $res['excerpt'] ?? '' ),
			'seo_title'       => (string) ( $res['seo_title'] ?? '' ),
			'seo_description' => (string) ( $res['seo_description'] ?? '' ),
			'category_id'     => isset( $res['category_id'] ) ? (int) $res['category_id'] : 0,
		)
	);
} );

/* ==========================================================
 * GENERATE IMAGES — pide URLs al Worker (no guarda)
 * ========================================================== */
add_action( 'wp_ajax_urlifywritergenerate_images', function () {
	if ( ! current_user_can( 'edit_posts' ) ) {
		wp_send_json_error(
			array( 'code' => 'forbidden', 'message' => __( 'You do not have permission.', 'urlifywriter' ) ),
			403
		);
	}
	check_ajax_referer( 'urlifywriterurl_flow', 'nonce' );

	if ( ! defined( 'URLIFYWRITER_IMG_API' ) || empty( URLIFYWRITER_IMG_API ) ) {
		wp_send_json_error(
			array( 'code' => 'no_endpoint', 'message' => __( 'Remote images endpoint is not configured.', 'urlifywriter' ) ),
			500
		);
	}
	if ( ! defined( 'URLIFYWRITER_LIC_HMAC' ) || empty( URLIFYWRITER_LIC_HMAC ) ) {
		wp_send_json_error(
			array( 'code' => 'no_hmac', 'message' => __( 'HMAC secret is not configured.', 'urlifywriter' ) ),
			500
		);
	}

	$mode = isset( $_POST['mode'] ) ? sanitize_text_field( wp_unslash( $_POST['mode'] ) ) : 'pixabay';
	$mode = in_array( $mode, array( 'pixabay', 'ai' ), true ) ? $mode : 'pixabay';

	$query = isset( $_POST['query'] ) ? wp_strip_all_tags( wp_unslash( $_POST['query'] ) ) : '';
	$query = trim( preg_replace( '/[#,;:|\/\\\\\[\]\(\)\{\}"\'`<>^~]+/u', ' ', $query ) );
	if ( '' === $query ) {
		$query = 'travel landscape';
	}

	$n = isset( $_POST['n'] ) ? max( 1, min( 10, absint( wp_unslash( $_POST['n'] ) ) ) ) : 1;

	$prefer_user_api = ! empty( $_POST['prefer_user_api'] );

	$user_px = (string) get_option( 'urlifywriterapi_pixabay_key', '' );
	$user_oa = (string) get_option( 'urlifywriterapi_openai_key', '' );

	/* ===== recoger secciones desde el POST (por keys) ===== */
	$sections = array();

	foreach ( array_keys( $_POST ) as $k ) {
		if ( preg_match( '/^sections\[\d+\]$/', (string) $k ) ) {

			$v = isset( $_POST[ $k ] )
				? sanitize_text_field( wp_unslash( $_POST[ $k ] ) )
				: '';

			if ( '' !== $v ) {
				$sections[] = $v;
			}
		}
	}

	// Lee y deslashea UNA vez. Si no existe, quedará ''.
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	$sections_json = isset( $_POST['sections_json'] ) ? wp_unslash( $_POST['sections_json'] ) : '';

	if ( empty( $sections ) && '' !== $sections_json ) {
		$decoded = json_decode( $sections_json, true );
		if ( JSON_ERROR_NONE === json_last_error() && is_array( $decoded ) ) {
			foreach ( $decoded as $s ) {
				if ( is_string( $s ) ) {
					$label = sanitize_text_field( $s );
					if ( '' !== $label ) {
						$sections[] = $label;
					}
				}
			}
		}
	}

	if ( count( $sections ) > 10 ) {
		$sections = array_slice( $sections, 0, 10 );
	}

	$URLIFYWRITER_DEBUG = defined( 'URLIFYWRITER_DEBUG' )
		? (bool) URLIFYWRITER_DEBUG
		: ( defined( 'WP_DEBUG' ) && WP_DEBUG );

	if ( $URLIFYWRITER_DEBUG ) {
		urlifywriterdebug_log(
			'[URLIFYWRITER] /urlifywritergenerate_images sections_count=' . count( $sections ) .
			' sample=' . wp_json_encode( array_slice( $sections, 0, 3 ) )
		);
	}

	/* ===== recoger STYLE si es IA (admite alias) ===== */
	$style = '';
	if ( 'ai' === $mode ) {
		$style_keys = array( 'style', 'ai_style', 'image_style', 'ai_style_hint_run' );
		foreach ( $style_keys as $sk ) {
			if ( $style ) {
				break;
			}
			if ( isset( $_POST[ $sk ] ) ) {
				$style = sanitize_text_field( wp_unslash( $_POST[ $sk ] ) );
			}
		}

		$style = trim( wp_strip_all_tags( $style ) );
		if ( '' === $style ) {
			$style = (string) get_option( 'urlifywriterai_style_hint', '' );
			$style = trim( wp_strip_all_tags( $style ) );
		}
		if ( '' !== $style ) {
			$style = mb_substr( $style, 0, 200 );
		}
	}

	$lic_opt = (string) get_option( 'urlifywriterlicense_key', '' );

	$home_parts  = wp_parse_url( home_url() );
	$site_domain = ( is_array( $home_parts ) && isset( $home_parts['host'] ) ) ? (string) $home_parts['host'] : '';

	$locale = get_locale();
	$lang2  = is_string( $locale ) ? substr( $locale, 0, 2 ) : 'en';
	$lang2  = $lang2 ? $lang2 : 'en';

	$payload = array(
		'ts'              => time() * 1000,
		'license_key'     => ( '' !== $lic_opt ? $lic_opt : '' ),
		'site_domain'     => $site_domain,
		'mode'            => $mode,
		'query'           => $query,
		'n'               => $n,
		'user_api'        => array(
			'pixabay' => $user_px ? $user_px : '',
			'openai'  => $user_oa ? $user_oa : '',
		),
		'prefer_user_api' => $prefer_user_api ? true : false,
		'lang'            => $lang2,
		'sections'        => $sections,
		'debug'           => ( defined( 'URLIFYWRITER_DEBUG' ) && URLIFYWRITER_DEBUG ) ? 1 : 0,
	);

	if ( 'ai' === $mode && '' !== $style ) {
		$payload['style'] = $style;
	}

	if ( $URLIFYWRITER_DEBUG ) {
		$log = $payload;
		if ( ! empty( $log['user_api']['pixabay'] ) ) { $log['user_api']['pixabay'] = '***' . substr( $log['user_api']['pixabay'], -4 ); }
		if ( ! empty( $log['user_api']['openai'] ) )  { $log['user_api']['openai']  = '***' . substr( $log['user_api']['openai'], -4 ); }
		urlifywriterdebug_log( '[URLIFYWRITER] /urlifywritergenerate_images payload=' . wp_json_encode( $log ) );
	}

	$res = urlifywritercloud_post( URLIFYWRITER_IMG_API, $payload );

	if ( is_wp_error( $res ) ) {
		$edata       = $res->get_error_data();
		$http_code   = isset( $edata['http_code'] ) ? (int) $edata['http_code'] : 0;
		$remote_code = isset( $edata['remote_code'] ) ? (string) $edata['remote_code'] : '';
		$remote_msg  = isset( $edata['remote_message'] ) ? (string) $edata['remote_message'] : '';
		$remote_prov = isset( $edata['provider'] ) ? (string) $edata['provider'] : '';

		if ( 400 === $http_code && 'content_policy_violation' === $remote_code ) {
			wp_send_json_error(
				array(
					'code'     => 'content_policy_violation',
					'provider' => $remote_prov ? $remote_prov : 'ai',
					'message'  => __( 'This request was blocked by the image provider\'s content policy.', 'urlifywriter' ),
					'reason'   => $remote_msg ? $remote_msg : '',
				),
				200
			);
		}

		if ( $URLIFYWRITER_DEBUG ) {
			urlifywriterdebug_log( '[URLIFYWRITER] /urlifywritergenerate_images ERROR ' . $res->get_error_code() . ': ' . $res->get_error_message() );
		}

		wp_send_json_error(
			array(
				'code'    => $res->get_error_code() ? $res->get_error_code() : 'remote_error',
				'message' => $res->get_error_message() ? $res->get_error_message() : __( 'Remote error', 'urlifywriter' ),
			),
			500
		);
	}

	if ( isset( $res['code'] ) && 'content_policy_violation' === $res['code'] ) {
		wp_send_json_error(
			array(
				'code'     => 'content_policy_violation',
				'provider' => (string) ( $res['provider'] ?? 'ai' ),
				'message'  => __( 'This request was blocked by the image provider\'s content policy.', 'urlifywriter' ),
				'reason'   => (string) ( $res['message'] ?? '' ),
			),
			200
		);
	}

	$urls       = (array) ( $res['urls'] ?? array() );
	$by_section = ( isset( $res['by_section'] ) && is_array( $res['by_section'] ) ) ? $res['by_section'] : array();
	$log_remote = ( isset( $res['log'] ) && ( is_array( $res['log'] ) || is_string( $res['log'] ) ) ) ? $res['log'] : null;

	if ( ! empty( $urls ) ) {
		wp_send_json_success(
			array(
				'provider'   => (string) ( $res['provider'] ?? $mode ),
				'urls'       => array_values( $urls ),
				'n_used'     => (int) ( $res['n_used'] ?? count( $urls ) ),
				'by_section' => $by_section,
				'log'        => $log_remote,
				'echo_style' => ( 'ai' === $mode ) ? $style : '',
			)
		);
	}

	$code = (string) ( $res['code'] ?? 'no_urls' );
	$msg  = (string) ( $res['error'] ?? __( 'No images returned', 'urlifywriter' ) );
	$http = (int) ( $res['http_status'] ?? 0 );

	if ( 'pixabay_no_results' === $code || 423 === $http || 'no_urls' === $code ) {
		wp_send_json_error(
			array(
				'code'       => $code,
				'message'    => $msg,
				'by_section' => $by_section,
				'log'        => $log_remote,
				'echo_style' => ( 'ai' === $mode ) ? $style : '',
			),
			200
		);
	}

	wp_send_json_error( array( 'code' => $code, 'message' => $msg ), 500 );
} );

/* ==========================================================
 * DRAFT: crea el borrador en WordPress (+ imágenes)
 * ========================================================== */
add_action( 'wp_ajax_urlifywritercreate_draft', function () {
	check_ajax_referer( 'urlifywriterurl_flow', 'nonce' );

	if ( ! ( current_user_can( 'edit_posts' ) || current_user_can( 'edit_pages' ) ) ) {
		wp_send_json_error( array( 'message' => __( 'You do not have permission.', 'urlifywriter' ) ) );
	}

	$URLIFYWRITER_DEBUG = defined( 'URLIFYWRITER_DEBUG' )
		? (bool) URLIFYWRITER_DEBUG
		: ( defined( 'WP_DEBUG' ) && WP_DEBUG );

	if ( $URLIFYWRITER_DEBUG ) {
		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		$keys = array_map( 'sanitize_key', array_keys( $_POST ) );
		urlifywriterdebug_log( '[URLIFYWRITER] /urlifywritercreate_draft POST keys: ' . implode( ', ', $keys ) );
	}

	/* ===================== Publicación / Programación ===================== */
	$publish_mode = isset( $_POST['publish_mode'] )
		? sanitize_text_field( wp_unslash( $_POST['publish_mode'] ) )
		: 'draft';

	$pub_allowed = array( 'draft', 'publish', 'schedule' );
	if ( ! in_array( $publish_mode, $pub_allowed, true ) ) {
		$publish_mode = 'draft';
	}

	$sched_freq = '';
	if ( isset( $_POST['schedule_freq'] ) ) {
		$sched_freq = sanitize_text_field( wp_unslash( $_POST['schedule_freq'] ) );
	} elseif ( isset( $_POST['schedule_frequency'] ) ) {
		$sched_freq = sanitize_text_field( wp_unslash( $_POST['schedule_frequency'] ) );
	}

	$sched_date  = isset( $_POST['schedule_start_date'] ) ? sanitize_text_field( wp_unslash( $_POST['schedule_start_date'] ) ) : '';
	$sched_time  = isset( $_POST['schedule_start_time'] ) ? sanitize_text_field( wp_unslash( $_POST['schedule_start_time'] ) ) : '';
	$sched_count = isset( $_POST['schedule_count'] ) ? absint( wp_unslash( $_POST['schedule_count'] ) ) : 1;
	$sched_hours = isset( $_POST['schedule_hours_between'] )
	? (float) sanitize_text_field( wp_unslash( $_POST['schedule_hours_between'] ) )
	: 0.0;


	$post_status = 'draft';
	$post_date   = current_time( 'mysql' ); // local WP

	if ( 'publish' === $publish_mode ) {
		$post_status = 'publish';
	} elseif ( 'schedule' === $publish_mode ) {
		$tz   = wp_timezone();
		$date = trim( (string) $sched_date );
		$time = trim( (string) $sched_time );
		$time = ( '' !== $time ) ? $time : '00:00';

		if ( 5 === strlen( $time ) ) { $time .= ':00'; }

		$dt = $date ? date_create_from_format( 'Y-m-d H:i:s', "$date $time", $tz ) : false;

		if ( $dt instanceof DateTimeInterface ) {
			$now = new DateTime( 'now', $tz );
			if ( $dt > $now ) {
				$post_status = 'future';
				$post_date   = $dt->format( 'Y-m-d H:i:s' );
			} else {
				$publish_mode = 'draft';
				if ( $URLIFYWRITER_DEBUG ) { urlifywriterdebug_log( '[URLIFYWRITER] schedule requested but datetime is past → keeping draft' ); }
			}
		} else {
			$publish_mode = 'draft';
			if ( $URLIFYWRITER_DEBUG ) { urlifywriterdebug_log( '[URLIFYWRITER] schedule requested but invalid datetime → keeping draft' ); }
		}
	}

	$post_type = isset( $_POST['post_type'] ) ? sanitize_key( wp_unslash( $_POST['post_type'] ) ) : 'post';
	if ( ! in_array( $post_type, array( 'post', 'page' ), true ) ) {
		$post_type = 'post';
	}
	if ( 'page' === $post_type && ! current_user_can( 'edit_pages' ) ) {
		wp_send_json_error( array( 'message' => __( 'You do not have permission to create pages.', 'urlifywriter' ) ) );
	}

	/* ========================= Datos del contenido ======================== */
	$url     = isset( $_POST['url'] ) ? esc_url_raw( wp_unslash( $_POST['url'] ) ) : '';
	$title   = isset( $_POST['title'] ) ? wp_strip_all_tags( wp_unslash( $_POST['title'] ) ) : '';
	$content = isset( $_POST['content'] ) ? wp_kses_post( wp_unslash( $_POST['content'] ) ) : '';
	$excerpt = isset( $_POST['excerpt'] ) ? wp_strip_all_tags( wp_unslash( $_POST['excerpt'] ) ) : '';
	$cat_id  = isset( $_POST['category_id'] ) ? absint( wp_unslash( $_POST['category_id'] ) ) : 0;

	$seo_t = isset( $_POST['seo_title'] ) ? wp_strip_all_tags( wp_unslash( $_POST['seo_title'] ) ) : '';
	$seo_d = isset( $_POST['seo_description'] ) ? wp_strip_all_tags( wp_unslash( $_POST['seo_description'] ) ) : '';

	// TAGS (array o CSV + tags[n]) — unslash + sanitize
	$tags = array();

	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	$raw_tags = isset( $_POST['tags'] ) ? wp_unslash( $_POST['tags'] ) : '';

	if ( '' !== $raw_tags ) {
		if ( is_array( $raw_tags ) ) {
			foreach ( $raw_tags as $t ) { $tags[] = sanitize_text_field( (string) $t ); }
		} elseif ( is_string( $raw_tags ) ) {
			$tags = array_map(
				'sanitize_text_field',
				array_map( 'trim', explode( ',', $raw_tags ) )
			);
		}
	}

	foreach ( array_keys( $_POST ) as $k ) {
		if ( preg_match( '/^tags\[\d+\]$/', (string) $k ) ) {
			$tags[] = isset( $_POST[ $k ] )
				? sanitize_text_field( wp_unslash( $_POST[ $k ] ) )
				: '';
		}
	}

	$tags = array_values( array_unique( array_filter( $tags ) ) );

	// Autor por defecto (0 => actual)
	$default_author = (int) get_option( 'urlifywriterarticle_author', 0 );
	$post_author_id = $default_author ? $default_author : get_current_user_id();

	$lang = (string) get_option(
		'urlifywriterarticle_language',
		function_exists( 'urlifywriterget_default_lang_from_wp' ) ? urlifywriterget_default_lang_from_wp() : 'en'
	);

	/* ============================ Crear post ============================== */
	$postarr = array(
		'post_title'   => $title ? $title : __( 'Untitled Draft', 'urlifywriter' ),
		'post_content' => $content ? $content : '',
		'post_excerpt' => $excerpt ? $excerpt : '',
		'post_status'  => $post_status,
		'post_author'  => $post_author_id,
		'post_date'    => $post_date,
		'post_type'    => $post_type,
	);

	$post_id = wp_insert_post( $postarr, true );
	if ( is_wp_error( $post_id ) ) {
		if ( $URLIFYWRITER_DEBUG ) {
			urlifywriterdebug_log( '[URLIFYWRITER] /urlifywritercreate_draft ERROR wp_insert_post: ' . $post_id->get_error_message() );
		}
		wp_send_json_error( array( 'message' => $post_id->get_error_message() ) );
	}

	if ( $URLIFYWRITER_DEBUG ) {
		urlifywriterdebug_log( '[URLIFYWRITER] /urlifywritercreate_draft created post_id=' . (int) $post_id . ' status=' . $post_status . ' date=' . $post_date );
	}

	update_post_meta( $post_id, '_urlifywriterlang', $lang );
	if ( $url ) {
		update_post_meta( $post_id, '_urlifywritersource_url', esc_url_raw( $url ) );
	}

	// Guardar metadatos informativos de programación
	update_post_meta( $post_id, '_urlifywriterpublish_mode', $publish_mode );
	if ( 'schedule' === $publish_mode ) {
		update_post_meta( $post_id, '_urlifywriterschedule_frequency', $sched_freq );
		update_post_meta( $post_id, '_urlifywriterschedule_start_date', $sched_date );
		update_post_meta( $post_id, '_urlifywriterschedule_start_time', $sched_time );
		update_post_meta( $post_id, '_urlifywriterschedule_count_per_interval', max( 1, (int) $sched_count ) );
		update_post_meta( $post_id, '_urlifywriterschedule_hours_between', (float) $sched_hours );
	}

	/* ======================== Categoría y Tags ============================ */
	$set_cat_id = 0;

	if ( 'post' === $post_type ) {
		if ( $cat_id ) {
			$exists = term_exists( $cat_id, 'category' );
			if ( $exists ) {
				wp_set_post_categories( $post_id, array( $cat_id ), false );
				$set_cat_id = $cat_id;
			}
		} else {
			$default_cat = (int) get_option( 'default_category' );
			if ( $default_cat ) {
				wp_set_post_categories( $post_id, array( $default_cat ), false );
				$set_cat_id = $default_cat;
			}
		}

		if ( ! empty( $tags ) ) {
			wp_set_post_tags( $post_id, $tags, false );
		}
	}

	/* ============================ SEO (Yoast) ============================= */
	$final_seo_title = $seo_t ? $seo_t : $title;
	if ( $final_seo_title ) {
		update_post_meta( $post_id, '_yoast_wpseo_title', $final_seo_title );
	}

	$final_seo_desc = $seo_d;
	if ( ! $final_seo_desc ) {
		$plain = $excerpt ? $excerpt : wp_strip_all_tags( $content );
		$plain = trim( preg_replace( '/\s+/', ' ', $plain ) );
		$final_seo_desc = mb_substr( $plain, 0, 155 );
		if ( mb_strlen( $plain ) > 155 ) {
			$final_seo_desc = preg_replace( '/\s+\S*$/u', '', mb_substr( $plain, 0, 155 ) ) . '…';
		}
	}
	update_post_meta( $post_id, '_yoast_wpseo_metadesc', $final_seo_desc );

	if ( $set_cat_id && 'post' === $post_type ) {
		update_post_meta( $post_id, '_yoast_wpseo_primary_category', $set_cat_id );
	}

	$focuskw = '';
	if ( ! empty( $tags ) ) { $focuskw = (string) reset( $tags ); }
	if ( ! $focuskw && $title ) { $focuskw = $title; }
	if ( $focuskw ) {
		update_post_meta( $post_id, '_yoast_wpseo_focuskw', sanitize_text_field( $focuskw ) );
	}

	/* =================== Imágenes =================== */
	$images_added    = 0;
	$ai_images_used  = 0;
	$featured_id     = 0;
	$attached_ids    = array();

	$images_enabled = isset( $_POST['images_enabled'] ) ? absint( wp_unslash( $_POST['images_enabled'] ) ) : 0;

	$image_source = isset( $_POST['image_source'] )
		? sanitize_text_field( wp_unslash( $_POST['image_source'] ) )
		: '';
	$image_source = in_array( $image_source, array( 'original', 'pixabay', 'ai' ), true ) ? $image_source : '';

	$image_original_ack = isset( $_POST['image_original_ack'] ) ? absint( wp_unslash( $_POST['image_original_ack'] ) ) : 0;

	$pixabay_n = isset( $_POST['pixabay_images_per_article'] ) ? max( 0, min( 10, absint( wp_unslash( $_POST['pixabay_images_per_article'] ) ) ) ) : 0;
	$ai_n      = isset( $_POST['ai_images_per_article'] ) ? max( 0, min( 10, absint( wp_unslash( $_POST['ai_images_per_article'] ) ) ) ) : 0;

	$image_insert = isset( $_POST['image_insert'] ) ? sanitize_text_field( wp_unslash( $_POST['image_insert'] ) ) : 'featured';
	if ( ! in_array( $image_insert, array( 'featured', 'inline', 'both' ), true ) ) { $image_insert = 'featured'; }
	if ( 'featured' === $image_insert ) { $image_insert = 'both'; } // tu comportamiento previo

	// Estilo IA opcional (alias)
	$ai_style = '';
	foreach ( array( 'style', 'ai_style', 'image_style', 'ai_style_hint_run' ) as $sk ) {
		if ( $ai_style ) { break; }
		if ( isset( $_POST[ $sk ] ) ) {
			$ai_style = sanitize_text_field( wp_unslash( $_POST[ $sk ] ) );
		}
	}
	$ai_style = trim( wp_strip_all_tags( $ai_style ) );
	if ( '' !== $ai_style ) { $ai_style = mb_substr( $ai_style, 0, 200 ); }

	// Recoger image_urls[] — unslash + esc_url_raw
	$image_urls = array();

	if ( isset( $_POST['image_urls'] ) ) {
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$raw_image_urls = wp_unslash( $_POST['image_urls'] );

		if ( is_array( $raw_image_urls ) ) {
			foreach ( $raw_image_urls as $u ) {
				$image_urls[] = esc_url_raw( (string) $u );
			}
		} elseif ( is_string( $raw_image_urls ) ) {
			$maybe_csv = array_map( 'trim', explode( ',', $raw_image_urls ) );
			foreach ( $maybe_csv as $u ) {
				if ( '' !== $u ) {
					$image_urls[] = esc_url_raw( $u );
				}
			}
		}
	}

	foreach ( array_keys( $_POST ) as $k ) {
		if ( preg_match( '/^image_urls\[\d+\]$/', (string) $k ) ) {

			$v = isset( $_POST[ $k ] )
				? sanitize_text_field( wp_unslash( $_POST[ $k ] ) )
				: '';

			$image_urls[] = esc_url_raw( $v );
		}
	}


	$image_urls = array_values( array_unique( array_filter( $image_urls ) ) );

	if ( $URLIFYWRITER_DEBUG ) {
		urlifywriterdebug_log(
			'[URLIFYWRITER] /urlifywritercreate_draft received image_urls count=' . count( $image_urls ) .
			' sample=' . wp_json_encode( array_slice( $image_urls, 0, 3 ) )
		);
	}

	$urlifywriterext_from_url = function( $u ) {
		if ( preg_match( '~\.(jpe?g|png|webp|gif)(?:\?.*)?$~i', (string) $u, $m ) ) {
			return strtolower( $m[1] );
		}
		return 'jpg';
	};

	$candidate_urls = array();
	$provider_is_ai = ( 'ai' === $image_source );

	if ( $images_enabled && $image_source ) {
		try {
			if ( 'original' === $image_source ) {
				if ( $image_original_ack && $url ) {
					$html = urlifywriterfetch_html( $url );
					if ( ! is_wp_error( $html ) ) {
						$img = urlifywriterextract_featured_image( $html );
						if ( $img ) { $candidate_urls[] = $img; }
					}
				}
				$provider_is_ai = false;
			} elseif ( 'pixabay' === $image_source ) {
				if ( ! empty( $image_urls ) ) {
					$candidate_urls = array_slice( $image_urls, 0, max( 1, $pixabay_n ) );
				}
				$provider_is_ai = false;
			} elseif ( 'ai' === $image_source ) {
				if ( ! empty( $image_urls ) ) {
					$candidate_urls = array_slice( $image_urls, 0, max( 1, $ai_n ) );
				}
				$provider_is_ai = true;
			}
		} catch ( Throwable $e ) {
			if ( $URLIFYWRITER_DEBUG ) { urlifywriterdebug_log( '[URLIFYWRITER] images discovery exception: ' . $e->getMessage() ); }
		}
	}

	// FEATURED
	if ( $images_enabled && in_array( $image_insert, array( 'featured', 'both' ), true ) && ! empty( $candidate_urls ) ) {
		$featured_url   = $candidate_urls[0];
		$ext            = $urlifywriterext_from_url( $featured_url );
		$featured_title = $title ? $title : ( $focuskw ? $focuskw : __( 'Image', 'urlifywriter' ) );
		$featured_slug  = urlifywriterslug( $featured_title );

		$att_id = urlifywritersideload_image_to_post(
			$featured_url,
			$post_id,
			$featured_title,
			array(
				'filename' => $featured_slug . '-featured.' . $ext,
				'title'    => $featured_title,
				'alt'      => $featured_title,
			)
		);

		if ( ! is_wp_error( $att_id ) ) {
			$attached_ids[] = (int) $att_id;
			set_post_thumbnail( $post_id, $att_id );
			$featured_id = (int) $att_id;
			$images_added++;
			if ( $provider_is_ai ) { $ai_images_used++; }
		}

		if ( 'both' === $image_insert ) {
			array_shift( $candidate_urls );
		}
	}

	// INLINE
	if ( $images_enabled && in_array( $image_insert, array( 'inline', 'both' ), true ) && ! empty( $candidate_urls ) ) {
		$content_cur = (string) get_post_field( 'post_content', $post_id );

		$parts = preg_split( '~(</p>)~i', $content_cur, -1, PREG_SPLIT_DELIM_CAPTURE );
		$paras = array();

		if ( $parts && count( $parts ) > 1 ) {
			for ( $i = 0; $i < count( $parts ); $i += 2 ) {
				$paras[] = $parts[ $i ] . ( $parts[ $i + 1 ] ?? '' );
			}
		} else {
			$paras = array( $content_cur );
		}

		$linear = $paras;

		$h_pattern = '~<h(2|3)[^>]*>(.*?)</h\1>~is';
		$matches   = array();
		preg_match_all( $h_pattern, $content_cur, $matches, PREG_OFFSET_CAPTURE );

		$next_h_ix  = 0;
		$next_h_pos = isset( $matches[0][ $next_h_ix ] ) ? (int) $matches[0][ $next_h_ix ][1] : PHP_INT_MAX;

		$section_titles_by_para_index = array();
		$running_section              = '';

		$para_positions = array();
		$offset         = 0;

		foreach ( $linear as $ix => $phtml ) {
			$pos_start = $offset;
			$offset   += strlen( (string) $phtml );
			$para_positions[ $ix ] = $pos_start;

			while ( $next_h_pos < $pos_start && isset( $matches[2][ $next_h_ix ] ) ) {
				$running_section = wp_strip_all_tags( (string) $matches[2][ $next_h_ix ][0] );
				$next_h_ix++;
				$next_h_pos = isset( $matches[0][ $next_h_ix ] ) ? (int) $matches[0][ $next_h_ix ][1] : PHP_INT_MAX;
			}
			$section_titles_by_para_index[ $ix ] = $running_section;
		}

		$numParas = count( $paras );
		$numImgs  = count( $candidate_urls );
		$step     = max( 2, (int) floor( $numParas / ( $numImgs + 1 ) ) );
		$nextPos  = $step;

		$mkFig = function( $id, $alt_text ) {
			$img = wp_get_attachment_image( $id, 'large', false, array( 'alt' => $alt_text ) );
			return $img ? '<figure class="wp-block-image size-large">' . $img . '</figure>' : '';
		};

		$new   = array();
		$imgIx = 0;

		for ( $i = 0; $i < $numParas; $i++ ) {
			$new[] = $paras[ $i ];

			if ( $imgIx < $numImgs && ( $i + 1 ) >= $nextPos ) {
				$section_base_title = trim( (string) ( $section_titles_by_para_index[ $i ] ?? '' ) );
				if ( '' === $section_base_title ) {
					$section_base_title = $title ? $title : ( $focuskw ? $focuskw : __( 'Image', 'urlifywriter' ) );
				}

				$focus_cap     = $focuskw ? ucfirst( mb_strtolower( $focuskw ) ) : '';
				$section_cap   = ucfirst( mb_strtolower( mb_substr( $section_base_title, 0, 1 ) ) ) . mb_substr( $section_base_title, 1 );
				$display_title = $focus_cap ? ( $focus_cap . ': ' . $section_cap ) : $section_cap;

				$sec_slug = urlifywriterslug( $display_title );
				$use_url  = $candidate_urls[ $imgIx ];
				$ext      = $urlifywriterext_from_url( $use_url );

				$att_id = urlifywritersideload_image_to_post(
					$use_url,
					$post_id,
					$display_title,
					array(
						'filename' => $sec_slug . '-img-' . ( $imgIx + 1 ) . '.' . $ext,
						'title'    => $display_title,
						'alt'      => $display_title,
					)
				);

				if ( ! is_wp_error( $att_id ) ) {
					$attached_ids[] = (int) $att_id;
					$images_added++;
					if ( $provider_is_ai ) { $ai_images_used++; }

					$new[] = "\n\n" . $mkFig( (int) $att_id, $display_title ) . "\n\n";
				}

				$imgIx++;
				$nextPos += $step;
			}
		}

		while ( $imgIx < $numImgs ) {
			$use_url = $candidate_urls[ $imgIx ];

			$section_base_title = $title ? $title : ( $focuskw ? $focuskw : __( 'Image', 'urlifywriter' ) );

			$focus_cap     = $focuskw ? ucfirst( mb_strtolower( $focuskw ) ) : '';
			$section_cap   = ucfirst( mb_strtolower( mb_substr( $section_base_title, 0, 1 ) ) ) . mb_substr( $section_base_title, 1 );
			$display_title = $focus_cap ? ( $focus_cap . ': ' . $section_cap ) : $section_cap;

			$sec_slug = urlifywriterslug( $display_title );
			$ext      = $urlifywriterext_from_url( $use_url );

			$att_id = urlifywritersideload_image_to_post(
				$use_url,
				$post_id,
				$display_title,
				array(
					'filename' => $sec_slug . '-img-' . ( $imgIx + 1 ) . '.' . $ext,
					'title'    => $display_title,
					'alt'      => $display_title,
				)
			);

			if ( ! is_wp_error( $att_id ) ) {
				$attached_ids[] = (int) $att_id;
				$images_added++;
				if ( $provider_is_ai ) { $ai_images_used++; }

				$new[] = "\n\n" . $mkFig( (int) $att_id, $display_title ) . "\n\n";
			}

			$imgIx++;
		}

		$new_content = implode( '', $new );
		wp_update_post( array( 'ID' => $post_id, 'post_content' => $new_content ) );
	}

	/* =================== Licencia y respuesta =================== */
	if ( $post_id && function_exists( 'urlifywriterreport_license_use' ) ) {
		try { urlifywriterreport_license_use( 1, (int) $ai_images_used ); } catch ( Throwable $e ) {}
	}

	$edit_link = get_edit_post_link( $post_id, '' );
	$view_link = get_permalink( $post_id );

	if ( $URLIFYWRITER_DEBUG ) {
		$thumb_now = (int) get_post_thumbnail_id( $post_id );
		urlifywriterdebug_log( '[URLIFYWRITER] images summary: added=' . (int) $images_added . ' featured=' . (int) $featured_id . ' thumb_now=' . $thumb_now );
	}

	wp_send_json_success(
		array(
			'edit'        => $edit_link ? $edit_link : '',
			'view'        => $view_link ? $view_link : '',
			'id'          => (int) $post_id,
			'status'      => $post_status,
			'post_date'   => $post_date,
			'publishMode' => $publish_mode,
			'images'      => array(
				'added'    => (int) $images_added,
				'featured' => (int) $featured_id,
				'ai_used'  => (int) $ai_images_used,
				'source'   => $image_source,
				'found'    => ( $featured_id || $images_added ) ? 1 : 0,
				'url'      => ! empty( $candidate_urls ) ? $candidate_urls[0] : '',
			),
		)
	);
} );


/* ==========================================================
 * TEST API: validadores simples para claves
 * ========================================================== */
add_action( 'wp_ajax_urlifywritertest_api', function () {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'ok' => false, 'error' => __( 'Unauthorized', 'urlifywriter' ) ), 403 );
	}

	check_admin_referer( 'urlifywritertest_api_nonce' );

	$opt = isset( $_POST['opt'] ) ? sanitize_text_field( wp_unslash( $_POST['opt'] ) ) : '';
	$key = isset( $_POST['key'] ) ? trim( wp_strip_all_tags( wp_unslash( $_POST['key'] ) ) ) : '';

	if ( '' === $key ) {
		wp_send_json_error( array( 'ok' => false, 'error' => __( 'Empty key', 'urlifywriter' ) ), 200 );
	}

	switch ( $opt ) {
		case 'urlifywriterapi_pixabay_key':
			if ( strlen( $key ) < 10 ) {
				wp_send_json_error( array( 'ok' => false, 'error' => __( 'Key too short', 'urlifywriter' ) ), 200 );
			}

			$resp = wp_remote_get(
				'https://pixabay.com/api/?key=' . rawurlencode( $key ) . '&q=test&per_page=3',
				array( 'timeout' => 12 )
			);

			if ( is_wp_error( $resp ) ) {
				wp_send_json_error( array( 'ok' => false, 'error' => $resp->get_error_message() ), 200 );
			}

			$code = (int) wp_remote_retrieve_response_code( $resp );
			$body = json_decode( (string) wp_remote_retrieve_body( $resp ), true );

			if ( 200 === $code && is_array( $body ) && isset( $body['totalHits'] ) ) {
				wp_send_json_success( array( 'ok' => true ), 200 );
			}

			wp_send_json_error( array( 'ok' => false, 'error' => __( 'Invalid or unauthorized key', 'urlifywriter' ) ), 200 );
			break;

		case 'urlifywriterapi_openai_key':
			$resp = wp_remote_get(
				'https://api.openai.com/v1/models',
				array(
					'timeout' => 12,
					'headers' => array( 'Authorization' => 'Bearer ' . $key ),
				)
			);

			if ( is_wp_error( $resp ) ) {
				wp_send_json_error( array( 'ok' => false, 'error' => $resp->get_error_message() ), 200 );
			}

			$code = (int) wp_remote_retrieve_response_code( $resp );
			if ( 200 === $code ) {
				wp_send_json_success( array( 'ok' => true ), 200 );
			}

			wp_send_json_error( array( 'ok' => false, 'error' => __( 'Unauthorized (check billing/permissions)', 'urlifywriter' ) ), 200 );
			break;

		default:
			wp_send_json_error( array( 'ok' => false, 'error' => __( 'Unknown option', 'urlifywriter' ) ), 400 );
	}
} );

/* ==========================================================
 * Helpers (APIs usuario y licencias)
 * ========================================================== */
if ( ! function_exists( 'urlifywriterget_user_pixabay_key' ) ) {
	function urlifywriterget_user_pixabay_key() {
		$k = get_option( 'urlifywriterapi_pixabay_key', '' );
		$k = is_string( $k ) ? trim( $k ) : '';
		return ( '' !== $k ) ? $k : '';
	}
}
if ( ! function_exists( 'urlifywriterget_user_openai_key' ) ) {
	function urlifywriterget_user_openai_key() {
		$k = get_option( 'urlifywriterapi_openai_key', '' );
		$k = is_string( $k ) ? trim( $k ) : '';
		return ( '' !== $k ) ? $k : '';
	}
}

/**
 * Reportar uso al servidor de licencias remoto.
 * Requiere URLIFYWRITER_LIC_BASE y URLIFYWRITER_LIC_HMAC.
 */
if ( ! function_exists( 'urlifywriterreport_license_use' ) ) {
	function urlifywriterreport_license_use( $add_articles, $add_images ) {
		if ( ! defined( 'URLIFYWRITER_LIC_HMAC' ) || empty( URLIFYWRITER_LIC_HMAC ) ) { return; }
		if ( ! defined( 'URLIFYWRITER_LIC_BASE' ) || empty( URLIFYWRITER_LIC_BASE ) ) { return; }

		$key = get_option( 'urlifywriterlicense_key', '' );
		if ( ! $key ) { return; }

		$home_parts = wp_parse_url( home_url() );
		$domain     = ( is_array( $home_parts ) && isset( $home_parts['host'] ) ) ? (string) $home_parts['host'] : '';

		$body = array(
			'license'      => $key,
			'domain'       => $domain,
			'add_articles' => max( 0, (int) $add_articles ),
			'add_images'   => max( 0, (int) $add_images ),
			'ts'           => time(),
		);

		$raw = wp_json_encode( $body );
		$sig = hash_hmac( 'sha256', $raw, URLIFYWRITER_LIC_HMAC );

		$url  = rtrim( URLIFYWRITER_LIC_BASE, '/' ) . '/wp-json/autopress/v1/license/use';
		$args = array(
			'method'  => 'POST',
			'timeout' => 15,
			'headers' => array(
				'Content-Type'             => 'application/json',
				'X-URLIFYWRITER-Signature' => $sig,
			),
			'body'    => $raw,
		);

		wp_remote_request( $url, $args );
	}
}

/* ==========================================================
 * Helpers para imágenes originales, URLs y sideload con filename/title/alt
 * ========================================================== */
if ( ! function_exists( 'urlifywritermake_absolute_url' ) ) {
	function urlifywritermake_absolute_url( $maybe, $base ) {
		if ( preg_match( '~^https?://~i', (string) $maybe ) ) { return $maybe; }

		$bp = wp_parse_url( $base );
		if ( ! $bp || empty( $bp['host'] ) ) { return $maybe; }

		$scheme = $bp['scheme'] ?? 'https';
		$host   = $bp['host'];
		$port   = isset( $bp['port'] ) ? ':' . $bp['port'] : '';
		$bpath  = $bp['path'] ?? '/';

		if ( 0 === strpos( (string) $maybe, '/' ) ) {
			$path = $maybe;
		} else {
			$dir  = preg_replace( '~/[^/]*$~', '/', $bpath ? $bpath : '/' );
			$path = $dir . $maybe;
		}

		$parts = array();
		foreach ( explode( '/', $path ) as $seg ) {
			if ( '' === $seg || '.' === $seg ) {
				$parts[] = ( '' === $seg ) ? '' : null;
				continue;
			}
			if ( '..' === $seg ) {
				if ( count( $parts ) > 1 ) { array_pop( $parts ); }
				continue;
			}
			$parts[] = $seg;
		}

		$canon = implode( '/', array_filter( $parts, static function( $v ) { return null !== $v; } ) );
		return $scheme . '://' . $host . $port . $canon;
	}
}

/**
 * Sideload con control de filename + alt + title.
 * $opts = ['filename' => 'mi-archivo.webp', 'title' => 'Mi título', 'alt' => 'Mi alt']
 */
if ( ! function_exists( 'urlifywritersideload_image_to_post' ) ) {
	function urlifywritersideload_image_to_post( $url, $post_id, $desc = '', $opts = array() ) {
		if ( ! function_exists( 'media_sideload_image' ) ) {
			require_once ABSPATH . 'wp-admin/includes/media.php';
			require_once ABSPATH . 'wp-admin/includes/file.php';
			require_once ABSPATH . 'wp-admin/includes/image.php';
		}

		$desired = isset( $opts['filename'] ) ? sanitize_file_name( $opts['filename'] ) : '';
		$filter  = null;

		if ( $desired ) {
			$filter = function( $file ) use ( $desired ) {
				$ext = '';
				if ( ! empty( $file['name'] ) && preg_match( '~\.(jpe?g|png|webp|gif)$~i', $file['name'], $m ) ) {
					$ext = strtolower( $m[0] ); // incluye el punto
				} elseif ( preg_match( '~\.(jpe?g|png|webp|gif)$~i', $desired, $m ) ) {
					$ext = strtolower( $m[0] );
				}
				$base        = preg_replace( '~\.(jpe?g|png|webp|gif)$~i', '', $desired );
				$file['name'] = sanitize_file_name( $base . ( $ext ? $ext : '.jpg' ) );
				return $file;
			};
			add_filter( 'wp_handle_sideload_prefilter', $filter, 10, 1 );
		}

		$att_id = media_sideload_image( $url, $post_id, $desc, 'id' );

		if ( $filter ) {
			remove_filter( 'wp_handle_sideload_prefilter', $filter, 10 );
		}

		if ( is_wp_error( $att_id ) ) { return $att_id; }

		$att_title = isset( $opts['title'] ) ? wp_strip_all_tags( $opts['title'] ) : $desc;
		if ( $att_title ) {
			wp_update_post(
				array(
					'ID'         => (int) $att_id,
					'post_title' => $att_title,
					'post_name'  => urlifywriterslug( $att_title ),
				)
			);
		}
		if ( ! empty( $opts['alt'] ) ) {
			update_post_meta( $att_id, '_wp_attachment_image_alt', wp_strip_all_tags( $opts['alt'] ) );
		}

		return $att_id;
	}
}

/* ==========================================================
 * Featured image extractor (kept as-is, but unchanged logic)
 * ========================================================== */
function urlifywriterextract_featured_image( $html ) {
	if ( ! is_string( $html ) || '' === $html ) { return ''; }

	$base_url = '';
	if ( preg_match( '~<base[^>]+href=["\']([^"\']+)~i', $html, $m ) ) {
		$base_url = trim( html_entity_decode( $m[1], ENT_QUOTES ) );
	}

	$abs = function( $u ) use ( $base_url ) {
		$u = trim( html_entity_decode( (string) $u, ENT_QUOTES ) );
		if ( '' === $u || 0 === stripos( $u, 'data:' ) ) { return ''; }
		if ( 0 === strpos( $u, '//' ) ) { return 'https:' . $u; }
		if ( ! preg_match( '~^https?://~i', $u ) ) {
			if ( $base_url && function_exists( 'urlifywritermake_absolute_url' ) ) {
				return urlifywritermake_absolute_url( $u, $base_url );
			}
			return '';
		}
		return $u;
	};

	// ✅ Ampliado: bloquear iconos/acciones sociales típicas (whatsapp, etc.)
	$badRe = '~(logo|icon|sprite|avatar|placeholder|thumb|banner|ad-|advert|promo|social|share|sharing|default|favicon|whatsapp|facebook|twitter|x-icon|linkedin|pinterest|telegram|reddit|mail|email)~i';

	$push  = function( $list, $url, $score, $w = 0, $h = 0 ) {
		$url = trim( (string) $url );
		if ( '' === $url || ! preg_match( '~\.(jpe?g|png|webp)(\?.*)?$~i', $url ) ) { return $list; }
		if ( isset( $list[ $url ] ) ) {
			$list[ $url ]['score'] = max( $list[ $url ]['score'], $score );
			return $list;
		}
		$list[ $url ] = array( 'score' => $score, 'w' => (int) $w, 'h' => (int) $h );
		return $list;
	};

	$candidates = array();

	if ( class_exists( 'DOMDocument' ) ) {
		$dom = new DOMDocument();
		libxml_use_internal_errors( true );
		@$dom->loadHTML( '<?xml encoding="utf-8" ?>' . $html );
		libxml_clear_errors();
		$xp = new DOMXPath( $dom );

		$roots = array();
		foreach ( array(
			'//article',
			'//*[@role="main"]',
			'//*[contains(@class,"entry-content") or contains(@class,"post-content") or contains(@class,"single-content") or contains(@id,"content") or contains(@class,"td-post-content") or contains(@class,"the-content") or contains(@class,"content__article-body")]',
		) as $q ) {
			foreach ( $xp->query( $q ) as $n ) { $roots[] = $n; }
		}
		if ( empty( $roots ) ) { $roots = array( $dom->documentElement ); }

		foreach ( $roots as $root ) {
			foreach ( $xp->query( './/img', $root ) as $img ) {
				$src = '';

				$srcset = $img->getAttribute( 'data-srcset' ) ?: $img->getAttribute( 'srcset' );
				if ( $srcset ) {
					$best  = '';
					$bestw = 0;
					foreach ( preg_split( '/\s*,\s*/', $srcset ) as $entry ) {
						if ( preg_match( '~\s+(\d+)w~', $entry, $m ) ) {
							$u = trim( str_replace( $m[0], '', $entry ) );
							$w = (int) $m[1];
							if ( $w > $bestw && $w <= 2400 ) { $bestw = $w; $best = $u; }
						}
					}
					if ( $best ) { $src = $best; }
				}

				if ( ! $src ) {
					foreach ( array( 'data-src', 'data-original', 'data-lazy-src', 'src' ) as $a ) {
						$v = $img->getAttribute( $a );
						if ( $v ) { $src = $v; break; }
					}
				}
				if ( ! $src ) { continue; }

				$src = $abs( $src );
				if ( ! $src ) { continue; }

				$cls = strtolower( $img->getAttribute( 'class' ) . ' ' . $img->getAttribute( 'id' ) );
				$alt = strtolower( $img->getAttribute( 'alt' ) );

				if ( preg_match( $badRe, $src ) || preg_match( $badRe, $cls ) || preg_match( '~\.gif(\?.*)?$~i', $src ) ) { continue; }

				// ✅ Ampliado: excluir más contenedores típicos de share plugins
				$okTree = true; $p = $img;
				while ( $p && $p->nodeType === XML_ELEMENT_NODE ) {
					$pcls = strtolower( $p->getAttribute( 'class' ) . ' ' . $p->getAttribute( 'id' ) );
					if ( preg_match( '~(sidebar|widget|related|nav|header|footer|logo|author|comments|breadcrumb|share|sharing|social|addtoany|sharedaddy|heateor|sassy|ssba|socials|follow|sticky-share)~', $pcls ) ) { $okTree = false; break; }
					if ( 'article' === $p->nodeName || preg_match( '~(entry-content|post-content|content__article-body)~', $pcls ) ) { break; }
					$p = $p->parentNode;
				}
				if ( ! $okTree ) { continue; }

				$w = (int) $img->getAttribute( 'width' );
				$h = (int) $img->getAttribute( 'height' );

				if ( ( ! $w || ! $h ) && preg_match( '~(\d{3,4})[xX](\d{3,4})~', $src, $m ) ) { $w = (int) $m[1]; $h = (int) $m[2]; }
				if ( ( ! $w || ! $h ) && preg_match( '~[?&]w=(\d{2,4}).*[?&]h=(\d{2,4})~', $src, $m ) ) { $w = (int) $m[1]; $h = (int) $m[2]; }

				// ✅ NUEVO: descartar iconos pequeños aunque estén dentro del contenido
				if ( $w && $h && ( $w <= 200 && $h <= 200 ) ) { continue; }

				if ( $w && $h ) {
					if ( $w < 320 || $h < 180 ) { continue; }
					$ratio = $w / max( 1, $h );
					if ( $ratio < 0.5 || $ratio > 2.3 ) { continue; }
				}

				$score = 50;
				if ( $w && $h ) { $score += min( 50, (int) ( ( ( $w * $h ) / ( 900 * 500 ) ) * 10 ) ); }
				if ( $alt && strlen( $alt ) > 4 ) { $score += 5; }
				if ( preg_match( '~/uploads/|/wp-content/|/images/|/media/~', $src ) ) { $score += 8; }

				$candidates = $push( $candidates, $src, $score, $w, $h );
			}
		}
	}

	if ( empty( $candidates ) && preg_match_all( '~<img[^>]+src=["\']([^"\']+)["\'][^>]*>~i', $html, $m ) ) {
		foreach ( $m[1] as $u ) {
			$u2 = $abs( $u );
			if ( ! $u2 ) { continue; }
			if ( preg_match( $badRe, $u2 ) || preg_match( '~\.gif(\?.*)?$~i', $u2 ) ) { continue; }
			$candidates = $push( $candidates, $u2, 30 );
		}
	}

	if ( empty( $candidates ) ) {
		if ( preg_match( '/<meta[^>]+property=["\']og:image(["\'][^>]*content=["\']([^"\']+)|[^>]*content=["\']([^"\']+)["\'][^>]*property=["\']og:image)["\']/i', $html, $m ) ) {
			$u = $abs( ! empty( $m[2] ) ? $m[2] : ( ! empty( $m[3] ) ? $m[3] : '' ) );
			if ( $u && ! preg_match( $badRe, $u ) && preg_match( '~\.(jpe?g|png|webp)(\?.*)?$~i', $u ) ) {
				$candidates = $push( $candidates, $u, 25 );
			}
		}
		if ( empty( $candidates ) && preg_match( '/<meta[^>]+name=["\']twitter:image(["\'][^>]*content=["\']([^"\']+)|[^>]*content=["\']([^"\']+)["\'][^>]*name=["\']twitter:image)["\']/i', $html, $m ) ) {
			$u = $abs( ! empty( $m[2] ) ? $m[2] : ( ! empty( $m[3] ) ? $m[3] : '' ) );
			if ( $u && ! preg_match( $badRe, $u ) && preg_match( '~\.(jpe?g|png|webp)(\?.*)?$~i', $u ) ) {
				$candidates = $push( $candidates, $u, 22 );
			}
		}
	}

	if ( empty( $candidates ) ) { return ''; }

	$items = $candidates;
	$keys  = array_keys( $items );
	$topn  = min( 6, count( $keys ) );

	for ( $i = 0; $i < $topn; $i++ ) {
		$k = $keys[ $i ];
		if ( empty( $items[ $k ]['w'] ) && function_exists( 'getimagesize' ) && @ini_get( 'allow_url_fopen' ) ) {
			$sz = @getimagesize( $k );
			if ( is_array( $sz ) && ! empty( $sz[0] ) && ! empty( $sz[1] ) ) {
				$items[ $k ]['w'] = (int) $sz[0];
				$items[ $k ]['h'] = (int) $sz[1];
				if ( $sz[0] >= 600 && $sz[1] >= 400 ) { $items[ $k ]['score'] += 12; }
			}
		}
	}

	uksort( $items, function( $a, $b ) use ( $items ) {
		return ( $items[ $b ]['score'] <=> $items[ $a ]['score'] );
	} );

	$best = array_key_first( $items );
	return $best ? esc_url_raw( $best ) : '';
}
