<?php
// admin/page-settings.php
if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * ===== Language helpers =====
 */
function urlifywriterlanguages_list() {
	return array(
		'ar'      => __( 'Arabic', 'urlifywriter' ),
		'bn'      => __( 'Bengali', 'urlifywriter' ),
		'zh'      => __( 'Chinese (Simplified)', 'urlifywriter' ),
		'zh-Hant' => __( 'Chinese (Traditional)', 'urlifywriter' ),
		'cs'      => __( 'Czech', 'urlifywriter' ),
		'da'      => __( 'Danish', 'urlifywriter' ),
		'nl'      => __( 'Dutch', 'urlifywriter' ),
		'en'      => __( 'English', 'urlifywriter' ),
		'fi'      => __( 'Finnish', 'urlifywriter' ),
		'fr'      => __( 'French', 'urlifywriter' ),
		'de'      => __( 'German', 'urlifywriter' ),
		'el'      => __( 'Greek', 'urlifywriter' ),
		'he'      => __( 'Hebrew', 'urlifywriter' ),
		'hi'      => __( 'Hindi', 'urlifywriter' ),
		'hu'      => __( 'Hungarian', 'urlifywriter' ),
		'id'      => __( 'Indonesian', 'urlifywriter' ),
		'it'      => __( 'Italian', 'urlifywriter' ),
		'ja'      => __( 'Japanese', 'urlifywriter' ),
		'ko'      => __( 'Korean', 'urlifywriter' ),
		'ms'      => __( 'Malay', 'urlifywriter' ),
		'mr'      => __( 'Marathi', 'urlifywriter' ),
		'no'      => __( 'Norwegian', 'urlifywriter' ),
		'fa'      => __( 'Persian', 'urlifywriter' ),
		'pl'      => __( 'Polish', 'urlifywriter' ),
		'pt'      => __( 'Portuguese', 'urlifywriter' ),
		'ru'      => __( 'Russian', 'urlifywriter' ),
		'es'      => __( 'Spanish', 'urlifywriter' ),
		'sv'      => __( 'Swedish', 'urlifywriter' ),
		'tl'      => __( 'Tagalog', 'urlifywriter' ),
		'ta'      => __( 'Tamil', 'urlifywriter' ),
		'te'      => __( 'Telugu', 'urlifywriter' ),
		'th'      => __( 'Thai', 'urlifywriter' ),
		'tr'      => __( 'Turkish', 'urlifywriter' ),
		'ur'      => __( 'Urdu', 'urlifywriter' ),
		'vi'      => __( 'Vietnamese', 'urlifywriter' ),
	);
}

/**
 * Map WP locale → language code
 */
function urlifywriterget_default_lang_from_wp() {
	$locale = get_locale();
	$locale = is_string( $locale ) ? $locale : 'en_US';
	$short  = strtolower( substr( $locale, 0, 2 ) );

	$map_specific = array(
		'zh_cn' => 'zh',
		'zh_sg' => 'zh',
		'zh_tw' => 'zh-Hant',
		'zh_hk' => 'zh-Hant',
		'he_il' => 'he',
		'iw_il' => 'he',
		'nb_no' => 'no',
		'pt_br' => 'pt',
		'pt_pt' => 'pt',
	);

	$key = strtolower( str_replace( '-', '_', $locale ) );
	if ( isset( $map_specific[ $key ] ) ) {
		return $map_specific[ $key ];
	}

	$langs = urlifywriterlanguages_list();
	if ( array_key_exists( $short, $langs ) ) {
		return $short;
	}

	return 'en';
}

/**
 * ===== Connections & License helpers =====
 */
function urlifywriterhas_pixabay_user_key() {
	$k = (string) get_option( 'urlifywriterapi_pixabay_key', '' );
	return $k !== '';
}

function urlifywriterhas_ai_user_key() {
	$openai = (string) get_option( 'urlifywriterapi_openai_key', '' );
	return ( $openai !== '');
}

/**
 * Returns: ['included'=>bool, 'max_per_article'=>int, 'monthly_cap'=>int]
 */
function urlifywriterlicense_ai_limits() {
	$state   = get_option( 'urlifywriterlicense_state', 'unknown' );
	$details = get_option( 'urlifywriterlicense_details', array() );

	$included = ( $state === 'valid' );
	$max_per  = 0;
	$monthly  = 0;

	if ( $included && is_array( $details ) ) {
		$caps = ( isset( $details['caps'] ) && is_array( $details['caps'] ) ) ? $details['caps'] : array();
		if ( isset( $caps['ai_images_per_article'] ) && is_numeric( $caps['ai_images_per_article'] ) ) {
			$max_per = max( 0, (int) $caps['ai_images_per_article'] );
		}
		if ( isset( $caps['articles_per_month'] ) && is_numeric( $caps['articles_per_month'] ) ) {
			$monthly = max( 0, (int) $caps['articles_per_month'] );
		}
	}

	return array(
		'included'       => $included,
		'max_per_article'=> $max_per,
		'monthly_cap'    => $monthly,
	);
}

/**
 * Helper: detect if THIS settings form is being submitted and nonce is valid.
 * We do NOT copy/loop whole $_POST. We just validate the nonce once.
 */
function urlifywriter_is_settings_post_valid() {

	$request_method = isset( $_SERVER['REQUEST_METHOD'] )
		? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_METHOD'] ) )
		: '';

	if ( strtoupper( $request_method ) !== 'POST' ) {
		return false;
	}

	// options.php submit will include our custom nonce field.
	$nonce = isset( $_POST['urlifywritersettings_nonce'] )
		? sanitize_text_field( wp_unslash( $_POST['urlifywritersettings_nonce'] ) )
		: '';

	if ( ! $nonce ) {
		return false;
	}

	return (bool) wp_verify_nonce( $nonce, 'urlifywritersettings_save' );
}

/**
 * ===== Settings page registration =====
 */
add_action( 'admin_init', function () {

	/**
	 * ---------- Section: General ----------
	 */
	add_settings_section(
		'urlifywritersettings_section',
		__( 'General', 'urlifywriter' ),
		function () {
			echo '<p class="description">' .
				esc_html__( 'Configure extraction and target length defaults used across the plugin.', 'urlifywriter' ) .
			'</p>';
		},
		'urlifywriter-settings'
	);

	// Sanitizers & bounds
	$sanitize_extract_max = function ( $val ) {
		$val = absint( $val );
		if ( $val < 200 )  { $val = 200; }
		if ( $val > 5000 ) { $val = 5000; }
		return $val;
	};

	$sanitize_min = function ( $val ) {
		$val = absint( $val );
		if ( $val < 100 )  { $val = 100; }
		if ( $val > 3000 ) { $val = 3000; }
		$max = (int) get_option( 'urlifywritertarget_max_words', 1200 );
		if ( $val > $max ) { $val = $max; }
		return $val;
	};

	$sanitize_max = function ( $val ) {
		$val = absint( $val );
		if ( $val < 200 )  { $val = 200; }
		if ( $val > 3000 ) { $val = 3000; }
		$min = (int) get_option( 'urlifywritertarget_min_words', 800 );
		if ( $val < $min ) { $val = $min; }
		return $val;
	};

	// 1) Max words to extract (pre-AI)
	register_setting( 'urlifywriter-settings', 'urlifywritermax_words', array(
		'type'              => 'integer',
		'sanitize_callback' => $sanitize_extract_max,
		'default'           => 1200,
	) );

	add_settings_field(
		'urlifywritermax_words',
		__( 'Maximum words to extract from the source URL', 'urlifywriter' ),
		function () {
			$val = (int) get_option( 'urlifywritermax_words', 1200 );
			?>
			<div class="urlifywriter-range">
				<input type="range" id="urlifywritermax_words" name="urlifywritermax_words" min="200" max="5000" step="50" value="<?php echo esc_attr( $val ); ?>">
				<div class="urlifywriter-range-meta">
					<span>200</span>
					<strong><span id="urlifywritermax_words_val"><?php echo esc_html( $val ); ?></span></strong>
					<span>5000</span>
				</div>
				<p class="description"><?php echo esc_html__( 'Extracted text will be truncated before any AI processing.', 'urlifywriter' ); ?></p>
			</div>
			<?php
		},
		'urlifywriter-settings',
		'urlifywritersettings_section'
	);

	// 2) Target min words (AI)
	register_setting( 'urlifywriter-settings', 'urlifywritertarget_min_words', array(
		'type'              => 'integer',
		'sanitize_callback' => $sanitize_min,
		'default'           => 800,
	) );

	add_settings_field(
		'urlifywritertarget_min_words',
		__( 'Target min words (AI output)', 'urlifywriter' ),
		function () {
			$val = (int) get_option( 'urlifywritertarget_min_words', 800 );
			?>
			<div class="urlifywriter-range">
				<input type="range" id="urlifywritertarget_min_words" name="urlifywritertarget_min_words" min="100" max="3000" step="50" value="<?php echo esc_attr( $val ); ?>">
				<div class="urlifywriter-range-meta">
					<span>100</span>
					<strong><span id="urlifywritertarget_min_words_val"><?php echo esc_html( $val ); ?></span></strong>
					<span>3000</span>
				</div>
				<p class="description"><?php echo esc_html__( 'Minimum target length for generated articles. Capped at 3000.', 'urlifywriter' ); ?></p>
			</div>
			<?php
		},
		'urlifywriter-settings',
		'urlifywritersettings_section'
	);

	// 3) Target max words (AI)
	register_setting( 'urlifywriter-settings', 'urlifywritertarget_max_words', array(
		'type'              => 'integer',
		'sanitize_callback' => $sanitize_max,
		'default'           => 1200,
	) );

	add_settings_field(
		'urlifywritertarget_max_words',
		__( 'Target max words (AI output)', 'urlifywriter' ),
		function () {
			$val = (int) get_option( 'urlifywritertarget_max_words', 1200 );
			?>
			<div class="urlifywriter-range">
				<input type="range" id="urlifywritertarget_max_words" name="urlifywritertarget_max_words" min="200" max="3000" step="50" value="<?php echo esc_attr( $val ); ?>">
				<div class="urlifywriter-range-meta">
					<span>200</span>
					<strong><span id="urlifywritertarget_max_words_val"><?php echo esc_html( $val ); ?></span></strong>
					<span>3000</span>
				</div>
				<p class="description"><?php echo esc_html__( 'Maximum target length for generated articles. Hard limit: 3000.', 'urlifywriter' ); ?></p>
			</div>
			<?php
		},
		'urlifywriter-settings',
		'urlifywritersettings_section'
	);

	/**
	 * ---------- Section: Article Defaults ----------
	 */
	add_settings_section(
		'urlifywriterarticle_defaults_section',
		__( 'Article Defaults', 'urlifywriter' ),
		function () {
			echo '<p class="description">' .
				esc_html__( 'Choose the default language and author that will be applied when creating articles.', 'urlifywriter' ) .
			'</p>';
		},
		'urlifywriter-settings'
	);

	$sanitize_lang = function ( $val ) {
		$val  = is_string( $val ) ? sanitize_text_field( $val ) : '';
		$list = urlifywriterlanguages_list();
		$val  = $val ?: urlifywriterget_default_lang_from_wp();
		return array_key_exists( $val, $list ) ? $val : urlifywriterget_default_lang_from_wp();
	};

	register_setting( 'urlifywriter-settings', 'urlifywriterarticle_language', array(
		'type'              => 'string',
		'sanitize_callback' => $sanitize_lang,
		'default'           => urlifywriterget_default_lang_from_wp(),
	) );

	add_settings_field(
		'urlifywriterarticle_language',
		__( 'Default Article Language', 'urlifywriter' ),
		function () {
			$value = get_option( 'urlifywriterarticle_language', urlifywriterget_default_lang_from_wp() );
			$langs = urlifywriterlanguages_list();
			?>
			<select name="urlifywriterarticle_language" id="urlifywriterarticle_language">
				<?php foreach ( $langs as $code => $label ) : ?>
					<option value="<?php echo esc_attr( $code ); ?>" <?php selected( $value, $code ); ?>>
						<?php echo esc_html( $label ); ?>
					</option>
				<?php endforeach; ?>
			</select>
			<p class="description"><?php echo esc_html__( 'Used when prompting AI and/or setting content language meta.', 'urlifywriter' ); ?></p>
			<?php
		},
		'urlifywriter-settings',
		'urlifywriterarticle_defaults_section'
	);

	$sanitize_author = function ( $val ) {
		$val = absint( $val );
		if ( $val === 0 ) { return 0; }
		$user = get_user_by( 'ID', $val );
		return $user ? $val : 0;
	};

	register_setting( 'urlifywriter-settings', 'urlifywriterarticle_author', array(
		'type'              => 'integer',
		'sanitize_callback' => $sanitize_author,
		'default'           => 0,
	) );

	add_settings_field(
		'urlifywriterarticle_author',
		__( 'Default Article Author', 'urlifywriter' ),
		function () {
			$value = (int) get_option( 'urlifywriterarticle_author', 0 );
			$users = get_users( array(
				'role__in' => array( 'administrator', 'editor', 'author' ),
				'orderby'  => 'display_name',
				'order'    => 'ASC',
				'fields'   => array( 'ID', 'display_name' ),
			) );
			?>
			<select name="urlifywriterarticle_author" id="urlifywriterarticle_author">
				<option value="0" <?php selected( $value, 0 ); ?>>
					<?php echo esc_html__( '— Keep post creator as author —', 'urlifywriter' ); ?>
				</option>
				<?php foreach ( $users as $u ) : ?>
					<option value="<?php echo esc_attr( $u->ID ); ?>" <?php selected( $value, $u->ID ); ?>>
						<?php echo esc_html( $u->display_name . ' (ID ' . $u->ID . ')' ); ?>
					</option>
				<?php endforeach; ?>
			</select>
			<p class="description"><?php echo esc_html__( 'If set, generated posts will be assigned to this user ID.', 'urlifywriter' ); ?></p>
			<?php
		},
		'urlifywriter-settings',
		'urlifywriterarticle_defaults_section'
	);

	/**
	 * ---------- Section: Images ----------
	 */
	add_settings_section(
		'urlifywriterimages_section',
		__( 'Images', 'urlifywriter' ),
		function () {
			echo '<p class="description">' .
				esc_html__( 'Automatically add images to generated articles. Configure the source below.', 'urlifywriter' ) .
			'</p>';
		},
		'urlifywriter-settings'
	);

	$sanitize_bool = function( $v ) {
		return $v ? 1 : 0;
	};

	// IMPORTANT: We ONLY peek minimal needed POST keys if nonce is valid.
	$is_valid_post = urlifywriter_is_settings_post_valid();

	$post_images_enabled = null;
	$post_source         = null;
	$post_ack            = null;

	if ( $is_valid_post ) {
		$post_images_enabled = ! empty( $_POST['urlifywriterimages_enabled'] );
		$post_source = isset( $_POST['urlifywriterimage_source'] )
			? sanitize_text_field( wp_unslash( $_POST['urlifywriterimage_source'] ) )
			: null;
		$post_ack = ! empty( $_POST['urlifywriterimage_original_ack'] );
	}

	$sanitize_source = function( $v ) use ( $is_valid_post, $post_images_enabled, $post_source, $post_ack ) {

		$v = in_array( $v, array( 'original', 'pixabay', 'ai' ), true ) ? $v : 'pixabay';

		if ( ! $is_valid_post ) {
			return $v;
		}

		// If images disabled in the submitted form, no need to enforce ACK.
		if ( ! $post_images_enabled ) {
			return $v;
		}

		// Use the submitted radio if available, otherwise fallback to $v
		$selected = is_string( $post_source ) ? $post_source : $v;

		if ( $selected === 'original' ) {
			if ( ! $post_ack ) {
				add_settings_error(
					'urlifywriter-settings',
					'urlifywriteroriginal_needs_ack',
					__( 'Using “From post URL (original image)” requires acknowledging the copyright disclaimer.', 'urlifywriter' ),
					'error'
				);
				$prev = get_option( 'urlifywriterimage_source', 'pixabay' );
				return ( $prev && $prev !== 'original' ) ? $prev : 'pixabay';
			}
		}

		return $v;
	};

	$sanitize_count_pixabay = function( $val ) use ( $is_valid_post, $post_images_enabled, $post_source ) {

		$val = max( 1, min( 5, absint( $val ) ) );

		if ( ! $is_valid_post ) {
			return $val;
		}

		$enabled  = (bool) $post_images_enabled;
		$selected = is_string( $post_source ) ? $post_source : get_option( 'urlifywriterimage_source', 'pixabay' );

		if ( ! ( $enabled && $selected === 'pixabay' ) ) {
			return $val;
		}

		if ( ! urlifywriterhas_pixabay_user_key() && $val > 1 ) {
			add_settings_error(
				'urlifywriter-settings',
				'urlifywriterpixabay_forced_one',
				__( 'More than 1 image per article with Pixabay requires your own API key. Limited to 1.', 'urlifywriter' ),
				'notice-warning'
			);
			return 1;
		}

		return $val;
	};

	$sanitize_count_ai = function( $val ) use ( $is_valid_post, $post_images_enabled, $post_source ) {

		$val = max( 0, min( 4, absint( $val ) ) );

		if ( ! $is_valid_post ) {
			return $val;
		}

		$enabled  = (bool) $post_images_enabled;
		$selected = is_string( $post_source ) ? $post_source : get_option( 'urlifywriterimage_source', 'pixabay' );

		if ( ! ( $enabled && $selected === 'ai' ) ) {
			return $val;
		}

		$lic = urlifywriterlicense_ai_limits();

		if ( $lic['included'] ) {
			$max = (int) $lic['max_per_article'];
			return min( $val ? $val : 1, $max );
		}

		if ( ! urlifywriterhas_ai_user_key() ) {
			add_settings_error(
				'urlifywriter-settings',
				'urlifywriterai_needs_api',
				__( 'Your license does not include AI images. Add your OpenAI API key or upgrade your plan.', 'urlifywriter' ),
				'error'
			);
			return 0;
		}

		return max( 1, min( 4, $val ? $val : 1 ) );
	};

	$sanitize_insert = function( $v ) {
		$v = is_string( $v ) ? strtolower( trim( $v ) ) : 'both';
		return in_array( $v, array( 'featured', 'inline', 'both' ), true ) ? $v : 'both';
	};

	$sanitize_ai_style = function( $v ) use ( $is_valid_post, $post_images_enabled, $post_source ) {

		if ( ! $is_valid_post ) {
			return is_string( $v ) ? sanitize_text_field( $v ) : '';
		}

		$selected = is_string( $post_source ) ? $post_source : get_option( 'urlifywriterimage_source', 'pixabay' );
		$enabled  = (bool) $post_images_enabled;

		if ( ! $enabled || $selected !== 'ai' ) {
			return '';
		}

		$v = is_string( $v ) ? wp_strip_all_tags( $v, true ) : '';
		$v = trim( preg_replace( '/\s+/', ' ', $v ) );

		if ( strlen( $v ) > 200 ) {
			$v = substr( $v, 0, 200 );
		}
		return $v;
	};

	// Register options
	register_setting( 'urlifywriter-settings', 'urlifywriterimages_enabled', array(
		'type'              => 'boolean',
		'sanitize_callback' => $sanitize_bool,
		'default'           => 0,
	) );

	register_setting( 'urlifywriter-settings', 'urlifywriterimage_source', array(
		'type'              => 'string',
		'sanitize_callback' => $sanitize_source,
		'default'           => 'pixabay',
	) );

	register_setting( 'urlifywriter-settings', 'urlifywriterimage_original_ack', array(
		'type'              => 'boolean',
		'sanitize_callback' => $sanitize_bool,
		'default'           => 0,
	) );

	register_setting( 'urlifywriter-settings', 'urlifywriterpixabay_images_per_article', array(
		'type'              => 'integer',
		'sanitize_callback' => $sanitize_count_pixabay,
		'default'           => 1,
	) );

	register_setting( 'urlifywriter-settings', 'urlifywriterai_images_per_article', array(
		'type'              => 'integer',
		'sanitize_callback' => $sanitize_count_ai,
		'default'           => 0,
	) );

	register_setting( 'urlifywriter-settings', 'urlifywriterimage_insert', array(
		'type'              => 'string',
		'sanitize_callback' => $sanitize_insert,
		'default'           => 'both',
	) );

	register_setting( 'urlifywriter-settings', 'urlifywriterai_style_hint', array(
		'type'              => 'string',
		'sanitize_callback' => $sanitize_ai_style,
		'default'           => '',
	) );

	// Field: Auto-add images
	add_settings_field(
		'urlifywriterimages_enabled',
		__( 'Auto-add images', 'urlifywriter' ),
		function () {
			$val = (int) get_option( 'urlifywriterimages_enabled', 0 );
			?>
			<label>
				<input type="checkbox" name="urlifywriterimages_enabled" id="urlifywriterimages_enabled" value="1" <?php checked( $val, 1 ); ?>>
				<?php echo esc_html__( 'Add images automatically to generated posts', 'urlifywriter' ); ?>
			</label>
			<p class="description"><?php echo esc_html__( 'If disabled, no images will be added.', 'urlifywriter' ); ?></p>
			<?php
		},
		'urlifywriter-settings',
		'urlifywriterimages_section'
	);

	// Field: Image source + suboptions
	add_settings_field(
		'urlifywriterimage_source',
		__( 'Image source', 'urlifywriter' ),
		function () {
			$enabled = (int) get_option( 'urlifywriterimages_enabled', 0 );
			$src     = (string) get_option( 'urlifywriterimage_source', 'pixabay' );

			$pixabay_has_key = urlifywriterhas_pixabay_user_key();
			$ai_has_key      = urlifywriterhas_ai_user_key();
			$lic             = urlifywriterlicense_ai_limits();

			$is_disabled_group = ! $enabled;
			?>
			<fieldset id="urlifywriterimage_source_group"<?php if ( $is_disabled_group ) { echo ' aria-disabled="true"'; } ?>>

				<label class="urlifywriter-radio-row">
					<input type="radio" name="urlifywriterimage_source" value="original" <?php checked( $src, 'original' ); ?> <?php disabled( $is_disabled_group ); ?>>
					<strong><?php echo esc_html__( 'From post URL (original image)', 'urlifywriter' ); ?></strong>
					<em class="urlifywriter-risk"><?php echo esc_html__( 'Use at your own risk (copyright).', 'urlifywriter' ); ?></em>
				</label>

				<div class="urlifywriter-subrow" data-for="original" style="<?php echo esc_attr( $src === 'original' ? '' : 'display:none;' ); ?>">
					<label>
						<input type="checkbox" name="urlifywriterimage_original_ack" value="1"
							<?php checked( (int) get_option( 'urlifywriterimage_original_ack', 0 ), 1 ); ?>
							<?php disabled( $is_disabled_group ); ?>>
						<?php echo esc_html__( 'I understand this may infringe copyrights and accept responsibility.', 'urlifywriter' ); ?>
					</label>
				</div>

				<label class="urlifywriter-radio-row">
					<input type="radio" name="urlifywriterimage_source" value="pixabay" <?php checked( $src, 'pixabay' ); ?> <?php disabled( $is_disabled_group ); ?>>
					<strong><?php echo esc_html__( 'Pixabay', 'urlifywriter' ); ?></strong>
					<span class="urlifywriter-muted">
						<?php
						echo $pixabay_has_key
							? esc_html__( 'Your API key is set: up to 5 images/article', 'urlifywriter' )
							: esc_html__( 'No user API: limited to 1 image/article (you can add your API in Connections)', 'urlifywriter' );
						?>
					</span>
				</label>

				<div class="urlifywriter-subrow" data-for="pixabay" style="<?php echo esc_attr( $src === 'pixabay' ? '' : 'display:none;' ); ?>">
					<label>
						<?php
						$pix_n = (int) get_option( 'urlifywriterpixabay_images_per_article', 1 );
						$max   = $pixabay_has_key ? 5 : 1;
						?>
						<?php echo esc_html__( 'Images per article:', 'urlifywriter' ); ?>
						<input type="number" min="1" max="<?php echo esc_attr( $max ); ?>" step="1"
							name="urlifywriterpixabay_images_per_article"
							value="<?php echo esc_attr( $pix_n ); ?>"
							<?php disabled( $is_disabled_group ); ?>>
						<span class="description urlifywriter-inline-desc">
							<?php
							echo $pixabay_has_key
								? esc_html__( 'Up to 5 with your Pixabay API key.', 'urlifywriter' )
								: esc_html__( 'Add your Pixabay API key (free) to allow more than 1.', 'urlifywriter' );
							?>
						</span>
					</label>
				</div>

				<label class="urlifywriter-radio-row">
					<input type="radio" name="urlifywriterimage_source" value="ai" <?php checked( $src, 'ai' ); ?> <?php disabled( $is_disabled_group ); ?>>
					<strong><?php echo esc_html__( 'AI (OpenAI)', 'urlifywriter' ); ?></strong>
					<span class="urlifywriter-muted">
						<?php
						if ( $lic['included'] ) {
							printf(
								/* translators: %d is the maximum number of AI images included per article by the license. */
								esc_html__( 'License includes AI images: up to %d per article', 'urlifywriter' ),
								(int) $lic['max_per_article']
							);
						} elseif ( $ai_has_key ) {
							echo esc_html__( 'User API present. You can generate AI images (1–4 per article).', 'urlifywriter' );
						} else {
							echo esc_html__( 'License does not include AI. Add your API in Connections or upgrade your plan.', 'urlifywriter' );
						}
						?>
					</span>
				</label>

				<div class="urlifywriter-subrow" data-for="ai" style="<?php echo esc_attr( $src === 'ai' ? '' : 'display:none;' ); ?>">
					<?php
					$ai_n   = (int) get_option( 'urlifywriterai_images_per_article', 0 );
					$ai_max = $lic['included'] ? max( 1, (int) $lic['max_per_article'] ) : ( $ai_has_key ? 4 : 0 );
					$ai_min = ( $ai_max === 0 ) ? 0 : 1;
					$ai_val = ( $ai_max === 0 ) ? 0 : max( 1, min( $ai_max, $ai_n ? $ai_n : 1 ) );
					$ai_disabled_combined = ( $ai_max === 0 ) || $is_disabled_group;
					?>
					<label>
						<?php echo esc_html__( 'Images per article:', 'urlifywriter' ); ?>
						<input type="number"
							min="<?php echo esc_attr( $ai_min ); ?>"
							max="<?php echo esc_attr( $ai_max ); ?>"
							step="1"
							name="urlifywriterai_images_per_article"
							value="<?php echo esc_attr( $ai_val ); ?>"
							<?php disabled( $ai_disabled_combined ); ?>>
					</label>

					<?php if ( $ai_max === 0 ) : ?>
						<p class="description urlifywriter-error">
							<?php echo esc_html__( 'Not available: add your OpenAI API in Connections or upgrade your license.', 'urlifywriter' ); ?>
						</p>
					<?php endif; ?>

					<div class="urlifywriter-stylehint">
						<label for="urlifywriterai_style_hint" class="urlifywriter-stylehint-label">
							<?php echo esc_html__( 'AI Style hint (optional)', 'urlifywriter' ); ?>
						</label>
						<input type="text"
							id="urlifywriterai_style_hint"
							name="urlifywriterai_style_hint"
							value="<?php echo esc_attr( get_option( 'urlifywriterai_style_hint', '' ) ); ?>"
							<?php disabled( $ai_disabled_combined ); ?>
							placeholder="<?php echo esc_attr__( 'e.g. cinematic, moody, 50mm, shallow depth of field', 'urlifywriter' ); ?>">
						<p class="description">
							<?php echo esc_html__( 'Short style guidance appended to the AI prompt. Keep it concise.', 'urlifywriter' ); ?>
						</p>
					</div>
				</div>

			</fieldset>
			<?php
		},
		'urlifywriter-settings',
		'urlifywriterimages_section'
	);

	// Default image placement
	add_settings_field(
		'urlifywriterimage_insert',
		__( 'Default image placement', 'urlifywriter' ),
		function () {
			$val = (string) get_option( 'urlifywriterimage_insert', 'both' );
			?>
			<select name="urlifywriterimage_insert" id="urlifywriterimage_insert">
				<option value="featured" <?php selected( $val, 'featured' ); ?>><?php echo esc_html__( 'Featured only', 'urlifywriter' ); ?></option>
				<option value="inline" <?php selected( $val, 'inline' ); ?>><?php echo esc_html__( 'Inline only', 'urlifywriter' ); ?></option>
				<option value="both" <?php selected( $val, 'both' ); ?>><?php echo esc_html__( 'Featured + Inline', 'urlifywriter' ); ?></option>
			</select>
			<p class="description"><?php echo esc_html__( 'Default placement for images added to generated posts.', 'urlifywriter' ); ?></p>
			<?php
		},
		'urlifywriter-settings',
		'urlifywriterimages_section'
	);

} );

/**
 * ===== Render settings page =====
 */
function urlifywriterrender_settings_page() {

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'urlifywriter' ) );
	}
	?>
	<div class="wrap urlifywriter-wrap">
		<div class="urlifywriter-hero">
			<div>
				<h1 class="urlifywriter-title"><?php echo esc_html__( 'UrlifyWriter — Settings', 'urlifywriter' ); ?></h1>
				<p class="urlifywriter-subtitle">
					<?php echo esc_html__( 'Configure how UrlifyWriter extracts source text, target length, and image behavior.', 'urlifywriter' ); ?>
				</p>
			</div>
			<span class="urlifywriter-badge">UrlifyWriter</span>
		</div>

		<div class="urlifywriter-card" style="max-width:760px;">
			<form method="post" action="options.php" class="urlifywriter-form">
				<?php
					settings_fields( 'urlifywriter-settings' );
					wp_nonce_field( 'urlifywritersettings_save', 'urlifywritersettings_nonce' );
					settings_errors( 'urlifywriter-settings' );
					do_settings_sections( 'urlifywriter-settings' );
					submit_button( __( 'Save Changes', 'urlifywriter' ) );
				?>
			</form>
		</div>
	</div>
	<?php
}
