<?php
// admin/page-license.php (PLUGIN CLIENTE)
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/** Option keys */
if ( ! defined( 'URLIFYWRITER_LICENSE_KEY_OPT' ) )     define( 'URLIFYWRITER_LICENSE_KEY_OPT',     'urlifywriterlicense_key' );
if ( ! defined( 'URLIFYWRITER_LICENSE_STATE_OPT' ) )   define( 'URLIFYWRITER_LICENSE_STATE_OPT',   'urlifywriterlicense_state' );   // valid | invalid | unknown
if ( ! defined( 'URLIFYWRITER_LICENSE_DETAILS_OPT' ) ) define( 'URLIFYWRITER_LICENSE_DETAILS_OPT', 'urlifywriterlicense_details' ); // array con plan/caps/usage

// Inicializa estado/detalles si no existen
add_action( 'admin_init', function () {
	if ( get_option( URLIFYWRITER_LICENSE_STATE_OPT, null ) === null ) {
		update_option( URLIFYWRITER_LICENSE_STATE_OPT, 'unknown', false );
	}
	if ( get_option( URLIFYWRITER_LICENSE_DETAILS_OPT, null ) === null ) {
		update_option( URLIFYWRITER_LICENSE_DETAILS_OPT, array(), false );
	}
} );

/** AJAX: Validar licencia vía Cloudflare Worker (y guardar la key introducida) */
add_action( 'wp_ajax_urlifywritervalidate_license', function () {

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'msg' => 'forbidden' ), 403 );
	}

	check_ajax_referer( 'urlifywritervalidate_license', 'nonce' );

	$key = isset( $_POST['key'] ) ? sanitize_text_field( wp_unslash( $_POST['key'] ) ) : '';

	if ( $key === '' ) {
		update_option( URLIFYWRITER_LICENSE_KEY_OPT, '', false );
		update_option( URLIFYWRITER_LICENSE_STATE_OPT, 'unknown', false );
		update_option( URLIFYWRITER_LICENSE_DETAILS_OPT, array(), false );

		wp_send_json_success(
			array(
				'state'   => 'unknown',
				'label'   => 'Unknown',
				'details' => null,
				'error'   => 'missing_key',
			)
		);
	}

	update_option( URLIFYWRITER_LICENSE_KEY_OPT, $key, false );

	$payload = array(
		'key'            => $key,
		'domain'         => (string) wp_parse_url( home_url(), PHP_URL_HOST ),
		'site_url'       => (string) home_url( '/' ),
		'plugin'         => 'urlifywriter',
		'plugin_version' => defined( 'URLIFYWRITER_VERSION' ) ? URLIFYWRITER_VERSION : '1.0.0',
		'wp_version'     => get_bloginfo( 'version' ),
		'php_version'    => PHP_VERSION,
		'ts'             => time(),
		'nonce'          => wp_generate_uuid4(),
	);

	$endpoint    = defined( 'URLIFYWRITER_LIC_API' )  ? URLIFYWRITER_LIC_API  : '';
	$hmac_secret = defined( 'URLIFYWRITER_LIC_HMAC' ) ? URLIFYWRITER_LIC_HMAC : '';

	if ( $endpoint === '' || $hmac_secret === '' ) {
		wp_send_json_success(
			array(
				'state'   => 'unknown',
				'label'   => 'Unknown',
				'details' => null,
				'error'   => 'no_endpoint_or_hmac',
			)
		);
	}

	$body      = wp_json_encode( $payload );
	$signature = hash_hmac( 'sha256', $body, $hmac_secret );

	$args = array(
		'timeout' => 15,
		'headers' => array(
			'Content-Type' => 'application/json',
			'X-Signature'  => $signature,
		),
		'body' => $body,
	);

	$response = wp_remote_post( $endpoint, $args );

	if ( is_wp_error( $response ) ) {
		update_option( 'urlifywriterlicense_last_error', $response->get_error_message(), false );
		update_option( URLIFYWRITER_LICENSE_STATE_OPT, 'unknown', false );
		update_option( URLIFYWRITER_LICENSE_DETAILS_OPT, array(), false );
		wp_send_json_success( array( 'state' => 'unknown', 'label' => 'Unknown', 'details' => null ) );
	}

	$code = (int) wp_remote_retrieve_response_code( $response );
	$raw  = (string) wp_remote_retrieve_body( $response );
	$json = json_decode( $raw, true );

	if ( $code === 200 && is_array( $json ) && ! empty( $json['ok'] ) ) {

		$state = ( ( $json['state'] ?? '' ) === 'valid' ) ? 'valid' : 'invalid';


		$details = array(
			'plan'       => $json['plan']       ?? null,
			'issued_at'  => $json['issued_at']  ?? null,
			'expires_at' => $json['expires_at'] ?? null,
			'caps'       => is_array( $json['caps'] ?? null ) ? $json['caps'] : array(),
			'usage'      => is_array( $json['usage'] ?? null ) ? $json['usage'] : array(),
			'notes'      => $json['notes']      ?? null,
			'meta'       => is_array( $json['meta'] ?? null ) ? $json['meta'] : array(),
		);

		update_option( URLIFYWRITER_LICENSE_STATE_OPT, $state, false );
		if ( $state === 'valid' ) {
			update_option( URLIFYWRITER_LICENSE_DETAILS_OPT, $details, false );
		} else {
			update_option( URLIFYWRITER_LICENSE_DETAILS_OPT, array(), false );
		}

		update_option( 'urlifywriterlicense_last_checked', time(), false );
		delete_option( 'urlifywriterlicense_last_error' );
		delete_option( 'urlifywriterlicense_last_error_http' );

		wp_send_json_success(
			array(
				'state'   => $state,
				'label'   => ucfirst( $state ),
				'details' => ( $state === 'valid' ) ? $details : null,
			)
		);
	}

	$msg = '';
	if ( is_array( $json ) && ! empty( $json['error'] ) ) {
		$msg = (string) $json['error'];
	}

	update_option( 'urlifywriterlicense_last_error_http', sprintf( 'HTTP %d %s', $code, $msg ), false );
	update_option( URLIFYWRITER_LICENSE_STATE_OPT, 'unknown', false );
	update_option( URLIFYWRITER_LICENSE_DETAILS_OPT, array(), false );

	wp_send_json_success( array( 'state' => 'unknown', 'label' => 'Unknown', 'details' => null ) );
} );

/** Render page (hero + card) */
function urlifywriterrender_license_page() {

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'urlifywriter' ) );
	}

	$key      = get_option( URLIFYWRITER_LICENSE_KEY_OPT, '' );
	$state    = get_option( URLIFYWRITER_LICENSE_STATE_OPT, 'unknown' );
	$details  = get_option( URLIFYWRITER_LICENSE_DETAILS_OPT, array() );
	$last_err = get_option( 'urlifywriterlicense_last_error' ) ?: get_option( 'urlifywriterlicense_last_error_http' );

	$get_i = function( $arr, $k, $default = null ) {
		return isset( $arr[ $k ] ) ? (int) $arr[ $k ] : $default;
	};

	$caps  = $details['caps']  ?? array();
	$usage = $details['usage'] ?? array();

	$issued_at_raw  = $details['issued_at']  ?? null;
	$expires_at_raw = $details['expires_at'] ?? null;

	$issued_at  = $issued_at_raw  ? date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( (string) $issued_at_raw ) ) : null;
	$expires_at = $expires_at_raw ? date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( (string) $expires_at_raw ) ) : null;

	$articles_per_month  = $get_i( $caps, 'articles_per_month', null );
	$articles_used_month = $get_i( $usage, 'articles_used_month', null );
	if ( ( $articles_per_month === null || $articles_used_month === null ) && ! empty( $details['notes'] ) ) {
		if ( preg_match( '/month_used=(\d+)\/(\d+)/', (string) $details['notes'], $m ) ) {
			$articles_used_month = (int) $m[1];
			$articles_per_month  = (int) $m[2];
		}
	}

	$ai_images_per_article = $get_i( $caps, 'ai_images_per_article', null );

	$pricing_url = defined( 'URLIFYWRITER_PRICING_URL' ) ? URLIFYWRITER_PRICING_URL : 'https://urlifywriter.com/pricing/';
	$terms_url   = defined( 'URLIFYWRITER_TERMS_URL' )   ? URLIFYWRITER_TERMS_URL   : 'https://urlifywriter.com/terms/';
	?>
	<div class="wrap urlifywriter-wrap">
		<div class="urlifywriter-hero">
			<div>
				<h1 class="urlifywriter-title"><?php echo esc_html__( 'UrlifyWriter — License', 'urlifywriter' ); ?></h1>
				<p class="urlifywriter-subtitle">
					<?php echo esc_html__( 'Enter your license key and click Validate. The key will be saved and verified.', 'urlifywriter' ); ?>
				</p>
			</div>
			<span class="urlifywriter-badge">UrlifyWriter</span>
		</div>

		<div class="urlifywriter-card" style="max-width:760px;">
			<div class="urlifywriter-form">
				<label for="urlifywriterlicense_key" style="display:block;margin-bottom:6px;font-weight:600;">
					<?php echo esc_html__( 'License Key', 'urlifywriter' ); ?>
				</label>

				<input type="text"
					id="urlifywriterlicense_key"
					value="<?php echo esc_attr( $key ); ?>"
					class="regular-text"
					autocomplete="off"
					style="width:420px;max-width:100%;"
				/>

				<button type="button" class="button button-secondary" id="urlifywriter-validate-btn" style="margin-left:8px;">
					<?php echo esc_html__( 'Validate', 'urlifywriter' ); ?>
				</button>

				<span id="urlifywriter-license-state" class="urlifywriter-status-badge <?php echo esc_attr( $state ); ?>" style="margin-left:8px;">
					<?php echo esc_html( ucfirst( $state ) ); ?>
				</span>

				<?php if ( $last_err ) : ?>
					<p class="description" style="margin-top:6px;color:#a00;">
						<?php echo esc_html__( 'Last error:', 'urlifywriter' ) . ' ' . esc_html( (string) $last_err ); ?>
					</p>
				<?php endif; ?>
			</div>
		</div>

		<div class="urlifywriter-card" style="max-width:760px;margin-top:16px;">
			<h2 style="margin-top:0;"><?php echo esc_html__( 'License Details', 'urlifywriter' ); ?></h2>
			<table class="widefat striped" style="margin-top:8px;">
				<tbody id="urlifywriter-license-details">

				<?php if ( ! empty( $details ) ) : ?>

					<?php if ( $issued_at ) : ?>
						<tr><td style="width:220px;"><strong><?php echo esc_html__( 'Issued At', 'urlifywriter' ); ?></strong></td><td><?php echo esc_html( $issued_at ); ?></td></tr>
					<?php endif; ?>

					<?php if ( $expires_at ) : ?>
						<tr><td><strong><?php echo esc_html__( 'Valid Until', 'urlifywriter' ); ?></strong></td><td><?php echo esc_html( $expires_at ); ?></td></tr>
					<?php endif; ?>

					<?php if ( $articles_per_month !== null || $articles_used_month !== null ) : ?>
						<tr>
							<td><strong><?php echo esc_html__( 'Articles per month — used', 'urlifywriter' ); ?></strong></td>
							<td>
								<?php
								$tot = ( $articles_per_month !== null ) ? number_format_i18n( $articles_per_month ) : '–';
								$usd = ( $articles_used_month !== null ) ? number_format_i18n( $articles_used_month ) : '–';
								echo esc_html( "$tot — used: $usd" );
								?>
							</td>
						</tr>
					<?php endif; ?>

					<?php if ( $ai_images_per_article !== null ) : ?>
						<tr>
							<td><strong><?php echo esc_html__( 'AI images per article', 'urlifywriter' ); ?></strong></td>
							<td><?php echo esc_html( number_format_i18n( $ai_images_per_article ) ); ?></td>
						</tr>
					<?php endif; ?>

				<?php else : ?>

					<tr>
						<td colspan="2">
							<p class="description" style="margin:8px 0;">
								<?php echo esc_html__(
									'No valid license found. A valid license key is required to connect to the UrlifyWriter external service.',
									'urlifywriter'
								); ?>
							</p>
							<p style="margin:8px 0;">
								<a href="<?php echo esc_url( $pricing_url ); ?>" target="_blank" rel="noopener noreferrer" class="button button-primary">
									<?php echo esc_html__( 'Get a license', 'urlifywriter' ); ?>
								</a>
								<a href="<?php echo esc_url( $terms_url ); ?>" target="_blank" rel="noopener noreferrer" class="button">
									<?php echo esc_html__( 'Terms & Conditions', 'urlifywriter' ); ?>
								</a>
							</p>
						</td>
					</tr>

				<?php endif; ?>

				</tbody>
			</table>
<?php
// CTA: Comprar / mejorar plan (aparece siempre, pero cambia el texto según estado)
$cta_title = '';
$cta_desc  = '';

if ( $state === 'valid' ) {
	$cta_title = __( 'Need more articles?', 'urlifywriter' );
	$cta_desc  = __( 'Upgrade your plan.', 'urlifywriter' );
} elseif ( $state === 'invalid' ) {
	$cta_title = __( 'This license is not valid for this site', 'urlifywriter' );
	$cta_desc  = __( 'You can upgrade to a paid plan or purchase another license key.', 'urlifywriter' );
} else { // unknown (o cualquier otro)
	$cta_title = __( 'No license yet?', 'urlifywriter' );
	$cta_desc  = __( 'Start with the Free plan: 10 articles per month. You can upgrade anytime.', 'urlifywriter' );
}
?>

<div class="urlifywriter-upgrade-cta" style="margin-top:12px;padding:12px;border:1px solid #e5e7eb;border-radius:10px;background:#fff;">
	<div style="display:flex;gap:12px;align-items:flex-start;justify-content:space-between;flex-wrap:wrap;">
		<div style="min-width:240px;">
			<div style="font-weight:700;margin-bottom:4px;">
				<?php echo esc_html( $cta_title ); ?>
			</div>
			<div class="description" style="margin:0;color:#475569;">
				<?php echo esc_html( $cta_desc ); ?>
			</div>
		</div>

		<div style="display:flex;gap:8px;align-items:center;">
			<a href="<?php echo esc_url( $pricing_url ); ?>" target="_blank" rel="noopener noreferrer" class="button button-primary">
				<?php echo esc_html__( 'View plans', 'urlifywriter' ); ?>
			</a>
		</div>
	</div>
</div>
		</div>
	</div>
	<?php
}
