<?php

/**
 * Plugin Name: URL Export Addon for TranslatePress
 * Description: A WooCommerce addon for TranslatePress that exports product URLs in multiple languages into a CSV file for SEO and marketplace submissions.
 * Author: ABN Junction
 * Author URI: https://www.abnjunction.com/
 * Version: 1.0.1
 * Text Domain: url-export-addon-for-translatepress
 * License: GPL-2.0+
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */


if (! defined('ABSPATH')) exit; // Exit if accessed directly

register_activation_hook(__FILE__, 'abn_translatepress_plugin_activation');
function abn_translatepress_plugin_activation()
{
    // Code to run during plugin activation, like database table creation or initial settings
}

// Register Deactivation Hook
register_deactivation_hook(__FILE__, 'abn_translatepress_plugin_deactivation');

function abn_translatepress_plugin_deactivation()
{
    // Code to run during plugin deactivation, like cleaning up settings or database
}

// Hook for adding admin menus
add_action('admin_menu', 'abn_translate_press_menu');
function abn_translate_press_menu()
{
    // Add a new top-level menu
    add_menu_page(
        __('Custom Translate Press', 'url-export-addon-for-translatepress'),
        __('Translate Press URL Translation', 'url-export-addon-for-translatepress'),
        'manage_options',
        'custom-translate-press',
        'abn_translate_press_page',
        'dashicons-translation',
        6
    );
}
function abn_translate_press_admin_scripts($hook)
{
    if ($hook !== 'toplevel_page_custom-translate-press') {
        return;
    }

    // Enqueue styles and scripts securely
    // Load Bootstrap CSS
    wp_enqueue_style(
        'abn-bootstrap-css',
        plugin_dir_url(__FILE__) . 'css/bootstrap.min.css',
        array(),
        '5.3.0',
        'all'
    );

    // Load Popper JS
    wp_enqueue_script(
        'abn-bootstrap-popper',
        plugin_dir_url(__FILE__) . 'js/popper.min.js',
        array(),
        '2.11.6', // Use actual version if known
        true
    );

    // Load Bootstrap JS
    wp_enqueue_script(
        'abn-bootstrap-js',
        plugin_dir_url(__FILE__) . 'js/bootstrap.min.js',
        array('jquery', 'abn-bootstrap-popper'), // Add Popper as dependency
        '5.3.0',
        true
    );

    // Load your plugin’s admin script
    wp_enqueue_script(
        'abn-admin-script',
        plugin_dir_url(__FILE__) . 'js/admin-script.js',
        array('jquery'),
        '1.0.0', // Replace with your script version
        true
    );

    // Pass Ajax URL and User ID to JS
    wp_localize_script('abn-admin-script', 'custom_admin_data', array(
        'ajax_url' => esc_url(admin_url('admin-ajax.php')),
        'user_id'  => get_current_user_id(),
    ));
}
add_action('admin_enqueue_scripts', 'abn_translate_press_admin_scripts');

function abn_translate_press_page()
{
?>
    <div class="wrap form_translation_data">
        <h1><?php esc_html_e('Custom Translate Press Admin', 'url-export-addon-for-translatepress'); ?></h1>

        <ul class="nav nav-pills mb-3" id="pills-tab" role="tablist">
            <li class="nav-item" role="presentation">
                <button class="nav-link active" id="general-setting-tab" data-bs-toggle="pill"
                    data-bs-target="#general-setting" type="button" role="tab" aria-controls="general-setting"
                    aria-selected="true">
                    <?php esc_html_e('General Setting', 'url-export-addon-for-translatepress'); ?>
                </button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="export-translate-tab" data-bs-toggle="pill"
                    data-bs-target="#export-translate" type="button" role="tab" aria-controls="export-translate"
                    aria-selected="false">
                    <?php esc_html_e('Export', 'url-export-addon-for-translatepress'); ?>
                </button>
            </li>
        </ul>

        <div class="tab-content" id="pills-tabContent">
            <div class="tab-pane fade show active" id="general-setting" role="tabpanel"
                aria-labelledby="general-setting-tab" tabindex="0">
                <div class="basic_form_translate">
                    <style>
                        .loader {
                            border: 16px solid #f3f3f3;
                            border-top: 16px solid #3498db;
                            border-radius: 50%;
                            width: 120px;
                            height: 120px;
                            animation: spin 2s linear infinite;
                        }

                        @keyframes spin {
                            0% {
                                transform: rotate(0deg);
                            }

                            100% {
                                transform: rotate(360deg);
                            }
                        }

                        p.success_message {
                            color: green;
                        }
                    </style>

                    <?php
                    $total_products = wp_count_posts('product');
                    $total_products_count = 0;
                    foreach ($total_products as $status => $count) {
                        $total_products_count += absint($count); // Ensure it's an integer
                    }

                    $posts_per_page = 50;
                    $total_pages = ceil($total_products_count / $posts_per_page);

                    $trp = TRP_Translate_Press::get_trp_instance();
                    $trp_settings = $trp->get_component('settings');
                    $settings = $trp_settings->get_settings();
                    $published_languages = $settings['publish-languages'];
                    ?>

                    <label for="selected_country_lang"><?php esc_html_e('Select Language', 'url-export-addon-for-translatepress'); ?></label>
                    <select name="selected_country" id="selected_country_lang" class="form-control">
                        <option value=""><?php esc_html_e('Select Language', 'url-export-addon-for-translatepress'); ?></option>
                        <?php
                        foreach ($published_languages as $language) {
                            echo "<option value='" . esc_attr($language) . "'>" . esc_html($language) . "</option>";
                        }
                        ?>
                    </select>

                    <div class="product_update_options my-2">
                        <input type="radio" name="update_prod_url" value="update_all" id="update_all" checked>
                        <label for="update_all"><?php esc_html_e('Update All', 'url-export-addon-for-translatepress'); ?></label>
                        &nbsp;&nbsp;
                        <input type="radio" name="update_prod_url" value="update_existing" id="update_existing">
                        <label for="update_existing"><?php esc_html_e('Update Existing', 'url-export-addon-for-translatepress'); ?></label>
                    </div>

                    <input type="hidden" class="total_page" value="<?php echo esc_attr($total_pages); ?>">
                    <button id="start_prod_translate" class="btn btn-success">
                        <?php esc_html_e('Start Translate Products', 'url-export-addon-for-translatepress'); ?>
                    </button>
                </div>
            </div>

            <div class="tab-pane fade" id="export-translate" role="tabpanel" aria-labelledby="export-translate-tab" tabindex="0">
                <label for="select_post_type"><?php esc_html_e('Select Product Status', 'url-export-addon-for-translatepress'); ?></label>
                <select name="" class="form-control mb-2" id="select_post_type">
                    <option value=""><?php esc_html_e('All', 'url-export-addon-for-translatepress'); ?></option>
                </select>

                <?php $current_datetime = current_datetime()->format('Ymd_His'); ?>

                <input type="hidden" name="export_file_name" id="export_file_name"
                    value="<?php echo esc_attr($current_datetime); ?>">
                <input type="hidden" name="export_file_name" id="total_pages"
                    value="<?php echo esc_attr($total_pages); ?>">

                <input type="button" class="btn btn-success" value="<?php esc_attr_e('Export products', 'url-export-addon-for-translatepress'); ?>"
                    id="export_products_csv">
                <a href="#" class="btn" id="view_file_csv" style="display:none;">
                    <?php esc_html_e('View File', 'url-export-addon-for-translatepress'); ?>
                </a>
            </div>
        </div>
    </div>
<?php
}

add_action('wp_ajax_start_translating_products', 'abn_translating_products_callback');
add_action('wp_ajax_nopriv_start_translating_products', 'abn_translating_products_callback');

function abn_translating_products_callback()
{
    // Ensure POST request
    if (!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] !== 'POST') {
        wp_send_json_error(['message' => 'Invalid request method'], 405);
    }

    // Verify nonce securely
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'start_translating_products_nonce')) {
        wp_send_json_error(['message' => 'Invalid request'], 403);
    }

    // Sanitize inputs
    $updated_option = isset($_POST['updated_option']) ? sanitize_text_field(wp_unslash($_POST['updated_option'])) : '';
    $selected_language = isset($_POST['selected_language']) ? sanitize_text_field(wp_unslash($_POST['selected_language'])) : '';
    $current_page = isset($_POST['current_page']) ? intval($_POST['current_page']) : 1;
    $user_ID = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;

    // Generate a new nonce for the REST API request
    $rest_nonce = wp_create_nonce('wp_rest');

    // Construct API URL securely
    $url = add_query_arg([
        'page_number'     => $current_page,
        'target_language' => $selected_language,
        'user_id'         => $user_ID,
        'nonce'           => $rest_nonce
    ], get_home_url() . '/wp-json/myplugin/v1/update_translatepress_plugin');

    // API Request Arguments
    $args = [
        'headers' => [
            'Content-Type' => 'application/json',
        ],
        'method' => 'GET',
    ];

    // Make GET Request
    $request = wp_remote_get($url, $args);

    // Handle API Errors
    if (is_wp_error($request)) {
        wp_send_json_error(['message' => 'Failed to fetch data', 'error' => $request->get_error_message()], 500);
    }

    $apiBody = json_decode(wp_remote_retrieve_body($request), true);

    // Validate Response
    if (empty($apiBody)) {
        wp_send_json_error(['message' => 'Invalid API response'], 500);
    }

    wp_send_json_success($apiBody);
}

add_action('rest_api_init', function () {
    register_rest_route('myplugin/v1', '/update_translatepress_plugin', array(
        'methods' => 'GET',
        'callback' => 'abn_translatepress_plugin_callback',
        'permission_callback' => function () {
            return current_user_can('manage_options');
        },
    ));
});

function abn_translatepress_plugin_callback(WP_REST_Request $request)
{
    // Verify the nonce
    $nonce = $request->get_param('nonce');
    if (!$nonce || !wp_verify_nonce($nonce, 'wp_rest')) {
        return new WP_REST_Response(['message' => 'Invalid nonce'], 403);
    }

    // Validate & Sanitize GET parameters
    $target_language = isset($_GET['target_language']) ? sanitize_text_field(wp_unslash($_GET['target_language'])) : '';
    $page_number = isset($_GET['page_number']) ? intval($_GET['page_number']) : 1;
    $user_id = isset($_GET['user_id']) ? intval($_GET['user_id']) : 0;

    // Return error if required parameters are missing
    if (empty($target_language) || empty($user_id)) {
        return wp_send_json_error(['message' => 'Missing required parameters'], 400);
    }

    // Check if user exists and is an administrator
    $user = get_userdata($user_id);
    if (!$user || !in_array('administrator', (array) $user->roles)) {
        return wp_send_json_error(['message' => 'Unauthorized access'], 403);
    }

    // Ensure TranslatePress is active
    if (!class_exists('TRP_Translate_Press')) {
        return wp_send_json_error(['message' => 'TranslatePress plugin is not active'], 500);
    }

    $trp = TRP_Translate_Press::get_trp_instance();
    $url_converter = $trp->get_component('url_converter');

    // Query products
    $args = [
        'post_type'      => 'product',
        'posts_per_page' => 50,
        'paged'          => $page_number,
        'post_status'    => ['publish', 'pending', 'draft', 'private'],
    ];

    $query = new WP_Query($args);
    $permalink_array = [];

    if ($query->have_posts()) {
        while ($query->have_posts()) {
            $query->the_post();
            $product_id = get_the_ID();
            $product = wc_get_product($product_id);
            if (!$product) continue;

            $permalink = get_permalink($product_id);
            if ($permalink) {
                $target_url = $url_converter->get_url_for_language($target_language, $permalink, '');
                update_post_meta($product_id, $target_language, $target_url);
                $permalink_array[$product_id] = ['url' => esc_url($target_url)];
            }

            // Process variations for variable products
            if ($product->is_type('variable')) {
                foreach ($product->get_children() as $variation_id) {
                    $variation_permalink = get_permalink($variation_id);
                    if ($variation_permalink) {
                        $variation_url = $url_converter->get_url_for_language($target_language, $variation_permalink, '');
                        update_post_meta($variation_id, $target_language, $variation_url);
                        $permalink_array[$variation_id] = ['url' => esc_url($variation_url)];
                    }
                }
            }
        }
        wp_reset_postdata();
    } else {
        return wp_send_json_error(['message' => 'No products found'], 404);
    }

    return wp_send_json_success(['translated_urls' => $permalink_array]);
}



add_action('wp_ajax_generate_csv_file', 'abn_csv_file_callback');
add_action('wp_ajax_nopriv_generate_csv_file', 'abn_csv_file_callback');

function abn_csv_file_callback()
{
    // Ensure nonce is set and verify it properly
    if (!isset($_GET['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['nonce'])), 'generate_csv_nonce')) {
        wp_send_json_error(["error" => "Invalid request."], 403);
    }

    // Ensure user has appropriate permissions
    if (!current_user_can('manage_options')) {
        wp_send_json_error(["error" => "Unauthorized request."], 403);
    }

    // Validate & sanitize GET parameters
    $page_number = isset($_GET['page_number']) ? intval($_GET['page_number']) : 1;
    if ($page_number < 1) {
        wp_send_json_error(["error" => "Invalid page number."], 400);
    }

    // Unsanitize and sanitize filename properly
    $filename = isset($_GET['file_name']) ? sanitize_file_name(wp_unslash($_GET['file_name'])) : 'export';
    $filename = preg_replace('/[^a-zA-Z0-9_-]/', '', $filename); // Ensure a safe filename

    $upload_dir = wp_upload_dir();
    if (empty($upload_dir['basedir']) || empty($upload_dir['baseurl'])) {
        wp_send_json_error(["error" => "Unable to determine upload directory."], 500);
    }

    $export_folder = trailingslashit($upload_dir['basedir']) . 'product_exports/';
    $file_url = trailingslashit($upload_dir['baseurl']) . "product_exports/products_{$filename}.csv";
    $filepath = $export_folder . "products_{$filename}.csv";

    // Load WordPress file system
    global $wp_filesystem;
    if (empty($wp_filesystem)) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem();
    }

    // Ensure the directory exists using wp_mkdir_p()
    if (!wp_mkdir_p($export_folder)) {
        wp_send_json_error(["error" => "Failed to create export directory."], 500);
    }

    // Ensure index.php file exists for security
    $index_path = $export_folder . 'index.php';
    if (!$wp_filesystem->exists($index_path)) {
        $wp_filesystem->put_contents($index_path, "<?php // Silence is golden.", FS_CHMOD_FILE);
    }

    // Fetch existing CSV data if appending
    $csv_content = '';
    if ($wp_filesystem->exists($filepath) && $page_number > 1) {
        $csv_content = $wp_filesystem->get_contents($filepath);
    }

    // Add UTF-8 BOM and header if first page
    if ($page_number == 1) {
        $csv_content .= "\xEF\xBB\xBF"; // UTF-8 BOM
        $csv_content .= implode(',', ['ID', 'Title', 'Product Type', 'SKU', 'Product URL', 'Regular Price', 'Sale Price', 'EAN', 'Delivery Time', 'Image URLs', 'Description', 'Brand Name', 'Brand Slug', 'Category', 'de_DE', 'en_US', 'es_ES', 'fr_FR']) . "\n";
    }

    // Fetch product data
    $query = new WP_Query([
        'post_type'      => 'product',
        'posts_per_page' => 50,
        'paged'          => $page_number,
        'post_status'    => 'publish',
    ]);

    while ($query->have_posts()) {
        $query->the_post();
        $product = wc_get_product(get_the_ID());

        $csv_content .= implode(',', [
            get_the_ID(),
            sanitize_text_field(get_the_title()),
            sanitize_text_field($product->get_type()),
            sanitize_text_field($product->get_sku()),
            esc_url(get_permalink()),
            floatval(get_post_meta(get_the_ID(), '_regular_price', true)),
            floatval($product->get_sale_price()),
            sanitize_text_field(get_post_meta(get_the_ID(), '_ts_gtin', true)),
            sanitize_text_field(get_post_meta(get_the_ID(), '_default_delivery_time', true)),
            '',
            sanitize_textarea_field(get_post_field('post_content', get_the_ID())),
            '',
            '',
            '',
            sanitize_text_field(get_post_meta(get_the_ID(), 'de_DE', true)),
            sanitize_text_field(get_post_meta(get_the_ID(), 'en_US', true)),
            sanitize_text_field(get_post_meta(get_the_ID(), 'es_ES', true)),
            sanitize_text_field(get_post_meta(get_the_ID(), 'fr_FR', true))
        ]) . "\n";
    }

    // Save CSV content using WP_Filesystem
    if (!$wp_filesystem->put_contents($filepath, $csv_content, FS_CHMOD_FILE)) {
        wp_send_json_error(["error" => "Failed to save file."], 500);
    }

    wp_send_json_success(['file_url' => esc_url($file_url)]);
}

?>