<?php

namespace Upsellar\Interaction;

use Upsellar\Admin\UpsellarSettingsManager;
use Upsellar\Utilities\UpsellarCookieUtilities;

// Check if WordPress environment is loaded
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class UpsellarInteractionTrackingController {

    public function __construct() {
        // Handle AJAX requests from JavaScript
        add_action('wp_ajax_upsellar_track_product_interaction_js', array($this, 'upsellar_track_product_interaction_js'));
        add_action('wp_ajax_nopriv_upsellar_track_product_interaction_js', array($this, 'upsellar_track_product_interaction_js'));

        // Handle direct PHP calls
        add_action('track_product_interaction_php', array($this, 'upsellar_handle_php_interaction'));

    }

    function upsellar_is_valid_interaction_type($type) {
        // Define an array of valid interaction types
        $valid_types = array('show', 'time_shown', 'view', 'time_viewed', 'click', 'impression', 'add_to_cart', 'purchase');
    
        // Check if the provided type is in the array of valid types
        return in_array($type, $valid_types, true);
    }

    public function upsellar_track_product_interaction_js() {
        check_ajax_referer('upsellar_track_product_interaction_nonce', 'nonce');
        
        // Sanitize and validate interaction data
        $actionType = isset($_POST['actionType']) ? sanitize_text_field(wp_unslash($_POST['actionType'])) : '';
        if (!$this->upsellar_is_valid_interaction_type($actionType)) {
            wp_send_json_error('Invalid interaction type');
            return;
        }

        $productId = isset($_POST['productId']) ? absint($_POST['productId']) : 0;
        if ($productId <= 0) {
            wp_send_json_error('Invalid product ID');
            return;
        }
    
        $parameter = isset($_POST['parameter']) ? sanitize_text_field(wp_unslash($_POST['parameter'])) : null;

        $timestamp = isset($_POST['timestamp']) ? sanitize_text_field(wp_unslash($_POST['timestamp'])) : '';
        if (empty($timestamp)) {
            wp_send_json_error('Invalid or missing timestamp.');
            return;
        }

        $interactionData = [
            'actionType' => $actionType,
            'productId' => $productId,
            'timestamp' => $timestamp,
        ];   
        
        // Include parameter only if it was provided
        if ($parameter !== null) {
            $interactionData['parameter'] = $parameter;
        }

        // Append user information
        $interactionData = $this->upsellar_append_user_information($interactionData);
    
        // Validate user interaction
        if (!$this->upsellar_validate_user_interaction($interactionData)) {
            wp_send_json_error('Invalid interaction type');
            return;
        }
    
        // Validate and append product information
        $interactionData = $this->upsellar_validate_and_append_product_information($interactionData);
        if (!$interactionData) {
            wp_send_json_error('Invalid interaction data');
            return; // Error response already sent in the function
        }

        if(!$this->upsellar_should_track_user()){
            return;
        }
    
        // Send data to main server
        $response = $this->upsellar_send_data_to_main_server($interactionData);

        if($response){

            return true;
        }
    
        wp_die();
    }
    
    private function upsellar_append_user_information($data) {

        $cookiesUtilties = new UpsellarCookieUtilities();
        $user_identifier = $cookiesUtilties->upsellar_get_user_identifier();
        $user_type = $cookiesUtilties->upsellar_get_user_type();

        $data['userIdentifier'] = $user_identifier;
        $data['userType'] = $user_type;
    
        return $data;
    }

    private function upsellar_should_track_user() {
        if (!is_user_logged_in()) {
            return true; // Always track guests
        }
    
        $user = wp_get_current_user();
        // List roles that should not be tracked
        $excluded_roles = ['administrator', 'editor', 'author', 'contributor', 'moderator'];
        
        // Check if user has any excluded roles
        foreach ($user->roles as $role) {
            if (in_array($role, $excluded_roles)) {
                return false; // Do not track if user has an excluded role
            }
        }
    
        return true; // Track if user doesn't have an excluded role
    }
    
    private function upsellar_validate_user_interaction($data) {
        return $this->upsellar_is_valid_interaction_type($data['actionType']);
    }
    
    private function upsellar_validate_and_append_product_information($data) {
        if (isset($data['productId']) && !empty($data['productId'])) {
            $product = wc_get_product($data['productId']);
            if (!$product) {
                wp_send_json_error('Invalid product ID');
                return false;
            }
    
            $categories = wp_get_post_terms($data['productId'], 'product_cat', array('fields' => 'ids'));
            if (!is_wp_error($categories)) {
                $data['categoryIds'] = $categories;
            } else {
                $data['categoryIds'] = [];
            }
        } else {
            wp_send_json_error('Product ID is missing');
            return false;
        }
        return $data;
    }

    public function upsellar_handle_php_interaction($interactionData, $custom_url = null) {
        
        $interactionData = $this->upsellar_append_user_information($interactionData);

        // Validate user interaction
        if (!$this->upsellar_validate_user_interaction($interactionData)) {
            wp_send_json_error('Invalid interaction type');
            return;
        }

        if(!$this->upsellar_should_track_user()){
            return;
        }

        $response = $this->upsellar_send_data_to_main_server($interactionData);


        return $response;
    }

    public function upsellar_handle_data_import($interactionData, $custom_url) {

        $response = $this->upsellar_send_data_to_main_server($interactionData, $custom_url);

        return $response;
    }

    /**
     * Send interation data to the Upsellar server.
     * 
     * See README.TXT for the API documentation.
     * 
     * @param array $data  The data being sent to the Upsellar server.
     * @param string $custom_url  Send the data to this URL.
     * @return array  Return the status of request.
     * @since 1.0.0
     */ 
    private function upsellar_send_data_to_main_server($data, $custom_url = null) {
        $settingsManager = new UpsellarSettingsManager();
    
        $defaultURL = 'https://upsellar.com/api/v1/track-interaction';
    
        // Determine the URL to use
        $url = $custom_url ?: $defaultURL;
        $apiKey = $settingsManager->upsellar_get_api_key();
        
        $response = wp_remote_post($url, array(
            'headers' => array(
                'Content-Type' => 'application/json; charset=utf-8',
                'Authorization' => 'Bearer ' . $apiKey,
                'accept' => 'application/json'
            ),
            'body' => wp_json_encode($data)
        ));
    
        // Check for WP_Error or retrieve the response body
        if (is_wp_error($response)) {
            // Log the error
            return array('error' => $response->get_error_message());
        } else {
            $body = wp_remote_retrieve_body($response);
            // Log the response
            return json_decode($body, true);
        }
    } 

    
}
