<?php 

namespace Upsellar\Engine\Connection;

use Upsellar\Admin\UpsellarSettingsManager;
use Upsellar\Utilities\UpsellarCookieUtilities;

defined('ABSPATH') || exit;

// Responsible for fetching product data
class UpsellarProductFetcher {

    private $cookie_utility;

    public function __construct() {
        $this->cookie_utility = new UpsellarCookieUtilities();
    }

    /**
     * Fetch product ids from Upsellar API.
     * 
     * See README.TXT for the API documentation.
     * 
     * @param int|array $currentProductId The current product id or array of product ids to get recommendations for.
     * @param int $limit The number of product ids to fetch.
     * @return array The product ids fetched from Upsellar API.
     * @since 1.0.0
     */
    public function upsellar_get_product_ids($currentProductId, $limit) {

        $settingsManager = new UpsellarSettingsManager();

        $api_url = 'https://upsellar.com/api/v1/get-recommendations';

        $apiKey = $settingsManager->upsellar_get_api_key(); 

        $user_id = $this->cookie_utility->upsellar_get_user_identifier();
        $user_type = $this->cookie_utility->upsellar_get_user_type();

        // Prepare the payload with additional details
        $payload = array(
            'user_id' => $user_id,
            'user_type' => $user_type,
            'current_product_id' => isset($currentProductId) ? $currentProductId : null,
            'limit' => $limit,
            //'recommendationType' => isset($attributes['recommendationType']) ? $attributes['recommendationType'] : 'default'
        );

        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Content-Type' => 'application/json; charset=utf-8', 
                'Authorization' => 'Bearer ' . $apiKey,
                'accept' => 'application/json'
            ),
            'body' => wp_json_encode($payload)
        ));

        if (is_wp_error($response)) {
            // Handle error accordingly
            return array();
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        return isset($data['products']) ? $data['products'] : array();
    }

    /**
     * Get the product ids from cart.
     * 
     * @return array of product ids in the cart.
     * @since 1.0.0
     */
    public function upsellar_get_cart_product_ids() {
        $product_ids = array();

        if (class_exists('WC_Cart') && WC()->cart) {
            foreach (WC()->cart->get_cart() as $cart_item) {

                $product = $cart_item['data'];
                
                $product_ids[] = $product->get_id();

            }
        }


        return $product_ids;
    }

    /**
     * Get the current product id.
     * 
     * @return int|null the product id or null if not in a single product view.
     * @since 1.0.0
     */
    public function upsellar_get_current_product_id() {
        global $product;
        return $product ? $product->get_id() : get_the_ID();
    }

    /**
     * Explodes the product id string into an array.
     * 
     * @param String $currentProductId the product id string.
     * @return Array of product ids.
     * @since 1.0.0
     */
    // TODO: This function is not used anywhere. Remove.
    public function upsellar_explode_product_array(String $currentProductId){
        if (strpos($currentProductId, ',') !== false) {
            $productIdsArray = explode(',', $currentProductId);
        } else {
            // If it's a single number, make it an array
            $productIdsArray = [$currentProductId];
        }

        return $productIdsArray;
    }

}

